

#ifndef _QCONFIG_H_
#define _QCONFIG_H_

/**@name QConfig - Class to read and write conf files easily in qt apps.
 *
 * <b>Copyright</b><br>
 * This library is free software; you can redistribute it and/or      
 * modify it under the terms of the GNU Library General Public        
 * License as published by the Free Software Foundation; either       
 * version 2 of the License, or (at your option) any later version.   
 * <p>                                                                   
 * This library is distributed in the hope that it will be useful,    
 * but WITHOUT ANY WARRANTY; without even the implied warranty of     
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  
 * Library General Public License for more details.                   
 * <p>                                                                   
 * You should have received a copy of the GNU Library General Public  
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @author Ricardo Villalba &lt;rvm@linuxfan.com&gt;
 */
//@{ 

#include <SimpleConfig.h>

#ifndef NO_GRAPHICS_AT_ALL
#include <qsize.h>
#include <qpoint.h>
#include <qfont.h>
#include <qcolor.h>
#endif

#include <qstring.h>

class SimpleConfig;

// ----------------------------------------------------------------------------
/**
  QConfig derives from SimpleConfig and adds methods for reading and
  writing some Qt objects.
  <p>
  <b>Descripcin en espaol</b>
  <p>
  QConfig permite leer y escribir objetos Qt. Actualmente slo
  soporta QSize y QPoint.
  He intentado que el API sea similar al del KConfig del KDE.
  <p>
  QConfig deriva de SimpleConfig, por tanto todo lo que se explica en
  la documentacin de dicha clase, se aplica aqu tambin.

  <p>
  Ejemplo:

  <pre>
  QConfig conf("testqconfig.conf");

  conf.setGroup("General");
  conf.writeSizeEntry("Size", QSize(10,10) );
  conf.writeNumEntry("Int", 10 );
  conf.writeBoolEntry("Bool", true );
  conf.writeEntry("String", "Hola" );

  conf.setGroup("Another_group");
  conf.writeBoolEntry("Check_list", false );
  conf.writePointEntry("Point", QPoint(5,200) );

  conf.sync();

  </pre>

  <p>
  Este ejemplo escribir en el fichero "testqconfig.conf" en el directorio
  actual, este contenido:

  <pre>
  [General]
  Size_width = 10
  Size_height = 10
  Int = 10
  Bool = true
  String = Hola
  [Another_group]
  Check_list = false
  Point_x = 5
  Point_y = 200
  </pre>

  <p>
  Luego el contenido de este fichero puede leerse as de fcil:

  <pre>
  conf.setGroup("General");
  QSize s( conf.readSizeEntry("Size", QSize(0,0) ) );

  conf.setGroup("Another_group");
  QPoint p( conf.readPointEntry("Point", QPoint(0,0) ) );
  </pre>

  <p>
  <b>Description in English</b>
  <p>
  QConfig allows you to read and write some Qt objects. At the moment
  it only supports QSize y QPoint.
  I tried that the API be similar to KDE's KConfig.
  <p>
  QConfig derives from SimpleConfig, so all explain for that class
  applies here too.

  <p>
  Example:

  <pre>
  QConfig conf("testqconfig.conf");

  conf.setGroup("General");
  conf.writeSizeEntry("Size", QSize(10,10) );
  conf.writeNumEntry("Int", 10 );
  conf.writeBoolEntry("Bool", true );
  conf.writeEntry("String", "Hola" );

  conf.setGroup("Another_group");
  conf.writeBoolEntry("Check_list", false );
  conf.writePointEntry("Point", QPoint(5,200) );

  conf.sync();

  </pre>

  <p>
  This example will write in the file "testqconfig.conf" in the current
  directory, these lines:

  <pre>
  [General]
  Size_width = 10
  Size_height = 10
  Int = 10
  Bool = true
  String = Hola
  [Another_group]
  Check_list = false
  Point_x = 5
  Point_y = 200
  </pre>

  <p>
  Then the content of that file can be read this way:

  <pre>
  conf.setGroup("General");
  QSize s( conf.readSizeEntry("Size", QSize(0,0) ) );

  conf.setGroup("Another_group");
  QPoint p( conf.readPointEntry("Point", QPoint(0,0) ) );
  </pre>
*/
// ----------------------------------------------------------------------------
 
class QConfig : public SimpleConfig
{

public:
	/** @name Constructor and destructor */
		//@{
        /**
         Constructor
		 ////@param fileName Nombre del fichero a usar como fichero de
         //// configuracin
         @param fileName Name of the file to be used as configuration file
        */
	QConfig(const char *fileName);
	//QConfig(const char *fileName, bool useHomeDir=FALSE);
		/**
		 Destructor
		*/
	~QConfig();
	//@}

	/**
	 ////Devuelve un nombre de fichero que puede ser
     ////usado como fichero de configuracin. Mtodo esttico. 
	 ////<p>
	 ////Ejemplo:
	 ////<pre>const char *nombre=QConfig::configName("test.conf");</pre>
	 ////<p>devolvera /home/ricardo/.test.conf
	 ////@param name Nombre simple de fichero
     ////@return El directorio $HOME+"/."+name
	
	 Returns a file name that can be used for a configuration file. 
     Static method. 
	 <p>
	 Example:
	 <pre>const char *nombre=QConfig::configName("test.conf");</pre>
	 <p>would return /home/ricardo/.test.conf
	 @param name Simple file name
     @return $HOME+"/."+name
	*/
	static QString configName(const char *name);

#ifndef NO_GRAPHICS_AT_ALL
	/**
     ////Guarda una entrada con los datos del objeto QSize especificado
	 ////@param key La clave
	 ////@param size Objeto QSize a guardar
	 ////@return siempre TRUE

     Stores an entry with data of the specified object QSize
	 @param key The key
	 @param size QSize object to store
	 @return always TRUE
	*/
	bool writeSizeEntry(const char *key, QSize size);

	/**
     ////Guarda una entrada con los datos del objeto QPoint especificado
	 ////@param key La clave
	 ////@param point Objeto QPoint a guardar
	 ////@return siempre TRUE

     Stores an entry with data of the specified object QPoint
	 @param key The key
	 @param point QPoint object to store
	 @return always TRUE
	*/
	bool writePointEntry(const char *key, QPoint point);

	bool writeFontEntry(const char *key, QFont font);

	bool writeColorEntry(const char *key, QColor color);

	/**
	 ////Recupera los datos de un objeto QSize guardado previamente
	 ////@param key La clave
	 ////@param defaultSize Dato por defecto
	 ////@return El valor de la clave o defaultSize si no se encontr

	 Retrieves data of an object QSize previously stored
	 @param key The key
	 @param defaultSize Default value
	 @return The value of the key or defaultSize if not found
	*/
	QSize readSizeEntry(const char *key, QSize defaultSize) const;

	/**
	 ////Recupera los datos de un objeto QPoint guardado previamente
	 ////@param key La clave
	 ////@param defaultPoint Dato por defecto
	 ////@return El valor de la clave o defaultPoint si no se encontr

	 Retrieves data of an object QPoint previously stored
	 @param key The key
	 @param defaultPoint Default value
	 @return The value of the key or defaultPoint if not found
	*/
	QPoint readPointEntry(const char *key, QPoint defaultPoint) const;

	QFont readFontEntry(const char *key, QFont defaultFont) const;

	QColor readColorEntry(const char *key, QColor defaultColor) const;

#endif
};

//@} 

#endif

