% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/uwot.R
\name{lvish}
\alias{lvish}
\title{Dimensionality Reduction with a LargeVis-like method}
\usage{
lvish(
  X,
  perplexity = 50,
  n_neighbors = perplexity * 3,
  n_components = 2,
  metric = "euclidean",
  n_epochs = -1,
  learning_rate = 1,
  scale = "maxabs",
  init = "lvrandom",
  init_sdev = NULL,
  repulsion_strength = 7,
  negative_sample_rate = 5,
  nn_method = NULL,
  n_trees = 50,
  search_k = 2 * n_neighbors * n_trees,
  n_threads = NULL,
  n_sgd_threads = 0,
  grain_size = 1,
  kernel = "gauss",
  pca = NULL,
  pca_center = TRUE,
  pcg_rand = TRUE,
  fast_sgd = FALSE,
  ret_nn = FALSE,
  ret_extra = c(),
  tmpdir = tempdir(),
  verbose = getOption("verbose", TRUE)
)
}
\arguments{
\item{X}{Input data. Can be a \code{\link{data.frame}}, \code{\link{matrix}},
\code{\link[stats]{dist}} object or \code{\link[Matrix]{sparseMatrix}}.
Matrix and data frames should contain one observation per row. Data frames
will have any non-numeric columns removed, although factor columns will be
used if explicitly included via \code{metric} (see the help for
\code{metric} for details). A sparse matrix is interpreted as a distance
matrix, and is assumed to be symmetric, so you can also pass in an
explicitly upper or lower triangular sparse matrix to save storage. There
must be at least \code{n_neighbors} non-zero distances for each row. Both
implicit and explicit zero entries are ignored. Set zero distances you want
to keep to an arbitrarily small non-zero value (e.g. \code{1e-10}).
\code{X} can also be \code{NULL} if pre-computed nearest neighbor data is
passed to \code{nn_method}, and \code{init} is not \code{"spca"} or
\code{"pca"}.}

\item{perplexity}{Controls the size of the local neighborhood used for
manifold approximation. This is the analogous to \code{n_neighbors} in
\code{\link{umap}}. Change this, rather than \code{n_neighbors}.}

\item{n_neighbors}{The number of neighbors to use when calculating the
\code{perplexity}. Usually set to three times the value of the
\code{perplexity}. Must be at least as large as \code{perplexity}.}

\item{n_components}{The dimension of the space to embed into. This defaults
to \code{2} to provide easy visualization, but can reasonably be set to any
integer value in the range \code{2} to \code{100}.}

\item{metric}{Type of distance metric to use to find nearest neighbors. One
  of:
\itemize{
  \item \code{"euclidean"} (the default)
  \item \code{"cosine"}
  \item \code{"manhattan"}
  \item \code{"hamming"}
  \item \code{"correlation"} (a distance based on the Pearson correlation)
  \item \code{"categorical"} (see below)
}
Only applies if \code{nn_method = "annoy"} (for \code{nn_method = "fnn"}, the
distance metric is always "euclidean").

If \code{X} is a data frame or matrix, then multiple metrics can be
specified, by passing a list to this argument, where the name of each item in
the list is one of the metric names above. The value of each list item should
be a vector giving the names or integer ids of the columns to be included in
a calculation, e.g. \code{metric = list(euclidean = 1:4, manhattan = 5:10)}.

Each metric calculation results in a separate fuzzy simplicial set, which are
intersected together to produce the final set. Metric names can be repeated.
Because non-numeric columns are removed from the data frame, it is safer to
use column names than integer ids.

Factor columns can also be used by specifying the metric name
\code{"categorical"}. Factor columns are treated different from numeric
columns and although multiple factor columns can be specified in a vector,
each factor column specified is processed individually. If you specify
a non-factor column, it will be coerced to a factor.

For a given data block, you may override the \code{pca} and \code{pca_center}
arguments for that block, by providing a list with one unnamed item
containing the column names or ids, and then any of the \code{pca} or
\code{pca_center} overrides as named items, e.g. \code{metric =
list(euclidean = 1:4, manhattan = list(5:10, pca_center = FALSE))}. This
exists to allow mixed binary and real-valued data to be included and to have
PCA applied to both, but with centering applied only to the real-valued data
(it is typical not to apply centering to binary data before PCA is applied).}

\item{n_epochs}{Number of epochs to use during the optimization of the
embedded coordinates. The default is calculate the number of epochs
dynamically based on dataset size, to give the same number of edge samples
as the LargeVis defaults. This is usually substantially larger than the
UMAP defaults. If \code{n_epochs = 0}, then coordinates determined by
\code{"init"} will be returned.}

\item{learning_rate}{Initial learning rate used in optimization of the
coordinates.}

\item{scale}{Scaling to apply to \code{X} if it is a data frame or matrix:
\itemize{
  \item{\code{"none"} or \code{FALSE} or \code{NULL}} No scaling.
  \item{\code{"Z"} or \code{"scale"} or \code{TRUE}} Scale each column to
  zero mean and variance 1.
  \item{\code{"maxabs"}} Center each column to mean 0, then divide each
  element by the maximum absolute value over the entire matrix.
  \item{\code{"range"}} Range scale the entire matrix, so the smallest
  element is 0 and the largest is 1.
  \item{\code{"colrange"}} Scale each column in the range (0,1).
}
For lvish, the default is \code{"maxabs"}, for consistency with LargeVis.}

\item{init}{Type of initialization for the coordinates. Options are:
 \itemize{
   \item \code{"spectral"} Spectral embedding using the normalized Laplacian
   of the fuzzy 1-skeleton, with Gaussian noise added.
   \item \code{"normlaplacian"}. Spectral embedding using the normalized
   Laplacian of the fuzzy 1-skeleton, without noise.
   \item \code{"random"}. Coordinates assigned using a uniform random
   distribution between -10 and 10.
   \item \code{"lvrandom"}. Coordinates assigned using a Gaussian
   distribution with standard deviation 1e-4, as used in LargeVis
   (Tang et al., 2016) and t-SNE.
   \item \code{"laplacian"}. Spectral embedding using the Laplacian Eigenmap
   (Belkin and Niyogi, 2002).
   \item \code{"pca"}. The first two principal components from PCA of
   \code{X} if \code{X} is a data frame, and from a 2-dimensional classical
   MDS if \code{X} is of class \code{"dist"}.
   \item \code{"spca"}. Like \code{"pca"}, but each dimension is then scaled
   so the standard deviation is 1e-4, to give a distribution similar to that
   used in t-SNE and LargeVis. This is an alias for \code{init = "pca",
   init_sdev = 1e-4}.
   \item \code{"agspectral"} An "approximate global" modification of
   \code{"spectral"} which all edges in the graph to a value of 1, and then
   sets a random number of edges (\code{negative_sample_rate} edges per
   vertex) to 0.1, to approximate the effect of non-local affinities.
   \item A matrix of initial coordinates.
 }
For spectral initializations, (\code{"spectral"}, \code{"normlaplacian"},
\code{"laplacian"}), if more than one connected component is identified,
each connected component is initialized separately and the results are
merged. If \code{verbose = TRUE} the number of connected components are
logged to the console. The existence of multiple connected components
implies that a global view of the data cannot be attained with this
initialization. Either a PCA-based initialization or increasing the value of
\code{n_neighbors} may be more appropriate.}

\item{init_sdev}{If non-\code{NULL}, scales each dimension of the initialized
coordinates (including any user-supplied matrix) to this standard
deviation. By default no scaling is carried out, except when \code{init =
"spca"}, in which case the value is \code{0.0001}. Scaling the input may
help if the unscaled versions result in initial coordinates with large
inter-point distances or outliers. This usually results in small gradients
during optimization and very little progress being made to the layout.
Shrinking the initial embedding by rescaling can help under these
circumstances. Scaling the result of \code{init = "pca"} is usually
recommended and \code{init = "spca"} as an alias for \code{init = "pca",
init_sdev = 1e-4} but for the spectral initializations the scaled versions
usually aren't necessary unless you are using a large value of
\code{n_neighbors} (e.g. \code{n_neighbors = 150} or higher).}

\item{repulsion_strength}{Weighting applied to negative samples in low
dimensional embedding optimization. Values higher than one will result in
greater weight being given to negative samples.}

\item{negative_sample_rate}{The number of negative edge/1-simplex samples to
use per positive edge/1-simplex sample in optimizing the low dimensional
embedding.}

\item{nn_method}{Method for finding nearest neighbors. Options are:
\itemize{
  \item \code{"fnn"}. Use exact nearest neighbors via the
    \href{https://cran.r-project.org/package=FNN}{FNN} package.
  \item \code{"annoy"} Use approximate nearest neighbors via the
    \href{https://cran.r-project.org/package=RcppAnnoy}{RcppAnnoy} package.
 }
By default, if \code{X} has less than 4,096 vertices, the exact nearest
neighbors are found. Otherwise, approximate nearest neighbors are used.
You may also pass precalculated nearest neighbor data to this argument. It
must be a list consisting of two elements:
\itemize{
  \item \code{"idx"}. A \code{n_vertices x n_neighbors} matrix
  containing the integer indexes of the nearest neighbors in \code{X}. Each
  vertex is considered to be its own nearest neighbor, i.e.
  \code{idx[, 1] == 1:n_vertices}.
  \item \code{"dist"}. A \code{n_vertices x n_neighbors} matrix
  containing the distances of the nearest neighbors.
}
Multiple nearest neighbor data (e.g. from two different precomputed
metrics) can be passed by passing a list containing the nearest neighbor
data lists as items.
The \code{n_neighbors} parameter is ignored when using precomputed
nearest neighbor data.}

\item{n_trees}{Number of trees to build when constructing the nearest
neighbor index. The more trees specified, the larger the index, but the
better the results. With \code{search_k}, determines the accuracy of the
Annoy nearest neighbor search. Only used if the \code{nn_method} is
\code{"annoy"}. Sensible values are between \code{10} to \code{100}.}

\item{search_k}{Number of nodes to search during the neighbor retrieval. The
larger k, the more the accurate results, but the longer the search takes.
With \code{n_trees}, determines the accuracy of the Annoy nearest neighbor
search. Only used if the \code{nn_method} is \code{"annoy"}.}

\item{n_threads}{Number of threads to use (except during stochastic gradient
descent). Default is half the number of concurrent threads supported by the
system. For nearest neighbor search, only applies if 
\code{nn_method = "annoy"}. If \code{n_threads > 1}, then the Annoy index 
will be temporarily written to disk in the location determined by
\code{\link[base]{tempfile}}.}

\item{n_sgd_threads}{Number of threads to use during stochastic gradient
descent. If set to > 1, then results will not be reproducible, even if
`set.seed` is called with a fixed seed before running. Set to
\code{"auto"} go use the same value as \code{n_threads}.}

\item{grain_size}{The minimum amount of work to do on each thread. If this
value is set high enough, then less than \code{n_threads} or
\code{n_sgd_threads} will be used for processing, which might give a
performance improvement if the overhead of thread management and context
switching was outweighing the improvement due to concurrent processing.
This should be left at default (\code{1}) and work will be spread evenly
over all the threads specified.}

\item{kernel}{Type of kernel function to create input probabilities. Can be
one of \code{"gauss"} (the default) or \code{"knn"}. \code{"gauss"} uses
the usual Gaussian weighted similarities. \code{"knn"} assigns equal
probabilities to every edge in the nearest neighbor graph, and zero
otherwise, using \code{perplexity} nearest neighbors. The \code{n_neighbors}
parameter is ignored in this case.}

\item{pca}{If set to a positive integer value, reduce data to this number of
columns using PCA. Doesn't applied if the distance \code{metric} is
\code{"hamming"}, or the dimensions of the data is larger than the
number specified (i.e. number of rows and columns must be larger than the
value of this parameter). If you have > 100 columns in a data frame or
matrix, reducing the number of columns in this way may substantially
increase the performance of the nearest neighbor search at the cost of a
potential decrease in accuracy. In many t-SNE applications, a value of 50
is recommended, although there's no guarantee that this is appropriate for
all settings.}

\item{pca_center}{If \code{TRUE}, center the columns of \code{X} before
carrying out PCA. For binary data, it's recommended to set this to
\code{FALSE}.}

\item{pcg_rand}{If \code{TRUE}, use the PCG random number generator (O'Neill,
2014) during optimization. Otherwise, use the faster (but probably less
statistically good) Tausworthe "taus88" generator. The default is
\code{TRUE}.}

\item{fast_sgd}{If \code{TRUE}, then the following combination of parameters
is set: \code{pcg_rand = TRUE} and \code{n_sgd_threads = "auto"}. The
default is \code{FALSE}. Setting this to \code{TRUE} will speed up the
stochastic optimization phase, but give a potentially less accurate
embedding, and which will not be exactly reproducible even with a fixed
seed. For visualization, \code{fast_sgd = TRUE} will give perfectly good
results. For more generic dimensionality reduction, it's safer to leave
\code{fast_sgd = FALSE}. If \code{fast_sgd = TRUE}, then user-supplied
values of \code{pcg_rand} and \code{n_sgd_threads}, are ignored.}

\item{ret_nn}{If \code{TRUE}, then in addition to the embedding, also return
nearest neighbor data that can be used as input to \code{nn_method} to
avoid the overhead of repeatedly calculating the nearest neighbors when
manipulating unrelated parameters (e.g. \code{min_dist}, \code{n_epochs},
\code{init}). See the "Value" section for the names of the list items. If
\code{FALSE}, just return the coordinates. Note that the nearest neighbors
could be sensitive to data scaling, so be wary of reusing nearest neighbor
data if modifying the \code{scale} parameter.}

\item{ret_extra}{A vector indicating what extra data to return. May contain
any combination of the following strings:
\itemize{
  \item \code{"nn"} same as setting `ret_nn = TRUE`.
  \item \code{"P"} the high dimensional probability matrix. The graph
  is returned as a sparse symmetric N x N matrix of class
  \link[Matrix]{dgCMatrix-class}, where a non-zero entry (i, j) gives the
  input probability (or similarity or affinity) of the edge connecting
  vertex i and vertex j. Note that the graph is further sparsified by
  removing edges with sufficiently low membership strength that they would
  not be sampled by the probabilistic edge sampling employed for
  optimization and therefore the number of non-zero elements in the matrix
  is dependent on \code{n_epochs}. If you are only interested in the fuzzy
  input graph (e.g. for clustering), setting `n_epochs = 0` will avoid any
  further sparsifying.
}}

\item{tmpdir}{Temporary directory to store nearest neighbor indexes during
nearest neighbor search. Default is \code{\link{tempdir}}. The index is
only written to disk if \code{n_threads > 1} and
\code{nn_method = "annoy"}; otherwise, this parameter is ignored.}

\item{verbose}{If \code{TRUE}, log details to the console.}
}
\value{
A matrix of optimized coordinates, or:
  \itemize{
    \item if \code{ret_nn = TRUE} (or \code{ret_extra} contains \code{"nn"}),
    returns the nearest neighbor data as a list called \code{nn}. This
    contains one list for each \code{metric} calculated, itself containing a
    matrix \code{idx} with the integer ids of the neighbors; and a matrix
    \code{dist} with the distances. The \code{nn} list (or a sub-list) can be
    used as input to the \code{nn_method} parameter.
    \item if \code{ret_extra} contains \code{"P"}, returns the high
    dimensional probability matrix as a sparse matrix called \code{P}, of
    type \link[Matrix]{dgCMatrix-class}.
  }
  The returned list contains the combined data from any combination of
  specifying \code{ret_nn} and \code{ret_extra}.
}
\description{
Carry out dimensionality reduction of a dataset using a method similar to
LargeVis (Tang et al., 2016).
}
\details{
\code{lvish} differs from the official LargeVis implementation in the
following:

\itemize{
  \item Only the nearest-neighbor index search phase is multi-threaded.
  \item Matrix input data is not normalized.
  \item The \code{n_trees} parameter cannot be dynamically chosen based on
  data set size.
  \item Nearest neighbor results are not refined via the
  neighbor-of-my-neighbor method. The \code{search_k} parameter is twice
  as large than default to compensate.
  \item Gradient values are clipped to \code{4.0} rather than \code{5.0}.
  \item Negative edges are generated by uniform sampling of vertexes rather
  than their degree ^ 0.75.
  \item The default number of samples is much reduced. The default number of
  epochs, \code{n_epochs}, is set to \code{5000}, much larger than for
  \code{\link{umap}}, but may need to be increased further depending on your
  dataset. Using \code{init = "spectral"} can help.
}

Note that the \code{grain_size} parameter no longer does anything and is
present to avoid break backwards compatibility only.
}
\examples{
# Default number of epochs is much larger than for UMAP, assumes random
# initialization. Use perplexity rather than n_neighbors to control the size
# of the local neighborhood 20 epochs may be too small for a random 
# initialization
iris_lvish <- lvish(iris,
  perplexity = 50, learning_rate = 0.5,
  init = "random", n_epochs = 20
)
}
\references{
Tang, J., Liu, J., Zhang, M., & Mei, Q. (2016, April).
Visualizing large-scale and high-dimensional data.
In \emph{Proceedings of the 25th International Conference on World Wide Web}
(pp. 287-297).
International World Wide Web Conferences Steering Committee.
\url{https://arxiv.org/abs/1602.00370}
}
