/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_P7B.h: interface for the PKCS7 class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_P7B_H
#define PKI_P7B_H

#include <PKI_ERR.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <HashTable/HashTable_String.h>
#include <PKI_RSA.h>
#include <PKI_CERT.h>
#include <mString.h>

#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/pkcs7.h>

/*!
	This class represents a P7B
*/
class PKI_P7B  
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_P7B();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other pksc7 to copy from.
	 */
	PKI_P7B(const PKI_P7B & other);

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_P7B();

	/*! \brief This function clears the internal datas.
	 *  \param CertsList [IN] Should the list of certificates be cleared as well.
	 */
	void Clear(bool CertsList = true);

	/*! \brief This function adds a certificate to the list.
	 *  \param Cert [IN] The certificate to add.
	 *  \return true on success, false on failure.
	 */
	bool AddCert(X509 * Cert);

	/*! \brief This function returns the P7B in PEM.
	 *  \return The P7B in PEM.
	 */
	const mString & GetPemP7B() const;

	/*! \brief This function generates the P7B.
	 *  \return true on success, false on failure.
	 */
	bool Generate();

	/*! \brief This function loads the PKCS7.
	 *  \param p7 [IN] The PKCS7 to load from.
	 *  \return true on success, false on failure.
	 */
	bool Load(const PKCS7 * p7);
	
	/*! \brief This function loads the PKCS7.
	 *  \param p7pem [IN] The PEM encoded PKCS7.
	 *  \return true on success, false on failure.
	 */
	bool Load(const char * p7pem);

	/*! \brief This function return the number of certificates.
	 *  \return The number of certificates.
	 */
	int GetNumCert() const;

	/*! \brief This function returns a certificate.
	 *  \param index [IN] The index of the certificate.
	 *  \param Duplicate [IN] Should the X509 be duplicated, then freed by calling function.
	 *  \return The certificates.
	 */
	X509 * GetCert(int index, bool Duplicate = false) const;

	/*! \brief This returns the PKCS7.
	 *  \param Duplicate [IN] Should the PKCS7 be duplicated, then freed by calling function.
	 *  \return The PKCS7.
	 */
	PKCS7 * GetPKCS7(bool Duplicate = false) const;

	/*! \brief This function copies the internal pkcs7.
	 *  \param c_p7 [OUT] The pkcs7.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKCS7 ** c_p7) const;

	/*! \brief This function sets the pkcs7.
	 *  \param c_p7 [IN] A pointer to an OpenSSL PKCS7 certificate.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKCS7 * c_p7);

	/*! \brief This operator copies one PKI_P7B into another.
	 *  \param other [IN] The other pkcs7 to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_P7B &other );

	/*! \brief This operator is used to know if the class is set.
	 *  \return return 1 when key is operable and 0 when not.
	 */
	operator int() const;

	/*! \brief This member is an empty instance of this class.
	 */
	static PKI_P7B EmptyInstance;
private:
	bool PKCS7ToString();
	bool StringToPKCS7(const char * p7pem);
	bool LoadDatas();
	STACK_OF(X509) * m_Certs;
	mString m_Pkcs7Pem;
	PKCS7 * m_p7b;
};

#endif
