/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CRL.h: interface for the PKI_CRL class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_CRL_H
#define PKI_CRL_H


#include <stdio.h>
#include <string.h>
#include <Objects.h>



#include <PKI_ERR.h>
#include <PEM_DER.h>

#include <PKI_CERT.h>
#include <PKI_EXT.h>

#include <openssl/x509.h>

#include <vector>
using namespace std;

/*!
	This structure represents a revoked certificate
*/
typedef struct st_REVOCATION_INFO
{
	unsigned long serial;		//!< The serial
	time_t rev_date;			//!< The revocation date
	long reason;			//!< The reason
}REVOCATION_INFO;

/*!
  This class represents a PKI_CRL (Certificate Revocation List)
*/
class PKI_CRL  
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_CRL();
	
	/*! \brief This is the constructor.
	 *  \param other [IN] The other crl to copy from.
	 */
	PKI_CRL(const PKI_CRL & other);

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_CRL();

	/*! \brief This function returns the PKI_CRL in the OpenSSL X509_CRL format.
	 *  \param dup [IN] Should the X509_CRL be duplicated, then freed by calling function.
	 *  \return The CRL in the OpenSSL X509 format.
	 */
	X509_CRL * GetX509_CRL(bool dup = false) const;

	/*! \brief This function generates the CRL.
	 *  \param CaCert [IN] The CA cert and private key to sign the CRL.
	 *  \param md [IN] The name of the hash algo to use.
	 *  \param ValidityLen [IN] The length in hours for the CRL validity.
	 *  \param Exts [IN] The potential extension list.
	 *  \return true on success, false on failure.
	 */
	bool Generate(const PKI_CERT & CaCert, const char * md, unsigned long ValidityLen, const HashTable_String * Exts);
	
	/*! \brief This function loads the CRL from a file.
	 *  \param File [IN] The filename to load the CRL from.
	 *  \return true on success, false on failure.
	 */
	bool LoadFromFile(const mString & File);

	/*! \brief This function set the CRL from PEM.
	 *  \param CrlPem [IN] The CRL in PEM format.
	 *  \return true on success, false on failure.
	 */
	bool SetCRL(const char * CrlPem);

	/*! \brief This function set the CRL.
	 *  \param Crl [IN] The CRL.
	 *  \return true on success, false on failure.
	 */
	bool SetCRL(const X509_CRL * Crl);

	/*! \brief This function set the CRL.
	 *  \param Crl [IN] The CRL.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const X509_CRL * Crl);

	/*! \brief This function copies the internal CRL.
	 *  \param Crl [OUT] The CRL.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(X509_CRL ** Crl) const;

	/*! \brief This function clears the internal datas.
	 */
	void Clear();

	/*! \brief This function returns the CRL in a PEM format.
	 *  \return The CRL in PEM formeat.
	 */
	const mString & GetPemCRL() const;
	
	/*! \brief This function returns the start time of the CRL.
	 *  \return The expiration (valid to) date of the CRL.
	 */
	time_t GetEndTime() const;

	/*! \brief This function returns the end time of the CRL.
	 *  \return The begining (valid from) date of the CRL.
	 */
	time_t GetStartTime() const;

	/*! \brief This function adds an revoked serial to list.
	 *  \param rev_inf [IN] The revocation information.
	 */
	void AddRevokedCert(REVOCATION_INFO * rev_inf);

	/*! \brief This function checks if a certificate is revoked.
	 *  \param cert [IN] The certificate to check.
	 *  \return true when revoked, false when not revoked.
	 */
	bool IsRevoked(const PKI_CERT & cert) const;

	/*! \brief This function returns the version of the certificate.
	 *  \return The version of the CRL.
	 */
	unsigned long GetVersion() const;

	/*! \brief This function returns the DN of the issuer of the certificate.
	 *  \return The DN of the issuer of the CRL.
	 */
	const HashTable_Dn & GetIssuerDN() const;

	/*! \brief This function returns the signature algorithm of the certificate.
	 *  \return The signature algorithm of the CRL.
	 */
	const char * GetSignatureAlg() const;

	/*! \brief This function returns the extensions of the certificate.
	 *  \return The extensions of the CRL.
	 */
	const HashTable_String & GetExtensions() const;	

	/*! \brief This function gets the revocation info.
	 *  \param pos [IN] The index in the list.
	 *  \return The revocation information.
	 */
	const REVOCATION_INFO * GetRevokedCert(size_t pos) const;

	/*! \brief This function returns the number of revoked certificates.
	 *  \return The number of revocation information.
	 */
	size_t RevokedCertsCount() const;

	/*! \brief This function return 1 when CRL is operable and 0 when not.
	 *  \return return 1 when CRL is operable and 0 when not.
	 */
	operator int() const;

	/*! \brief This operator copies one PKI_CRL into another.
	 *  \param other [IN] The other crl to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_CRL &other );

	/*! \brief This member is an empty instance of this class.
	 */
	static PKI_CRL EmptyInstance;
private:
	HashTable_Dn IssuerDN;
	bool LoadIssuerDN();
	PKI_EXT InternalExts;
	void ClearPointers();
	void Reset();
	mString lePEM_CRL;
	X509_CRL * crl;
	bool StringToX509Crl(const char * CrlPem);
	bool X509CrlToString();
	vector<REVOCATION_INFO> RevokedCerts;
	bool CommonLoad();
};

#endif //PKI_CRL_H

