/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1REQ_H
#define ASN1REQ_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/rand.h>
#include <openssl/evp.h>
#include <openssl/err.h>
#include "Asn1Logs.h"
#include "Asn1Entity.h"
#include "Asn1User.h"
#include "Asn1Conf.h"
#include "Asn1Cert.h"
#include <PKI_P7B.h>


/*!
	This structure defines a password change request for a webuser
*/
typedef struct st_WEBUSER_CHANGE_PASSWD
{
	ASN1_INTEGER * user_id;			//!< The user id
	ASN1_UTF8STRING * old_password;		//!< The old password
	ASN1_UTF8STRING * password;		//!< The new password
}WEBUSER_CHANGE_PASSWD;
DECLARE_ASN1_ITEM(WEBUSER_CHANGE_PASSWD)

/*!
	This structure represents a login request.
*/
typedef struct st_ADMIN_REQ_LOGIN
{
	ASN1_UTF8STRING * entity;		//!< The entity name to log into
	ASN1_UTF8STRING * username;		//!< The optional username
	ASN1_UTF8STRING * password;		//!< The optional password
}ADMIN_REQ_LOGIN;
DECLARE_ASN1_ITEM(ADMIN_REQ_LOGIN)


/*!
	This structure represents a web user login request.
*/
typedef struct st_WEB_REQ_LOGIN
{
	ASN1_UTF8STRING * email;		//!< The email
	ASN1_UTF8STRING * password;		//!< The password
}WEB_REQ_LOGIN;
DECLARE_ASN1_ITEM(WEB_REQ_LOGIN)


/*!
	This structure represents a web user creating request.
*/
typedef struct st_WEB_REQ_CREATE
{
	ASN1_UTF8STRING * email;		//!< The email
	ASN1_UTF8STRING * password;		//!< The password
	X509_NAME * dn;					//!< The DN
}WEB_REQ_CREATE;
DECLARE_ASN1_ITEM(WEB_REQ_CREATE)


/*!
	This structure represents an entity email configuration modification.
*/
typedef struct st_SET_ENTITY_MAIL_CONF
{
	X509 * entity_cert;			//!< The certificate of the entity to modify the conf from
	EMAIL_CONF * mail_conf;		//!< The new email conf
}S_ENTITY_MAIL_CONF;
DECLARE_ASN1_ITEM(S_ENTITY_MAIL_CONF)

/*!
	This structure represents an entity ACLs configuration modification.
*/
typedef struct st_SET_ENTITY_ACL
{
	X509 * entity_cert;						//!< The certificate of the entity to modify the conf from
	STACK_OF(ACL_ENTRY) * acl_entries;		//!< The ACLs
}S_ENTITY_ACL;
DECLARE_ASN1_ITEM(S_ENTITY_ACL)


/*!
	This structure represents an entity configuration modification.
*/
typedef struct st_SET_ENTITY_CONF
{
	X509 * entity_cert;						//!< The certificate of the entity to modify the conf from
	ENTITY_CONF_BODY * conf;				//!< The conf
}S_ENTITY_CONF;
DECLARE_ASN1_ITEM(S_ENTITY_CONF)


/*!
	This structure represents an entity Audits configuration modification.
*/
typedef struct st_SET_ENTITY_AUDITS
{
	X509 * entity_cert;						//!< The certificate of the entity to modify the conf from
	STACK_OF(ENTITY_AUDIT_ENTRY) * audits;	//!< The audits
}S_ENTITY_AUDITS;
DECLARE_ASN1_ITEM(S_ENTITY_AUDITS)


/*!
	This structure represents an email.
*/
typedef struct st_MAIL_DATAS
{
	ASN1_UTF8STRING * Subject;		//!< The subject
	ASN1_UTF8STRING * Body;			//!< The body
	ASN1_INTEGER * SignMail;		//!< Should the email be signed by the entity
	ASN1_OCTET_STRING * Attach;		//!< An attachement
	ASN1_UTF8STRING * AttachName;	//!< The file name of the attachement
	ASN1_UTF8STRING * AttachType;	//!< The MIME type of the attachement
}MAIL_DATAS;
DECLARE_ASN1_ITEM(MAIL_DATAS)


/*!
	This structure represents an object enumeration.
*/
typedef struct st_ENUM_OBJECTS
{
	ASN1_INTEGER * state;		//!< An optional state
	ASN1_INTEGER * index;		//!< The index
	ASN1_INTEGER * num;			//!< The maximum number of objects to return
	ASN1_UTF8STRING * filter;	//!< An optional filter
}OBJECTS_ENUM;
DECLARE_ASN1_ITEM(OBJECTS_ENUM)


/*!
	[CA] This structure represents a certificate signature
*/
typedef struct st_SIGN_CSR
{
	ASN1_INTEGER * days;		//!< The validity in days for the certificate
	X509_REQ * request;			//!< The certificat request
	ASN1_UTF8STRING * uid;		//!< The optional LDAP uid
}SIGN_CSR;
DECLARE_ASN1_ITEM(SIGN_CSR)



/*!
	This structure represents a group renaming request.
*/
typedef struct st_RENAME_GROUP
{
	ASN1_INTEGER * id;			//!< The ID of the group to rename
	ASN1_UTF8STRING * name;		//!< The new name
}S_RENAME_GROUP;
DECLARE_ASN1_ITEM(S_RENAME_GROUP)

/*!
	This structure represents the change of UID for a profile.
*/
typedef struct st_PROFILE_CHANGE_UID
{
	ASN1_INTEGER * profile_id;			//!< The profile ID
	ASN1_UTF8STRING * ldap_uid;			//!< The LDAP UID
}PROFILE_CHANGE_UID;
DECLARE_ASN1_ITEM(PROFILE_CHANGE_UID)


/*!
	This structure represents the change of owner for a profile.
*/
typedef struct st_PROFILE_CHANGE_OWNER
{
	ASN1_INTEGER * profile_id;			//!< The profile ID
	ASN1_INTEGER * owner_group_serial;	//!< The serial of the owning group, or 0 if the owner is a user
}PROFILE_CHANGE_OWNER;
DECLARE_ASN1_ITEM(PROFILE_CHANGE_OWNER)


/*!
	This structure represents the change of DN for a profile.
*/
typedef struct st_PROFILE_CHANGE_DN
{
	ASN1_INTEGER * profile_id;			//!< The profile ID
	X509_NAME * dn;						//!< The DN of the profile
}PROFILE_CHANGE_DN;
DECLARE_ASN1_ITEM(PROFILE_CHANGE_DN)


/*!
	This structure represents the rejection of a profile
*/
typedef struct st_PROFILE_REJECT
{
	ASN1_INTEGER * profile_id;			//!< The profile ID
	ASN1_UTF8STRING * reason;			//!< The reson
}PROFILE_REJECT;
DECLARE_ASN1_ITEM(PROFILE_REJECT)


/*!
	This structure represents the importation of a PKCS#12.
*/
typedef struct st_CA_P12_IMPORT
{
	PKCS12 * p12;					//!< The PKCS#12 to import
	ASN1_UTF8STRING * password;		//!< The PKCS#12's password
}CA_P12_IMPORT;
DECLARE_ASN1_ITEM(CA_P12_IMPORT)


/*!
	This structure represents the revocation of a certificate by an EE user
*/
typedef struct st_WEBUSER_REVOKE_CERT
{
	ASN1_INTEGER * user_id;			//!< The user ID
	ASN1_INTEGER * cert_id;			//!< The certificate ID
}WEBUSER_REVOKE_CERT;
DECLARE_ASN1_ITEM(WEBUSER_REVOKE_CERT)


/*!
	This structure represents the validation of a DN.
*/
typedef struct st_DN_VALIDATION
{
	ASN1_UTF8STRING * id;		//!< The id
	X509_NAME * dn;				//!< The DN
}DN_VALIDATION;
DECLARE_ASN1_ITEM(DN_VALIDATION)




#define ENUM_EE_USERS_NOT_RA_VALIDATED		1
#define ENUM_EE_USERS_NOT_ACTIVE			2
#define ENUM_EE_USERS_VALID					3


#define ADMIN_REQ_TYPE_LOGIN					0	//!< The request is a login
#define ADMIN_REQ_TYPE_CREATE_ENTITY			1	//!< The request is to create an entity
#define ADMIN_REQ_TYPE_DELETE_ENTITY			2	//!< The request is to delete an entity
#define ADMIN_REQ_TYPE_ENUM_ENTITIES			3	//!< The request is to list the entities
#define ADMIN_REQ_TYPE_CREATE_USER				4	//!< The request is to create a server user
#define ADMIN_REQ_TYPE_ENUM_USERS				5	//!< The request is to list the users
#define ADMIN_REQ_TYPE_UPDATE_USER				6	//!< The request is to update a server user
#define ADMIN_REQ_TYPE_ENUM_LOGS				7	//!< The request is to enumerate the logs
#define ADMIN_REQ_TYPE_CHG_USER_PASSWD			8	//!< The request is for changing a server user's password
#define ADMIN_REQ_TYPE_CHG_PASSWD				9	//!< The request is for a server user to change its password
#define ADMIN_REQ_TYPE_SIGN_ENTITY				10	//!< The request is to sign an entity
#define ADMIN_REQ_TYPE_CREATE_PKI_USER			11	//!< The request is to create a PKI user
#define ADMIN_REQ_TYPE_REVOKE_USER				12	//!< The request is for revoking a PKI user
#define ADMIN_REQ_TYPE_REVOKE_ENTITY			13	//!< The request is revoking an entity
#define ADMIN_REQ_TYPE_PUSH_CONFIG				14	//!< The request is for pushing the PKI Conf on a repository
#define ADMIN_REQ_TYPE_SET_OFFLINE				15	//!< The request is setting the PKI Entity offline
#define ADMIN_REQ_TYPE_GET_OFFLINE				16	//!< The request is getting the PKI Offline Status
#define ADMIN_REQ_TYPE_GET_MY_ACL				17	//!< The request is user to get its ACL
#define ADMIN_REQ_TYPE_GET_CONFIG				18	//!< The request is for exporting the PKI Conf
#define ADMIN_REQ_TYPE_GET_MY_CONFIG			19	//!< The request is for an entity to get its conf from a repository
#define ADMIN_REQ_TYPE_INIT_ENTITY				20	//!< The request is intializing an entity
#define ADMIN_REQ_TYPE_GET_ENTITY_CONF			21	//!< The request is for an entity GUI to get the entity Conf
#define ADMIN_REQ_TYPE_GET_REPOSITORIES			22	//!< The request is to get the list of known repositories
#define ADMIN_REQ_TYPE_SET_REPOSITORIES			23	//!< The request is to modify the repositories information (host:port)
#define ADMIN_REQ_TYPE_GET_LOGS_TYPE			24	//!< The request is getting the list of known logs type on a server entity
#define ADMIN_REQ_TYPE_LOCK_RESOURCE			25	//!< The request is for locking a PKI resource
#define ADMIN_REQ_TYPE_UNLOCK_RESOURCE			26	//!< The request is for unlocking a resource
#define ADMIN_REQ_TYPE_GET_ENTITY_ACL			27	//!< The request is to get the ACLs of an entity
#define ADMIN_REQ_TYPE_GET_ENTITY_MAIL_CONF		28	//!< The request is to get the Email Conf of an entity
#define ADMIN_REQ_TYPE_SET_ENTITY_ACL			29	//!< The request is to modify the ACLs of an entity
#define ADMIN_REQ_TYPE_SET_ENTITY_MAIL_CONF		30	//!< The request is to modify the Email Conf of an entity
#define ADMIN_REQ_TYPE_SET_ENTITY_CONF			31	//!< The request is to modify the Conf of an entity
#define ADMIN_REQ_TYPE_SEND_ADMIN_MAIL			32	//!< The request is for sending a administrative email
#define ADMIN_REQ_TYPE_GET_ENTITY_AUDITS		33	//!< The request is to get the Audits of an entity
#define ADMIN_REQ_TYPE_SET_ENTITY_AUDITS		34	//!< The request is to modify the Audits of an entity
#define ADMIN_REQ_TYPE_GET_ENTITY_LOGS_TYPE		35	//!< The request is to get the list of known logs type on a entity
#define ADMIN_REQ_TYPE_GET_ENTITIES_LINKS		36	//!< The request is to get the entities links
#define ADMIN_REQ_TYPE_SET_ENTITIES_LINKS		37	//!< The request is to modify the entities links
#define ADMIN_REQ_TYPE_PUSH_NEWPKI_OBJECTS		38	//!< The request is for pushing a newpki object between two repositories
#define ADMIN_REQ_TYPE_CREATE_ROOT_CA			39	//!< The request is to create a ROOT CA
#define ADMIN_REQ_TYPE_GET_CA_STATUS			40	//!< The request is to get the status of a CA
#define ADMIN_REQ_TYPE_GET_MY_REQS				41	//!< The request is to get the new inter-entities requests that were sent to an entity
#define ADMIN_REQ_TYPE_ENUM_CERTS				42	//!< The request is for enumerating the CA certs
#define ADMIN_REQ_TYPE_ENUM_CRLS				43	//!< The request is for enumerating the CA crls
#define ADMIN_REQ_TYPE_SIGN_CSR					44	//!< The request is to sign a CSR on the CA
#define ADMIN_REQ_TYPE_CREATE_CHILD_CA			45	//!< The request is to create a Child CA
#define ADMIN_REQ_TYPE_IMPORT_CHILD_CA_CERT		46	//!< The request is for importing the Child CA certificate
#define ADMIN_REQ_TYPE_REVOKE_CERT				47	//!< The request is for revoking a cert in the CA
#define ADMIN_REQ_TYPE_GET_GROUPS				48	//!< The request is to get the groups
#define ADMIN_REQ_TYPE_SET_GROUPS				49	//!< The request is to set the groups
#define ADMIN_REQ_TYPE_ADD_GROUP				50	//!< The request is to add a group
#define ADMIN_REQ_TYPE_DEL_GROUP				51	//!< The request is to delete a group
#define ADMIN_REQ_TYPE_REN_GROUP				52	//!< The request is to rename a group
#define ADMIN_REQ_TYPE_CHECK_LOGS				53	//!< The request is to verify the integrity of the logs
#define ADMIN_REQ_TYPE_IMPORT_PROFILE			54	//!< The request is to import a profile
#define ADMIN_REQ_TYPE_GET_LOCAL_CONF			55	//!< The request is to get the complete conf of an entity
#define ADMIN_REQ_TYPE_PUSH_REQ					56	//!< The request is to send a inter-entities request to a repository
#define ADMIN_REQ_TYPE_PUSH_RESP				57	//!< The request is to send a inter-entities response to a repository
#define ADMIN_REQ_TYPE_GET_MY_RESPS				58	//!< The request is for an entity to get the inter-entities responses
#define ADMIN_REQ_TYPE_ENUM_PROFILES			59	//!< The request is for enumerating the profiles
#define ADMIN_REQ_TYPE_REQUEST_CERT				60	//!< The request is for a RA Operator to request a certificate for a profile
#define ADMIN_REQ_TYPE_GENERATE_CRL				61	//!< The request is for a CA Operator to request the generation of a CRL
#define ADMIN_REQ_TYPE_SYNCH_OBJECTS			62	//!< The request is for two repositories to synchronize their objects
#define ADMIN_REQ_TYPE_GET_CA_P7B				63	//!< The request is to get the certificate of a CA
#define ADMIN_REQ_TYPE_DELETE_PKCS12			64	//!< The request is to delete the PKCS#12
#define ADMIN_REQ_TYPE_GET_LOGS_COUNT			65	//!< The request is to get the number of logs
#define ADMIN_REQ_TYPE_SEARCH_LDAP				66	//!< The request is to search the LDAP
#define ADMIN_REQ_TYPE_DELETE_RESPONSE			67	//!< The request is for the deletion of a response
#define ADMIN_REQ_TYPE_CHANGE_PROFILE_UID		68	//!< The request is for changing a profile's UID
#define ADMIN_REQ_TYPE_CHANGE_PROFILE_OWNER		69	//!< The request is for changing a profile's OWNER
#define ADMIN_REQ_TYPE_DELETE_PROFILE			70	//!< The request is for the deletion of a profile
#define ADMIN_REQ_TYPE_CHANGE_PROFILE_DN		71	//!< The request is for changing the DN of a profile
#define ADMIN_REQ_TYPE_LOAD_ENTITY				72	//!< The request id for loading an entity
#define ADMIN_REQ_TYPE_UNLOAD_ENTITY			73	//!< The request id for unloading an entity
#define ADMIN_REQ_TYPE_DELETE_REQUEST			74	//!< The request is for the deletion of a request
#define ADMIN_REQ_TYPE_IMPORT_CA_P12			75	//!< The request is for importing a CA certificate in PKCS#12
#define ADMIN_REQ_TYPE_DN_VALIDATION			76	//!< The request is for validation of a DN
#define ADMIN_REQ_TYPE_GET_DN_VALIDATION		77	//!< The request is for getting a DN validation request
#define ADMIN_REQ_TYPE_WEBUSER_CREATE			78	//!< The request is for creating a web user
#define ADMIN_REQ_TYPE_WEBUSER_LOGIN			79	//!< The request is for loging-in a web user
#define ADMIN_REQ_TYPE_WEBUSER_ACTIVATE			80	//!< The request is for activating a web user
#define ADMIN_REQ_TYPE_ACCEPT_PROFILE			81	//!< The request is for accepting a profile
#define ADMIN_REQ_TYPE_REJECT_PROFILE			82	//!< The request is for rejecting a profile
#define ADMIN_REQ_TYPE_EE_REQUEST_CERT			83	//!< The request is for a EE user to request a certificate
#define ADMIN_REQ_TYPE_WEBUSER_ENUM_CERTS		84	//!< The request is for a EE user to enumerate its certs
#define ADMIN_REQ_TYPE_SUSPEND_CERT				85	//!< The request is for suspending a cert in the CA
#define ADMIN_REQ_TYPE_UNSUSPEND_CERT			86	//!< The request is for unsuspending a cert in the CA
#define ADMIN_REQ_TYPE_WEBUSER_GET_DN			87	//!< The request is for a EE user to get its DN
#define ADMIN_REQ_TYPE_EE_REVOKE_CERT			88	//!< The request is for a EE user to revoke a cert
#define ADMIN_REQ_TYPE_ENUM_EE_USERS			89	//!< The request is for enumerating the EE users
#define ADMIN_REQ_TYPE_WEBUSER_CHANGE_PASSWD	90	//!< The request is for change the password of a web user 

/*!
	This structure represents the body of a request.
*/
typedef struct st_ADMIN_REQUEST_BODY
{
	int type;
	union
	{
		ADMIN_REQ_LOGIN * login;					//!< Login request
		ENTITY_CREATION_REQ * cre_entity;			//!< Entity creation request
		ASN1_UTF8STRING * entity_name;				//!< Entity suppresion request
		USER_INFO * cre_user;						//!< Server user creation request
		USER_ENTRY * upd_user;						//!< Server user update request
		ADMIN_REQ_ENUM_LOGS * enum_logs;			//!< Logs enumeration request
		CHANGE_USER_PASSWD * chg_user_pwd;			//!< Server user password changing request
		CHANGE_PASSWD * chg_pwd;					//!< Server user password changing request
		ENTITY_SIGNATURE_REQ * sign_entity;			//!< Entity signature request
		CREATE_PKI_USER_REQUEST * create_pki_user;	//!< PKI User creation request
		EXPORTED_PKI_CONF * conf;					//!< PKI conf synchronization request
		ASN1_INTEGER * serial;						//!< A serial related request
		ASN1_INTEGER * offline;						//!< Setting PKI offline status request
		ENTITY_INIT_REQ * init_entity;				//!< Entity initialization request
		X509 * entity_cert;							//!< Entity certificate related request
		STACK_OF(REP_ENTRY_INFO) * repositories;	//!< Setting the repositories info request
		ASN1_UTF8STRING * resource_name;			//!< A resource name related request
		S_ENTITY_ACL * entity_acl;					//!< Setting the ACLs request
		S_ENTITY_MAIL_CONF * entity_mail_conf;		//!< Setting the email conf request
		S_ENTITY_CONF * entity_conf;				//!< Setting the entity Conf request
		S_ENTITY_AUDITS * entity_audits;			//!< Setting the entity audits request
		MAIL_DATAS * admin_mail;					//!< Sending a admin mail request
		ASN1_INTEGER * entity_type;					//!< Getting the logs type request
		STACK_OF(ENTITY_LINKS) * entities_links;	//!< Setting the entities links requets
		WAITING_NEWPKI_OBJECT * waiting_obj;		//!< Synchronization the repositories objects request
		REQ_CREATE_ROOT_CA * create_root_ca;		//!< Creating a ROOT CA request
		TRANSACTION_IDS * transactions_ids;			//!< Transaction IDs related request
		OBJECTS_ENUM * enum_objects;				//!< Objects enumeration request
		SIGN_CSR * sign_csr;						//!< CSR signature request
		REQ_CREATE_CHILD_CA * create_child_ca;		//!< Creating a CHILD CA request
		STACK_OF(USERS_GROUP) * groups;				//!< Setting the groups request
		ASN1_UTF8STRING * group_name;				//!< Group name related request
		ASN1_INTEGER * group_id;					//!< Group ID related request
		S_RENAME_GROUP * rename_group;				//!< Renaming a group request
		NEWPKI_PROFILE * profile;					//!< Profile importation request
		CRYPTED_NEWPKI_REQUEST * new_req;			//!< Inter-entities request publication request
		CRYPTED_NEWPKI_RESPONSE * new_resp;			//!< Inter-entities response publication request
		NEWPKI_REQUEST_CERT * cert_request;			//!< RA certification request
		OBJECTS_LIST_SYNCHRO * known_objects;		//!< Inter repositories synchronization request
		PKCS7 * p7b;								//!< Child CA certificate importation request
		ASN1_UTF8STRING * ldap_search;				//!< A LDAP search request
		ASN1_OCTET_STRING * transaction_id;			//!< A transaction ID related request
		PROFILE_CHANGE_OWNER * profile_owner;		//!< Changing a profile's owner
		PROFILE_CHANGE_UID * profile_uid;			//!< Changing a profile's uid
		PROFILE_CHANGE_DN * profile_dn;				//!< Changing a profile's dn
		PROFILE_REJECT * profile_reject;			//!< Rejecting a profile
		ASN1_INTEGER * profile_id;					//!< A profile's ID
		ASN1_INTEGER * request_id;					//!< A request's ID
		CA_P12_IMPORT * p12_import;					//!< Importing a CA's certificate from a PKCS#12
		ASN1_UTF8STRING * dn_id;					//!< The DN's Id
		DN_VALIDATION * dn_val;						//!< Validating a DN
		WEB_REQ_CREATE * webcreate;					//!< Web user create request
		WEB_REQ_LOGIN * weblogin;					//!< Web user login request
		ASN1_UTF8STRING * webactivate;				//!< Web user activation request
		PROFILE_REJECT * reject_profile;			//!< Profile rejection request
		NEWPKI_EE_REQUEST_CERT * ee_cert_request;	//!< EE certification request
		WEBUSER_REVOKE_CERT * ee_cert_revoke;		//!< Web user certificate revocation
		WEBUSER_CHANGE_PASSWD * web_chg_passwd;		//!< Changing the password of a web user
		ASN1_NULL * other;							//!< Any other request
	}d;
}ADMIN_REQUEST_BODY;
DECLARE_ASN1_ITEM(ADMIN_REQUEST_BODY)

/*!
	This structure represents a request.
*/
typedef struct st_ADMIN_REQUEST
{
	ADMIN_REQUEST_BODY * body;			//!< The body
}ADMIN_REQUEST;
DECLARE_ASN1_ITEM(ADMIN_REQUEST)




#define d2i_ADMIN_REQUEST_bio(bp,msg) (ADMIN_REQUEST *)ASN1_item_d2i_bio(AdminRequest::get_ASN1_ITEM(), bp,(unsigned char **)(msg))

#define i2d_ADMIN_REQUEST_bio(bp,msg) ASN1_item_i2d_bio(AdminRequest::get_ASN1_ITEM(), bp,(unsigned char *)msg)


/*!
  This class is the representation of WEBUSER_CHANGE_PASSWD
*/
class WebuserChangePasswd : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WebuserChangePasswd();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WebuserChangePasswd(const WebuserChangePasswd & other);

	/*! \brief This is the destructor.
	 */
	~WebuserChangePasswd();


	/*! \brief This function converts a WEBUSER_CHANGE_PASSWD to a WebuserChangePasswd.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WEBUSER_CHANGE_PASSWD * Datas);

	/*! \brief This function converts a WebuserChangePasswd to a WEBUSER_CHANGE_PASSWD.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WEBUSER_CHANGE_PASSWD ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WEBUSER_CHANGE_PASSWD.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::old_password.
	 *  \param c_oldPassword [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_oldPassword(const mString & c_oldPassword);

	/*! \brief This function returns ASN1_UTF8STRING::old_password.
	 *  \return The value of ASN1_UTF8STRING::old_password.
	 */
	const mString & get_oldPassword() const;

	/*! \brief This function returns ASN1_UTF8STRING::old_password.
	 *  \return The value of ASN1_UTF8STRING::old_password.
	 */
	mString & get_oldPassword();


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function sets ASN1_INTEGER::user_id.
	 *  \param c_userId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userId(unsigned long c_userId);

	/*! \brief This function returns ASN1_INTEGER::user_id.
	 *  \return The value of ASN1_INTEGER::user_id.
	 */
	unsigned long get_userId() const;


	/*! \brief This function copies a WebuserChangePasswd to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WebuserChangePasswd & other);
	/*! \brief This member is an empty instance of WebuserChangePasswd.
	 */
	static WebuserChangePasswd EmptyInstance;

private:
	mString m_oldPassword;
	mString m_password;
	unsigned long m_userId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CA_P12_IMPORT
*/
class CaP12Import : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CaP12Import();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CaP12Import(const CaP12Import & other);

	/*! \brief This is the destructor.
	 */
	~CaP12Import();


	/*! \brief This function converts a CA_P12_IMPORT to a CaP12Import.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CA_P12_IMPORT * Datas);

	/*! \brief This function converts a CaP12Import to a CA_P12_IMPORT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CA_P12_IMPORT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CA_P12_IMPORT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets PKCS12::p12.
	 *  \param c_p12 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12(const PKI_PKCS12 & c_p12);

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	const PKI_PKCS12 & get_p12() const;

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	PKI_PKCS12 & get_p12();


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function copies a CaP12Import to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CaP12Import & other);
	/*! \brief This member is an empty instance of CaP12Import.
	 */
	static CaP12Import EmptyInstance;

private:
	PKI_PKCS12 m_p12;
	mString m_password;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of DN_VALIDATION
*/
class DnValidation : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	DnValidation();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	DnValidation(const DnValidation & other);

	/*! \brief This is the destructor.
	 */
	~DnValidation();


	/*! \brief This function converts a DN_VALIDATION to a DnValidation.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const DN_VALIDATION * Datas);

	/*! \brief This function converts a DnValidation to a DN_VALIDATION.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(DN_VALIDATION ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of DN_VALIDATION.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_UTF8STRING::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(const mString & c_id);

	/*! \brief This function returns ASN1_UTF8STRING::id.
	 *  \return The value of ASN1_UTF8STRING::id.
	 */
	const mString & get_id() const;

	/*! \brief This function returns ASN1_UTF8STRING::id.
	 *  \return The value of ASN1_UTF8STRING::id.
	 */
	mString & get_id();


	/*! \brief This function copies a DnValidation to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const DnValidation & other);
	/*! \brief This member is an empty instance of DnValidation.
	 */
	static DnValidation EmptyInstance;

private:
	X509_NAME * m_dn;
	mString m_id;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_REQ_LOGIN
*/
class AdminReqLogin : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminReqLogin();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminReqLogin(const AdminReqLogin & other);

	/*! \brief This is the destructor.
	 */
	~AdminReqLogin();


	/*! \brief This function converts a ADMIN_REQ_LOGIN to a AdminReqLogin.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_REQ_LOGIN * Datas);

	/*! \brief This function converts a AdminReqLogin to a ADMIN_REQ_LOGIN.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_REQ_LOGIN ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_REQ_LOGIN.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::entity.
	 *  \param c_entity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entity(const mString & c_entity);

	/*! \brief This function returns ASN1_UTF8STRING::entity.
	 *  \return The value of ASN1_UTF8STRING::entity.
	 */
	const mString & get_entity() const;

	/*! \brief This function returns ASN1_UTF8STRING::entity.
	 *  \return The value of ASN1_UTF8STRING::entity.
	 */
	mString & get_entity();


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function sets ASN1_UTF8STRING::username.
	 *  \param c_username [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_username(const mString & c_username);

	/*! \brief This function returns ASN1_UTF8STRING::username.
	 *  \return The value of ASN1_UTF8STRING::username.
	 */
	const mString & get_username() const;

	/*! \brief This function returns ASN1_UTF8STRING::username.
	 *  \return The value of ASN1_UTF8STRING::username.
	 */
	mString & get_username();


	/*! \brief This function copies a AdminReqLogin to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminReqLogin & other);
	/*! \brief This member is an empty instance of AdminReqLogin.
	 */
	static AdminReqLogin EmptyInstance;

private:
	mString m_entity;
	mString m_password;
	mString m_username;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of WEB_REQ_LOGIN
*/
class WebReqLogin : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WebReqLogin();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WebReqLogin(const WebReqLogin & other);

	/*! \brief This is the destructor.
	 */
	~WebReqLogin();


	/*! \brief This function converts a WEB_REQ_LOGIN to a WebReqLogin.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WEB_REQ_LOGIN * Datas);

	/*! \brief This function converts a WebReqLogin to a WEB_REQ_LOGIN.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WEB_REQ_LOGIN ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WEB_REQ_LOGIN.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function copies a WebReqLogin to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WebReqLogin & other);
	/*! \brief This member is an empty instance of WebReqLogin.
	 */
	static WebReqLogin EmptyInstance;

private:
	mString m_email;
	mString m_password;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of WEB_REQ_CREATE
*/
class WebReqCreate : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WebReqCreate();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WebReqCreate(const WebReqCreate & other);

	/*! \brief This is the destructor.
	 */
	~WebReqCreate();


	/*! \brief This function converts a WEB_REQ_CREATE to a WebReqCreate.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WEB_REQ_CREATE * Datas);

	/*! \brief This function converts a WebReqCreate to a WEB_REQ_CREATE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WEB_REQ_CREATE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WEB_REQ_CREATE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_UTF8STRING::email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_UTF8STRING::password.
	 *  \param c_password [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_password(const mString & c_password);

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	const mString & get_password() const;

	/*! \brief This function returns ASN1_UTF8STRING::password.
	 *  \return The value of ASN1_UTF8STRING::password.
	 */
	mString & get_password();


	/*! \brief This function copies a WebReqCreate to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WebReqCreate & other);
	/*! \brief This member is an empty instance of WebReqCreate.
	 */
	static WebReqCreate EmptyInstance;

private:
	X509_NAME * m_dn;
	mString m_email;
	mString m_password;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of S_ENTITY_MAIL_CONF
*/
class SEntityMailConf : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SEntityMailConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SEntityMailConf(const SEntityMailConf & other);

	/*! \brief This is the destructor.
	 */
	~SEntityMailConf();


	/*! \brief This function converts a S_ENTITY_MAIL_CONF to a SEntityMailConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const S_ENTITY_MAIL_CONF * Datas);

	/*! \brief This function converts a SEntityMailConf to a S_ENTITY_MAIL_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(S_ENTITY_MAIL_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of S_ENTITY_MAIL_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const PKI_CERT & c_entityCert);

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	const PKI_CERT & get_entityCert() const;

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	PKI_CERT & get_entityCert();


	/*! \brief This function sets EMAIL_CONF::mail_conf.
	 *  \param c_mailConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_mailConf(const EmailConf & c_mailConf);

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	const EmailConf & get_mailConf() const;

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	EmailConf & get_mailConf();


	/*! \brief This function copies a SEntityMailConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SEntityMailConf & other);
	/*! \brief This member is an empty instance of SEntityMailConf.
	 */
	static SEntityMailConf EmptyInstance;

private:
	PKI_CERT m_entityCert;
	EmailConf m_mailConf;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of S_ENTITY_ACL
*/
class SEntityAcl : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SEntityAcl();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SEntityAcl(const SEntityAcl & other);

	/*! \brief This is the destructor.
	 */
	~SEntityAcl();


	/*! \brief This function converts a S_ENTITY_ACL to a SEntityAcl.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const S_ENTITY_ACL * Datas);

	/*! \brief This function converts a SEntityAcl to a S_ENTITY_ACL.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(S_ENTITY_ACL ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of S_ENTITY_ACL.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ACL_ENTRY::acl_entries.
	 *  \param c_aclEntries [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_aclEntries(const mVector< AclEntry > & c_aclEntries);

	/*! \brief This function returns ACL_ENTRY::acl_entries.
	 *  \return The value of ACL_ENTRY::acl_entries.
	 */
	const mVector< AclEntry > & get_aclEntries() const;

	/*! \brief This function returns ACL_ENTRY::acl_entries.
	 *  \return The value of ACL_ENTRY::acl_entries.
	 */
	mVector< AclEntry > & get_aclEntries();


	/*! \brief This function sets X509::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const PKI_CERT & c_entityCert);

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	const PKI_CERT & get_entityCert() const;

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	PKI_CERT & get_entityCert();


	/*! \brief This function copies a SEntityAcl to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SEntityAcl & other);
	/*! \brief This member is an empty instance of SEntityAcl.
	 */
	static SEntityAcl EmptyInstance;

private:
	mVector< AclEntry > m_aclEntries;
	PKI_CERT m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of S_ENTITY_CONF
*/
class SEntityConf : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SEntityConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SEntityConf(const SEntityConf & other);

	/*! \brief This is the destructor.
	 */
	~SEntityConf();


	/*! \brief This function converts a S_ENTITY_CONF to a SEntityConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const S_ENTITY_CONF * Datas);

	/*! \brief This function converts a SEntityConf to a S_ENTITY_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(S_ENTITY_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of S_ENTITY_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_CONF_BODY::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConfBody & c_conf);

	/*! \brief This function returns ENTITY_CONF_BODY::conf.
	 *  \return The value of ENTITY_CONF_BODY::conf.
	 */
	const EntityConfBody & get_conf() const;

	/*! \brief This function returns ENTITY_CONF_BODY::conf.
	 *  \return The value of ENTITY_CONF_BODY::conf.
	 */
	EntityConfBody & get_conf();


	/*! \brief This function sets X509::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const PKI_CERT & c_entityCert);

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	const PKI_CERT & get_entityCert() const;

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	PKI_CERT & get_entityCert();


	/*! \brief This function copies a SEntityConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SEntityConf & other);
	/*! \brief This member is an empty instance of SEntityConf.
	 */
	static SEntityConf EmptyInstance;

private:
	EntityConfBody m_conf;
	PKI_CERT m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of MAIL_DATAS
*/
class MailDatas : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	MailDatas();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	MailDatas(const MailDatas & other);

	/*! \brief This is the destructor.
	 */
	~MailDatas();


	/*! \brief This function converts a MAIL_DATAS to a MailDatas.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const MAIL_DATAS * Datas);

	/*! \brief This function converts a MailDatas to a MAIL_DATAS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(MAIL_DATAS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of MAIL_DATAS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_OCTET_STRING::Attach.
	 *  \param c_attach [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_attach(const Asn1OctetString & c_attach);

	/*! \brief This function returns ASN1_OCTET_STRING::Attach.
	 *  \return The value of ASN1_OCTET_STRING::Attach.
	 */
	const Asn1OctetString & get_attach() const;

	/*! \brief This function returns ASN1_OCTET_STRING::Attach.
	 *  \return The value of ASN1_OCTET_STRING::Attach.
	 */
	Asn1OctetString & get_attach();


	/*! \brief This function sets ASN1_UTF8STRING::AttachName.
	 *  \param c_attachname [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_attachname(const mString & c_attachname);

	/*! \brief This function returns ASN1_UTF8STRING::AttachName.
	 *  \return The value of ASN1_UTF8STRING::AttachName.
	 */
	const mString & get_attachname() const;

	/*! \brief This function returns ASN1_UTF8STRING::AttachName.
	 *  \return The value of ASN1_UTF8STRING::AttachName.
	 */
	mString & get_attachname();


	/*! \brief This function sets ASN1_UTF8STRING::AttachType.
	 *  \param c_attachtype [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_attachtype(const mString & c_attachtype);

	/*! \brief This function returns ASN1_UTF8STRING::AttachType.
	 *  \return The value of ASN1_UTF8STRING::AttachType.
	 */
	const mString & get_attachtype() const;

	/*! \brief This function returns ASN1_UTF8STRING::AttachType.
	 *  \return The value of ASN1_UTF8STRING::AttachType.
	 */
	mString & get_attachtype();


	/*! \brief This function sets ASN1_UTF8STRING::Body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const mString & c_body);

	/*! \brief This function returns ASN1_UTF8STRING::Body.
	 *  \return The value of ASN1_UTF8STRING::Body.
	 */
	const mString & get_body() const;

	/*! \brief This function returns ASN1_UTF8STRING::Body.
	 *  \return The value of ASN1_UTF8STRING::Body.
	 */
	mString & get_body();


	/*! \brief This function sets ASN1_INTEGER::SignMail.
	 *  \param c_signmail [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signmail(unsigned long c_signmail);

	/*! \brief This function returns ASN1_INTEGER::SignMail.
	 *  \return The value of ASN1_INTEGER::SignMail.
	 */
	unsigned long get_signmail() const;


	/*! \brief This function sets ASN1_UTF8STRING::Subject.
	 *  \param c_subject [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_subject(const mString & c_subject);

	/*! \brief This function returns ASN1_UTF8STRING::Subject.
	 *  \return The value of ASN1_UTF8STRING::Subject.
	 */
	const mString & get_subject() const;

	/*! \brief This function returns ASN1_UTF8STRING::Subject.
	 *  \return The value of ASN1_UTF8STRING::Subject.
	 */
	mString & get_subject();


	/*! \brief This function copies a MailDatas to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const MailDatas & other);
	/*! \brief This member is an empty instance of MailDatas.
	 */
	static MailDatas EmptyInstance;

private:
	Asn1OctetString m_attach;
	mString m_attachname;
	mString m_attachtype;
	mString m_body;
	unsigned long m_signmail;
	mString m_subject;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of S_ENTITY_AUDITS
*/
class SEntityAudits : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SEntityAudits();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SEntityAudits(const SEntityAudits & other);

	/*! \brief This is the destructor.
	 */
	~SEntityAudits();


	/*! \brief This function converts a S_ENTITY_AUDITS to a SEntityAudits.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const S_ENTITY_AUDITS * Datas);

	/*! \brief This function converts a SEntityAudits to a S_ENTITY_AUDITS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(S_ENTITY_AUDITS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of S_ENTITY_AUDITS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_AUDIT_ENTRY::audits.
	 *  \param c_audits [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_audits(const mVector< EntityAuditEntry > & c_audits);

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	const mVector< EntityAuditEntry > & get_audits() const;

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	mVector< EntityAuditEntry > & get_audits();


	/*! \brief This function sets X509::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const PKI_CERT & c_entityCert);

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	const PKI_CERT & get_entityCert() const;

	/*! \brief This function returns X509::entity_cert.
	 *  \return The value of X509::entity_cert.
	 */
	PKI_CERT & get_entityCert();


	/*! \brief This function copies a SEntityAudits to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SEntityAudits & other);
	/*! \brief This member is an empty instance of SEntityAudits.
	 */
	static SEntityAudits EmptyInstance;

private:
	mVector< EntityAuditEntry > m_audits;
	PKI_CERT m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of OBJECTS_ENUM
*/
class ObjectsEnum : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ObjectsEnum();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ObjectsEnum(const ObjectsEnum & other);

	/*! \brief This is the destructor.
	 */
	~ObjectsEnum();


	/*! \brief This function converts a OBJECTS_ENUM to a ObjectsEnum.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const OBJECTS_ENUM * Datas);

	/*! \brief This function converts a ObjectsEnum to a OBJECTS_ENUM.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(OBJECTS_ENUM ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of OBJECTS_ENUM.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::filter.
	 *  \param c_filter [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_filter(const mString & c_filter);

	/*! \brief This function returns ASN1_UTF8STRING::filter.
	 *  \return The value of ASN1_UTF8STRING::filter.
	 */
	const mString & get_filter() const;

	/*! \brief This function returns ASN1_UTF8STRING::filter.
	 *  \return The value of ASN1_UTF8STRING::filter.
	 */
	mString & get_filter();


	/*! \brief This function sets ASN1_INTEGER::index.
	 *  \param c_index [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_index(unsigned long c_index);

	/*! \brief This function returns ASN1_INTEGER::index.
	 *  \return The value of ASN1_INTEGER::index.
	 */
	unsigned long get_index() const;


	/*! \brief This function sets ASN1_INTEGER::num.
	 *  \param c_num [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_num(unsigned long c_num);

	/*! \brief This function returns ASN1_INTEGER::num.
	 *  \return The value of ASN1_INTEGER::num.
	 */
	unsigned long get_num() const;


	/*! \brief This function sets ASN1_INTEGER::state.
	 *  \param c_state [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_state(unsigned long c_state);

	/*! \brief This function returns ASN1_INTEGER::state.
	 *  \return The value of ASN1_INTEGER::state.
	 */
	unsigned long get_state() const;


	/*! \brief This function copies a ObjectsEnum to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ObjectsEnum & other);
	/*! \brief This member is an empty instance of ObjectsEnum.
	 */
	static ObjectsEnum EmptyInstance;

private:
	mString m_filter;
	unsigned long m_index;
	unsigned long m_num;
	unsigned long m_state;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of SIGN_CSR
*/
class SignCsr : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SignCsr();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SignCsr(const SignCsr & other);

	/*! \brief This is the destructor.
	 */
	~SignCsr();


	/*! \brief This function converts a SIGN_CSR to a SignCsr.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const SIGN_CSR * Datas);

	/*! \brief This function converts a SignCsr to a SIGN_CSR.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(SIGN_CSR ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of SIGN_CSR.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::days.
	 *  \param c_days [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_days(unsigned long c_days);

	/*! \brief This function returns ASN1_INTEGER::days.
	 *  \return The value of ASN1_INTEGER::days.
	 */
	unsigned long get_days() const;


	/*! \brief This function sets X509_REQ::request.
	 *  \param c_request [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_request(const PKI_CSR & c_request);

	/*! \brief This function returns X509_REQ::request.
	 *  \return The value of X509_REQ::request.
	 */
	const PKI_CSR & get_request() const;

	/*! \brief This function returns X509_REQ::request.
	 *  \return The value of X509_REQ::request.
	 */
	PKI_CSR & get_request();


	/*! \brief This function sets ASN1_UTF8STRING::uid.
	 *  \param c_uid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_uid(const mString & c_uid);

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	const mString & get_uid() const;

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	mString & get_uid();


	/*! \brief This function copies a SignCsr to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SignCsr & other);
	/*! \brief This member is an empty instance of SignCsr.
	 */
	static SignCsr EmptyInstance;

private:
	unsigned long m_days;
	PKI_CSR m_request;
	mString m_uid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of S_RENAME_GROUP
*/
class SRenameGroup : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	SRenameGroup();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	SRenameGroup(const SRenameGroup & other);

	/*! \brief This is the destructor.
	 */
	~SRenameGroup();


	/*! \brief This function converts a S_RENAME_GROUP to a SRenameGroup.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const S_RENAME_GROUP * Datas);

	/*! \brief This function converts a SRenameGroup to a S_RENAME_GROUP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(S_RENAME_GROUP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of S_RENAME_GROUP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::id.
	 *  \return The value of ASN1_INTEGER::id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function copies a SRenameGroup to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const SRenameGroup & other);
	/*! \brief This member is an empty instance of SRenameGroup.
	 */
	static SRenameGroup EmptyInstance;

private:
	unsigned long m_id;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PROFILE_CHANGE_UID
*/
class ProfileChangeUid : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ProfileChangeUid();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ProfileChangeUid(const ProfileChangeUid & other);

	/*! \brief This is the destructor.
	 */
	~ProfileChangeUid();


	/*! \brief This function converts a PROFILE_CHANGE_UID to a ProfileChangeUid.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PROFILE_CHANGE_UID * Datas);

	/*! \brief This function converts a ProfileChangeUid to a PROFILE_CHANGE_UID.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PROFILE_CHANGE_UID ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PROFILE_CHANGE_UID.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function copies a ProfileChangeUid to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ProfileChangeUid & other);
	/*! \brief This member is an empty instance of ProfileChangeUid.
	 */
	static ProfileChangeUid EmptyInstance;

private:
	mString m_ldapUid;
	unsigned long m_profileId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PROFILE_CHANGE_OWNER
*/
class ProfileChangeOwner : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ProfileChangeOwner();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ProfileChangeOwner(const ProfileChangeOwner & other);

	/*! \brief This is the destructor.
	 */
	~ProfileChangeOwner();


	/*! \brief This function converts a PROFILE_CHANGE_OWNER to a ProfileChangeOwner.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PROFILE_CHANGE_OWNER * Datas);

	/*! \brief This function converts a ProfileChangeOwner to a PROFILE_CHANGE_OWNER.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PROFILE_CHANGE_OWNER ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PROFILE_CHANGE_OWNER.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::owner_group_serial.
	 *  \param c_ownerGroupSerial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ownerGroupSerial(unsigned long c_ownerGroupSerial);

	/*! \brief This function returns ASN1_INTEGER::owner_group_serial.
	 *  \return The value of ASN1_INTEGER::owner_group_serial.
	 */
	unsigned long get_ownerGroupSerial() const;


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function copies a ProfileChangeOwner to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ProfileChangeOwner & other);
	/*! \brief This member is an empty instance of ProfileChangeOwner.
	 */
	static ProfileChangeOwner EmptyInstance;

private:
	unsigned long m_ownerGroupSerial;
	unsigned long m_profileId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PROFILE_CHANGE_DN
*/
class ProfileChangeDn : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ProfileChangeDn();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ProfileChangeDn(const ProfileChangeDn & other);

	/*! \brief This is the destructor.
	 */
	~ProfileChangeDn();


	/*! \brief This function converts a PROFILE_CHANGE_DN to a ProfileChangeDn.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PROFILE_CHANGE_DN * Datas);

	/*! \brief This function converts a ProfileChangeDn to a PROFILE_CHANGE_DN.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PROFILE_CHANGE_DN ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PROFILE_CHANGE_DN.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function copies a ProfileChangeDn to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ProfileChangeDn & other);
	/*! \brief This member is an empty instance of ProfileChangeDn.
	 */
	static ProfileChangeDn EmptyInstance;

private:
	X509_NAME * m_dn;
	unsigned long m_profileId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PROFILE_REJECT
*/
class ProfileReject : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ProfileReject();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ProfileReject(const ProfileReject & other);

	/*! \brief This is the destructor.
	 */
	~ProfileReject();


	/*! \brief This function converts a PROFILE_REJECT to a ProfileReject.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PROFILE_REJECT * Datas);

	/*! \brief This function converts a ProfileReject to a PROFILE_REJECT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PROFILE_REJECT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PROFILE_REJECT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::profile_id.
	 *  \return The value of ASN1_INTEGER::profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function sets ASN1_UTF8STRING::reason.
	 *  \param c_reason [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_reason(const mString & c_reason);

	/*! \brief This function returns ASN1_UTF8STRING::reason.
	 *  \return The value of ASN1_UTF8STRING::reason.
	 */
	const mString & get_reason() const;

	/*! \brief This function returns ASN1_UTF8STRING::reason.
	 *  \return The value of ASN1_UTF8STRING::reason.
	 */
	mString & get_reason();


	/*! \brief This function copies a ProfileReject to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ProfileReject & other);
	/*! \brief This member is an empty instance of ProfileReject.
	 */
	static ProfileReject EmptyInstance;

private:
	unsigned long m_profileId;
	mString m_reason;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of WEBUSER_REVOKE_CERT
*/
class WebuserRevokeCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	WebuserRevokeCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	WebuserRevokeCert(const WebuserRevokeCert & other);

	/*! \brief This is the destructor.
	 */
	~WebuserRevokeCert();


	/*! \brief This function converts a WEBUSER_REVOKE_CERT to a WebuserRevokeCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const WEBUSER_REVOKE_CERT * Datas);

	/*! \brief This function converts a WebuserRevokeCert to a WEBUSER_REVOKE_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(WEBUSER_REVOKE_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of WEBUSER_REVOKE_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::cert_id.
	 *  \param c_certId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certId(unsigned long c_certId);

	/*! \brief This function returns ASN1_INTEGER::cert_id.
	 *  \return The value of ASN1_INTEGER::cert_id.
	 */
	unsigned long get_certId() const;


	/*! \brief This function sets ASN1_INTEGER::user_id.
	 *  \param c_userId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userId(unsigned long c_userId);

	/*! \brief This function returns ASN1_INTEGER::user_id.
	 *  \return The value of ASN1_INTEGER::user_id.
	 */
	unsigned long get_userId() const;


	/*! \brief This function copies a WebuserRevokeCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const WebuserRevokeCert & other);
	/*! \brief This member is an empty instance of WebuserRevokeCert.
	 */
	static WebuserRevokeCert EmptyInstance;

private:
	unsigned long m_certId;
	unsigned long m_userId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_REQUEST_BODY
*/
class AdminRequestBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminRequestBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminRequestBody(const AdminRequestBody & other);

	/*! \brief This is the destructor.
	 */
	~AdminRequestBody();


	/*! \brief This function converts a ADMIN_REQUEST_BODY to a AdminRequestBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_REQUEST_BODY * Datas);

	/*! \brief This function converts a AdminRequestBody to a ADMIN_REQUEST_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_REQUEST_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_REQUEST_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets MAIL_DATAS::d.admin_mail.
	 *  \param c_adminMail [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_adminMail(const MailDatas & c_adminMail);

	/*! \brief This function returns MAIL_DATAS::d.admin_mail.
	 *  \return The value of MAIL_DATAS::d.admin_mail.
	 */
	const MailDatas & get_adminMail() const;

	/*! \brief This function returns MAIL_DATAS::d.admin_mail.
	 *  \return The value of MAIL_DATAS::d.admin_mail.
	 */
	MailDatas & get_adminMail();


	/*! \brief This function sets NEWPKI_REQUEST_CERT::d.cert_request.
	 *  \param c_certRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certRequest(const NewpkiRequestCert & c_certRequest);

	/*! \brief This function returns NEWPKI_REQUEST_CERT::d.cert_request.
	 *  \return The value of NEWPKI_REQUEST_CERT::d.cert_request.
	 */
	const NewpkiRequestCert & get_certRequest() const;

	/*! \brief This function returns NEWPKI_REQUEST_CERT::d.cert_request.
	 *  \return The value of NEWPKI_REQUEST_CERT::d.cert_request.
	 */
	NewpkiRequestCert & get_certRequest();


	/*! \brief This function sets CHANGE_PASSWD::d.chg_pwd.
	 *  \param c_chgPwd [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_chgPwd(const ChangePasswd & c_chgPwd);

	/*! \brief This function returns CHANGE_PASSWD::d.chg_pwd.
	 *  \return The value of CHANGE_PASSWD::d.chg_pwd.
	 */
	const ChangePasswd & get_chgPwd() const;

	/*! \brief This function returns CHANGE_PASSWD::d.chg_pwd.
	 *  \return The value of CHANGE_PASSWD::d.chg_pwd.
	 */
	ChangePasswd & get_chgPwd();


	/*! \brief This function sets CHANGE_USER_PASSWD::d.chg_user_pwd.
	 *  \param c_chgUserPwd [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_chgUserPwd(const ChangeUserPasswd & c_chgUserPwd);

	/*! \brief This function returns CHANGE_USER_PASSWD::d.chg_user_pwd.
	 *  \return The value of CHANGE_USER_PASSWD::d.chg_user_pwd.
	 */
	const ChangeUserPasswd & get_chgUserPwd() const;

	/*! \brief This function returns CHANGE_USER_PASSWD::d.chg_user_pwd.
	 *  \return The value of CHANGE_USER_PASSWD::d.chg_user_pwd.
	 */
	ChangeUserPasswd & get_chgUserPwd();


	/*! \brief This function sets EXPORTED_PKI_CONF::d.conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const ExportedPkiConf & c_conf);

	/*! \brief This function returns EXPORTED_PKI_CONF::d.conf.
	 *  \return The value of EXPORTED_PKI_CONF::d.conf.
	 */
	const ExportedPkiConf & get_conf() const;

	/*! \brief This function returns EXPORTED_PKI_CONF::d.conf.
	 *  \return The value of EXPORTED_PKI_CONF::d.conf.
	 */
	ExportedPkiConf & get_conf();


	/*! \brief This function sets ENTITY_CREATION_REQ::d.cre_entity.
	 *  \param c_creEntity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_creEntity(const EntityCreationReq & c_creEntity);

	/*! \brief This function returns ENTITY_CREATION_REQ::d.cre_entity.
	 *  \return The value of ENTITY_CREATION_REQ::d.cre_entity.
	 */
	const EntityCreationReq & get_creEntity() const;

	/*! \brief This function returns ENTITY_CREATION_REQ::d.cre_entity.
	 *  \return The value of ENTITY_CREATION_REQ::d.cre_entity.
	 */
	EntityCreationReq & get_creEntity();


	/*! \brief This function sets USER_INFO::d.cre_user.
	 *  \param c_creUser [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_creUser(const UserInfo & c_creUser);

	/*! \brief This function returns USER_INFO::d.cre_user.
	 *  \return The value of USER_INFO::d.cre_user.
	 */
	const UserInfo & get_creUser() const;

	/*! \brief This function returns USER_INFO::d.cre_user.
	 *  \return The value of USER_INFO::d.cre_user.
	 */
	UserInfo & get_creUser();


	/*! \brief This function sets REQ_CREATE_CHILD_CA::d.create_child_ca.
	 *  \param c_createChildCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_createChildCa(const ReqCreateChildCa & c_createChildCa);

	/*! \brief This function returns REQ_CREATE_CHILD_CA::d.create_child_ca.
	 *  \return The value of REQ_CREATE_CHILD_CA::d.create_child_ca.
	 */
	const ReqCreateChildCa & get_createChildCa() const;

	/*! \brief This function returns REQ_CREATE_CHILD_CA::d.create_child_ca.
	 *  \return The value of REQ_CREATE_CHILD_CA::d.create_child_ca.
	 */
	ReqCreateChildCa & get_createChildCa();


	/*! \brief This function sets CREATE_PKI_USER_REQUEST::d.create_pki_user.
	 *  \param c_createPkiUser [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_createPkiUser(const CreatePkiUserRequest & c_createPkiUser);

	/*! \brief This function returns CREATE_PKI_USER_REQUEST::d.create_pki_user.
	 *  \return The value of CREATE_PKI_USER_REQUEST::d.create_pki_user.
	 */
	const CreatePkiUserRequest & get_createPkiUser() const;

	/*! \brief This function returns CREATE_PKI_USER_REQUEST::d.create_pki_user.
	 *  \return The value of CREATE_PKI_USER_REQUEST::d.create_pki_user.
	 */
	CreatePkiUserRequest & get_createPkiUser();


	/*! \brief This function sets REQ_CREATE_ROOT_CA::d.create_root_ca.
	 *  \param c_createRootCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_createRootCa(const ReqCreateRootCa & c_createRootCa);

	/*! \brief This function returns REQ_CREATE_ROOT_CA::d.create_root_ca.
	 *  \return The value of REQ_CREATE_ROOT_CA::d.create_root_ca.
	 */
	const ReqCreateRootCa & get_createRootCa() const;

	/*! \brief This function returns REQ_CREATE_ROOT_CA::d.create_root_ca.
	 *  \return The value of REQ_CREATE_ROOT_CA::d.create_root_ca.
	 */
	ReqCreateRootCa & get_createRootCa();


	/*! \brief This function sets ASN1_UTF8STRING::d.dn_id.
	 *  \param c_dnId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnId(const mString & c_dnId);

	/*! \brief This function returns ASN1_UTF8STRING::d.dn_id.
	 *  \return The value of ASN1_UTF8STRING::d.dn_id.
	 */
	const mString & get_dnId() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.dn_id.
	 *  \return The value of ASN1_UTF8STRING::d.dn_id.
	 */
	mString & get_dnId();


	/*! \brief This function sets DN_VALIDATION::d.dn_val.
	 *  \param c_dnVal [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnVal(const DnValidation & c_dnVal);

	/*! \brief This function returns DN_VALIDATION::d.dn_val.
	 *  \return The value of DN_VALIDATION::d.dn_val.
	 */
	const DnValidation & get_dnVal() const;

	/*! \brief This function returns DN_VALIDATION::d.dn_val.
	 *  \return The value of DN_VALIDATION::d.dn_val.
	 */
	DnValidation & get_dnVal();


	/*! \brief This function sets NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \param c_eeCertRequest [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertRequest(const NewpkiEeRequestCert & c_eeCertRequest);

	/*! \brief This function returns NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \return The value of NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 */
	const NewpkiEeRequestCert & get_eeCertRequest() const;

	/*! \brief This function returns NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 *  \return The value of NEWPKI_EE_REQUEST_CERT::d.ee_cert_request.
	 */
	NewpkiEeRequestCert & get_eeCertRequest();


	/*! \brief This function sets WEBUSER_REVOKE_CERT::d.ee_cert_revoke.
	 *  \param c_eeCertRevoke [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeCertRevoke(const WebuserRevokeCert & c_eeCertRevoke);

	/*! \brief This function returns WEBUSER_REVOKE_CERT::d.ee_cert_revoke.
	 *  \return The value of WEBUSER_REVOKE_CERT::d.ee_cert_revoke.
	 */
	const WebuserRevokeCert & get_eeCertRevoke() const;

	/*! \brief This function returns WEBUSER_REVOKE_CERT::d.ee_cert_revoke.
	 *  \return The value of WEBUSER_REVOKE_CERT::d.ee_cert_revoke.
	 */
	WebuserRevokeCert & get_eeCertRevoke();


	/*! \brief This function sets ENTITY_LINKS::d.entities_links.
	 *  \param c_entitiesLinks [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitiesLinks(const mVector< EntityLinks > & c_entitiesLinks);

	/*! \brief This function returns ENTITY_LINKS::d.entities_links.
	 *  \return The value of ENTITY_LINKS::d.entities_links.
	 */
	const mVector< EntityLinks > & get_entitiesLinks() const;

	/*! \brief This function returns ENTITY_LINKS::d.entities_links.
	 *  \return The value of ENTITY_LINKS::d.entities_links.
	 */
	mVector< EntityLinks > & get_entitiesLinks();


	/*! \brief This function sets S_ENTITY_ACL::d.entity_acl.
	 *  \param c_entityAcl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityAcl(const SEntityAcl & c_entityAcl);

	/*! \brief This function returns S_ENTITY_ACL::d.entity_acl.
	 *  \return The value of S_ENTITY_ACL::d.entity_acl.
	 */
	const SEntityAcl & get_entityAcl() const;

	/*! \brief This function returns S_ENTITY_ACL::d.entity_acl.
	 *  \return The value of S_ENTITY_ACL::d.entity_acl.
	 */
	SEntityAcl & get_entityAcl();


	/*! \brief This function sets S_ENTITY_AUDITS::d.entity_audits.
	 *  \param c_entityAudits [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityAudits(const SEntityAudits & c_entityAudits);

	/*! \brief This function returns S_ENTITY_AUDITS::d.entity_audits.
	 *  \return The value of S_ENTITY_AUDITS::d.entity_audits.
	 */
	const SEntityAudits & get_entityAudits() const;

	/*! \brief This function returns S_ENTITY_AUDITS::d.entity_audits.
	 *  \return The value of S_ENTITY_AUDITS::d.entity_audits.
	 */
	SEntityAudits & get_entityAudits();


	/*! \brief This function sets X509::d.entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const PKI_CERT & c_entityCert);

	/*! \brief This function returns X509::d.entity_cert.
	 *  \return The value of X509::d.entity_cert.
	 */
	const PKI_CERT & get_entityCert() const;

	/*! \brief This function returns X509::d.entity_cert.
	 *  \return The value of X509::d.entity_cert.
	 */
	PKI_CERT & get_entityCert();


	/*! \brief This function sets S_ENTITY_CONF::d.entity_conf.
	 *  \param c_entityConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityConf(const SEntityConf & c_entityConf);

	/*! \brief This function returns S_ENTITY_CONF::d.entity_conf.
	 *  \return The value of S_ENTITY_CONF::d.entity_conf.
	 */
	const SEntityConf & get_entityConf() const;

	/*! \brief This function returns S_ENTITY_CONF::d.entity_conf.
	 *  \return The value of S_ENTITY_CONF::d.entity_conf.
	 */
	SEntityConf & get_entityConf();


	/*! \brief This function sets S_ENTITY_MAIL_CONF::d.entity_mail_conf.
	 *  \param c_entityMailConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityMailConf(const SEntityMailConf & c_entityMailConf);

	/*! \brief This function returns S_ENTITY_MAIL_CONF::d.entity_mail_conf.
	 *  \return The value of S_ENTITY_MAIL_CONF::d.entity_mail_conf.
	 */
	const SEntityMailConf & get_entityMailConf() const;

	/*! \brief This function returns S_ENTITY_MAIL_CONF::d.entity_mail_conf.
	 *  \return The value of S_ENTITY_MAIL_CONF::d.entity_mail_conf.
	 */
	SEntityMailConf & get_entityMailConf();


	/*! \brief This function sets ASN1_UTF8STRING::d.entity_name.
	 *  \param c_entityName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityName(const mString & c_entityName);

	/*! \brief This function returns ASN1_UTF8STRING::d.entity_name.
	 *  \return The value of ASN1_UTF8STRING::d.entity_name.
	 */
	const mString & get_entityName() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.entity_name.
	 *  \return The value of ASN1_UTF8STRING::d.entity_name.
	 */
	mString & get_entityName();


	/*! \brief This function sets ASN1_INTEGER::d.entity_type.
	 *  \param c_entityType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityType(unsigned long c_entityType);

	/*! \brief This function returns ASN1_INTEGER::d.entity_type.
	 *  \return The value of ASN1_INTEGER::d.entity_type.
	 */
	unsigned long get_entityType() const;


	/*! \brief This function sets ADMIN_REQ_ENUM_LOGS::d.enum_logs.
	 *  \param c_enumLogs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_enumLogs(const AdminReqEnumLogs & c_enumLogs);

	/*! \brief This function returns ADMIN_REQ_ENUM_LOGS::d.enum_logs.
	 *  \return The value of ADMIN_REQ_ENUM_LOGS::d.enum_logs.
	 */
	const AdminReqEnumLogs & get_enumLogs() const;

	/*! \brief This function returns ADMIN_REQ_ENUM_LOGS::d.enum_logs.
	 *  \return The value of ADMIN_REQ_ENUM_LOGS::d.enum_logs.
	 */
	AdminReqEnumLogs & get_enumLogs();


	/*! \brief This function sets OBJECTS_ENUM::d.enum_objects.
	 *  \param c_enumObjects [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_enumObjects(const ObjectsEnum & c_enumObjects);

	/*! \brief This function returns OBJECTS_ENUM::d.enum_objects.
	 *  \return The value of OBJECTS_ENUM::d.enum_objects.
	 */
	const ObjectsEnum & get_enumObjects() const;

	/*! \brief This function returns OBJECTS_ENUM::d.enum_objects.
	 *  \return The value of OBJECTS_ENUM::d.enum_objects.
	 */
	ObjectsEnum & get_enumObjects();


	/*! \brief This function sets ASN1_INTEGER::d.group_id.
	 *  \param c_groupId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groupId(unsigned long c_groupId);

	/*! \brief This function returns ASN1_INTEGER::d.group_id.
	 *  \return The value of ASN1_INTEGER::d.group_id.
	 */
	unsigned long get_groupId() const;


	/*! \brief This function sets ASN1_UTF8STRING::d.group_name.
	 *  \param c_groupName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groupName(const mString & c_groupName);

	/*! \brief This function returns ASN1_UTF8STRING::d.group_name.
	 *  \return The value of ASN1_UTF8STRING::d.group_name.
	 */
	const mString & get_groupName() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.group_name.
	 *  \return The value of ASN1_UTF8STRING::d.group_name.
	 */
	mString & get_groupName();


	/*! \brief This function sets USERS_GROUP::d.groups.
	 *  \param c_groups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groups(const mVector< UsersGroup > & c_groups);

	/*! \brief This function returns USERS_GROUP::d.groups.
	 *  \return The value of USERS_GROUP::d.groups.
	 */
	const mVector< UsersGroup > & get_groups() const;

	/*! \brief This function returns USERS_GROUP::d.groups.
	 *  \return The value of USERS_GROUP::d.groups.
	 */
	mVector< UsersGroup > & get_groups();


	/*! \brief This function sets ENTITY_INIT_REQ::d.init_entity.
	 *  \param c_initEntity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_initEntity(const EntityInitReq & c_initEntity);

	/*! \brief This function returns ENTITY_INIT_REQ::d.init_entity.
	 *  \return The value of ENTITY_INIT_REQ::d.init_entity.
	 */
	const EntityInitReq & get_initEntity() const;

	/*! \brief This function returns ENTITY_INIT_REQ::d.init_entity.
	 *  \return The value of ENTITY_INIT_REQ::d.init_entity.
	 */
	EntityInitReq & get_initEntity();


	/*! \brief This function sets OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \param c_knownObjects [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_knownObjects(const ObjectsListSynchro & c_knownObjects);

	/*! \brief This function returns OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \return The value of OBJECTS_LIST_SYNCHRO::d.known_objects.
	 */
	const ObjectsListSynchro & get_knownObjects() const;

	/*! \brief This function returns OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \return The value of OBJECTS_LIST_SYNCHRO::d.known_objects.
	 */
	ObjectsListSynchro & get_knownObjects();


	/*! \brief This function sets ASN1_UTF8STRING::d.ldap_search.
	 *  \param c_ldapSearch [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapSearch(const mString & c_ldapSearch);

	/*! \brief This function returns ASN1_UTF8STRING::d.ldap_search.
	 *  \return The value of ASN1_UTF8STRING::d.ldap_search.
	 */
	const mString & get_ldapSearch() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.ldap_search.
	 *  \return The value of ASN1_UTF8STRING::d.ldap_search.
	 */
	mString & get_ldapSearch();


	/*! \brief This function sets ADMIN_REQ_LOGIN::d.login.
	 *  \param c_login [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_login(const AdminReqLogin & c_login);

	/*! \brief This function returns ADMIN_REQ_LOGIN::d.login.
	 *  \return The value of ADMIN_REQ_LOGIN::d.login.
	 */
	const AdminReqLogin & get_login() const;

	/*! \brief This function returns ADMIN_REQ_LOGIN::d.login.
	 *  \return The value of ADMIN_REQ_LOGIN::d.login.
	 */
	AdminReqLogin & get_login();


	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::d.new_req.
	 *  \param c_newReq [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_newReq(const CryptedNewpkiRequest & c_newReq);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::d.new_req.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::d.new_req.
	 */
	const CryptedNewpkiRequest & get_newReq() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::d.new_req.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::d.new_req.
	 */
	CryptedNewpkiRequest & get_newReq();


	/*! \brief This function sets CRYPTED_NEWPKI_RESPONSE::d.new_resp.
	 *  \param c_newResp [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_newResp(const CryptedNewpkiResponse & c_newResp);

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::d.new_resp.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::d.new_resp.
	 */
	const CryptedNewpkiResponse & get_newResp() const;

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSE::d.new_resp.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSE::d.new_resp.
	 */
	CryptedNewpkiResponse & get_newResp();


	/*! \brief This function sets ASN1_INTEGER::d.offline.
	 *  \param c_offline [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_offline(unsigned long c_offline);

	/*! \brief This function returns ASN1_INTEGER::d.offline.
	 *  \return The value of ASN1_INTEGER::d.offline.
	 */
	unsigned long get_offline() const;


	/*! \brief This function sets ASN1_NULL::d.other.
	 *  \param c_other [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_other(const ASN1_NULL * c_other);

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	const ASN1_NULL * get_other() const;

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	ASN1_NULL * get_other();


	/*! \brief This function sets CA_P12_IMPORT::d.p12_import.
	 *  \param c_p12Import [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12Import(const CaP12Import & c_p12Import);

	/*! \brief This function returns CA_P12_IMPORT::d.p12_import.
	 *  \return The value of CA_P12_IMPORT::d.p12_import.
	 */
	const CaP12Import & get_p12Import() const;

	/*! \brief This function returns CA_P12_IMPORT::d.p12_import.
	 *  \return The value of CA_P12_IMPORT::d.p12_import.
	 */
	CaP12Import & get_p12Import();


	/*! \brief This function sets PKCS7::d.p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function sets NEWPKI_PROFILE::d.profile.
	 *  \param c_profile [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profile(const NewpkiProfile & c_profile);

	/*! \brief This function returns NEWPKI_PROFILE::d.profile.
	 *  \return The value of NEWPKI_PROFILE::d.profile.
	 */
	const NewpkiProfile & get_profile() const;

	/*! \brief This function returns NEWPKI_PROFILE::d.profile.
	 *  \return The value of NEWPKI_PROFILE::d.profile.
	 */
	NewpkiProfile & get_profile();


	/*! \brief This function sets PROFILE_CHANGE_DN::d.profile_dn.
	 *  \param c_profileDn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileDn(const ProfileChangeDn & c_profileDn);

	/*! \brief This function returns PROFILE_CHANGE_DN::d.profile_dn.
	 *  \return The value of PROFILE_CHANGE_DN::d.profile_dn.
	 */
	const ProfileChangeDn & get_profileDn() const;

	/*! \brief This function returns PROFILE_CHANGE_DN::d.profile_dn.
	 *  \return The value of PROFILE_CHANGE_DN::d.profile_dn.
	 */
	ProfileChangeDn & get_profileDn();


	/*! \brief This function sets ASN1_INTEGER::d.profile_id.
	 *  \param c_profileId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileId(unsigned long c_profileId);

	/*! \brief This function returns ASN1_INTEGER::d.profile_id.
	 *  \return The value of ASN1_INTEGER::d.profile_id.
	 */
	unsigned long get_profileId() const;


	/*! \brief This function sets PROFILE_CHANGE_OWNER::d.profile_owner.
	 *  \param c_profileOwner [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileOwner(const ProfileChangeOwner & c_profileOwner);

	/*! \brief This function returns PROFILE_CHANGE_OWNER::d.profile_owner.
	 *  \return The value of PROFILE_CHANGE_OWNER::d.profile_owner.
	 */
	const ProfileChangeOwner & get_profileOwner() const;

	/*! \brief This function returns PROFILE_CHANGE_OWNER::d.profile_owner.
	 *  \return The value of PROFILE_CHANGE_OWNER::d.profile_owner.
	 */
	ProfileChangeOwner & get_profileOwner();


	/*! \brief This function sets PROFILE_CHANGE_UID::d.profile_uid.
	 *  \param c_profileUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profileUid(const ProfileChangeUid & c_profileUid);

	/*! \brief This function returns PROFILE_CHANGE_UID::d.profile_uid.
	 *  \return The value of PROFILE_CHANGE_UID::d.profile_uid.
	 */
	const ProfileChangeUid & get_profileUid() const;

	/*! \brief This function returns PROFILE_CHANGE_UID::d.profile_uid.
	 *  \return The value of PROFILE_CHANGE_UID::d.profile_uid.
	 */
	ProfileChangeUid & get_profileUid();


	/*! \brief This function sets PROFILE_REJECT::d.reject_profile.
	 *  \param c_rejectProfile [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rejectProfile(const ProfileReject & c_rejectProfile);

	/*! \brief This function returns PROFILE_REJECT::d.reject_profile.
	 *  \return The value of PROFILE_REJECT::d.reject_profile.
	 */
	const ProfileReject & get_rejectProfile() const;

	/*! \brief This function returns PROFILE_REJECT::d.reject_profile.
	 *  \return The value of PROFILE_REJECT::d.reject_profile.
	 */
	ProfileReject & get_rejectProfile();


	/*! \brief This function sets S_RENAME_GROUP::d.rename_group.
	 *  \param c_renameGroup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_renameGroup(const SRenameGroup & c_renameGroup);

	/*! \brief This function returns S_RENAME_GROUP::d.rename_group.
	 *  \return The value of S_RENAME_GROUP::d.rename_group.
	 */
	const SRenameGroup & get_renameGroup() const;

	/*! \brief This function returns S_RENAME_GROUP::d.rename_group.
	 *  \return The value of S_RENAME_GROUP::d.rename_group.
	 */
	SRenameGroup & get_renameGroup();


	/*! \brief This function sets REP_ENTRY_INFO::d.repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::d.repositories.
	 *  \return The value of REP_ENTRY_INFO::d.repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::d.repositories.
	 *  \return The value of REP_ENTRY_INFO::d.repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets ASN1_INTEGER::d.request_id.
	 *  \param c_requestId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_requestId(unsigned long c_requestId);

	/*! \brief This function returns ASN1_INTEGER::d.request_id.
	 *  \return The value of ASN1_INTEGER::d.request_id.
	 */
	unsigned long get_requestId() const;


	/*! \brief This function sets ASN1_UTF8STRING::d.resource_name.
	 *  \param c_resourceName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_resourceName(const mString & c_resourceName);

	/*! \brief This function returns ASN1_UTF8STRING::d.resource_name.
	 *  \return The value of ASN1_UTF8STRING::d.resource_name.
	 */
	const mString & get_resourceName() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.resource_name.
	 *  \return The value of ASN1_UTF8STRING::d.resource_name.
	 */
	mString & get_resourceName();


	/*! \brief This function sets ASN1_INTEGER::d.serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::d.serial.
	 *  \return The value of ASN1_INTEGER::d.serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function sets SIGN_CSR::d.sign_csr.
	 *  \param c_signCsr [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signCsr(const SignCsr & c_signCsr);

	/*! \brief This function returns SIGN_CSR::d.sign_csr.
	 *  \return The value of SIGN_CSR::d.sign_csr.
	 */
	const SignCsr & get_signCsr() const;

	/*! \brief This function returns SIGN_CSR::d.sign_csr.
	 *  \return The value of SIGN_CSR::d.sign_csr.
	 */
	SignCsr & get_signCsr();


	/*! \brief This function sets ENTITY_SIGNATURE_REQ::d.sign_entity.
	 *  \param c_signEntity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signEntity(const EntitySignatureReq & c_signEntity);

	/*! \brief This function returns ENTITY_SIGNATURE_REQ::d.sign_entity.
	 *  \return The value of ENTITY_SIGNATURE_REQ::d.sign_entity.
	 */
	const EntitySignatureReq & get_signEntity() const;

	/*! \brief This function returns ENTITY_SIGNATURE_REQ::d.sign_entity.
	 *  \return The value of ENTITY_SIGNATURE_REQ::d.sign_entity.
	 */
	EntitySignatureReq & get_signEntity();


	/*! \brief This function sets ASN1_OCTET_STRING::d.transaction_id.
	 *  \param c_transactionId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionId(const Asn1OctetString & c_transactionId);

	/*! \brief This function returns ASN1_OCTET_STRING::d.transaction_id.
	 *  \return The value of ASN1_OCTET_STRING::d.transaction_id.
	 */
	const Asn1OctetString & get_transactionId() const;

	/*! \brief This function returns ASN1_OCTET_STRING::d.transaction_id.
	 *  \return The value of ASN1_OCTET_STRING::d.transaction_id.
	 */
	Asn1OctetString & get_transactionId();


	/*! \brief This function sets TRANSACTION_IDS::d.transactions_ids.
	 *  \param c_transactionsIds [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionsIds(const TransactionIds & c_transactionsIds);

	/*! \brief This function returns TRANSACTION_IDS::d.transactions_ids.
	 *  \return The value of TRANSACTION_IDS::d.transactions_ids.
	 */
	const TransactionIds & get_transactionsIds() const;

	/*! \brief This function returns TRANSACTION_IDS::d.transactions_ids.
	 *  \return The value of TRANSACTION_IDS::d.transactions_ids.
	 */
	TransactionIds & get_transactionsIds();


	/*! \brief This function sets USER_ENTRY::d.upd_user.
	 *  \param c_updUser [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_updUser(const UserEntry & c_updUser);

	/*! \brief This function returns USER_ENTRY::d.upd_user.
	 *  \return The value of USER_ENTRY::d.upd_user.
	 */
	const UserEntry & get_updUser() const;

	/*! \brief This function returns USER_ENTRY::d.upd_user.
	 *  \return The value of USER_ENTRY::d.upd_user.
	 */
	UserEntry & get_updUser();


	/*! \brief This function sets WAITING_NEWPKI_OBJECT::d.waiting_obj.
	 *  \param c_waitingObj [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_waitingObj(const WaitingNewpkiObject & c_waitingObj);

	/*! \brief This function returns WAITING_NEWPKI_OBJECT::d.waiting_obj.
	 *  \return The value of WAITING_NEWPKI_OBJECT::d.waiting_obj.
	 */
	const WaitingNewpkiObject & get_waitingObj() const;

	/*! \brief This function returns WAITING_NEWPKI_OBJECT::d.waiting_obj.
	 *  \return The value of WAITING_NEWPKI_OBJECT::d.waiting_obj.
	 */
	WaitingNewpkiObject & get_waitingObj();


	/*! \brief This function sets WEBUSER_CHANGE_PASSWD::d.web_chg_passwd.
	 *  \param c_webChgPasswd [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_webChgPasswd(const WebuserChangePasswd & c_webChgPasswd);

	/*! \brief This function returns WEBUSER_CHANGE_PASSWD::d.web_chg_passwd.
	 *  \return The value of WEBUSER_CHANGE_PASSWD::d.web_chg_passwd.
	 */
	const WebuserChangePasswd & get_webChgPasswd() const;

	/*! \brief This function returns WEBUSER_CHANGE_PASSWD::d.web_chg_passwd.
	 *  \return The value of WEBUSER_CHANGE_PASSWD::d.web_chg_passwd.
	 */
	WebuserChangePasswd & get_webChgPasswd();


	/*! \brief This function sets ASN1_UTF8STRING::d.webactivate.
	 *  \param c_webactivate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_webactivate(const mString & c_webactivate);

	/*! \brief This function returns ASN1_UTF8STRING::d.webactivate.
	 *  \return The value of ASN1_UTF8STRING::d.webactivate.
	 */
	const mString & get_webactivate() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.webactivate.
	 *  \return The value of ASN1_UTF8STRING::d.webactivate.
	 */
	mString & get_webactivate();


	/*! \brief This function sets WEB_REQ_CREATE::d.webcreate.
	 *  \param c_webcreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_webcreate(const WebReqCreate & c_webcreate);

	/*! \brief This function returns WEB_REQ_CREATE::d.webcreate.
	 *  \return The value of WEB_REQ_CREATE::d.webcreate.
	 */
	const WebReqCreate & get_webcreate() const;

	/*! \brief This function returns WEB_REQ_CREATE::d.webcreate.
	 *  \return The value of WEB_REQ_CREATE::d.webcreate.
	 */
	WebReqCreate & get_webcreate();


	/*! \brief This function sets WEB_REQ_LOGIN::d.weblogin.
	 *  \param c_weblogin [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_weblogin(const WebReqLogin & c_weblogin);

	/*! \brief This function returns WEB_REQ_LOGIN::d.weblogin.
	 *  \return The value of WEB_REQ_LOGIN::d.weblogin.
	 */
	const WebReqLogin & get_weblogin() const;

	/*! \brief This function returns WEB_REQ_LOGIN::d.weblogin.
	 *  \return The value of WEB_REQ_LOGIN::d.weblogin.
	 */
	WebReqLogin & get_weblogin();


	/*! \brief This function copies a AdminRequestBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminRequestBody & other);
	/*! \brief This member is an empty instance of AdminRequestBody.
	 */
	static AdminRequestBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	MailDatas * m_adminMail;
	NewpkiRequestCert * m_certRequest;
	ChangePasswd * m_chgPwd;
	ChangeUserPasswd * m_chgUserPwd;
	ExportedPkiConf * m_conf;
	EntityCreationReq * m_creEntity;
	UserInfo * m_creUser;
	ReqCreateChildCa * m_createChildCa;
	CreatePkiUserRequest * m_createPkiUser;
	ReqCreateRootCa * m_createRootCa;
	mString * m_dnId;
	DnValidation * m_dnVal;
	NewpkiEeRequestCert * m_eeCertRequest;
	WebuserRevokeCert * m_eeCertRevoke;
	mVector< EntityLinks > * m_entitiesLinks;
	SEntityAcl * m_entityAcl;
	SEntityAudits * m_entityAudits;
	PKI_CERT * m_entityCert;
	SEntityConf * m_entityConf;
	SEntityMailConf * m_entityMailConf;
	mString * m_entityName;
	unsigned long * m_entityType;
	AdminReqEnumLogs * m_enumLogs;
	ObjectsEnum * m_enumObjects;
	unsigned long * m_groupId;
	mString * m_groupName;
	mVector< UsersGroup > * m_groups;
	EntityInitReq * m_initEntity;
	ObjectsListSynchro * m_knownObjects;
	mString * m_ldapSearch;
	AdminReqLogin * m_login;
	CryptedNewpkiRequest * m_newReq;
	CryptedNewpkiResponse * m_newResp;
	unsigned long * m_offline;
	ASN1_NULL * m_other;
	CaP12Import * m_p12Import;
	PKI_P7B * m_p7b;
	NewpkiProfile * m_profile;
	ProfileChangeDn * m_profileDn;
	unsigned long * m_profileId;
	ProfileChangeOwner * m_profileOwner;
	ProfileChangeUid * m_profileUid;
	ProfileReject * m_rejectProfile;
	SRenameGroup * m_renameGroup;
	mVector< RepEntryInfo > * m_repositories;
	unsigned long * m_requestId;
	mString * m_resourceName;
	unsigned long * m_serial;
	SignCsr * m_signCsr;
	EntitySignatureReq * m_signEntity;
	Asn1OctetString * m_transactionId;
	TransactionIds * m_transactionsIds;
	UserEntry * m_updUser;
	WaitingNewpkiObject * m_waitingObj;
	WebuserChangePasswd * m_webChgPasswd;
	mString * m_webactivate;
	WebReqCreate * m_webcreate;
	WebReqLogin * m_weblogin;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_REQUEST
*/
class AdminRequest : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminRequest();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminRequest(const AdminRequest & other);

	/*! \brief This is the destructor.
	 */
	~AdminRequest();


	/*! \brief This function converts a ADMIN_REQUEST to a AdminRequest.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_REQUEST * Datas);

	/*! \brief This function converts a AdminRequest to a ADMIN_REQUEST.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_REQUEST ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_REQUEST.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ADMIN_REQUEST_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const AdminRequestBody & c_body);

	/*! \brief This function returns ADMIN_REQUEST_BODY::body.
	 *  \return The value of ADMIN_REQUEST_BODY::body.
	 */
	const AdminRequestBody & get_body() const;

	/*! \brief This function returns ADMIN_REQUEST_BODY::body.
	 *  \return The value of ADMIN_REQUEST_BODY::body.
	 */
	AdminRequestBody & get_body();


	/*! \brief This function copies a AdminRequest to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminRequest & other);
	/*! \brief This member is an empty instance of AdminRequest.
	 */
	static AdminRequest EmptyInstance;

private:
	AdminRequestBody m_body;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASN1REQ_H
