#pragma interface
#ifndef DIALOG_H
#define DIALOG_H

#include <curses.h>
#include <stdarg.h>

#include "dialog_def.h"

#ifndef MISC_H
	#include <misc.h>
#endif
	
class BUTTONS_INFO;
class SSTRINGS;
class HELP_FILE;

#include "private_msg.h"

class FIELD_MSG{
public:
	char is_loaded;		// Is there a message to process
	void *key;			// Generally a pointer to a common
						// data used by field to talk to each other
	int  int_val;		// Value of the message
						// It is expected to see other type
						// to communicate more complex messages
						// between fields.
	/*~PROTOBEG~ FIELD_MSG */
public:
	FIELD_MSG (void);
	/*~PROTOEND~ FIELD_MSG */
};

class FIELD: public ARRAY_OBJ{
protected:
	bool readonly;		// This field can't be modified
	bool maybeempty;
	bool may_select;	// For Menu field
	bool is_head;		// Is a heading
	friend class DIALOG;
	friend class DIALOG_LISTE;
	friend class DIALOG_RECORDS;
	friend class DIALOG_MENU;
	char *prompt;
	int  vsize;		// Number of lines used by the field (normally 1)
					// may be 0
	bool donotcheckold;	// In HTML mode, do not check if the current
						// value of the field match the hidden (old) value
						// Used for the clock dialog
	PRIVATE_MESSAGE *msg;	// Used to trigger help dialog on that field
	const char *original_prompt;	// Use for the variable system
	const char *regkey;
	SSTRING guiparms;	// Extra GUI parameters
						// Set with DIALOG::set_guiparms();
	SSTRING guipath;	// Relative path of the FIELD in the dialog
						// without its ID.
	SSTRING guipath_sufix;
public:
	struct {
		int x;	// Horizontal position of the input boxes
		int width;
		int y;	// Vertical coordinate of the input boxes
	} box;
public:		     	
	virtual void drawtxt(WINDOW *, int offset)=0;
	virtual MENU_STATUS dokey(WINDOW *, int key, FIELD_MSG &msg, bool &grab)=0;
	virtual void save()=0;
	virtual void restore()=0;
	virtual void reload(const char *dianame, int nof)=0;
	virtual void html_draw(int nofield)=0;
	virtual void gui_draw(int nofield, SSTRINGS &subs)=0;
	virtual MENU_STATUS gui_get(int nofield, const char *fid, const char *actionid)=0;
	virtual char getidprefix ()=0;
	virtual int html_validate(int nofield)=0;
	virtual const char *get_registry_value()=0;
	virtual void set_registry_value (const char *)=0;
	/*~PROTOBEG~ FIELD */
public:
	FIELD (const char *_prompt);
	void draw (WINDOW *dialog, int offset);
	virtual void format_htmlkey (char *key, int nof);
protected:
	const char *formatpath (char tmp[1000],
		 const char *diapath);
public:
	virtual const char *get_registry_key (void);
	virtual bool getflags (unsigned &);
	virtual const char *getmenuicon (void)const;
	virtual const char *getmenustr (void)const;
	virtual int getnotepadlevel (void)const;
protected:
	virtual int getwidths (int [], int &);
	void guisendprompt (void);
public:
	virtual bool is_passthrough (void);
	bool is_readonly (void);
	virtual bool is_selectable (void);
	virtual void popup_draw (int , int &);
	virtual int post_validate (void);
protected:
	virtual void processmsg (WINDOW *, FIELD_MSG&, int);
public:
	virtual const char *return_prompt (void);
protected:
	void sendval (const char *dianame,
		 int nof,
		 char type,
		 const char *val);
	void sendval (const char *dianame,
		 int nof,
		 char type,
		 int val);
public:
	void set_donotcheckold (void);
	void set_guiparms (const char *_parms);
	void set_guipath (const char *_path);
	void set_helpdia (PRIVATE_MESSAGE&_msg);
	virtual void set_noempty (void);
	void set_readonly (void);
	virtual void set_registry_key (const char *key);
	virtual void set_selectable (bool _may_select);
	virtual void setcursor (WINDOW *dialog, int);
protected:
	virtual void setwidths (int , int []);
public:
	virtual void unselect (WINDOW *, int);
	virtual ~FIELD (void);
	/*~PROTOEND~ FIELD */
};

class FIELD_STRING_BASE: public FIELD{
protected:
	friend class DIALOG;
	struct {
		int input;	// Current position in window */
		int scroll;	// Amount of horizontal scrolling so far */
	}x;
	int size;
	char *buf;		// For editing
protected:
	int password_mode;	// Echo on or not
	int fwidth;		// Recommended visible width of the text field
	/*~PROTOBEG~ FIELD_STRING_BASE */
protected:
	FIELD_STRING_BASE (const char *_prompt,
		 const char *_str,
		 int maxsiz);
	FIELD_STRING_BASE (const char *_prompt,
		 int maxsiz);
public:
	MENU_STATUS dokey (WINDOW *dialog,
		 int key,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *dialog, int);
	const char *get_registry_value (void);
	char getidprefix (void);
	void gui_draw (int nof, SSTRINGS&);
	MENU_STATUS gui_get (int nof,
		 const char *,
		 const char *);
	void html_draw (int nof);
	int html_validate (int nof);
private:
	void init (int maxsiz);
public:
	bool is_empty (void);
	int post_validate (void);
	void set_registry_value (const char *value);
	void setcursor (WINDOW *dialog, int);
	~FIELD_STRING_BASE (void);
	/*~PROTOEND~ FIELD_STRING_BASE */
};

class FIELD_STRING: public FIELD_STRING_BASE{
protected:
	bool mayreload;	// Is str a buffer that we can use
					// DIALOG::newf_info expect any temp string
					// instead of a buffer
	char *str;		// String to edit
	char *backup;
	/*~PROTOBEG~ FIELD_STRING */
public:
	FIELD_STRING (const char *_prompt,
		 char *_str,
		 int _maxsiz,
		 bool _mayreload);
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
	~FIELD_STRING (void);
	/*~PROTOEND~ FIELD_STRING */
};

class SSTRING;
class FIELD_SSTRING: public FIELD_STRING_BASE{
protected:
	SSTRING &str;
	SSTRING backup;
	/*~PROTOBEG~ FIELD_SSTRING */
public:
	FIELD_SSTRING (const char *_prompt,
		 SSTRING&_str,
		 int field_width);
	void html_draw (int nof);
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
	/*~PROTOEND~ FIELD_SSTRING */
};

class FIELD_PASSWORD: public FIELD_SSTRING{
	/*~PROTOBEG~ FIELD_PASSWORD */
public:
	FIELD_PASSWORD (const char *_prompt,
		 SSTRING&_str);
	/*~PROTOEND~ FIELD_PASSWORD */
};


// FIELD_STRING with a helper hot key
// generally used to setup combo-box
class FIELD_STRING_HELP: public FIELD_SSTRING{
protected:
	virtual void assist(WINDOW *dialog)=0;
	/*~PROTOBEG~ FIELD_STRING_HELP */
public:
	FIELD_STRING_HELP (const char *_prompt,
		 SSTRING&_str);
	FIELD_STRING_HELP (const char *_prompt,
		 SSTRING&_str,
		 int _maxsiz);
	MENU_STATUS dokey (WINDOW *dialog,
		 int key,
		 FIELD_MSG&msg,
		 bool&grab);
	void drawtxt (WINDOW *dialog, int offset);
	/*~PROTOEND~ FIELD_STRING_HELP */
};

class LIST_STR;
class FIELD_COMBO: public FIELD_STRING_HELP{
protected:
	LIST_STR *opts;
	int listvalueonly;
	/*~PROTOBEG~ FIELD_COMBO */
public:
	FIELD_COMBO (const char *_prompt,
		 SSTRING&_str,
		 int maxsiz);
	void addopt (const char *str);
	void addopt (const char *value,
		 const char *verbose);
	void addopts (const SSTRINGS&lst);
	void addopts (const SSTRING_KEYS&lst);
protected:
	void assist (WINDOW *dialog);
public:
	char getidprefix (void);
	void gui_draw (int nof, SSTRINGS&);
	MENU_STATUS gui_get (int nof,
		 const char *,
		 const char *);
	void html_draw (int nof);
	int html_validate (int nof);
	~FIELD_COMBO (void);
	/*~PROTOEND~ FIELD_COMBO */
};
class FIELD_LIST: public FIELD_COMBO{
	/*~PROTOBEG~ FIELD_LIST */
public:
	FIELD_LIST (const char *_prompt, SSTRING&_str);
	void addopt (const char *str);
	void addopt (const char *value,
		 const char *shown,
		 const char *verbose);
	void addopt (const char *value,
		 const char *verbose);
	MENU_STATUS dokey (WINDOW *dialog,
		 int ,
		 FIELD_MSG&,
		 bool&);
	virtual int post_validate (void);
	void save (void);
	/*~PROTOEND~ FIELD_LIST */
};
class FIELD_ENUM: public FIELD_LIST{
	int &sel;
	int backup_sel;
	/*~PROTOBEG~ FIELD_ENUM */
public:
	FIELD_ENUM (const char *_prompt, int &_sel);
	void addopt (const char *str);
	void addopt (const char *value,
		 const char *verbose);
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
	/*~PROTOEND~ FIELD_ENUM */
};

class FIELD_COMBO_MANY: public FIELD_COMBO{
	/*~PROTOBEG~ FIELD_COMBO_MANY */
public:
	FIELD_COMBO_MANY (const char *_prompt,
		 SSTRING&_str);
	void addopt (const char *str, char &selected);
protected:
	void assist (WINDOW *dialog);
public:
	/*~PROTOEND~ FIELD_COMBO_MANY */
};


class FIELD_CHECK_BASE: public FIELD{
protected:
	char *var;
	char val;
	char backup;
	char registry_buf[5];
	/*~PROTOBEG~ FIELD_CHECK_BASE */
public:
	FIELD_CHECK_BASE (const char *_prompt,
		 char &_var);
	const char *get_registry_value (void);
	void set_registry_value (const char *v);
	/*~PROTOEND~ FIELD_CHECK_BASE */
};


class FIELD_CHECK_RADIO: public FIELD_CHECK_BASE{
protected:
	char *title;
	const char *original_title;
	/*~PROTOBEG~ FIELD_CHECK_RADIO */
public:
	FIELD_CHECK_RADIO (const char *_prompt,
		 char &_var,
		 const char *_title);
	void drawtxt_check (WINDOW *dialog,
		 char openchar,
		 char closechar,
		 char selchar);
	void format_htmlkey (char *key, int nof);
	void reload (const char *dianame, int nof);
	void restore (void);
	const char *return_prompt (void);
	void save (void);
	void setcursor (WINDOW *dialog, int);
	~FIELD_CHECK_RADIO (void);
	/*~PROTOEND~ FIELD_CHECK_RADIO */
};


class FIELD_CHECK: public FIELD_CHECK_RADIO{
	/*~PROTOBEG~ FIELD_CHECK */
public:
	FIELD_CHECK (const char *_prompt,
		 char &_var,
		 const char *_title);
	MENU_STATUS dokey (WINDOW *dialog,
		 int key,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *dialog, int);
	char getidprefix (void);
	void gui_draw (int nof, SSTRINGS&);
	MENU_STATUS gui_get (int nof,
		 const char *,
		 const char *);
	void html_draw (int nof);
	int html_validate (int nof);
	/*~PROTOEND~ FIELD_CHECK */
};
class FIELD_RADIO: public FIELD_CHECK_RADIO{
	char instance_val;
	FIELD_RADIO *next;		// All field radio are in a linked list
							// see radio.c
	int nof;
	/*~PROTOBEG~ FIELD_RADIO */
public:
	FIELD_RADIO (const char *_prompt,
		 char &_var,
		 char _instance_val,
		 const char *_title);
	MENU_STATUS dokey (WINDOW *,
		 int key,
		 FIELD_MSG&msg,
		 bool&);
	void drawtxt (WINDOW *dialog, int);
	char getidprefix (void);
	void gui_draw (int _nof, SSTRINGS&);
	MENU_STATUS gui_get (int ,
		 const char *,
		 const char *);
	void html_draw (int);
	int html_validate (int);
private:
	FIELD_RADIO *locate_key (char *key);
protected:
	void processmsg (WINDOW *dialog,
		 FIELD_MSG&msg,
		 int drawok);
public:
	void reload (const char *dianame, int nof);
	~FIELD_RADIO (void);
	/*~PROTOEND~ FIELD_RADIO */
};
#define DIALOG_MAXCOL	20

class FIELD_CHECK_MULTI: public FIELD_CHECK_BASE{
protected:
	SSTRINGS options;
	/*~PROTOBEG~ FIELD_CHECK_MULTI */
public:
	FIELD_CHECK_MULTI (const char *_prompt,
		 char &_var,
		 const char *_options[]);
	MENU_STATUS dokey (WINDOW *dialog,
		 int key,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *dialog, int);
	char getidprefix (void);
	void gui_draw (int nof, SSTRINGS&);
	MENU_STATUS gui_get (int nof,
		 const char *,
		 const char *);
	void html_draw (int nof);
	int html_validate (int nof);
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
	/*~PROTOEND~ FIELD_CHECK_MULTI */
};


class FIELD_CLIST: public FIELD{
	class FIELD_CLIST_PRIVATE *priv;
	/*~PROTOBEG~ FIELD_CLIST */
public:
	FIELD_CLIST (const char *_prompt,
		 int nbvisible,
		 DIALOG *dia,
		 PRIVATE_MESSAGE&_msg,
		 int &sel);
	void addrecord (const char *s);
	void addrecordf (const char *s, ...);
private:
	MENU_STATUS dokey (WINDOW *,
		 int ,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *, int);
public:
	const char *get_registry_value (void);
	char getidprefix (void);
	const char *getline (int no)const;
	int getnb (void)const;
private:
	void gui_draw (int nof, SSTRINGS&tb);
public:
	MENU_STATUS gui_get (int no,
		 const char *field_id,
		 const char *actionid);
	void html_draw (int);
	int html_validate (int);
	void insrecord (int pos, const char *s);
	void mayclickhead (void);
	void reload (const char *, int);
	void remove (int n);
	void remove_last (int n);
	void restore (void);
	void save (void);
private:
	void sendhead (int guicmd, const char *prefix);
public:
	void set_registry_value (const char *);
	void setcursor (int pos, bool focus);
	void setdcs (const char *dcs);
	void setheader (const char *s);
	void sethsign (const char *hsign);
	void sethsign (int column, const char letter);
	void setnextdcs (const char *dcs);
	void setrecord (int no, const char *s);
	void setrecordf (int pos, const char *s, ...);
private:
	void update_head (void);
public:
	void vaddrecordf (const char *s, va_list list);
	void vsetrecordf (int pos,
		 const char *s,
		 va_list list);
	int whichcolumn (void);
	~FIELD_CLIST (void);
	/*~PROTOEND~ FIELD_CLIST */
};

class FIELD_BUTTON: public FIELD{
protected:
	class FIELD_BUTTON_PRIVATE *priv;
	/*~PROTOBEG~ FIELD_BUTTON */
protected:
	FIELD_BUTTON (const char *_str,
		 const char *_icon,
		 const char *_help,
		 PRIVATE_MESSAGE&_msg,
		 bool enter_action,
		 DIALOG *_dia);
public:
	MENU_STATUS dokey (WINDOW *,
		 int key,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *win, int);
	const char *get_registry_value (void);
	char getidprefix (void);
	void gui_draw (int nof, SSTRINGS&);
	MENU_STATUS gui_get (int nof,
		 const char *,
		 const char *actionid);
	void html_draw (int);
	int html_validate (int);
	void reload (const char *, int);
	void restore (void);
	void save (void);
protected:
	void sendval (const char *val);
public:
	void set_registry_value (const char *);
	~FIELD_BUTTON (void);
	/*~PROTOEND~ FIELD_BUTTON */
};

class FIELD_BUTTON_TEXT: public FIELD_BUTTON{
	/*~PROTOBEG~ FIELD_BUTTON_TEXT */
public:
	FIELD_BUTTON_TEXT (const char *_str,
		 const char *_help,
		 PRIVATE_MESSAGE&_msg,
		 bool enter_action,
		 DIALOG *_dia);
	void settext (const char *_str);
	/*~PROTOEND~ FIELD_BUTTON_TEXT */
};
class FIELD_BUTTON_ICON: public FIELD_BUTTON{
	/*~PROTOBEG~ FIELD_BUTTON_ICON */
public:
	FIELD_BUTTON_ICON (const char *_icon,
		 const char *_help,
		 PRIVATE_MESSAGE&_msg,
		 bool enter_action,
		 DIALOG *_dia);
	void seticon (const char *_icon);
	/*~PROTOEND~ FIELD_BUTTON_ICON */
};


class BUTTONS_INFO;

class DIALOG: public ARRAY{
	friend class DIALOG_LISTE;
	friend void multi_setlistening();
	friend void multi_touchwins();
protected:
	class DIALOG_INTERNAL *internal;
	/*~PROTOBEG~ DIALOG */
public:
	DIALOG (void);
protected:
	void add (FIELD *f);
public:
	void addhelp (HELP_FILE&help, const char *title);
	void addwhat (const char *help);
	void auto_newline (bool mode);
	void delwhat (const char *help);
private:
	void dokeydown (int &nof, WINDOW *dialog);
	void dokeyup (int &nof, WINDOW *dialog);
	void draw (WINDOW *dialog);
	void drawarrow_if (WINDOW *win,
		 bool condition,
		 bool&flag,
		 bool top,
		 chtype carac);
protected:
	void drawf (WINDOW *dialog);
public:
	MENU_STATUS edit (const char *_title,
		 const char *_intro,
		 HELP_FILE&helpfile,
		 int &nof,
		 int but_options);
	MENU_STATUS edit (const char *_title,
		 const char *intro,
		 HELP_FILE&helpfile);
	MENU_STATUS edit (const char *_title,
		 const char *intro,
		 HELP_FILE&helpfile,
		 int &nof);
private:
	MENU_STATUS editgui (int &nof, int but_options);
	MENU_STATUS editgui_thread (int &nof, int but_options);
	MENU_STATUS edithtml (int &nof);
public:
	virtual MENU_STATUS editmenu (const char *title,
		 const char *prompt,
		 HELP_FILE&helpfile,
		 int &sel,
		 int options);
private:
	MENU_STATUS editterm (int &nof, int but_options);
	void fixwidth1 (void);
public:
	FIELD *getitem (int no);
	const char *getmenustr (int choice);
	int getoffset (void)const;
	void gui_dispolast (GUI_H_DISPO dispoh,
		 int nbcellh,
		 GUI_V_DISPO dispov,
		 int nbcellv);
	void gui_end (void);
	void gui_form (void);
	void gui_group (const char *title);
	void gui_groupfit (const char *title);
	void gui_label (const char *ctl, ...);
	void gui_labelv (const char *args, va_list list);
	void gui_passthrough (int command,
		 const char *args,
		 ...);
	void gui_passthroughv (int command,
		 const char *args,
		 va_list list);
private:
	void guidelete (void);
public:
	void hide (void);
	void html_body (const char *ctl, ...);
private:
	void html_draw (DIALOG *spc, int nof);
	void html_draw (int nof);
	void html_draw_end (void);
	void html_draw_fields (int nof);
	void html_draw_form (int nof);
	void html_draw_intro (void);
	void html_draw_top (void);
public:
	void html_end (const char *ctl, ...);
	void html_intro (const char *ctl, ...);
	void html_top (const char *ctl, ...);
private:
	int html_validate (SSTRING&msg);
public:
	void inswhat (const char *help);
protected:
	virtual int keymove (WINDOW *dialog,
		 int key,
		 int &nof);
	int keymove_scroll (WINDOW *dialog,
		 int key,
		 int &nof);
public:
	void last_noempty (void);
	FIELD_BUTTON_TEXT *new_button (const char *str,
		 const char *help,
		 PRIVATE_MESSAGE&msg);
	FIELD_BUTTON_TEXT *new_button (const char *str,
		 const char *help,
		 PRIVATE_MESSAGE&msg,
		 bool enter_action);
	FIELD_BUTTON_ICON *new_button_icon (const char *icon,
		 const char *help,
		 PRIVATE_MESSAGE&msg);
	void new_inputgrid (int x,
		 int y,
		 int cellwidth,
		 int cellheight,
		 int nbcellh,
		 int nbcellv,
		 PRIVATE_MESSAGE&msg);
	void new_menuinfo (const char *prompt1,
		 const char *prompt2);
	void new_menuitem (const SSTRING&prompt1,
		 const SSTRING&prompt2);
	void new_menuitem (const char *icon,
		 const char *prompt1,
		 const char *prompt2);
	void new_menuitem (const char *prompt1,
		 const SSTRING&prompt2);
	void new_menuitem (const char *prompt1,
		 const char *prompt2);
private:
	void new_menuitem_parse (const char *item1,
		 const char *item2);
public:
	void new_menuitems (const char *items[], int item_no);
	void new_menuitems (const char *opt[]);
	void new_menuline (const char *icon,
		 const char *prompt1,
		 const char *prompt2,
		 bool may_select);
	FIELD_CHECK *newf_chk (const char *prompt,
		 char &var,
		 const char *title);
	FIELD_CHECK_MULTI *newf_chkm (const char *prompt,
		 char &var,
		 const char *title[]);
	FIELD_CHECK_MULTI *newf_chkm_col (const char *prompt,
		 char &var,
		 const char *title[]);
	FIELD_CHECK_MULTI *newf_chkm_hexnum (const char *prompt,
		 int &var,
		 const int defvar,
		 const int vals[],
		 const char *options[]);
	FIELD_CHECK_MULTI *newf_chkm_hexnum (const char *prompt,
		 int &var,
		 const int vals[],
		 const char *options[]);
	FIELD_CHECK_MULTI *newf_chkm_num (const char *prompt,
		 int &var,
		 const int defvar,
		 const int vals[],
		 const char *options[]);
	FIELD_CHECK_MULTI *newf_chkm_num (const char *prompt,
		 int &var,
		 const int vals[],
		 const char *options[]);
	FIELD_CHECK_MULTI *newf_chkm_str (const char *prompt,
		 int &var,
		 SSTRING&bufstr,
		 const int vals[],
		 const char *options[]);
	FIELD_CLIST *newf_clist (const char *prompt,
		 int nbvisible,
		 PRIVATE_MESSAGE&msg,
		 int &sel);
	FIELD_COMBO *newf_combo (const char *prompt,
		 SSTRING&str);
	FIELD_COMBO *newf_combo (const char *prompt,
		 SSTRING&str,
		 int maxsiz);
	FIELD_COMBO_MANY *newf_combo_many (const char *prompt,
		 SSTRING&str);
	FIELD *newf_dbl (const char *prompt,
		 double &val,
		 int nbdecimals);
	FIELD_ENUM *newf_enum (const char *prompt, int &sel);
	FIELD *newf_gauge (const char *prompt,
		 int &val,
		 int range);
	virtual void newf_head (const char *prompt,
		 const char *msg);
	FIELD *newf_hexnum (const char *prompt, int &val);
	void newf_info (const char *prompt, const char *str);
	FIELD_SSTRING *newf_ipnum (const char *prompt,
		 SSTRING&ip);
	FIELD_LIST *newf_list (const char *prompt,
		 SSTRING&str);
	FIELD *newf_num (const char *prompt, int &val);
	FIELD *newf_octnum (const char *prompt, int &val);
	FIELD_PASSWORD *newf_pass (const char *prompt,
		 SSTRING&str);
	FIELD_RADIO *newf_radio (const char *prompt,
		 char &var,
		 char instance_val,
		 const char *title);
	void newf_sheet (const char *prompt,
		 const char *titles[],
		 SSTRINGS&tb);
	FIELD *newf_slider (const char *prompt,
		 int &val,
		 int minval,
		 int maxval);
	FIELD_SSTRING *newf_str (const char *prompt,
		 SSTRING&str);
	FIELD_SSTRING *newf_str (const char *prompt,
		 SSTRING&str,
		 int width);
	FIELD_STRING *newf_str (const char *prompt,
		 char *str,
		 int maxsiz);
	void newf_textarea (const char *prompt,
		 SSTRING&text,
		 int width,
		 int height);
	void newf_title (const char *pad,
		 int level,
		 const char *prompt,
		 const char *msg);
	void newf_title (const char *prompt, const char *msg);
	void newline (void);
private:
	void processmsg (WINDOW *dialog, FIELD_MSG&msg);
public:
	void reload (int no);
	void reload (int start, int end);
	void reload (void);
	void remove_all (void);
	int remove_del (FIELD *f);
	int remove_del (int no);
	void remove_last (int cut);
	void request_dump (void);
	void reset_guidone (void);
	void restore (void);
	void save (void);
	void savewhat (const char *help);
private:
	void sendintro (void);
public:
	void set_alt_title (const char *_title);
	void set_button_on_side (void);
	void set_donotcheckold (void);
	void set_formparms (const char *parms, ...);
	void set_guiparms (const char *parms, ...);
	void set_helpdia (PRIVATE_MESSAGE&msg);
	void set_lastreadonly (void);
	void set_menuitem (int no,
		 const char *prompt1,
		 const char *prompt2);
	void set_nextfield (int no);
	void set_readonly (void);
	void set_registry_id (const char *id);
	void set_registry_key (const char *key);
	void set_selected_button (int button);
	void setbutinfo (int id,
		 const char *title,
		 const char *icon);
	void setcontext (const char *s);
	const char *setguiname (char tmp[200]);
	void setheight_hint (void);
	void seticon (const char *_icon);
	void setoffset (int newoff);
	void setsidetitle (const char *ctl, ...);
	void settype (DIALOG_TYPE type);
private:
	void setup (void);
public:
	void show (const char *_title,
		 const char *_intro,
		 HELP_FILE&helpfile,
		 int &nof,
		 int but_options);
private:
	void showgui (int &nof, int but_options);
	void showterm (int &nof, int but_options);
	void showtimeout (WINDOW *win);
	void skipprotect (int &nof,
		 int suggestkey,
		 WINDOW *dialog);
public:
	void vhtml_body (const char *ctl, va_list list);
	void vhtml_end (const char *ctl, va_list list);
	void vhtml_intro (const char *ctl, va_list list);
	void vhtml_top (const char *ctl, va_list list);
	void vset_formparms (const char *parms, va_list list);
	void vset_guiparms (const char *parms, va_list list);
	void vsetsidetitle (const char *ctl, va_list list);
	void waitfor (PRIVATE_MESSAGE&msg);
	void waitfor (const char *msg);
	void waitfortimer (PRIVATE_MESSAGE&priv);
	void waitfortimer (const char *id);
	int was_modified (void);
	~DIALOG (void);
	/*~PROTOEND~ DIALOG */
};

class DIALOG_RECORDS: public DIALOG{
	/*~PROTOBEG~ DIALOG_RECORDS */
public:
	DIALOG_RECORDS (void);
	MENU_STATUS editmenu (const char *title,
		 const char *prompt,
		 HELP_FILE&helpfile,
		 int &sel,
		 int options);
protected:
	int keymove (WINDOW *dialog,
		 int key,
		 int &nof);
public:
	void newf_head (const char *prompt,
		 const char *msg);
	void setkeyformat (HTML_KEY_TYPE key_type);
	/*~PROTOEND~ DIALOG_RECORDS */
};

class DIALOG_LISTE: public DIALOG_RECORDS{
	/*~PROTOBEG~ DIALOG_LISTE */
public:
	DIALOG_LISTE (void);
	MENU_STATUS editmenu (const char *title,
		 const char *prompt,
		 HELP_FILE&helpfile,
		 int &sel,
		 int options);
	void set_menuitem (int no,
		 const char *prompt1,
		 const char *prompt2);
	~DIALOG_LISTE (void);
	/*~PROTOEND~ DIALOG_LISTE */
};

class DIALOG_MENU: public DIALOG{
	/*~PROTOBEG~ DIALOG_MENU */
public:
	virtual MENU_STATUS editmenu (const char *title,
		 const char *prompt,
		 HELP_FILE&helpfile,
		 int &sel,
		 int options);
	void setmenu (MENU_CONTEXT ctx);
	/*~PROTOEND~ DIALOG_MENU */
};

class  DIALOG_TEXTBOX: public DIALOG{
	/*~PROTOBEG~ DIALOG_TEXTBOX */
public:
	DIALOG_TEXTBOX (void);
protected:
	int keymove (WINDOW *dialog,
		 int key,
		 int &nof);
public:
	void newf_text (const char *,
		 const SSTRINGS&strs);
	void newf_text (const char *prompt,
		 const char *buftab);
	/*~PROTOEND~ DIALOG_TEXTBOX */
};

class DIALOG_MENUPOPUP: public DIALOG_MENU{
	/*~PROTOBEG~ DIALOG_MENUPOPUP */
public:
	DIALOG_MENUPOPUP (void);
	MENU_STATUS editmenu (const char *title,
		 int &sel);
	/*~PROTOEND~ DIALOG_MENUPOPUP */
};

extern int uithread_id;

class HTML_VARVAL;
class POPENFD;

enum FIELDEDIT_TYPE{
	FIELDEDIT_ANY,
	FIELDEDIT_DIGIT,
	FIELDEDIT_HEXNUM
};


#include "dialog.p"

#endif
