/*
 *  
 *  $Id$
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
//#define _GINKGO_TRACE
#include "globals.h"
#include "ilock.h"
#include <wx/thread.h>
#include <main/controllers/controladorlog.h>
#include <iostream>

#if defined(_WINDOWS)
	#define SUSPEND_WAKEUP()
	#define RESUME_WAKEUP()

#else
	#include <signal.h>

	#define SUSPEND_WAKEUP()\
		siginterrupt(SIGUSR2, 0)

	#define RESUME_WAKEUP()\
		siginterrupt(SIGUSR2, 1)
#endif

//----------------------------------------------------------------------------------------------------
//region Interfaz de soporte de cerrojos

//----------------------------------------------------------------------------------------------------
//region Constructor y destructor

GNC::GCS::ILockable::ILockable()
{
	GTRACE("GNC::GCS::ILockable()");
	m_pLocker = NULL;
	m_IsLocked = false;
	m_pCS = new wxCriticalSection();
}

GNC::GCS::ILockable::~ILockable()
{
	GTRACE("GNC::GCS::~ILockable()");
	if (m_IsLocked) {
		if (m_pLocker != NULL) {
			GTRACE("\tAl cerrar ILockable: El cerrojo continua autobloqueado por " << m_pLocker << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str());
		}
		else if (m_LocBloqueo.size() != 0) {
			GTRACE("\tAl cerrar ILockable: El cerrojo continua bloqueado por una llamada en " << m_LocBloqueo.c_str());
		}
		else {
			GTRACE("\tAl cerrar ILockable: El cerrojo continua autobloqueado por NULL");
		}
	}
	if (m_pCS != NULL) {
		delete m_pCS;
	}
	m_pCS = NULL;

}

//endregion

//----------------------------------------------------------------------------------------------------
//region Interfaz de exclusion mutua

void GNC::GCS::ILockable::Lock(const std::string& loc)
{

	SUSPEND_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	// Entrada a la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (m_IsLocked) {
		GTRACE("GNC::GCS::Lock() en " << loc.c_str() << " Esperando liberacion desde bloqueo previo en " << m_LocBloqueo.c_str());
		m_pCS->Enter();
		m_LocBloqueo = loc;
		m_IsLocked = true;
		GTRACE("GNC::GCS::Lock() en " << loc.c_str());
	}
	else {
		m_pCS->Enter();
		m_LocBloqueo = loc;
		m_IsLocked = true;
		GTRACE("GNC::GCS::Lock() en " << loc.c_str());
	}
	RESUME_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------
}

void GNC::GCS::ILockable::UnLock(const std::string& loc)
{
	//----------------------------------------------------------------------------------------------------
	// Salida de la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (!m_IsLocked) {
		std::cerr << "Error: El cerrojo no estaba bloqueado. (Tratado de liberar en " << loc.c_str() << ")" ;
	}
	else if (m_pLocker != NULL) {
		std::cerr << "Error: El cerrojo estaba auto bloqueado previamente por " << m_pLocker << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str() << std::endl;

	}
	else {
		GTRACE("GNC::GCS::UnLock() en " << loc.c_str());
		GTRACE("	>> Liberado bloqueo desde " << m_LocBloqueo.c_str());
		m_LocBloqueo = "";
		m_IsLocked = false;
		m_pCS->Leave();
	}
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------
}

bool GNC::GCS::ILockable::IsLocked() const
{
	return m_IsLocked;
}

//endregion

//----------------------------------------------------------------------------------------------------
//region Interfaz de exclusion mutua

void GNC::GCS::ILockable::AutoLock(ILocker* pLocker, const std::string& loc)
{
	//----------------------------------------------------------------------------------------------------
	// Entrada a la seccion Critica
	//----------------------------------------------------------------------------------------------------
	SUSPEND_WAKEUP();
	m_pCS->Enter();
	m_IsLocked = true;
	m_pLocker = pLocker;
	m_LocBloqueo = loc;
	#if defined(_GINKGO_TRACE)
	if (pLocker != NULL) {
		GTRACE("GNC::GCS::Lock() automatico por " << pLocker << " instanciado en " << pLocker->m_LocInstanciacion.c_str());
	}
	else {
		GTRACE("GNC::GCS::Lock() automatico por NULL (¿Error?)");
	}
	#endif
	RESUME_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------
}

void GNC::GCS::ILockable::AutoUnLock(ILocker* pLocker)
{
	//----------------------------------------------------------------------------------------------------
	// Salida de la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (!m_IsLocked) {
		std::cerr << "Error: El cerrojo no estaba bloqueado. (Tratado de liberar automaticamente por " << pLocker;
		if (pLocker != NULL) {
			std::cerr << " instanciado en " << pLocker->m_LocInstanciacion.c_str() << ")" << std::endl;
		}
		else {
			std::cerr << ")" << std::endl;
		}
	}
	else if (m_pLocker != pLocker) {
		std::cerr << "Error: Se ha liberado un bloqueo desde un Locker automatico distinto del que lo inicio: " << std::endl;
		std::cerr << "\tIniciado por " << m_pLocker;
		if (m_pLocker != NULL) {
			std::cerr << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str() << std::endl;
		}
		else {
			std::cerr << std::endl;
		}
		std::cerr << "\tTratado de liberar  por " << pLocker;
		if (pLocker != NULL) {
			std::cerr << " instanciado en " << pLocker->m_LocInstanciacion.c_str() << std::endl;
		}
		else {
			std::cerr << std::endl;
		}
	}
	else {
		#if defined (_GINKGO_TRACE)
		if (pLocker != NULL) {
			GTRACE("GNC::GCS::UnLock() automatico por " << pLocker << " instanciado en " << pLocker->m_LocInstanciacion.c_str());
		}
		else {
			GTRACE("GNC::GCS::UnLock() automatico por " << pLocker);
		}
		#endif
		m_LocBloqueo = "";
		m_IsLocked = false;
		m_pCS->Leave();
	}
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------
}

//endregion

//endregion

//----------------------------------------------------------------------------------------------------
//region Helper de bloqueo automatico

//----------------------------------------------------------------------------------------------------
//region Construccion y destruccion

GNC::GCS::ILocker::ILocker(GNC::GCS::ILockable& pLockable, const std::string& loc)
{
	m_pLockable = &pLockable;
	m_LocInstanciacion = loc;
	GTRACE("GNC::GCS::ILockable()");
	m_pLockable->AutoLock(this, m_LocInstanciacion);

}

GNC::GCS::ILocker::ILocker(GNC::GCS::ILockable* pLockable, const std::string& loc)
{
	m_pLockable = pLockable;
	m_LocInstanciacion = loc;
	GTRACE("GNC::GCS::ILockable()");
	m_pLockable->AutoLock(this, loc);

}

GNC::GCS::ILocker::~ILocker()
{
	m_pLockable->AutoUnLock(this);
	GTRACE("GNC::GCS::~ILockable()");
}

//endregion

//endregion


//----------------------------------------------------------------------------------------------------
//region Cola de espera
//----------------------------------------------------------------------------------------------------

//----------------------------------------------------------------------------------------------------
//region Constructor y destructor

GNC::GCS::WaitQueue::WaitQueue()
{
	m_pSignaler = new wxSemaphore(0, 0);
	m_Size = 0;
}

GNC::GCS::WaitQueue::~WaitQueue()
{
	GNC::GCS::ILocker q_Locker(this, GLOC());
	if (m_Size > 0 ) {
		LOG_ERROR("WaitQueue", "La cola de espera termino teniendo tareas pendientes sin señalar. Pudiera provocarse un estado inconsistente");
	}
	delete m_pSignaler;
	m_pSignaler = NULL;
}
//endregion

//----------------------------------------------------------------------------------------------------
//region Interfaz general

GNC::GCS::TipoListaEspera::size_type GNC::GCS::WaitQueue::Size(const std::string& loc)
{
	ILocker q_lock(this, loc);
	const TipoListaEspera::size_type size =  m_Size;
	return size;
}

bool GNC::GCS::WaitQueue::Empty(const std::string& loc)
{
	ILocker q_lock(this, loc);
	const bool empty = (m_Size == 0);
	return empty;
}

bool GNC::GCS::WaitQueue::NotEmpty(const std::string& loc)
{
	ILocker q_lock(this, loc);
	const bool not_empty = (m_Size != 0);
	return not_empty;
}

GNC::GCS::TipoListaEspera::size_type GNC::GCS::WaitQueue::Pending(const std::string& loc)
{
	ILocker q_lock(this, loc);
	const TipoListaEspera::size_type npending = m_Size - m_ListaEspera.size();
	return npending;
}


bool GNC::GCS::WaitQueue::Wait(const std::string& loc)
{
	if (loc == "")
	{
		//std::cout << std::endl;
	}
	SUSPEND_WAKEUP();
	wxSemaError err = m_pSignaler->Wait();
	GNC::GCS::ILocker q_Locker(this, loc);
	if (err != wxSEMA_NO_ERROR) {
		RESUME_WAKEUP();
		return false;
	}
	else {
		m_Size--;
	}
	RESUME_WAKEUP();
	return true;
}

bool GNC::GCS::WaitQueue::Wait(long msecs, const std::string& loc)
{
	if (loc == "")
	{
		//std::cout << std::endl;
	}
	SUSPEND_WAKEUP();
	wxSemaError err = m_pSignaler->WaitTimeout(msecs);
	GNC::GCS::ILocker q_Locker(this, loc);
	if (err != wxSEMA_NO_ERROR) {
		RESUME_WAKEUP();
		return false;
	}
	else {
		m_Size--;
	}
	RESUME_WAKEUP();
	return true;
}

void GNC::GCS::WaitQueue::RegistrarEspera(WaitQueueTask* task, const std::string& loc)
{
	if (loc == "") {
		//std::cout << std::endl;
	}
	GNC::GCS::ILocker q_Locker(this, loc);
	GNC::GCS::ILocker t_Locker(task, loc);
	task->m_ListaNotificacion.push_back(this);
	m_ListaEspera.push_back(task);
	m_Size++;
}

void GNC::GCS::WaitQueue::TerminarPendientes(const std::string& loc)
{
	if (loc == "")
	{
		//std::cout << std::endl;
	}
	GNC::GCS::ILocker q_locker(this, loc);

	for (TipoListaEspera::iterator it = m_ListaEspera.begin(); it != m_ListaEspera.end(); it++)
	{
		GNC::GCS::WaitQueueTask* wqt = *it;
		GNC::GCS::ILocker t_locker(wqt, loc);
		if (wqt->m_NotificarTerminacion) {
			wqt->Terminar();
		}
	}
}
//endregion

//----------------------------------------------------------------------------------------------------
//region Interfaz privada

//endregion

//endregion

//----------------------------------------------------------------------------------------------------
// Tarea de la cola de espera
//----------------------------------------------------------------------------------------------------

//----------------------------------------------------------------------------------------------------
//region Constructor y destructor

GNC::GCS::WaitQueueTask::WaitQueueTask()
{
	m_NotificarTerminacion = false;
}

GNC::GCS::WaitQueueTask::~WaitQueueTask()
{
}
//endregion

//----------------------------------------------------------------------------------------------------
//region Interfaz general

void GNC::GCS::WaitQueueTask::Signal(const std::string& loc)
{
	if (loc == "")
	{
		//std::cout << std::endl;
	}
	GTRACE("GNC::GCS::WaitQueueTask::Signal() " << this);
	//GNC::GCS::ILocker t_locker(this, loc);
	GTRACE("Señalando a "<< m_ListaNotificacion.size()<< " tareas " << this);
	for (TipoListaNotificaciones::iterator it_q = m_ListaNotificacion.begin(); it_q != m_ListaNotificacion.end(); it_q++)
	{
		WaitQueue* wq = (*it_q);
		ILocker q_locker(wq, loc);

		bool notfound = true;
		for (TipoListaEspera::iterator it_t = wq->m_ListaEspera.begin(); it_t != wq->m_ListaEspera.end(); it_t++) {
			WaitQueueTask* qt = *it_t;
			if ( qt == this) {
				wq->m_ListaEspera.remove(this);
				wq->m_pSignaler->Post();
				notfound = false;
				break;
			}
		}

		if (notfound) {
			LOG_ERROR("WaitQueue", "Error: Inconsistencia en mecanismo de colas de espera. Se trató de notificar una dependencia no registrada o ya notificada");
		}

	}
	m_ListaNotificacion.clear();
}

void GNC::GCS::WaitQueueTask::Terminar()
{
	m_NotificarTerminacion = true;
	DoTerminar();
}
//endregion

//region Interfaz privada

//endregion

//endregion
