/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  April 1994                                */
/*************************************************************************/

#include "EST_speech_class.h"
#include "sigpr/EST_sigpr_utt.h"
#include "sigpr/EST_filter.h"
#include "srpd.h"
#include "EST_error.h"

int read_next_wave_segment (EST_Wave &sig, struct Srpd_Op *paras, 
			    SEGMENT_ *p_seg);

void srpd(EST_Wave &sig, EST_Track &fz, Srpd_Op &srpd_op);
struct Srpd_Op *default_srpd_op(struct Srpd_Op *srpd);
void parse_srpd_list(EST_Option &a_list, struct Srpd_Op *srpd);

void pda(EST_Wave &sig, EST_Track &fz, EST_Option &op, EST_String method)
{
    if (method == "")
    {
	if (op.present("pda_method"))
	    method = op.val("pda_method");
    }
    if (method == "")	
	do_srpd_fz(sig, fz, op);
    else if  (method == "srpd")
	do_srpd_fz(sig, fz, op);
    else
	EST_error("Unknown pda %s\n", (const char *)method);
}

void icda(EST_Wave &sig, EST_Track &fz, EST_Track &speech, EST_Option &op, 
	       EST_String method)
{ // intonation contour detection algorithm
    EST_Track raw_fz;
    if (method == "")
    {
	if (op.present("pda_method"))
	    method = op.val("pda_method");
    }
    if (method == "")	
	do_srpd_fz(sig, raw_fz, op);
    else if  (method == "srpd")
	do_srpd_fz(sig, raw_fz, op);
    else
	EST_error("Unknown pda %s\n", (const char *)method);

    smooth_phrase(raw_fz, speech, op, fz);
}

void do_srpd_fz(EST_Wave &sig, EST_Track &fz, EST_Option &op)
{
    Srpd_Op srpd_op;

    default_srpd_op(&srpd_op); // default values
    parse_srpd_list(op, &srpd_op); // override with options

    if (op.val("do_low_pass", 0) == "true")
	FIRlowpass_filter(sig, op.ival("lpf_cutoff",1),
			  op.ival("lpf_order",1));

    srpd(sig, fz, srpd_op);
}

void do_srpd_fz(EST_Wave &sig, EST_Track &fz)
{
    Srpd_Op srpd_op;
    default_srpd_op(&srpd_op);
    srpd(sig, fz, srpd_op);
}

void srpd(EST_Wave &sig, EST_Track &fz, Srpd_Op &srpd_op)
{
    int i, rns, tracklen, j = 0;
    SEGMENT_ segment;
    CROSS_CORR_ cc;
    STATUS_ pda_status, held_status;
    srpd_op.sample_freq = sig.sample_rate();
    float min, max;
    min = srpd_op.min_pitch; // must store as set up routines corrupt
    max = srpd_op.max_pitch;

    initialise_structures (&srpd_op, &segment, &cc);
    initialise_status (&srpd_op, &pda_status);
    initialise_status (&srpd_op, &held_status);

    tracklen = (sig.num_samples() - segment.length) / segment.shift + 1;
    fz.resize(tracklen,1);
    fz.set_channel_name("F0", 0);
    fz.set_contour_type(ct_f0);
    fz.fill_time(srpd_op.shift/1000);

    if (!fz.equal_space())
	EST_error("Pitch tracking algorithm must have equal spaced track\n");
    
    while ((rns = read_next_wave_segment (sig, &srpd_op, &segment)) != 0) 
    {
	if (rns == 2) 
	{
	    for (i = 0; i < cc.size; cc.coeff[i++] = 0.0);
	    initialise_status (&srpd_op, &pda_status);
	}
	else
	    super_resolution_pda (&srpd_op, segment, &cc, &pda_status);
	if (pda_status.s_h == HOLD) 
	{
	    held_status.pitch_freq = pda_status.pitch_freq;
	    held_status.v_uv = VOICED;
	    held_status.s_h = HELD;
	    held_status.cc_max = pda_status.cc_max;
	    held_status.threshold = pda_status.threshold;
	    continue;
	}
	if (held_status.s_h == HELD) 
	{
	    if (pda_status.pitch_freq == BREAK_NUMBER) 
	    {
		held_status.pitch_freq = BREAK_NUMBER;
		held_status.v_uv = UNVOICED;
	    }
	    held_status.s_h = SENT;
	    if (held_status.v_uv != VOICED) 
		fz.set_break(j);
	    fz.a(j++) = held_status.pitch_freq;
	}
	if (pda_status.v_uv != VOICED) 
	    fz.set_break(j);
	fz.a(j++) = pda_status.pitch_freq;
    }
    if (held_status.s_h == HELD) 
    {
	held_status.pitch_freq = BREAK_NUMBER;
	held_status.v_uv = UNVOICED;
	fz.set_break(j);
	fz.a(j++) = held_status.pitch_freq;
    }
    end_structure_use (&segment, &cc);
}

struct Srpd_Op *default_srpd_op(struct Srpd_Op *srpd)
{ 
    srpd->L = DEFAULT_DECIMATION;
    srpd->min_pitch = DEFAULT_MIN_PITCH;
    srpd->max_pitch = DEFAULT_MAX_PITCH;
    srpd->shift = DEFAULT_SHIFT;
    srpd->length = DEFAULT_LENGTH;
    srpd->Tsilent = DEFAULT_TSILENT;
    srpd->Tmin = DEFAULT_TMIN;
    srpd->Tmax_ratio = DEFAULT_TMAX_RATIO;
    srpd->Thigh = DEFAULT_THIGH;
    srpd->Tdh = DEFAULT_TDH;
    srpd->make_ascii = 0;
    srpd->peak_tracking = 0;
    srpd->sample_freq = DEFAULT_SF;
      /* p_par->Nmax and p_par->Nmin cannot be initialised */
    return(srpd);
}

void parse_srpd_list(EST_Option &al, struct Srpd_Op *srpd)
{ 
    if (al.present("decimation"))
	srpd->L = al.ival("decimation");
    if (al.present("min_pitch"))
	srpd->min_pitch = al.fval("min_pitch");
    if (al.present("max_pitch"))
	srpd->max_pitch = al.fval("max_pitch");    
    if (al.present("pda_frame_shift"))
	srpd->shift = al.fval("pda_frame_shift") * 1000.0;
    if (al.present("pda_frame_length"))
	srpd->length = al.fval("pda_frame_length") * 1000.0;
    if (al.present("noise_floor"))
	srpd->Tsilent = al.ival("noise_floor");
    if (al.present("v2uv_coeff_thresh"))
	srpd->Thigh = al.dval("v2uv_coef_thresh");
    if (al.present("min_v2uv_coef_thresh"))
	srpd->Tmin = al.dval("min_v2uv_coef_thresh");
    if (al.present("v2uv_coef_thresh_ratio"))
	srpd->Tmax_ratio = al.dval("v2uv_coef_thresh_ratio");
    if (al.present("anti_doubling_thresh"))
	srpd->Tdh = al.dval("anti_doubling_thresh");
    if (al.present("peak_tracking"))
	srpd->peak_tracking = al.ival("peak_tracking");
    if (al.present("sample_frequency"))
	srpd->sample_freq = al.ival("sample_frequency");
}

void default_pda_options(EST_Option &al)
{
    al.add_item("min_pitch", "40.0");
    EST_String m = "max_pitch";
    al.add_item(m, "900.0");
//    al.add_item("max_pitch", "400.0");
    al.add_item("pda_frame_shift", "0.05");
    al.add_fitem("pda_frame_length", DEFAULT_LENGTH / 1000.0);
    al.add_item("lpf_cutoff", "600");
    al.add_item("lpf_order", "49");
    al.add_item("f0_file_type", "esps");
    al.add_fitem("decimation", DEFAULT_DECIMATION);
    al.add_fitem("noise_floor", DEFAULT_TSILENT);
    al.add_fitem("min_v2uv_coef_thresh", DEFAULT_TMIN);
    al.add_fitem("v2uv_coef_thresh_ratio", DEFAULT_TMAX_RATIO);
    al.add_fitem("v2uv_coef_thresh", DEFAULT_THIGH);
    al.add_fitem("anti_doubling_thresh", DEFAULT_TDH);
    al.add_iitem("peak_tracking", 0);
}

EST_String options_pda_general(void)
{
    // The standard waveform input options 
    return
	EST_String("")+
        "-c <int>         Select channel from input data (starts from 0)\n"+
        "-n <int>         Number of channels of raw data\n"+
	"-L               perform low pass filtering on input\n" + 
	"-P               perform peak tracking\n" +
	"-R               don't perform low pass filtering on input\n" +
	"-fmin <float>      miniumum F0 value\n" +
	"-fmax <float>      maxiumum F0 value\n" +
	"-j <string>      <\"on\" \"off\"> join points in output\n" +
	"-l <float>       frame length in seconds.\n" +
	"-forder <int>    lp filter order (as an exponent of 2, eg 5)\n" +
	"-s <float>       frame shift in seconds.\n" +
	"-srpd            use srpd method\n" +
	"-u <int>         lp filter cutoff (eg 600)\n";
}

EST_String options_pda_srpd(void)
{
    // The standard waveform input options 
    return
	EST_String("")+
	"-d <float>       decimation threshold (srpd)\n" +
	"-n <float>       noise floor\n" +
	"-H <float>       unvoiced to voiced coeff threshold\n" +
	"-m <float>       min voiced to unvoiced coeff threshold\n" +
	"-r <float>       voiced to unvoiced coeff threshold-ratio\n" +
	"-t <float>       anti pitch doubling/halving threshold\n";
}


