/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Alan W Black                              */
/*                   Date   :  May 1998                                  */
/*-----------------------------------------------------------------------*/
/*                EST_Utterance class source file                        */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_error.h"
#include "EST_string_aux.h"
#include "ling_class/EST_Utterance.h"

static void clear_up_sisilist(KVL<EST_Item_Content *,EST_Item *> &s);
static EST_Item *map_ling_item(EST_Item *si,
				    KVL<EST_Item_Content *,
				        EST_Item *> &s);
static void copy_relation(EST_Item *to,EST_Item *from,
			  KVL<EST_Item_Content *,EST_Item *> &slist);

#if defined(INSTANTIATE_TEMPLATES)

#include "../base_class/EST_TList.cc"
#include "../base_class/EST_KV.cc"

template class KVL<EST_Item_Content *, EST_Item *>;
template class KVI<EST_Item_Content *, EST_Item *>;
template class EST_TList<KVI<EST_Item_Content *, EST_Item *> >;
template class EST_TItem<KVI<EST_Item_Content *, EST_Item *> >;

#endif

EST_Utterance::EST_Utterance()
{
}

void EST_Utterance::init()
{
}

void EST_Utterance::clear()
{
    for (EST_Litem *p=relations.list.head(); p; p=next(p))
	delete relations.list(p).v;
}

void EST_Utterance::clear_relations()
{
    for (EST_Litem *p=relations.list.head(); p; p=next(p))
	relations.list(p).v->clear();
}

EST_Relation *EST_Utterance::create_relation(const EST_String &n)
{
    EST_Relation *r = relation(n,FALSE);
    if (r)   // there is one already, so clear it
	r->clear();
    else
    {
	r = new EST_Relation(n);
	r->set_utt(this);
	relations.add_item(n,r);
    }

    return r;
}

void EST_Utterance::delete_relation(const EST_String &n)
{
    EST_Relation *r = relation(n,FALSE);

    if (r != 0)
    {
	relations.remove_item(n);
	delete r;
    }
}

EST_Relation *EST_Utterance::relation(int i)
{
    return relations.list.nth(i).v;
}

EST_Relation *EST_Utterance::relation(const EST_String &name, int err)
{
    EST_Litem *p;
    
    for (p=relations.list.head(); p; p=next(p))
	if (relations.list(p).k == name)
	    return relations.list(p).v;
    if (err)
	EST_error("utterance: no relation %s found\n", (const char *)name);

    return 0;
}

EST_Relation *EST_Utterance::relation(const char *name,int err)
{
    EST_Litem *p;
    
    for (p=relations.list.head(); p; p=next(p))
	if (relations.list(p).k == name)
	    return relations.list(p).v;
    if (err)
	EST_error("utterance: no relation %s found\n", (const char *)name);

    return 0;
}

bool EST_Utterance::has_relation(const EST_String name) const
{
    if (!name.contains("("))
	return relations.present(name);
    EST_StrList s;
    BracketStringtoStrList(name, s);
    return has_relation(s);
}

bool EST_Utterance::has_relation(EST_StrList &names) const
{
    for (EST_Litem *p = names.head(); p ; p = next(p))
	if (!relations.present(names(p)))
	    return false;
    return true;
}

EST_Utterance &EST_Utterance::operator=(const EST_Utterance &s)
{
    copy(s);
    return *this;
}

ostream& operator << (ostream &st, const EST_Utterance &u)
{
    u.save(st,"est_ascii");
    return st;
}

void EST_Utterance::copy(const EST_Utterance &u)
{
    // Make a copy of the utterance
    KVL<EST_Item_Content *,EST_Item *> sisilist;
    EST_Relation *nrel;
    EST_Item *rnode;

    clear();
    f = u.f;

    for (EST_Litem *r=u.relations.head(); r != 0; r=next(r))
    {
	EST_Relation *rr = u.relations.list(r).v;
	nrel = create_relation(rr->name());
	nrel->f = rr->f;
	if (rr->head() != 0)
	{
	    rnode = nrel->append(map_ling_item(rr->head(),sisilist));
	    copy_relation(rnode,rr->head(),sisilist);
	}
    }
    clear_up_sisilist(sisilist);
}

static void extra_sub_utterance(EST_Utterance &u,EST_Item *i)
{
    sub_utterance(u,i);
}

void EST_Utterance::sub_utterance(EST_Item *i)
{
    extra_sub_utterance(*this,i);
}

int utterance_merge(EST_Utterance &utt,
		    EST_Utterance &sub_utt,
		    EST_Item *at,
		    const EST_String &relname)
{
    // Joins sub_utt to utt at ling_item at, merging the root
    // of relname in sub_utt with ling_item at.  All other relations
    // in sub_utt get their root's appended (not merged) with the
    // corresponding relations in utt (and created if necessary).
    KVL<EST_Item_Content *,EST_Item *> sisilist;
    EST_Item *sub_root, *utt_root, *rnode;
    EST_Relation *nrel;

    if (!utt.has_relation(relname))
    {
	cerr << "utterance_merge: utt has no " << relname<<" relation" << endl;
	return FALSE;
    }
    if (!sub_utt.has_relation(relname))
    {
	cerr << "utterance_merge: sub_utt has no " << 
	    relname <<" relation" << endl;
	return FALSE;
    }
    utt_root = at->as_relation(relname);
    sub_root = sub_utt.relation(relname)->head();

    if (sub_root == 0)
    {
	cerr << "utterance_merge: at node not in " << 
	    relname <<" relation" << endl;
	return FALSE;
    }

    merge_features(utt_root->features(),sub_root->features());
    // in case root item in sub is referenced elsewhere in the structure
    sisilist.add_item(sub_root->contents(),utt_root);
    copy_relation(utt_root,sub_root,sisilist);

    for (EST_Litem *r=sub_utt.relations.head(); r != 0; r=next(r))
    {
	EST_Relation *rr = sub_utt.relations.list(r).v;
	if ((rr->name() != relname) &&
	    (rr->name() != "Word"))  // hacked 
	{
	    if (!utt.has_relation(rr->name()))
		nrel = utt.create_relation(rr->name());
	    else
		nrel = utt.relation(rr->name());
	    if (rr->head() != 0)
	    {
		EST_Item *nn = map_ling_item(rr->head(),sisilist);
		rnode = nrel->append(nn);
		copy_relation(rnode,rr->head(),sisilist);
	    }
	}
    }
    sisilist.remove_item(sub_root->contents());
    clear_up_sisilist(sisilist);
    return TRUE;
}

static void copy_relation(EST_Item *to,EST_Item *from,
 		          KVL<EST_Item_Content *,EST_Item *> &slist)
{
    // Construct next and down nodes of from, into to, mapping
    // stream_items through slist

    if (next(from))
	copy_relation(to->insert_after(map_ling_item(next(from),slist)),
		      next(from),
		      slist);
    if (down(from))
	copy_relation(to->insert_below(map_ling_item(down(from),slist)),
		      down(from),
		      slist);
}

static EST_Item *map_ling_item(EST_Item *si,
			       KVL<EST_Item_Content *,EST_Item *> &s)
{
    // If si is already in s return its map otherwise copy
    // si and add it to the list
    EST_Item *msi;
    EST_Item *def = 0;
    
    msi = s.val_def(si->contents(),def);
    if (msi == def)
    {   // First time, so copy it and add to map list
	msi = new EST_Item(*si);
	s.add_item(si->contents(),msi);
    }
    return msi;
}

static void clear_up_sisilist(KVL<EST_Item_Content *,EST_Item *> &s)
{
    // The EST_Items in the value of this need to be freed, its
    // contents however will not be freed as they will be referenced
    // somewhere in the copied utterance
    
    for (EST_Litem *r=s.list.head(); r != 0; r=next(r))
	delete s.list(r).v;

}

static EST_Item *mapped_parent(EST_Item *i,const EST_String &relname,
			       KVL<EST_Item_Content *,EST_Item *> &s)
{
    EST_Item *p;

    if ((p=parent(i,relname)) == 0)
	return 0;
    else if (s.present(p->contents()))
	return map_ling_item(p,s)->as_relation(relname);
    else
	return 0;
}

static void sub_utt_copy(EST_Utterance &sub,EST_Item *i,
			 KVL<EST_Item_Content *,EST_Item *> &s)
{
    if (s.present(i->contents()))
	return;
    else
    {
	EST_Item *np,*d;
	EST_Litem *r;
	EST_Item *ni = map_ling_item(i,s);
	for (r = i->relations().list.head(); r; r = r=next(r))
	{
	    EST_String relname = i->relations().list(r).k;
	    if (!sub.has_relation(relname))
		sub.create_relation(relname)->append(ni);
	    else if ((np=mapped_parent(i,relname,s)) != 0)
		np->append_daughter(ni);
	    else
		sub.relation(relname)->append(ni);

	    // Do its daughters
	    for (d = i->as_relation(relname)->daughter1(); d ; d=next(d))
		sub_utt_copy(sub,d,s);
	}
    }
}

void sub_utterance(EST_Utterance &sub,EST_Item *i)
{
    // Extract i and all its relations, and daughters ... to build
    // a new utterance in sub.
    KVL<EST_Item_Content *,EST_Item *> sisilist;

    sub.clear();
    sub_utt_copy(sub,i,sisilist);
    
    clear_up_sisilist(sisilist);
}

