/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1995                                 */
/*-----------------------------------------------------------------------*/
/*            Stream_Item class (separated out Feb 98)                   */
/*   Contains lingusitic information for a node                          */
/*=======================================================================*/
#ifndef __EST_STREAM_ITEM_H__
#define __EST_STREAM_ITEM_H__

#include "EST_common.h"
#include "EST_String.h"
#include "EST_TList.h"
#include "EST_KV.h"
#include "EST_Val.h"
#include "EST_types.h"

#include "EST_Features.h"

/** A class for containing individual linguistic objects such as
words or phones.

Stream_items consist of a set of features and a set of links to
\Ref{EST_Relation}s that they are in.  Features consist of 
a name an values.  Values are \Ref{EST_Val}s thus allowing them
to be strings, floats, or any other arbitrary object.

Relations between stream items may be followed through the provided
member functions, and, more easily, convenience functions provided.

*/
class EST_Stream_Item {
 private:
    EST_String p_stream_name;
    EST_String p_name;
    int a;

    // Copy a stream_item, this only copies the features, not the relations
    void copy(const EST_Stream_Item &x);
 public:

    /**@name Constructor Functions */
    //@{
    /// Default constructor
    EST_Stream_Item() {init("");}
    /// Copy constructor
    EST_Stream_Item(const EST_Stream_Item &item) { copy(item); }
    /// destructor
    ~EST_Stream_Item();
    //@}

    /// General features for this item
    EST_Features f;

    /** return the name of the item, e.g. the name of the phone or the
	text of the word*/
    const EST_String &name() const {return p_name;}
    
    /// set name
    void set_name(const EST_String &s) {p_name = s;}    

    EST_Stream_Item& operator = (const EST_Stream_Item& a);
    friend ostream& operator << (ostream &s, const EST_Stream_Item &a);

    // These methods are due for deletion
    /**@name Initialisation Functions */
    //@{
    /// Initialise Stream_Item to be an item of type {\tt stream_name}
    void init(const EST_String &stream_name); 
    //@}
    /**@name Access Functions */
    //@{
    /** name of stream that this item is in, e.g. word or phone */
    const EST_String &stream_name() const {return p_stream_name;}
    /** Return the end time of the item */
    float end() const {return f("end");}
    /** Return linguistic address of this item*/
    int addr() const {return a;}	
    /// start time in seconds of item, derived from end of previous item
    float start() const;
    /// mid point in seconds of item
    float mid() const;
    /// duration of item
    float dur() const;
    /// set end time
    void set_end(float e) {f.set("end",e);}
    /// set linguistic address
    void set_addr(int x) {a = x;}

    /// set stream name to {\tt s}
    void set_stream_name(const EST_String &s) {p_stream_name = s;}

    EST_Stream_Item *n;		// next pointer
    EST_Stream_Item *p;		// previous pointer

    //@}

};

inline EST_Stream_Item *next(EST_Stream_Item *ptr) {return (ptr == 0 ? 0 : ptr->n);}
inline EST_Stream_Item *prev(EST_Stream_Item *ptr) {return (ptr == 0 ? 0 : ptr->p);}

inline float dur(EST_Stream_Item &e)
{ return e.f.F("end") - e.f.F("start");}

#endif				//  __EST_STREAM_ITEM_H__
