 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                         Copyright (c) 1996                            */
 /*                        All Rights Reserved.                           */
 /*                                                                       */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*                                                                       */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                     Author :  Paul Taylor                             */
 /*                     Date   :  April 1996                              */
 /* --------------------------------------------------------------------- */
 /*                        Matrix class                                   */
 /*                                                                       */
 /*************************************************************************/

#ifndef __FMatrix_H__
#define __FMatrix_H__

#include "EST_TSimpleMatrix.h"
#include "EST_TSimpleVector.h"


class EST_FVector;

/** A matrix class for floating point numbers. EST_FMatrix x should be
used instead of float **x wherever possible.*/
class EST_FMatrix : public EST_TSimpleMatrix<float> {
private:
public:
    /// size constructor
    EST_FMatrix(int m, int n):EST_TSimpleMatrix<float>(m, n)  {}
    /// copy constructor
    EST_FMatrix(const EST_FMatrix &a):EST_TSimpleMatrix<float>(a)  {}

    /// CHECK  - what does this do???
    EST_FMatrix(const EST_FMatrix &a, int b);
    /// default constructor
    EST_FMatrix():EST_TSimpleMatrix<float>()  {}

    /// Save in file (ascii or binary)
    EST_write_status save(const EST_String &filename,
			  const EST_String &type = "est_ascii" );
    /// Load from file (ascii or binary as defined in file)
    EST_read_status load(const EST_String &filename);
    /// Save in file in est format
    EST_write_status est_save(const EST_String &filename,
			      const EST_String &type);
    /// Load from file in est format (binary/ascii defined in file itself)
    EST_read_status est_load(const EST_String &filename);

    /// Copy 2-d array {\tt x} of size {\tt rows x cols} into matrix.
    void copyin(float **x, int rows, int cols);

    /// Add elements of 2 same sized matrices.
    EST_FMatrix &operator+=(const EST_FMatrix &a);

    /// Multiply all elements of matrix by {\tt x}.
    friend EST_FMatrix operator*(const EST_FMatrix &a, const float x);

    /// Multiply matrix by vector.
    friend EST_FVector operator*(const EST_FMatrix &a, const EST_FVector &v);

    /// Multiply vector by matrix
    friend EST_FVector operator*(const EST_FVector &v,const EST_FMatrix &a);

    /// Multiply matrix by matrix.
    friend EST_FMatrix operator*(const EST_FMatrix &a, const EST_FMatrix &b);
};

/** A matrix class for integers. {\tt EST_IMatrix x} should be
    used instead of {\tt int **x} wherever possible.
*/
class EST_IMatrix: public EST_TSimpleMatrix<int> {
private:
public:
    /// size constructor
    EST_IMatrix(int m, int n):EST_TSimpleMatrix<int>(m, n) {}
    /// copy constructor
    EST_IMatrix(EST_IMatrix &a):EST_TSimpleMatrix<int>(a) {}
    /// CHECK  - what does this do???
    EST_IMatrix(EST_IMatrix &a, int b);
    /// default constructor
    EST_IMatrix():EST_TSimpleMatrix<int>() {}
};

class EST_SMatrix: public EST_TSimpleMatrix<short> {
private:
public:
    /// size constructor
    EST_SMatrix(int m, int n):EST_TSimpleMatrix<short>(m, n) {}
    /// copy constructor
    EST_SMatrix(EST_SMatrix &a):EST_TSimpleMatrix<short>(a) {}
    /// CHECK  - what does this do???
    EST_SMatrix(EST_SMatrix &a, int b);
    /// default constructor
    EST_SMatrix():EST_TSimpleMatrix<short>() {}

  int rateconv(int old_sr, int new_sr);
};

/** A vector class for floating point numbers. 
    {\tt EST_FVector x} should be used instead of {\tt float *x}
    wherever possible.
*/
class EST_FVector: public EST_TSimpleVector<float> {
public:
    /// Size constructor.
  EST_FVector(int n): EST_TSimpleVector<float>(n) {}
    /// Copy constructor.
    EST_FVector(const EST_FVector &a): EST_TSimpleVector<float>(a) {}
    /// Default constructor.
    EST_FVector(): EST_TSimpleVector<float>() {}

    /// elementwise multiply
    EST_FVector &operator*=(const EST_FVector &s); 

    /// load vector from file <tt> filename</tt>.
    EST_read_status load(EST_String &filename);
};

/** A vector class for double precision floating point
    numbers. {\tt EST_DVector x} should be used instead of 
    {\tt float *x} wherever possible.
*/
class EST_DVector: public EST_TSimpleVector<double> {
public:
    /// Size constructor.
    EST_DVector(int n): EST_TSimpleVector<double>(n) {}
    /// Copy constructor.
    EST_DVector(EST_DVector &a): EST_TSimpleVector<double>(a) {}
    /// Default constructor.
    EST_DVector(): EST_TSimpleVector<double>() {}

    /// load vector from file <tt> filename</tt>.
    EST_read_status load(EST_String &filename);
};

/// find largest element
int matrix_max(const EST_IMatrix &a);
/// find largest element
float matrix_max(const EST_FMatrix &a);

int square(const EST_FMatrix &a);
/// inverse
int inverse(const EST_FMatrix &a, EST_FMatrix &inv);
int inverse(const EST_FMatrix &a, EST_FMatrix &inv, int &singularity);
/// pseudo inverse (for non-square matrices)
int pseudo_inverse(const EST_FMatrix &a, EST_FMatrix &inv); 
int pseudo_inverse(const EST_FMatrix &a, EST_FMatrix &inv,int &singularity); 

/// some useful matrix creators
/// make an identity matrix of dimension n
void eye(EST_FMatrix &a, const int n);

/// the user should use est_seed48 to seed the random number generator
void est_seed48();
/// used for variance
void make_random_diagonal_matrix(EST_FMatrix &M, const float scale);
/// used for covariance
void make_random_symmetric_matrix(EST_FMatrix &M, const float scale);

void make_poly_basis_function(EST_FMatrix &T, EST_FVector t);

/// elementwise add
EST_FVector add(const EST_FVector &a,const EST_FVector &b);
/// elementwise subtract
EST_FVector subtract(const EST_FVector &a,const EST_FVector &b);

/// enforce symmetry
void symmetrize(EST_FMatrix &a);
/// stack columns on top of each other to make a vector
void stack_matrix(const EST_FMatrix &M, EST_FVector &v);


float determinant(const EST_FMatrix &a);
/// not implemented ??
int singular(EST_FMatrix &a);
/// exchange rows and columns
void transpose(const EST_FMatrix &a,EST_FMatrix &b);
EST_FMatrix triangulate(const EST_FMatrix &a);

/// extract leading diagonal as a matrix
EST_FMatrix diagonalise(const EST_FMatrix &a);
/// extract leading diagonal as a vector
EST_FVector diagonal(const EST_FMatrix &a);
/// sum of elements
float sum(const EST_FMatrix &a);
void multiply(const EST_FMatrix &a, const EST_FMatrix &b, EST_FMatrix &c);

EST_FMatrix operator*(const EST_FMatrix &a, const EST_FMatrix &b);
EST_FMatrix operator-(const EST_FMatrix &a, const EST_FMatrix &b);
EST_FMatrix operator+(const EST_FMatrix &a, const EST_FMatrix &b);

EST_FMatrix sub(const EST_FMatrix &a, int row, int col);
EST_FMatrix fmatrix_abs(const EST_FMatrix &a);

EST_FMatrix row(const EST_FMatrix &a, int row);
EST_FMatrix column(const EST_FMatrix &a, int col);


/// least squares fit
bool
polynomial_fit(EST_FVector &x, EST_FVector &y, EST_FVector &co_effs, int order);

/// weighted least squares fit
bool
polynomial_fit(EST_FVector &x, EST_FVector &y, EST_FVector &co_effs, 
	       EST_FVector &weights, int order);

float
polynomial_value(const EST_FVector &coeffs, const float x);

/// vector dot product
float operator*(const EST_FVector &v1, const EST_FVector &v2);

#endif
