/*================================================================
 * txtplay:
 *	play midi event text from miditext via stdin
 *
 * usage: txtplay [-options]
 *
 * for example, use this as
 *	% miditext test.mid | txtplay
 *
 * Copyright (C) 1996,1997 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include "channel.h"
#include "seq.h"
#include "midievent.h"
#include "util.h"
#include "controls.h"
#include "options.h"

/*
 * prototypes
 */
static void clean_up(int sig);
static void cvt_cb(MidiEvent *ev, int ch, int v1, int v2);
static void cvt_cd(MidiEvent *ev, int ch, int v1, int v2);
static void cvt_nrpn(MidiEvent *ev, int ch, int v1, int v2);
static void cvt_tag(MidiEvent *ev, int ch, int v1, int v2);
static void cvt_tempo(MidiEvent *ev, int ch, int v1, int v2);
static int parse_text_file(char *buf, MidiEvent *ev, MidiInfo *mp);

#define OPTION_FLAGS	"hvc:r:D:POCM:s"
static awe_option_args long_options[] = {
	{"help", 1, 0, OPT_HELP},
	{"verbose", 2, 0, OPT_VERBOSE},
	{"chorus", 1, 0, OPT_CHORUS},
	{"reberb", 1, 0, OPT_REVERB},
	{"drum", 1, 0, OPT_DRUM},
	{"drumflag", 1, 0, OPT_DRUMFLAG},
	{"realpan", 2, 0, OPT_REALTIME_PAN},
	{"acceptall", 2, 0, OPT_ACCEPTALL},
	{"samecsec", 2, 0, OPT_SAMECSEC},
	{"mt32", 1, 0, OPT_MT32},
	{"autoskip", 2, 0, OPT_AUTOSKIP},
	{"chorusdepth", 1, 0, OPT_CHORUSDEPTH},
	{"reverbdepth", 1, 0, OPT_REVERBDEPTH},
	{"tuning", 2, 0, OPT_TUNING},
	{"chnprior", 2, 0, OPT_CHN_PRIOR},
	{"usefx", 2, 0, OPT_USEFX},
	{0, 0, 0, 0},
};
static int option_index;

MidiInfo glinfo;
int verbose = 0, debug = 0;

extern ControlMode dumb_control_mode;
ControlMode *ctl = &dumb_control_mode;

void print_usage(void)
{
	printf("txtplay -- play midi event text from stdin\n");
	printf("     copyright (c) 1996,1997 by Takashi Iwai\n");
	printf("usage: txtplay [-options]\n");
	print_general_options(stdout, OPTION_FLAGS, long_options);
}


/* dummy function for preload.c */
void midi_close(MidiInfo *mp) {}

#define MAX_DONE	32
#define MAX_PUSHLIST	32

static int npushed, ndone;
static MidiEvent ev_done[MAX_DONE], ev_pushed[MAX_PUSHLIST];

static void push_clear(void)
{
	npushed = 0;
	ndone = 0;
}

static int check_same_csec(MidiEvent *ev, MidiInfo *mp)
{
	int i;

	if (! mp->check_same_csec) return FALSE;

	if (ev->type == ME_NOTEON && VelOf(ev) != 0) {
		if (ndone < MAX_DONE)
			ev_done[ndone++] = *ev;
		return FALSE;
	}

	if ((ev->type != ME_NOTEON || VelOf(ev) != 0) && ev->type != ME_NOTEOFF)
		return FALSE;
	if (npushed >= MAX_PUSHLIST)
		return FALSE;

	for (i = 0; i < ndone; i++) {
		if (ev_done[i].channel == ev->channel &&
		    KeyOf(ev_done + i) == KeyOf(ev)) {
			ev_pushed[npushed++] = *ev;
			return TRUE;
		}
	}
	return FALSE;
}

static int pop_event(MidiInfo *mp)
{
	int i, rc;

	ndone = 0;
	if (npushed > 0) {
		seq_wait(1);
		mp->curcs++;
		for (i = 0; i < npushed; i++) {
			rc = do_midi_event(&ev_pushed[i], mp, EV_PLAY);
			if (rc != RC_NONE) {
				npushed = 0;
				return rc;
			}
		}
		npushed = 0;
	}
	return RC_NONE;
}


void main(int argc, char **argv)
{
	char buf[256];
	MidiEvent ev;
	FILE *fp;
	int cp;
	int note_started;

	memcpy(&glinfo, &gl_default, sizeof(glinfo));

	while ((cp = awe_getopt(argc, argv, OPTION_FLAGS, long_options, &option_index)) != -1) {
		general_options(OPTION_FLAGS, cp);
	}

	seq_blocking_mode(TRUE);
	seq_init(FALSE);
	signal(SIGTERM, clean_up);
	signal(SIGINT, clean_up);
	signal(SIGQUIT, clean_up);

	seq_set_drumchannels(glinfo.drumflag);
	seq_set_chorus(glinfo.chorus);
	seq_set_reverb(glinfo.reverb);
	seq_equalizer(glinfo.bass_level, glinfo.treble_level);
	seq_change_volume(glinfo.master_volume);
	seq_set_realtime_pan(glinfo.realtime_pan);
	seqbuf_dump();

	channel_init(&glinfo);

	glinfo.curcs = 0;
	glinfo.tempo = MIDI_DEFAULT_TEMPO;
	seq_clear(0);
	push_clear();

	fp = stdin;
	note_started = !glinfo.skip_blank_head;

	while (fgets(buf, sizeof(buf), fp)) {
		if (parse_text_file(buf, &ev, &glinfo) == -1)
			continue;
		if (ev.csec > glinfo.curcs) {
			if (pop_event(&glinfo) != RC_NONE)
				break;
			/* curcs may be changed in pop_event */
			if (note_started && ev.csec > glinfo.curcs)
				seq_wait(ev.csec - glinfo.curcs);
			glinfo.curcs = ev.csec;
		}

		if (ev.type == ME_NOTEON && !note_started) {
			seq_clear(0);
			note_started = TRUE;
		}

		if (!check_same_csec(&ev, &glinfo))
			do_midi_event(&ev, &glinfo, EV_PLAY);

		seqbuf_dump();
	}
	seqbuf_dump();
	seq_wait_time(glinfo.curcs);

	exit(0);
}

int play_midi_file(MidiInfo *mp)
{
	return RC_NONE;
}

static void clean_up(int sig)
{
	/*fprintf(stderr, "\nterminating..");*/
	seq_terminate_all();
	seq_end();
	exit(0);
}


typedef struct _CvtFmt {
	char *tag;
	int type;
	void (*convert)(MidiEvent *ev, int chan, int v1, int v2);
} CvtFmt;

static CvtFmt formats[] = {
	{"note_on", ME_NOTEON, cvt_cd},
	{"note_off", ME_NOTEOFF, cvt_cd},
	{"key_press", ME_KEYPRESSURE, cvt_cd},
	{"pitch_sens", ME_PITCH_SENS, cvt_cb},
	{"pitch_wheel", ME_PITCHWHEEL, cvt_cb},
	{"main_volume", ME_CONTROL, cvt_cd},
	{"pan", ME_CONTROL, cvt_cd},
	{"expression", ME_CONTROL, cvt_cd},
	{"sustain", ME_CONTROL, cvt_cd},
	{"chorus", ME_CONTROL, cvt_cd},
	{"reverb", ME_CONTROL, cvt_cd},
	{"sostenuto", ME_CONTROL, cvt_cd},
	{"modwheel", ME_CONTROL, cvt_cd},
	{"control", ME_CONTROL, cvt_cd},
	{"program", ME_PROGRAM, cvt_cb},
	{"reset_control", ME_RESET_CONTROLLERS, cvt_cb},
	{"all_notes_off", ME_ALL_NOTES_OFF, cvt_tag},
	{"all_sounds_off", ME_ALL_SOUNDS_OFF, cvt_tag},
	{"tone_bank", ME_TONE_BANK, cvt_cd},
	{"chn_press", ME_CHNPRESSURE, cvt_cb},
	{"tempo", ME_TEMPO, cvt_tempo},
	{"lyric", ME_NONE, cvt_tag},
	{"master_volume", ME_MASTER_VOLUME, cvt_cb},
	{"gs_nrpn", ME_GS_FX, cvt_cd},
	{"coarse_tune", ME_COARSETUNE, cvt_cb},
	{"end_of_tune", ME_EOT, cvt_tag},
	{"e1_delay", ME_AWE_FX, cvt_nrpn},
	{"e1_attack", ME_AWE_FX, cvt_nrpn},
	{"e1_hold", ME_AWE_FX, cvt_nrpn},
	{"e1_decay", ME_AWE_FX, cvt_nrpn},
	{"e1_release", ME_AWE_FX, cvt_nrpn},
	{"e1_sustain", ME_AWE_FX, cvt_nrpn},
	{"e1_pitch", ME_AWE_FX, cvt_nrpn},
	{"e1_cutoff", ME_AWE_FX, cvt_nrpn},
	{"e2_delay", ME_AWE_FX, cvt_nrpn},
	{"e2_attack", ME_AWE_FX, cvt_nrpn},
	{"e2_hold", ME_AWE_FX, cvt_nrpn},
	{"e2_decay", ME_AWE_FX, cvt_nrpn},
	{"e2_release", ME_AWE_FX, cvt_nrpn},
	{"e2_sustain", ME_AWE_FX, cvt_nrpn},
	{"L1_delay", ME_AWE_FX, cvt_nrpn},
	{"L1_freq", ME_AWE_FX, cvt_nrpn},
	{"L1_volume", ME_AWE_FX, cvt_nrpn},
	{"L1_pitch", ME_AWE_FX, cvt_nrpn},
	{"L1_cutoff", ME_AWE_FX, cvt_nrpn},
	{"L2_delay", ME_AWE_FX, cvt_nrpn},
	{"L2_freq", ME_AWE_FX, cvt_nrpn},
	{"L2_pitch", ME_AWE_FX, cvt_nrpn},
	{"fx_init_pitch", ME_AWE_FX, cvt_nrpn},
	{"fx_chorus", ME_AWE_FX, cvt_nrpn},
	{"fx_reverb", ME_AWE_FX, cvt_nrpn},
	{"fx_cutoff", ME_AWE_FX, cvt_nrpn},
	{"fx_filterQ", ME_AWE_FX, cvt_nrpn},
};


static void cvt_cb(MidiEvent *ev, int ch, int v1, int v2)
{
	ev->channel = ch;
	ev->p.val = v1;
}

static void cvt_cd(MidiEvent *ev, int ch, int v1, int v2)
{
	ev->channel = ch;
	ev->p.par[0] = v1;
	ev->p.par[1] = v2;
}

static void cvt_nrpn(MidiEvent *ev, int ch, int v1, int v2)
{
	ev->channel = ch;
	ev->p.par[0] = v1;
	ev->p.par[1] = v2 + 8192;
}

static void cvt_tag(MidiEvent *ev, int ch, int v1, int v2)
{
}

static void cvt_tempo(MidiEvent *ev, int ch, int v1, int v2)
{
	ev->p.val = ch;
}


static int parse_text_file(char *buf, MidiEvent *ev, MidiInfo *mp)
{
	int i;
	char *time, *cmd, *channel, *val1, *val2;
	int chn_val, val1_val, val2_val;

	/* number of midi events */
	if (*buf == 'm')
		return -1;

	/* special commands */
	if (*buf == '#') {
		cmd = strtok(buf, " \n");
		val1 = strtok(NULL, " \n");
		chn_val = (int)strtol(val1, NULL, 0);
		if (strcmp(buf, "#drumflag") == 0) {
			mp->drumflag = chn_val;
			channel_init(mp);
			channel_set(mp);
		} else if (strcmp(buf, "#midimode") == 0) {
			mp->midi_mode = chn_val;
			channel_init(mp);
		} else if (strcmp(buf, "#chorus") == 0) {
			mp->chorus = chn_val;
		} else if (strcmp(buf, "#reverb") == 0) {
			mp->reverb = chn_val;
		}
		return -1;
	}

	chn_val = 0;
	val1_val = 0;
	val2_val = 0;

	time = strtok(buf, " \n");
	if (time)
		ev->csec = atoi(time);
	time = strtok(NULL, " \n");
	if (time)
		ev->time = atoi(time);
	cmd = strtok(NULL, " \n");
	channel = strtok(NULL, " \n");
	if (channel) {
		if (*channel == '(')
			chn_val = (channel[1] - '0') * 10 + (channel[2] - '0');
		else /* tempo */
			chn_val = atoi(channel);
	}
	val1 = strtok(NULL, " \n");
	if (val1)
		val1_val = atoi(val1);
	val2 = strtok(NULL, " \n");
	if (val2)
		val2_val = atoi(val2);

	if (strcmp(cmd, "unknown") == 0)
		return -1;

	for (i = 0; i < numberof(formats); i++) {
		if (strcmp(cmd, formats[i].tag) == 0) {
			ev->type = formats[i].type;
			formats[i].convert(ev, chn_val, val1_val, val2_val);
			return 0;
		}
	}
	fprintf(stderr, "unknown cmd=%s\n", cmd);
	return -1;
}
