/*================================================================
 * initialize / finish exterenal control
 *================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "util.h"
#include "sighandle.h"
#include "ext_c.h"

/*----------------------------------------------------------------*/

static void child_killed(void);
static int shm_alloc(void);
static void shm_free(void);
static void start_player(int pin, int pout, int o_argc, char **o_argv);

/*----------------------------------------------------------------*/

PanelInfo *extpanel = NULL;
int pipe_in = -1, pipe_out = -1;
int shmid = -1;
int child_pid;

int ext_control_init(int write_pipe, int read_pipe, int do_shmid,
		     int redirect, int argc, char **argv)
{
	int rc, a_pipe[2], p_pipe[2];

	a_pipe[0] = a_pipe[1] = -1;
	p_pipe[0] = p_pipe[1] = -1;
		
	if (do_shmid) {
		if (! shm_alloc())
			return 0;
	}
	if (write_pipe) {
		if ((rc = pipe(p_pipe)) != 0)  {
			shm_free();
			return 0;
		}
	}
	if (read_pipe) {
		if ((rc = pipe(a_pipe)) != 0)  {
			shm_free();
			if (write_pipe) {
				close(p_pipe[0]);
				close(p_pipe[1]);
			}
			return 0;
		}
	}

	if ((child_pid = fork()) == 0) {
		/* child */
		if (write_pipe) close(a_pipe[0]);
		if (read_pipe) close(p_pipe[1]);
		start_player(p_pipe[0], a_pipe[1], argc, argv);
		fprintf(stderr, "fatal error: can't start drvmidi player\n");
		exit(1);
	} else if (child_pid == -1) {
		shm_free();
		if (write_pipe) {
			close(p_pipe[0]); close(p_pipe[1]);
		}
		if (read_pipe) {
			close(a_pipe[0]); close(a_pipe[1]);
		}
		return 0;
	}

	if (write_pipe) {
		close(a_pipe[1]);
		pipe_in = a_pipe[0];
	}
	if (read_pipe) {
		close(p_pipe[0]);
		pipe_out = p_pipe[1];
	}

	if (redirect) {
		dup2(pipe_in, fileno(stdin));
		close(pipe_in);
		pipe_in = fileno(stdin);
		dup2(pipe_out, fileno(stdout));
		close(pipe_out);
		pipe_out = fileno(stdout);
	}
	add_signal(SIGTERM, ext_control_end, 1);
	add_signal(SIGINT, ext_control_end, 1);
	add_signal(SIGCHLD, child_killed, 0);

	return 1;
}

static void child_killed(void)
{
	int st;
	waitpid(child_pid, &st, WNOHANG);
	if (! WIFEXITED(st)) {
		reset_signal(SIGCHLD);
		return;
	}
	child_pid = -1;
	ext_control_end();
	exit(0);
}

void ext_control_end(void)
{
	if (child_pid != -1) {
		int st;
		signal(SIGCHLD, SIG_DFL);
		kill(child_pid, SIGTERM);
		waitpid(child_pid, &st, 0);
	}
	shm_free();
	if (pipe_in >= 0) {
		close(pipe_in);
		pipe_in = -1;
	}
	if (pipe_out >= 0) {
		close(pipe_out);
		pipe_out = -1;
	}
}

static int shm_alloc(void)
{
	shmid = shmget(IPC_PRIVATE, sizeof(PanelInfo),
		       IPC_CREAT|0600);
	if (shmid < 0) {
		fprintf(stderr, "can't allocate shared memory\n");
		return 0;
	}

	extpanel = (PanelInfo *)shmat(shmid, 0, 0);
	extpanel->reset_panel = 0;
	extpanel->multi_part = 0;

	return  1;
}

static void shm_free(void)
{
	if (shmid >= 0) {
		shmctl(shmid, IPC_RMID,NULL);
		shmdt((char*)extpanel);
		shmid = -1;
	}
}

#define PLAYER_PROGRAM	"drvmidi"

static void start_player(int pin, int pout, int o_argc, char **o_argv)
{
	int i, argc;
	char **argv;

	if ((argv = (char**)malloc(sizeof(char*) * (o_argc + 6))) == NULL) {
		fprintf(stderr, "can't malloc\n"); exit(1);
	}

	argc = 0;
	argv[argc++] = PLAYER_PROGRAM;
	argv[argc++] = "-ip";

	if (shmid >= 0) {
		static char tmp[32];
		sprintf(tmp, "--shmid=%d", shmid);
		argv[argc++] = tmp;
		argv[argc++] = "--trace=on";
	}
	if (pin >= 0) {
		static char tmp[32];
		sprintf(tmp, "--pipein=%d", pin);
		argv[argc++] = tmp;
	}
	if (pout >= 0) {
		static char tmp[32];
		sprintf(tmp, "--pipeout=%d", pout);
		argv[argc++] = tmp;
	}
	
	for (i = 0; i < o_argc; i++)
		argv[argc++] = o_argv[i];

	argv[argc++] = NULL;
	execvp(PLAYER_PROGRAM, argv);
}

/*================================================================
 *================================================================*/

void ext_pipe_printf(char *fmt, ...)
{
	char buf[1024];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	ext_pipe_puts(buf);
}

void ext_pipe_puts(char *str)
{
	int len;
	char lf = '\n';
	len = strlen(str);
	write(pipe_out, str, len);
	write(pipe_out, &lf, 1);
}

/*================================================================
 *================================================================*/

void ext_control_load(char *name)
{
	ext_pipe_puts("LOAD");
	ext_pipe_puts(name);
}

void ext_control_next(void)
{
	ext_pipe_puts("NEXT");
}

void ext_control_prev(void)
{
	ext_pipe_puts("PREV");
}

void ext_control_restart(void)
{
	ext_pipe_puts("RSTA");
}

void ext_control_stop(void)
{
	ext_pipe_puts("QUIT");
}

void ext_control_pause(void)
{
	ext_pipe_puts("STOP");
}

void ext_control_kill(void)
{
	ext_pipe_puts("ZIPP");
}

void ext_control_fwrd(void)
{
	ext_pipe_puts("FWRD");
}

void ext_control_back(void)
{
	ext_pipe_puts("BACK");
}

void ext_control_jump(int csec)
{
	ext_pipe_printf("JUMP %d", csec);
}

void ext_control_move(int csec)
{
	ext_pipe_printf("MOVE %d", csec);
}

void ext_control_chorus(int mode)
{
	ext_pipe_printf("CHRS %d", mode);
}

void ext_control_reverb(int mode)
{
	ext_pipe_printf("EVRB %d", mode);
}

void ext_control_volume(int vol)
{
	ext_pipe_printf("VOLM %d", vol);
}

void ext_control_bass(int vol)
{
	ext_pipe_printf("VBAS %d", vol);
}

void ext_control_treble(int vol)
{
	ext_pipe_printf("VTRB %d", vol);
}

void ext_control_keychange(int offset)
{
	ext_pipe_printf("OFST %d", offset);
}

