/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996, 1997, 1998 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    bhor0533@lehr.chem.TU-Berlin.DE
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#include <sys/types.h>
#endif
#include <ctype.h>
#include <dirent.h>
#if defined Linux || defined NEXTSTEP
#include <sys/dir.h>
#endif
#include "main.h"

extern void set_tty(int);
extern int dlastentry;
extern int tlastentry;
extern char *tfiles[HUNAME]; /* holds the selected files */
extern char *dfiles[HUNAME]; /* holds the selected files */
extern int set_entry_in_files(char **, char *, int *);
extern void rm_entry_from_files(char **, char *, int *);
extern void display_pa(char **, char *, char *, int);
extern char *tabdir(char *);
short mycmp(char *, char *);

extern char *headline;

short create_dir(char *s, char *isdir)
{
   char c, *tmp=NULL, *cdir=NULL;
   struct stat buf;
   short last=0;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));

   for (tmp=cdir; (*tmp = *s); tmp++, s++);
   if (*--tmp == '/' && tmp != cdir) *tmp = '\0';
   else tmp++;
   if (!isdir) {
      for (; *tmp != '/' && tmp != cdir; tmp--);
      if (tmp == cdir) ++tmp;
      *tmp = '\0';
   }
   if (stat(cdir, &buf)) {
      fprintf(stderr, "\n** `%s' does not exist!\n** Create that directory (y|n)? ", cdir);
      while((c = fgetc(stdin)) != 'y' && c != 'Y' && c != 'n' && c != 'N');
      if (c == 'n' || c == 'N') {
         if (cdir) free(cdir);
         return 0;
      }
   }
   if (isdir) strcat(cdir, isdir);
   
   for (tmp=cdir+1; *(tmp-1); tmp++) {
      if (*tmp == '/' || !*tmp) {
         if (!*tmp) last = 1;
         *tmp = '\0';
         if (stat(cdir, &buf)) {
            if (mkdir(cdir, 0755) < 0) {
               fprintf(stderr, "\n** %s: %s!\n", cdir, _PFTP_ERROR_ARRAY_);
               fprintf(stderr, "*** Type any key to continue ***");
               fgetc(stdin);
               if (cdir) free(cdir);
               return 0;
            }
         }
         if (!last) *tmp = '/';
      }
   }
   if (cdir) free(cdir);
   return 1;
}


short copy_file(char *s, char *d)
{
   FILE *sfp=NULL, *dfp=NULL;
   size_t n=0;
   extern char *saving_str;
   char str[BUFSIZE], c, *stmp=NULL;
   struct stat buf;

   stmp = d;
   while (!access(stmp, F_OK) && !(*statstr)->OVERWRITE) {
      if ((*statstr)->_SKIP_) return 1;
      fprintf(stderr, "** File %s exists!\n** Overwrite(y|n) all(Y), skip(s) all(S), cancel(c) ? ", stmp);
      c = fgetc(stdin);
      fprintf(stderr, "\n%s", saving_str);
      if(c == 'Y' || c == 'y') {
         if (stat(stmp, &buf) < 0) {
            fprintf(stderr, "** stat: %s: %s\n", \
            stmp, _PFTP_ERROR_ARRAY_);
            exit(1);
         }
         else if ((buf.st_mode & 0200) != 0200) {
            if (chmod(stmp, buf.st_mode|0200) < 0) {
               fprintf(stderr, "** chmod: %s: %s\n", \
               stmp, _PFTP_ERROR_ARRAY_);
               exit(1);
            }
         }
         if (c == 'Y') (*statstr)->OVERWRITE = BIT_ONE;
         break;
      }
      else if(c == 's') return 1;
      else if(c == 'S') {
         (*statstr)->_SKIP_ = BIT_ONE;
         return 1;
      }
      else if (c == 'c') return 0;
      else {
         strcpy(str, stmp);
         if (!(stmp = tabdir(str))) return 0;
         fprintf(stderr, "\n%s", saving_str);
      }   
   }

   /*
    * Open the source and the destination files.
    */
   if ((sfp = fopen(s, "r")) == NULL) {
      fprintf(stderr, "\n** %s:\n** %s\n", s, _PFTP_ERROR_ARRAY_);
      fprintf(stderr, "*** Type any key to continue ***");
      fgetc(stdin);
      return 0;
   }
   if ((dfp = fopen(stmp, "w")) == NULL) {
      fprintf(stderr, "\n** %s:\n** %s\n", d, _PFTP_ERROR_ARRAY_);
      fprintf(stderr, "*** Type any key to continue ***");
      fgetc(stdin);
      return 0;
   }
   
   /*
    * Copy source file.
    */
   while ((n = fread(str, sizeof(char), BUFSIZE, sfp)) > 0) {
      fwrite(str, sizeof(char), n, dfp);
   }
   fclose(dfp);
   fclose(sfp);

   /*
    * Set the permissions.
    */
   if (stat(s, &buf) < 0) {
      fprintf(stderr, "\n** stat: %s:\n** %s\n", s, _PFTP_ERROR_ARRAY_);
      fprintf(stderr, "*** Type any key to continue ***");
      fgetc(stdin);
      return 0;
   }
   if (chmod(d, buf.st_mode) < 0) {
      fprintf(stderr, "\n** chmod: %s:\n** %s\n", d, _PFTP_ERROR_ARRAY_);
      fprintf(stderr, "*** Type any key to continue ***");
      fgetc(stdin);
      return 0;
   }

   /*
    * Mark files for being removed.
    */
   set_entry_in_files(dfiles, s, &dlastentry);

   return 1;
}


short save_dir(char *s, char *d)
{
   char *sdir=NULL;
   char *ddir=NULL;
   char *str[HUNAME];
   char **tmp=NULL;
   struct stat buf;
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   MEM_CHECK((ddir = (char *)calloc(LONAME, sizeof(char))));
   MEM_CHECK((sdir = (char *)calloc(LONAME, sizeof(char))));
   tmp = str;

   if (!stat(s, &buf)) {
      if (buf.st_mode & S_IFDIR) {
         if (!(dp = opendir(s))) {
            fprintf(stdout, "\n** %s:\n** %s\n", s, _PFTP_ERROR_ARRAY_);
            fprintf(stderr, "*** Type any key to continue ***");
            fgetc(stdin);
            if (sdir) free(sdir);
            if (ddir) free(ddir);
            return 0;
         }
         while ((dir = readdir(dp)))
         {
            if (dir->d_ino && mycmp(dir->d_name, "..") && mycmp(dir->d_name, ".") \
               && mycmp("._pftp_info_file", dir->d_name)) {
               MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
               strcpy(*tmp++, dir->d_name);
            }
         }
      }
      else {
         if (!copy_file(sdir, ddir)) return 0;
         if (sdir) free(sdir);
         if (ddir) free(ddir);
         return 1;
      }
      closedir(dp);
      for (tmp--; tmp+1 != str; tmp--) {
         sprintf(sdir, "%s/%s", s, *tmp);
         sprintf(ddir, "%s/%s", d, *tmp);
         free(*tmp);
         stat(sdir, &buf);
         if (buf.st_mode & S_IFDIR) {
            if (stat(ddir, &buf)) {
               if (mkdir(ddir, 0755) < 0) {
                  fprintf(stderr, "\n** %s:\n** %s!\n", ddir, _PFTP_ERROR_ARRAY_);
                  fprintf(stderr, "*** Type any key to continue ***");
                  fgetc(stdin);
                  if (sdir) free(sdir);
                  if (ddir) free(ddir);
                  return 0;
               }
            }
            else if (!(buf.st_mode & S_IFDIR)) {
               fprintf(stderr, "\n** `%s' is not a directory!\n", ddir);
               fprintf(stderr, "*** Type any key to continue ***");
               fgetc(stdin);
               if (sdir) free(sdir);
               if (ddir) free(ddir);
               return 0;
            }
            save_dir(sdir, ddir);
         }
         else if (!copy_file(sdir, ddir)) return 0;
      }
      if (sdir) free(sdir);
      if (ddir) free(ddir);
      return 1;
   }
   if (sdir) free(sdir);
   if (ddir) free(ddir);
   return 0;
}


short save_tags(char *d)
{
   int i=0;
   char **tag=NULL, *tmp=NULL;
   char *cdir=NULL;
   struct stat buf;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));

   for (tag=tfiles; tag-tfiles <= tlastentry; tag++) {
      if (*tag) {
         if (!stat(*tag, &buf) && (buf.st_mode & S_IFREG)) {
            for (i=0, tmp=*tag; i < 2 && *tmp; tmp++) if (*tmp == '/') i++;
            sprintf(cdir, "%s/%s", d, tmp);
            for (tmp=cdir+1; *tmp; tmp++) {
               if (*tmp == '/') {
                  *tmp = '\0';
                  if (stat(cdir, &buf)) {
                     if (mkdir(cdir, 0755) < 0) {
                        fprintf(stderr, "\n** %s:\n** %s!\n", cdir, _PFTP_ERROR_ARRAY_);
                        fprintf(stderr, "*** Type any key to continue ***");
                        fgetc(stdin);
                        if (cdir) free(cdir);
                        return 0;
                     }
                  }
                  *tmp = '/';
               }
            }
            if (!copy_file(*tag, cdir)) return 0;
         }
         rm_entry_from_files(tfiles, *tag, &tlastentry);
      }
   }
   if (cdir) free(cdir);
   return 1;
}

#define HELPLINES 25

void mhelp(char *secondline) {
   extern char *headline;
   char *helpstr[HELPLINES+1] = {
      "     <l>, <right>     Enter directory.                               \n",
      "     <h>, <left>      Leave directory.                               \n",
      "     <j>, <down>      Go to the next entry.                          \n",
      "     <k>, <up>        Go to the previous entry.                      \n",
      "     <m>              Go back to the main menu.                      \n",
      "     <S>              Main menu: Save the current message file.      \n",
      "     <s>              Save the current file/directory recursively or \n",
      "                      if taged files exist select and save.  Within  \n",
      "                      this option the following keys are recognized: \n",
      "                      <tab>       Expand file and directory names    \n",
      "                                  and list directory entries.        \n",
      "                      <return>    Accept current file/directory.     \n",
      "                      <C-G>       Abort selection and do not save.   \n",
      "                      <C-L>       Clear screen.                      \n",
      "                      <key>       Print <key>.                       \n",
      "     <t>              Tag current file/directory recursively.        \n",
      "                      Tagged files/directories can be saved with <s>.\n",
      "     <d>              Mark current file/directory recursively as     \n",
      "                      to be deleted on exit.                         \n",
      "     <return>         Main menu: Display the message file.           \n",
      "                      Submenu: Display the current file.             \n",
      "     <n>              Main menu: Mark current entry as new.          \n",
      "     <space>          Go to the next page.                           \n",
      "     <b>              Go to the previous page.                       \n",
      "     <w>              Display warranty.                              \n",
      "     <?>              Print this help list.                          \n"
   };
   
   fprintf(stdout, "[H[J");
   display_pa(helpstr, headline, secondline, HELPLINES);  
}


char *get_subject(char *subj, char *user, char *date)
{
   FILE *fp=NULL;
   char *str=NULL;
   struct stat buf;

   MEM_CHECK((str = (char *)calloc(LONAME, sizeof(char))));
   sprintf(str, "%s/%s/._pftp_info_file", user, date);
   if (!stat(str, &buf)) {
      if (!(fp = fopen(str, "r"))) {
         free(str);
         return NULL;
      }
      while (fgets(str, SONAME, fp)) {
         if (!strncmp("Subject:", str, 8)) {
            strcpy(subj, str+9);
            free(str);
            fclose(fp);
            return subj;
         }
      }
      fclose(fp);
   }
   return NULL;
}


short pftp_pager(char *cdir, char *secondline, char *blankline)
{
   char *ltmp[HUNAME], *dtmp=NULL;
   char **tmp, *str=NULL, *ctmp=NULL;
   short islonger=0;
   long tmpend=0;
   FILE *fp=NULL;

   MEM_CHECK((str = (char *)calloc(LONAME, sizeof(char))));

   if (!(fp = fopen(cdir, "r"))) {
      fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
      fprintf(stdout, "[%dH** %s: %s\n",
      _WINROWS_, cdir, _PFTP_ERROR_ARRAY_);
      return 0;
   }
   for (tmp=ltmp; fgets(str, LONAME, fp);) {
      for (ctmp=str, islonger=0; *ctmp && tmpend < HUNAME; islonger=1) {
         tmpend++;
         MEM_CHECK((*tmp++ = (char *)calloc(LONAME, sizeof(char))));
         for (dtmp=*(tmp-1); *ctmp && dtmp - *(tmp-1) < _WINCOLS_ - 1 - islonger; dtmp++, ctmp++) {
            if (islonger) {
               *dtmp = '+';
               ctmp--;
               islonger=0;
            }
            else *dtmp = isprint(*ctmp) ? *ctmp: ' ';
         }
         if (!*ctmp) for (; dtmp - *(tmp-1) < _WINCOLS_ - 1; dtmp++) *dtmp = ' ';
         *dtmp = '\n';
         *++dtmp = '\0';
      }
   }
   *tmp = NULL;
   fclose(fp);
   free(str);
   display_pa(ltmp, headline, secondline, --tmpend);  
   for (tmp--; tmp != ltmp; tmp--) free(*tmp);
   return 1;
}

#define WHELPLINES 14

void warranty(void) {
   char *firstline=NULL, *blankline=NULL, *ctmp=NULL;
   char *warrantystr[WHELPLINES+1] = {
      "Copyright (C) 1996, 1997, 1998 Ben Schluricke                          \n",
      "                                                                       \n",
      "This program is free software; you can redistribute it and/or modify   \n",
      "it under the terms of the GNU General Public License as published by   \n",
      "the Free Software Foundation; either version 2 of the License, or (at  \n",
      "your option) any later version.                                        \n",
      "                                                                       \n",
      "This program is distributed in the hope that it will be useful, but    \n",
      "WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-   \n",
      "ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General       \n",
      "Public License for more details.                                       \n",
      "                                                                       \n",
      "You should have received a copy of the GNU General Public License      \n",
      "along with this program; if not, write to the Free Software Foundation,\n",
      "Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                          \n"
   };

   MEM_CHECK((firstline = (char *)calloc(LONAME, sizeof(char))));
   sprintf(firstline, "*** Port-FTP Version %s Copyright (C) 1996, 1997, 1998 Ben Schluricke ***", VERSION); 
   MEM_CHECK((blankline = (char *)calloc(SONAME, sizeof(char))));
   for (ctmp=blankline; ctmp - blankline < _WINCOLS_; ctmp++) *ctmp = ' ';
   *ctmp='\r';
   *++ctmp='\0';
   
   fprintf(stdout, "[H[J");
   display_pa(warrantystr, firstline, blankline, WHELPLINES);  
   free(blankline);
   free(firstline);
}
