/*------------------------------------------------------------------------*
 * $Id: writer.c,v 1.2 1995/06/25 23:20:45 donovan Exp donovan $
 *
 * Purpose:	Write out pages and unswizzle pointers
 *
 * Notes:
 *------------------------------------------------------------------------*
 * $Log: writer.c,v $
 * Revision 1.2  1995/06/25  23:20:45  donovan
 * checkpoint
 *
 *------------------------------------------------------------------------*/

#include "rstoret.h"
#include "compress.h"
#include <string.h>
#include <rscheme/hashfn.h>
#include <rscheme/hashmain.h>
#include <rscheme/scheme.h>
#include "alloc.h"
#include "swizzler.h"

/*-------------------------------------------------------------------*/

/* 
 * save the translated pointer in the page's context
 * so we don't have to go through all this work again
 * if we decide to write the page out after all
 */

static void an_xlated_ptr( write_ctx_t *ctx, obj x )
{
  if (ctx->xlated_ptr)
    {
	  /* (but skip the saving if we're not going to write,
	   * ie, if we're just noticing an object's refs
	   */
      assert( ctx->xlated_ptr < &ctx->xlated_ptrs[MM_PAGE_SIZE/sizeof(obj)] );
      *ctx->xlated_ptr++ = x;
    }
}

/*-------------------------------------------------------------------*/

static void notice_gvec_refs( write_ctx_t *ctx, struct PHeapHdr *p,
			      obj *start, UINT_32 len );
static void notice_template_refs( write_ctx_t *ctx, struct PHeapHdr *p,
				 obj *start, UINT_32 len );

/*-------------------------------------------------------------------*/

static void init_ref_tbl( struct ref_tbl *reft )
{
    reft->num_refs = 0;
    reft->cap_refs = QUICK_NREFS;
    reft->refs = reft->temp;
}

static void close_ref_tbl( struct ref_tbl *reft )
{
    if (reft->refs != reft->temp)
	free( reft->refs );
}

static struct PageRef *alloc_page_ref( struct ref_tbl *reft )
{
    if (reft->num_refs >= reft->cap_refs)
    {
	reft->cap_refs *= 2;
	if (reft->refs == reft->temp)
	{
	    reft->refs = ALLOCN( struct PageRef, reft->cap_refs );
	    memcpy( reft->refs, reft->temp, 
	    	    sizeof(struct PageRef) * QUICK_NREFS);
	}
	else
	    REALLOCN( reft->refs, struct PageRef, reft->cap_refs );
    }
    return &reft->refs[ reft->num_refs++ ];
}

static unsigned get_ref( struct ref_tbl *reft, 
			 struct VMPageRecord *into )
{
struct PageRef *p;
unsigned i;

    for (i=0, p=reft->refs; i<reft->num_refs; i++, p++)
	if ((p->base_page_num == into->ref.base_page_num)
	    && (p->nth_page == into->ref.nth_page))
	{
	    return i;
	}
    p = alloc_page_ref( reft );
    *p = into->ref;
    return i;
}

static unsigned get_pivot_ref( struct ref_tbl *reft, 
			       UINT_32 pivot_page_num )
{
struct PageRef *p;
unsigned i;

    for (i=0, p=reft->refs; i<reft->num_refs; i++, p++)
	if (p->base_page_num == pivot_page_num)
	{
	    return i;
	}
    p = alloc_page_ref( reft );
    memset( p, 0, sizeof( struct PageRef ) );
    p->base_page_num = pivot_page_num;
    p->first = 1;
    p->indirect = 1;
    return i;
}


/* actually translate a PTR to it's persistent representation

   (obviously a NOP unless it's a PTR)
   and record the failure if it can't be translated
   into a persistent pointer representation.

   the translation is stored in the context's xlated_ptrs
   table to avoid the lookup on the second pass

   will replace a pointer to the stale copy of a relocated object 
   with a pointer to the new object
*/

/* failing_unit is the object to add to the failure list
 * in case the lookup fails
 */
   
static void table_lookup_or_fail( write_ctx_t *ctx,
				  obj lookup_table,
				  obj failing_unit,
				  obj key_value )
{
  obj refnum;

  refnum = objecttable_lookup( lookup_table,
			       obj_hash(key_value),
			       key_value );
  if (EQ(refnum,FALSE_OBJ))
    {
      ctx->failed = cons( failing_unit, ctx->failed );
    }
  else
    {
      unsigned n;
      obj item;

      /* a reference number was provided.  bind it to an
       * indirect page reference from the current page
       * and build the appropriate pointer which includes
       * a POINTER_TAG so it will be recognized as an entity 
       * deserving translation at load time
       *
       * (note that the refnum is a FIXNUM, and appears so
       * on the scheme side)
       */
      n = get_pivot_ref( &ctx->refs, VAL(refnum) >> 8 );
      item  = OBJ( (VAL(refnum) & 0xFF) 
		  + (n << 16) 
		  + POINTER_TAG );
      an_xlated_ptr( ctx, item );
    }
}

void notice_page_ref( write_ctx_t *ctx, obj *pitem )
{
struct VMPageRecord *vmp;
unsigned n;
obj rel, item;

    item = *pitem;
    if (!OBJ_ISA_PTR(item))
      return;

 again:
    /* check for an into-store ptr */
    vmp = addr_to_vm_page_record( ctx->store, PTR_TO_PHH(item) );
    if (vmp)
    {
      /* it's into another page */
      /*  [CR 600] need to subtract off offsets so we're pointing
       *  to real data before we mask; otherwise, we could be masking
       *  off a significant page bit
       */
      UINT_32 addr = (UINT_32)(PTR_TO_PHH(item));
      UINT_32 offset = VAL(PHH_TO_PTR(addr & MM_PAGE_MASK));

      n = get_ref( &ctx->refs, vmp );
      item = OBJ( (n << 16) + offset );
      goto success;
    }
    else if (!EQ((rel = objecttable_lookup( ctx->store->reloc_table,
					    obj_hash(item),
					    item )),FALSE_OBJ))
    {
      /* store the new pointer in the object making the reference,
	 and try again 
      */
      item = *pitem = rel;
      goto again;
    }
    else
    {
      table_lookup_or_fail( ctx,
			    ctx->store->pivot_table,
			    item, 
			    item );
      return;
    }
    /* failed! */
    ctx->failed = cons( item, ctx->failed );
    return;
 success:
    an_xlated_ptr( ctx, item );
}

/*  specialized functions for the first two slots in a template
 *  analagous to notice_page_ref, but treat object references
 *  specially, as via the local_code_ptrs and local_fn_descrs
 *  tables
 */

				 

static void notice_code_ptr( write_ctx_t *ctx, obj inside, obj *pitem )
{
  table_lookup_or_fail( ctx, 
		        ctx->store->local_code_ptrs,
		        inside,
		        *pitem );
}

static void notice_fn_descr_ptr( write_ctx_t *ctx, obj inside, obj *pitem )
{
  table_lookup_or_fail( ctx, 
		        ctx->store->local_fn_descrs,
		        inside,
		        *pitem );
}


struct PHeapHdr *large_object_hdr( struct VMPageRecord *page )
{
    return (struct PHeapHdr *)((char *)page->mem_address
			        + sizeof(struct FirstPageHdr)
    				- MM_PAGE_SIZE * page->ref.nth_page);
}

struct PHeapHdr *first_on_first( struct VMPageRecord *page )
{
  assert( page->ref.first );

  return (struct PHeapHdr *)((char *)page->mem_address 
			     + sizeof(struct FirstPageHdr) );
}

/* this just accesses the already-translated pointer */

void unswiz_and_compress( write_ctx_t *ctx, obj thing )
{
    if (OBJ_ISA_PTR(thing))
      thing = *(ctx->xlated_ptr)++;
    compress_obj( &ctx->c, thing );
}

static void compress_LR( struct Compressor *c, struct LocationRef lr )
{
  union { 
    struct LocationRef lr;
    UINT_32 w[2];
  } overload;

  overload.lr = lr;
  compress_word( c, overload.w[0] );
  compress_word( c, overload.w[1] );
}

enum trav_mode {
  TRAV_NOTICE,
  TRAV_WRITE
};

static void traverse_pob( write_ctx_t *ctx, 
			  struct PHeapHdr *hdr,
			  void *src, UINT_32 from_start, 
			  UINT_32 len, enum trav_mode mode )
{
  UINT_32 i = 0;
  obj *gvecp = src;
  enum SwizzleMode m = mode_for_object(hdr);

  switch (m)
    {
    case SWIZ_MODE_TEMPLATE:

      if (from_start == 0)
	{
	  if (mode == TRAV_NOTICE)
	    {
	      assert( len >= SLOT(2) );
	      notice_code_ptr( ctx, PHH_TO_PTR(hdr), gvecp++ );
	      notice_fn_descr_ptr( ctx, PHH_TO_PTR(hdr), gvecp++ );
	    }
	  else
	    {
	      compress_obj( &ctx->c, *(ctx->xlated_ptr)++ );
	      compress_obj( &ctx->c, *(ctx->xlated_ptr)++ );
	    }
	  i = SLOT(2);
	}
      /* fall through into gvec. */

    case SWIZ_MODE_GVEC:
      for (; i<len; i+=SLOT(1), gvecp++)
	{
	  if (mode == TRAV_NOTICE)
	    notice_page_ref( ctx, gvecp );
	  else
	    unswiz_and_compress( ctx, *gvecp );
	}
      break;

    case SWIZ_MODE_ALLOC_AREA:
      { 
	PAllocArea *aa = src;
	assert( from_start == 0 );
	
	if (mode == TRAV_NOTICE)
	  {
	    notice_page_ref( ctx, &aa->entry );
	    notice_page_ref( ctx, &aa->reserved );
	  }
	else
	  {
	    UINT_32 *p;

	    unswiz_and_compress( ctx, aa->entry );
	    unswiz_and_compress( ctx, aa->reserved );

	    compress_LR( &ctx->c, aa->current_LR );
	    compress_LR( &ctx->c, aa->parent_LR );

	    i = 2 * sizeof(struct LocationRef)
	        + 2 * sizeof(obj);

	    p = (UINT_32 *)((char *)src + i);
	    for (; i<len; i+=SLOT(1))
	      {
		compress_word( &ctx->c, *p++ );
	      }
	  }
      }
      break;

      
    case SWIZ_MODE_PADDR_VEC:
      /* just verify that the owner is this pstore, and
       * skip the first two words 
       */
      if (from_start == 0)
	{
	  if (((struct PAddrVec *)src)->owner != ctx->store)
	    {
	      scheme_error( "<persistent-addr> in different store: ~s",
			    1, PHH_TO_PTR(hdr) );
	    }
	  i = SLOT(2);
	  src = (char *)src + SLOT(2);
	}

    case SWIZ_MODE_FLOAT:
    case SWIZ_MODE_UINT32:
    case SWIZ_MODE_BVEC:

      if (mode == TRAV_WRITE)
	{
	  UINT_32 *p = src;

	  for (; i<len; i+=SLOT(1))
	    {
	      compress_word( &ctx->c, *p++ );
	    }
	}
      break;
    default:
      {
	struct swiz_mode_handler *h = get_swiz_mode_handler( ctx->store, m );
	if (mode == TRAV_WRITE)
	  h->trav_write( h, ctx, hdr, src, from_start, len );
	else
	  h->trav_notice( h, ctx, hdr, src, from_start, len );
	break;
      }
    }
}

static void traverse_page( write_ctx_t *ctx, 
			   struct VMPageRecord *page,
			   enum trav_mode mode )
{
  struct PHeapHdr *p;
  unsigned i;
  
  if (page->ref.first)
    {
      struct PHeapHdr *limit;
      struct FirstPageHdr *fph = (struct FirstPageHdr *)page->mem_address; 
      obj tmp;

      /*
       *  write out the page's allocation-area pointer
       */

      tmp = DATAPTR_TO_PTR(fph->area);
      
      if (mode == TRAV_NOTICE)
	{
	  notice_page_ref( ctx, &tmp );
	}
      else
	{
	  unswiz_and_compress( ctx, tmp );
	}

      /*
       *  traverse / write out the objects on the page
       */

      p = first_on_first(page);
      limit = (struct PHeapHdr *)((char *)page->mem_address + MM_PAGE_SIZE);
      while (p < limit)
	{
	  UINT_32 N, i;

	  /* check for an early end of the page */
	  if (p->mem_size == 0)
	    {
	      if (mode == TRAV_WRITE)
		{
		  compress_word( &ctx->c, 0 );
		}
	      break;
	    }

	  /* p points to the PHeapHdr of an object on this page */

	  if (mode == TRAV_WRITE)
	    {
	      compress_word( &ctx->c, p->mem_size );
	      compress_word( &ctx->c, p->pstore_flags );
	    }


	  if (mode == TRAV_NOTICE)
	    {
	      notice_page_ref( ctx, &p->rs_header.pob_class );
	    }
	  else
	    {
	      unswiz_and_compress( ctx, p->rs_header.pob_class );
	    }

	  N = p->rs_header.pob_size;
	  if (mode == TRAV_WRITE)
	    {
	      compress_word( &ctx->c, N );
	    }
	  
	  if (page->ref.nth_page > 1)
	    {
	      /* clip the length to go only to the end of the page */
	      N = MM_PAGE_SIZE - sizeof(struct PHeapHdr) 
		            - sizeof(struct FirstPageHdr);
	    }

	  traverse_pob( ctx, p, p+1, 0, N, mode );

	  /* go on to the next object */
	  p = (struct PHeapHdr *)((char *)p + p->mem_size);
	}
    }
  else
    {
      UINT_32 i, M, N;
      struct PHeapHdr *p;
      
      p = large_object_hdr( page ); 
      
      /* figure out how many bytes to decode */

      M = page->ref.nth_page * MM_PAGE_SIZE;
      N = p->rs_header.pob_size 
	  + sizeof(struct PHeapHdr)
	  + sizeof(struct FirstPageHdr);

/*      printf( "save interior page %08x: at %08x\n",
	     page->ref.base_page_num + page->ref.nth_page,
	     page->mem_address );
  */    
      if (N > M)
	{
	  N -= M;
	  
	  /* there is at least SOMETHING to do */
	  
	  if (N >= MM_PAGE_SIZE)
	    {
	      /*  we're completely inside, so do only this page worth */
	      N = MM_PAGE_SIZE;
	    }
	  traverse_pob( ctx, p, 
		       page->mem_address, 
		       (char *)page->mem_address - (char *)(p+1),
		       N, mode );
	}
    }
}

obj write_page( struct RStore *store, struct VMPageRecord *page )
{
  struct PHeapHdr *p;
  unsigned i;
  write_ctx_t ctx;
  struct PageRef *pr;

  ctx.c.ptr = NULL;
  ctx.failed = NIL_OBJ;
  ctx.store = store;
  ctx.xlated_ptr = ctx.xlated_ptrs;

  init_ref_tbl( &ctx.refs );
  
  /* pass I -- find the pages that this refers to, and notice
     if any of the objects can't be unswizzled
  */
     
  traverse_page( &ctx, page, TRAV_NOTICE );
  
  if (!EQ(ctx.failed,NIL_OBJ))
    {
      /* some objects couldn't be translated into persistent ptrs... */
      close_ref_tbl( &ctx.refs );
      return ctx.failed;
    }

  /* pass II -- actually write the data out */

  /* start over to read the already-translated PTRs */

  ctx.xlated_ptr = ctx.xlated_ptrs;

  /* intialize the compressor stream */

  init_compressor( &ctx.c );
  
  /* (A) Write out the ref table... */

  compress_word( &ctx.c, ctx.refs.num_refs );

  pr = ctx.refs.refs;
  for (i=0; i<ctx.refs.num_refs; i++, pr++)
    {
      UINT_32 w;
      
      compress_word( &ctx.c, pr->base_page_num );
      w = (pr->first ? 1 : 0)
	  + (pr->indirect ? 2 : 0)
	  + (pr->nth_page << 2);
      compress_word( &ctx.c, w );
    }
	
  /* (B) Write out the page contents */

  traverse_page( &ctx, page, TRAV_WRITE );

  /* step III -- write the data to the lss */

  stop_compressor( &ctx.c );
  lss_write( store->lss, 
	     page->ref.first 
 	       ? page->ref.base_page_num
 	       : page->ref.base_page_num + page->ref.nth_page,
	     compressed_data( &ctx.c ),
	     compressed_size( &ctx.c ) );
  close_compressor( &ctx.c );
  close_ref_tbl( &ctx.refs );
  return NIL_OBJ;
}

obj write_dirty_pages( struct RStore *store )
{
struct VMPageRecord *page;
obj unresolved;

    while ((page = store->first_dirty))
    {
	unresolved = write_page( store, page );
	if (!EQ(unresolved,NIL_OBJ))
	{
	    /* note that if there was anything unresolved,
	       we do NOT mark this page as clean, because it
	       wasn't written
	    */
	    return unresolved;
	}
	page->ref.dirty = 0;
	store->num_dirty--;
	if (!(store->first_dirty = page->next_dirty))
	  {
	    store->last_dirty = NULL;
	  }
	page->next_dirty = NULL;
	mm_set_prot( page->mem_address, MM_PAGE_SIZE, MM_MODE_READ_ONLY );
    }
    return NIL_OBJ;
}

obj notice_object_refs( struct RStore *store, obj ptr, obj other_failures )
{
  struct PHeapHdr *p;
  UINT_32 i, N;
  write_ctx_t ctx;

  assert( OBJ_ISA_PTR(ptr) );

  ctx.c.ptr = NULL;
  ctx.failed = other_failures;
  ctx.store = store;

  /* set the save pointer to NULL to tell us not to bother
   * remembering the translated pointers.  If we didn't
   * do this, we would have to make a bigger buffer, because
   * ctx.xlated_ptrs is only big enough for a page's worth of
   * pointers, and an object may be quite a bit larger and full
   * of pointers
   */
  ctx.xlated_ptr = NULL;

  init_ref_tbl( &ctx.refs );

  p = PTR_TO_PHH(ptr);

  notice_page_ref( &ctx, &p->rs_header.pob_class );
  N = p->rs_header.pob_size;
  
  traverse_pob( &ctx, p, p+1, 0, N, TRAV_NOTICE );

  close_ref_tbl( &ctx.refs );
  return ctx.failed;
}

