#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/objsys/makeinst.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.18
 | File mod date:    1997.11.29 23:10:38
 | System build:     v0.7.2, 97.12.21
 | Owned by module:  objsys
 |
 | Purpose:          support for creating instances using "make"
 `------------------------------------------------------------------------|#

(define-class <keyword-list-malformed> (<condition>)
  kle-error-at
  kle-error-type
  kle-keyword-list)
  
;;

(define-class <initializer-error> (<condition>)
  target-class
  slot-descriptor)

(define-class <initializer-type-error> (<initializer-error>)
  given-value)

(define-class <initializer-missing> (<initializer-error>))

(define-method display-object ((self <keyword-list-malformed>) port)
  (__format port "malformed keyword list from ~s:\n" (kle-error-at self))
  (__format port ">> ~a\n" (kle-error-type self))
  (__format port ">> list = ~#@*60s\n" (kle-keyword-list self)))

(define-method display-object ((self <initializer-missing>) port)
  (__format port "make ~s: missing initializer for slot ~a\n"
	    (class-name (target-class self))
	    (name (slot-descriptor self))))

(define-method display-object ((self <initializer-type-error>) port)
  (__format port "make ~s: incorrect type of initial value for ~a~a\n"
	    (class-name (target-class self))
	    (if (slot-descriptor self)
		"slot "
		"allocation area")
	    (if (slot-descriptor self)
		(name (slot-descriptor self))
		""))
  (__format port ">> given: ~#*@60s\n" (given-value self))
  (__format port ">> required type: ~a\n"
	    (if (slot-descriptor self)
		(type-restriction (slot-descriptor self))
		<allocation-area>)))


;;;
;;;  parses a list representation alternating keywords and values
;;;  into a vector suitable for vassq operations (ie, also alternating
;;;  keywords and values).
;;;
;;;  signals the error <keyword-list-malformed> if the input is not a 
;;;  proper list, the keys are not all keywords, or there is a missing
;;;  value for a keyword (ie, the list has an odd length)
;;;

(define (keyword-value-list->vector kv-list)
  (let ((fnd (%make <vector> 
		    (gvec-alloc <vector> 16 #f)
		    4
		    0
		    <vector>)))
    (let loop ((p kv-list))
      (if (pair? p)
	  (let (((h <fixnum>) (transient->hash p)))
	    (if (object-table-lookup fnd h p)
		(signal
		 (make <keyword-list-malformed>
		       kle-keyword-list: kv-list
		       kle-error-at: #f
		       kle-error-type: "not a proper list"))
		(begin
		  (object-table-insert! fnd h p p)
		  (if (keyword? (car p))
		      (if (pair? (cdr p))
			  (loop (cdr (cdr p)))
			  (signal
			   (make <keyword-list-malformed>
				 kle-keyword-list: kv-list
				 kle-error-at: (car p)
				 kle-error-type: "missing value for keyword")))
		      (signal
		       (make <keyword-list-malformed>
			     kle-keyword-list: kv-list
			     kle-error-at: (car p)
			     kle-error-type: "key element not a keyword"))))))
	  (if (null? p)
	      (list->vector kv-list)
	      (signal
	       (make <keyword-list-malformed>
		     kle-keyword-list: kv-list
		     kle-error-at: #f
		     kle-error-type: "not a proper list")))))))

#|
   roughly 1/5 as fast as the above implementation...

(define (keyword-value-list->vector kv-list)
  (if (list? kv-list)
      (let* (((v <vector>) (list->vector kv-list))
	     ((n <fixnum>) (vector-length v)))
	(let loop (((i <fixnum>) 0))
	  (if (fixnum<? i n)
	      (if (keyword? (vector-ref v i))
		  (loop (fixnum+ i 2))
		  (signal
		   (make <keyword-list-malformed>
			 kle-keyword-list: kv-list
			 kle-error-at: (vector-ref v i)
			 kle-error-type: "key element not a keyword")))
	      (if (eq? i n)
		  v
		  (signal
		   (make <keyword-list-malformed>
			 kle-keyword-list: kv-list
			 kle-error-at: (vector-ref v (fixnum- n 1))
			 kle-error-type: "missing value for keyword"))))))
      (signal
       (make <keyword-list-malformed>
	     kle-keyword-list: kv-list
	     kle-error-at: #f
	     kle-error-type: "not a proper list"))))
|#

;;;
;;;  construct a list of "remainder" keyword/values, where the "used"
;;;  entries have a key of #f
;;;

(define (remainder->list (v <vector>))
  (let loop ((r '())
	     ((k <fixnum>) (gvec-length v)))
    (if (eq? k 0)
	r
	(let (((i <fixnum>) (fixnum- k 2)))
	  (loop (if (vector-ref v i)
		    (cons* (vector-ref v i)
			   (vector-ref v (add1 i))
			   r)
		    r)
		i)))))

;;;
;;;  look up the given keyword (`kwd') in the given 
;;;  keyword/value vector (`v') and invoke the `found-proc'
;;;  with the associated value if present (after clobbering
;;;  all occurrences of the key in the vector with #f).  If
;;;  not present, invoke the `notfound-proc' with no arguments.

(%strategy ccode
(define (using-keyword-value kwd (v <vector>) found-proc notfound-proc)
  (let ((i (vassq kwd v)))
    (if (fixnum? i)
	(let ((a (gvec-ref v i)))
	  (let loop ((i i))
	    (if (fixnum? i)
		(begin
		  (gvec-set! v (sub1 i) #f)
		  (loop (vassq kwd v)))
		(found-proc a))))
	(notfound-proc))))
)

(define (get-keyword-value (kvv <vector>) keyword default)
  (using-keyword-value
   keyword
   kvv
   (lambda (item)
     item)
   (lambda ()
     default)))

;;;

(%early-once-only
 (define $alloc-area-kwd (symbol->keyword '%alloc-area)))

(define (get-allocation-area c v)
  (using-keyword-value 
   $alloc-area-kwd
   v
   (lambda (a)
     (if (instance? a <allocation-area>)
	 a
	 (signal
	  (make <initializer-type-error>
		target-class: c
		slot-descriptor: #f
		given-value: a))))
   (lambda ()
     *default-allocation-area*)))

;;

(%strategy ccode
(define-method default-slot-value ((self <slot-descriptor>)
				   (target <object>)
				   (inits <vector>))
  (case (initialization-mode self)
    ((optional prohibited)
     (init-value self))
    ((function)
     ((init-value self)))
    ((required)
     (signal
      (make <initializer-missing>
	    target-class: (object-class target)
	    slot-descriptor: self)))))

(define-method initialize-slot! ((self <slot-descriptor>)
				 (target <object>)
				 (inits <vector>))
  (if (init-keyword self)
      (using-keyword-value
       (init-keyword self)
       inits
       (lambda (val)
	 (if (instance? val (type-restriction self))
	     (begin
	       (gvec-set! target (index self) val)
	       (values))
	     (signal
	      (make <initializer-type-error>
			 target-class: (object-class target)
			 slot-descriptor: self
			 given-value: val))))
       (lambda ()
	 (begin
	   (gvec-set! target 
		      (index self)
		      (default-slot-value self target inits))
	   (values))))
      (begin
	(gvec-set! target 
		   (index self)
		   (default-slot-value self target inits))
	(values))))

(define finish-initialization
  (lambda 'finish-initialization (instance more-inits)
    (if (null? more-inits)
	(initialize instance)
	(apply* instance more-inits initialize))
    ;; return the instance independent of what `initialize'
    ;; returns -- per Dylan spec [DIRM 94 p.80]
    instance))

;; 
;;  make-instance
;;
;;  support for creating of instances
;;  for which the class is known only at compile time
;;

(define (make-instance (class <<class>>) . inits)
  (if (not (or (eq? (heap-type class) 0)
	       (eq? (heap-type class) 4)))
      (if (eq? (heap-type class) 3)
	  (error "class ~s is abstract; instantiation is not permitted" class)
	  (error "cannot instantiate class ~s; heap type = ~d"
		 class (heap-type class))))
  (let* ((slots (class-compute-slots class))
	 ((v <vector>) (keyword-value-list->vector inits))
	 (instance (gvec-alloc-in-area (get-allocation-area class v)
				       class
				       (class-instance-size class)
				       #f)))
    ;;
    ;; fill in the initial values
    ;;
    (let loop ((slots slots))
      (if (null? slots)
	  ;; done processing all slots; finish the initialization
	  ;; (which involves invoking the `initialize' procedure)
	  (finish-initialization instance (remainder->list v))
	  (begin
	    (initialize-slot! (car slots) instance v)
	    (loop (cdr slots)))))))
)

(define (set-finish-initialization-proc! (proc <function>))
  (set! finish-initialization proc))

(define (slot-index slot)
  (gvec-ref slot 4))

(define (slot-init-kwd slot)
  (gvec-ref slot 5))
