(* MMM specific run-time environment for applets *)
open Safestd
open Safeapplets


module Html : sig
  (* HTML tokens *)
  type attribute_name = string 
  type attribute_value = string
  type attributes = (attribute_name * attribute_value) list

  type tag = {
    tag_name : string;
    attributes: attributes
  }

  type token =
     PCData of string
   | CData of string
   | OpenTag of tag
   | CloseTag of string
   | Comment of string
   | Doctype of string
   | EOF

  type location = Loc of int * int

  exception Html_Lexing of string * int
  exception Invalid_Html of string

  val get_entity : string -> string
    (* [get_entity "amp"] returns "&" *)

  val get_attribute : tag -> string -> string
    (* [get_attribute tag attrib_name] *)

  val has_attribute : tag -> string -> bool
    (* [has_attribute tag attrib_name] *)
end


module Dtd : sig
  type t
  val dtd20 : t
  val dtd32 : t
end

module ParseHTML : sig

 type minimization =
   Legal | Illegal of string

 val sgml_lexer :  Dtd.t -> Lexing.lexbuf -> (minimization * Html.token list * Html.location)

end
  



type fontInfo = 
   Family of string
 | Weight of string
 | Slant of string
 | FontIndex of int
 | FontDelta of int

type gattr = 
     Margin of int
  |  Justification of string
  |  Font of fontInfo		        (* mostly size and face *)
  |  FgColor of string
  |  BgColor of string
  |  Underlined
  |  Striked
  |  Spacing of int
  |  Superscript
  |  Lowerscript

type objmap = 
    ClientSide of Hyper.link		(* usemap link *)
  | ServerSide of Hyper.link		(* ismap *)
  | Direct of Hyper.link			(* inside an anchor *)
  | NoMap				(* no additionnal navigation *)
  | FormMap of (int * int -> Hyper.link)


type embobject = {
  embed_hlink : Hyper.link;               (* hyperlink to the object *)
  embed_frame : Widget.widget;  
     (* the frame where the viewers can do their stuff *)
  embed_context : Viewers.context;
  embed_map : objmap;                  (* associated map *)
  embed_alt : string
 }

class virtual imgloader (unit) =
  virtual add_image : embobject -> unit	 (* add one image *)
  virtual flush_images : unit	         (* flush when document is loaded *)
  virtual load_images : unit		 (* manual flush *)
end

class virtual extracontext (unit) =
  virtual set_title : string -> unit
  virtual add_link : string -> Hyper.link -> unit
end

type formatter = {
  (* Text primitives of the device *)
  new_paragraph: unit -> unit;  	(* Open a new paragraph *)
    (* make sure the following text will start on a new line *)
  close_paragraph: unit -> unit;  	(* Close a paragraph *)
    (* make sure there is an eol after the current text *)
  print_newline : bool -> unit;		(* Force a line break *)
  print_verbatim : string -> unit;	(* Print as-is *)
  format_string : string -> unit;	(* Line wrap, newlines don't count *)
  flush : unit -> unit;			(* Flush the device *)
  (* Predefined Images *)
  hr : unit -> unit;		(* could be embedded ? *)
  bullet : string -> unit;
  (* Graphical attributes *)
  set_defaults : gattr list -> unit;     (* bg, fg, links *)
  push_attr : gattr list -> unit;
  pop_attr : gattr list -> unit;
  (* Structure primitives *)
  isindex : string -> string -> unit;		(* <ISINDEX> *)
  start_anchor : unit -> unit;
  end_anchor : Hyper.link -> unit;
  add_mark : string -> unit;
  (* Embedding primitives *)
  create_embedded : 
     string -> int option -> int option -> Widget.widget;
       (* [create_embedded align w h ]: 
          returns a widget that we can pass as target to the embed manager.
          Should respect background color ?
        *)
  cell_formatter : Widget.widget -> formatter;
  (* Re-centering on a fragment *)
  see_frag : string option -> unit
  }

class virtual machine (unit) =
  virtual formatter : formatter
  virtual imgmanager : imgloader
  virtual tcontext : extracontext
  virtual base : string
  virtual ctx : Viewers.context
  virtual add_tag :
    string -> (formatter -> Html.tag -> unit) -> (formatter -> unit) -> unit
  virtual remove_tag : string -> unit
  virtual push_action : (string -> unit) -> unit
  virtual pop_action : unit
  virtual push_formatter : formatter -> unit
  virtual pop_formatter : unit
  virtual send : Html.token -> unit
end

module Get(C: sig val capabilities: Capabilities.t end) : sig
   val add_html_display_hook : (machine -> unit) -> unit
   val add_object : embobject -> unit
     (* queue an embedded object *)


   val add_user_navigation : string -> Hyper.func -> unit
   val add_user_menu : string -> (Viewers.context -> unit) -> unit

   (* The following allows demonstrations/replay *)
   type navigator
   val new_window : Url.t -> navigator option
   val follow_link : navigator -> Hyper.link -> unit
   val destroy_window : navigator -> unit

   (* Extra windows *)
   val get_global_widget : unit -> Widget.widget

   (* Embedded Viewers *)
   val add_embedded_viewer : 
     string * string ->
       ((string * string) list -> Widget.widget -> Viewers.context ->
	 Document.handle -> unit)
       -> unit
     (* [add_embedded_viewer media_type f]
	where [f media_parameters widget context document]
      *) 

end
