/*
 * draws the day view. PostScript printing is also here because it uses
 * the same high-level routines, only the low-level output is diverted.
 *
 *	clicked_day_calendar(x, y)	Press on the day display detected.
 *	draw_day_calendar(region)	If there is a day menu, resize, and
 *					redraw
 *	print_day_calendar(fp,l)	print PostScript day to fp
 */

#include <stdio.h>
#include <time.h>
#include <Xm/Xm.h>
#include "cal.h"

#define DBLTIME		1000		/* doubleclick time [ms] */

#define CUTOFF	(config.day_gap-1)	/* corner cut-off distance */
#define MINLEN		14		/* no bar is shorter than MINLEN */

#define TOD(t)		((t)%86400)
#define BOUND(t,a,b)	((t)<(a) ? (a) : (t)>(b) ? (b) : (t));
#define YPOS(t)		(((t) - c->day_minhour*3600) * c->day_hourheight/3600)

static void gen_day(), draw_bar(), draw_octagon(), draw_boxtext();

#ifdef JAPAN
int truncate_strpack();
void g_draw_strpack();
#endif

extern Display		*display;	/* everybody uses the same server */
extern XFontStruct	*font[NFONTS];	/* fonts: FONT_* */
extern struct config	config;		/* global configuration data */
extern struct mainmenu	mainmenu;	/* all important main window widgets */
extern struct user	*user;		/* user list (from file_r.c) */
extern time_t		curr_day;	/* day being displayed, time in sec */
extern struct week	day;		/* info on day view */
extern struct plist	*mainlist;	/* list of all schedule entries */
extern struct holiday	holiday[366];	/* info for each day, separate for */
extern struct holiday	sm_holiday[366];/* full-line texts under, and small */

extern char *weekday_name[];
extern char *monthname[];


/*
 * got a click in the calendar area. Traverse the node tree, first down until
 * the correct line is found, and then horizontally until the node is found.
 */

void clicked_day_calendar(xc, yc, time)
	int			xc, yc;		/* pixel pos clicked */
	Time			time;		/* current time in ms */
{
	register struct config	*c = &config;
	struct weeknode		*vert, *horz;	/* day node scan pointers */
	register struct entry	*ep;		/* entry to test */
	static struct entry	*last_ep;	/* last entry found (dblclk?)*/
	static Time		last_time;	/* time of last press */
	int			d;		/* day counter */
	int			x0, y0;		/* XY start coord of day box*/
	int			x, y;		/* upper left of day bk box */
	int			xs;		/* width of bar col w/ gaps */
	time_t			btime;		/* start time of bar */
	int			b, e, l;	/* Y start end end pos of bar*/
	int			yend;		/* bottom margin of chart */
	int			u;		/* index into user[] array */

	x0 = c->day_margin + c->day_hourwidth + 2;
	y0 = c->day_margin + c->day_title + c->day_headline + 2;
	xs = c->day_barwidth + c->day_gap;
	yend = y + c->day_hourheight * (c->day_maxhour - c->day_minhour);

	for (d=0; d < c->day_ndays; d++) {
	    x = x0;
	    for (y=y0, vert=day.tree[d]; vert; vert=vert->down, x+=xs) {
		if (xc < x || xc >= x+xs)
			continue;
		if (yc < y) {
			create_list_popup(mainlist, curr_day + d*86400,
							86400, 0, 0, 0, 0);
			return;
		}
		for (horz=vert; horz; horz=horz->next) {
			u  = name_to_user(horz->entry->user);
			ep = horz->entry;
			btime = BOUND(TOD(ep->time), c->day_minhour*3600,
						     c->day_maxhour*3600);
			e  = BOUND(y + YPOS(btime), y0, yend);
			if (config.weekwarn)
				btime -= ep->early_warn > ep->late_warn
						? ep->early_warn
						: ep->late_warn;
			b = BOUND(y + YPOS(btime), y0, yend);
			if (ep->notime)
				e = config.week_bignotime
					? y + YPOS(c->day_maxhour*3600)
					: b + YPOS((c->day_minhour+1)*3600);
			else {
				l  = ep->length * c->day_hourheight/3600;
				e += BOUND(l, MINLEN, 9999);
			}
			if (yc >= b && yc < e ||
			    yc >= yend-MINLEN && b >= yend) {
				if (time-last_time < DBLTIME && ep == last_ep){
					if (u >= 0 && user[u].readonly)
						print_button(day.info,
						 "Cannot edit, no permission");
					else
						create_list_popup(mainlist,
							0, 0, 0, ep, 0, 0);
					last_ep = 0;
					return;
				} else if (horz->entry->user)
					print_button(day.info, "%s %s: %s",
						mktimestring(ep->time, FALSE),
						horz->entry->user,
						ep->note ? ep->note : "");
				else
					print_button(day.info, "%s %s",
						mktimestring(ep->time, FALSE),
						ep->note ? ep->note : "");
				last_ep   = ep;
				last_time = time;
				return;
			}
		}
	    }
	    x0 += day.nlines[d] * xs + c->day_margin;
	}
	print_button(day.info, " ");
}


/*---------------------------------- drawing front-ends ---------------------*/
/*
 * draw one day into the current window, in the specified region (all if null).
 */

void draw_day_calendar(region)
	Region region;
{
	if (!g_init_window(&day, region))
		return;
	g_setcolor(COL_WBACK);
	g_fillrect(0, 0, day.canvas_xs, day.canvas_ys);
	gen_day();
	g_exit_window();
}


/*
 * print PostScript day calendar to file fp.
 */

void print_day_calendar(fp, landscape)
	FILE	*fp;
	BOOL	landscape;
{
	time_t time = get_time();
	struct tm *tm = time_to_tm(time);
	time -= (tm->tm_wday + 6 + config.sunday_first)%7 * 86400;
	if (!curr_day)
		curr_day = time;
	build_day(config.pr_omit_priv, config.pr_omit_appts);
	g_init_ps(&day, fp, landscape);
	gen_day();
	g_exit_ps();
}


/*---------------------------------- high-level drawing ---------------------*/
/*
 * draw the day, using low-level routines
 */

static void gen_day()
{
	register struct config	*c = &config;
	time_t			today;		/* today's date */
	char			buf[40], *p;	/* text to print */
	int			d, h, l;	/* day, hour, text length */
	int			x, xs;		/* appt bar pos, col width */
	int			x0, y0;		/* top left inside line box */
	int			dx, dy;		/* size inside day line box */
	int			cy_h, cy_d;	/* character height */
	struct weeknode		*vert, *horz;	/* day node scan pointers */

	x0 = c->day_margin;
	y0 = c->day_margin + c->day_title + c->day_headline + 2;
	xs = c->day_barwidth + c->day_gap;
	dy = c->day_hourheight * (c->day_maxhour - c->day_minhour);
	cy_h = font[FONT_WHOUR]->max_bounds.ascent;
	cy_d = font[FONT_WDAY]->max_bounds.ascent;

	strcpy(buf, mkdatestring(curr_day));			/* title */
	strcat(buf, " - ");
	strcat(buf, mkdatestring(curr_day + (c->day_ndays-1)*86400));
	g_setcolor(COL_WTITLE);
	g_setfont(FONT_WTITLE);
	g_drawtext((day.canvas_xs - strlen_in_pixels(buf,FONT_WTITLE))/2, 
		   c->day_margin + c->day_title * 3/4,
		   buf, strlen(buf));

	g_setcolor(COL_WDAY);					/* hour col */
	g_setfont(FONT_WHOUR);
	for (h=0; h <= c->day_maxhour-c->day_minhour; h++) {
		p = mktimestring((h + c->day_minhour) * 3600, FALSE);
		l = strlen_in_pixels(p, FONT_WDAY);
		g_drawtext(x0 + (c->day_hourwidth - l) / 2,
			   y0 + c->day_hourheight * h + cy_d/2 -2,
			   p, strlen(p));
	}
	x0 += c->day_hourwidth + 2;

	for (d=0; d < c->day_ndays; d++) {			/* day loop */
		if (!(dx = day.nlines[d] * xs))
			continue;

		today = get_time() - (curr_day + d*86400);	/* green? */
		if (today >= 0 && today < 86400) {
			g_setcolor(COL_CALTODAY);
			g_fillrect(x0-2, y0-2-c->day_headline,
				   dx+4, c->day_headline);
		}
		g_setcolor(COL_WDAY);				/* date */
		g_setfont(FONT_WHOUR);
		p = mkdatestring(curr_day + d * 86400);
		truncate_string(p, dx, FONT_WHOUR);
		g_drawtext(x0, y0-4, p, strlen(p));

		g_setcolor(COL_WBOXBACK);			/* box bgnd */
		g_fillrect(x0,    y0,    dx,   dy);

		g_setcolor(COL_WGRID);				/* box lines */
		g_fillrect(x0-2,  y0-2,  dx+4, 2);
		g_fillrect(x0-2,  y0+dy, dx+4, 2);
		g_fillrect(x0-2,  y0,    2,    dy);
		g_fillrect(x0+dx, y0,    2,    dy);
		for (h=1; h <= c->day_maxhour-c->day_minhour; h++)
			g_fillrect(x0, y0 + c->day_hourheight * h, dx, 1);

		x = x0 + c->day_gap/2;				/* appt bars */
		for (vert=day.tree[d]; vert; vert=vert->down) {
			for (horz=vert; horz; horz=horz->next) {
				int u = name_to_user(horz->entry->user);
				draw_bar(horz, u>0 ? user[u].color : 0, x, y0);
			}
			x += xs;
		}
		x0 += dx + c->day_margin;
	}
	g_setcolor(COL_STD);
}


/*
 * draw one entry bar.
 */

static void draw_bar(node, color, x, y)
	struct weeknode		*node;		/* entry node to print */
	int			color;		/* color, 0..7 */
	int			x, y;		/* top left pos in canvas */
{
	register struct entry	*ep = node->entry;
	register struct config	*c = &config;
	time_t			tod;		/* clipped appt time of day */
	int			yend;		/* right margin of chart */
	int			w[2], b, e;	/* early/late, begin, end */
	int			i;		/* warning counter, 0..1 */
	char			msg[256];

	tod  = BOUND(TOD(ep->time), c->day_minhour*3600, c->day_maxhour*3600);
	yend = y + c->day_hourheight * (c->day_maxhour - c->day_minhour);
	i = ep->early_warn > ep->late_warn;
	w[!i] = tod - ep->early_warn;
	w[ i] = tod - ep->late_warn;
	b     = tod;
	e     = tod + ep->length;
	w[0]  = y + YPOS(w[0]);
	w[1]  = y + YPOS(w[1]);
	b     = y + YPOS(b);
	e     = y + YPOS(e);
	if (ep->notime) {
		b = y + YPOS(c->day_minhour*3600);
		e = config.week_bignotime ? y + YPOS(c->day_maxhour*3600)
					  : b + YPOS((c->day_minhour+1)*3600);
	}
	if (config.weekwarn && !ep->notime)
		for (i=0; i < 2; i++)
			draw_octagon(COL_WWARN, x, w[i], e, yend);

	draw_octagon(ep->suspended ? COL_WWARN : COL_WUSER_0 + color,
						x, b, e, yend);

	*msg = 0;
	if (!ep->notime) {
		strcpy(msg, mktimestring(node->trigger, FALSE));
		if (ep->length)
			sprintf(msg+strlen(msg), "-%s", mktimestring(
					node->trigger + ep->length, FALSE));
	}
	if (ep->user && strcmp(ep->user, user[0].name))
		sprintf(msg+strlen(msg), " (%s)", ep->user);
	strcat(msg, " ");
	strcat(msg, node->text);

	draw_boxtext(x, b, c->day_barwidth, e-b < MINLEN ? MINLEN : e-b,
						msg, FALSE);
}


/*
 * draw one octagon (rectangle with the corners cut off, to make it easier
 * to see where one box ends and the next begins). Allow a bottom limit to
 * allow clipping, and to show flat bottoms where an end time isn't known.
 * The 9th polygon vertex closes the polygon. It runs counter-clockwise
 * from the topmost vertex at the left edge (10-o'clock position)
 */

static void draw_octagon(color, x, y, ye, yend)
	int	color;		/* color, 0..7 */
	int	x, y;		/* top left XY pos in canvas */
	int	ye;		/* Y bottom in canvas */
	int	yend;		/* bottom Y clip limit */
{
	XPoint	point[9];	/* polygon vertex list */

	if (ye < y + MINLEN) {		/* too narrow: draw flat bottom */
		ye = y + 9999;
		yend = y + MINLEN;
	}
	point[8].x =
	point[0].x =
	point[1].x = x;
	point[7].x =
	point[2].x = x + CUTOFF;
	point[6].x =
	point[3].x = x + config.day_barwidth - CUTOFF;
	point[5].x =
	point[4].x = x + config.day_barwidth;

	point[7].y =
	point[6].y = y;
	point[8].y =
	point[0].y =
	point[5].y = y + CUTOFF;
	point[1].y =
	point[4].y = BOUND(ye - CUTOFF, y, yend);
	point[2].y =
	point[3].y = BOUND(ye, y, yend);

	g_setcolor(color);
	g_drawpoly(point, 8);
	g_setcolor(COL_WFRAME);
	g_drawlines(point, 9);
}


/*
 * draw a text into the given space, optionally splitting the text into
 * multiple lines. Use the small note font.
 */

#ifdef JAPAN
#define BOXFONT FONT_JNOTE
#else
#define BOXFONT FONT_NOTE
#endif
#define ISSPACE(c) ((c)==' ' || (c)=='\t')
#define MAXSPLIT 6

static void draw_boxtext(x, y, xs, ys, text, center)
	int		x, y;		/* top left corner of free space */
	int		xs, ys;		/* size of free space */
	char		*text;		/* text to print */
	BOOL		center;		/* center text? */
{
	char		*beg, *p, *p_s;	/* soln, curr char, since last space */
	int		len, len_s;	/* curr text length in pixels */
	int		cys, lys;	/* character height, line height */
	int		i;

	g_setfont(BOXFONT);
	cys = font[BOXFONT]->max_bounds.ascent;
	lys = cys * 5/4;
	i   = (ys-cys) / lys;
	y  += (ys - cys - i*lys) / 2 + cys;
	x  += 3;
	xs -= 5;

	for (len_s=len=0, beg=p_s=p=text; *p; p++) {
		if (*p == ' ' || *p == '\t') {
			p_s   = p;
			len_s = len;
		}
		len += CHARWIDTH(BOXFONT, *p);
		if (!p[1])
			p++;
		if (len > xs || !*p) {
			if (len > xs && p_s != beg
				     && p_s >= p-MAXSPLIT && ys >= lys+cys) {
				p   = p_s;
				len = len_s;
			}
			i = center ? x + (xs - len)/2 : x;
			g_drawtext(i, y, beg, p-beg);
			while (ISSPACE(*p)) p++;
			beg = p_s   = p--;
			len = len_s = 0;
			y += lys;
			if ((ys -= lys) < cys)
				break;
		}
	}
}
