// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLScene.h

  Stephane Rehel

  December 8 1996
*/

#ifndef __MLScene_h
#define __MLScene_h

#ifndef __IBOOL_h
#include "tools/IBOOL.h"
#endif

#ifndef __MLName_h
#include "MLName.h"
#endif

class MLObject;
class MLNullObject;
class MLLight;
class MLMaterial;

class OString;
class MLObjectAllocator;
class MLAbstractMeshAllocator;
class MLAbstractCurveAllocator;
class MLMaterialAllocator;
class MLLightTable;
class MLLightPool;
class MLPoolTable;
class MLGlobalLighting;
class MLMaterialImages;
class MLRendererOptions;

/////////////////////////////////////////////////////////////////////////////

class MLScene
{
  friend class MLObject;
  friend class MLAbstractMesh;
  friend class MLAbstractCurve;
  friend class MLMaterial;
  friend class MLSceneIO;
  friend class MLSaveProject;

protected:
  MLObjectAllocator* objectAllocator;
  MLAbstractMeshAllocator* abstractAllocator;
  MLAbstractCurveAllocator* abstractCurveAllocator;
  MLMaterialAllocator* materialAllocator;
  MLLightTable* lights;
  MLPoolTable* poolTable;
  MLGlobalLighting* globalLighting;
  MLMaterialImages* materialImages;
  MLRendererOptions* rendererOptions;

  MLObject* rootObject; // this is a NullObject

  MLName name;

public:
  IBOOL dirtyPoolsLighting; // need to refresh some pool lightings
  IBOOL dirtyObjectsLighting; // need to refresh some object lightings
  IBOOL dirtyFinalLighting; // i.e. please refresh illum

public:
  MLScene();
  virtual ~MLScene();

  const MLName& getName() const
    {
    return name;
    }
  void setName( const OString& _name );
  void setName( const MLName& _name );

  MLObject* getRootObject() const
    {
    return rootObject;
    }

  int getNObjects() const;
  MLObject* getObject( int id, int object_type = 0 );
  MLObject* getDetachedObject( int id );
  MLObject* getObject( const MLName& name, int object_type = 0 );
  MLObject* getObjectByFilename( const OString& filename ) const;
  MLObjectAllocator* getObjectAllocator() const
    {
    return objectAllocator;
    }

  MLAbstractMesh* getAbstractObject( int id ) const;
  MLAbstractMesh* getAbstractObject( const MLName& name ) const;
  MLAbstractMesh* getAbstractByFilename( const OString& filename ) const;
  MLAbstractMeshAllocator* getAbstractAllocator() const
    {
    return abstractAllocator;
    }

  MLAbstractCurve* getAbstractCurve( int id ) const;
  MLAbstractCurve* getAbstractCurve( const MLName& name ) const;
  MLAbstractCurve* getAbstractCurveByFilename( const OString& filename ) const;
  MLAbstractCurveAllocator* getAbstractCurveAllocator() const
    {
    return abstractCurveAllocator;
    }

  int getNLights() const;
  MLLight* getLight( int light_index ) const;
  MLLight* getLight( const MLName& name ) const;
  MLLightTable* getLightTable() const
    {
    return lights;
    }
  void compileLights();

  int getNMaterials() const;
  MLMaterial* getMaterial( int id );
  MLMaterial* getMaterial( const OString& name );
  MLMaterial* getDefaultMaterial() const;

  // Allow this since raytracing must get materials very quickllyyyy
  MLMaterialAllocator* getMaterialAllocator() const
    {
    return materialAllocator;
    }

  void updateTransformations();

  void computeFinalLighting();

  int getNPools() const;
  MLLightPool* getPool( int pool_index ) const;
  MLLightPool* getPool( const OString& name ) const;
  MLPoolTable* getPoolTable() const
    {
    return poolTable;
    }

  MLGlobalLighting* getGlobalLighting() const
    {
    return globalLighting;
    }

  MLMaterialImages* getMaterialImages() const
    {
    return materialImages;
    }

  MLRendererOptions* getRendererOptions() const
    {
    return rendererOptions;
    }

  void clear();

  // Check for boundings/hashboxes to update
  void updateBoundingHashBox();

protected:
  void createRootObject();
};

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __MLScene_h
