// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLStartup.C

  Stephane Rehel
  March 25 1997
*/

#include "tools/zlib/MLLibZ.h"

#include "graphics/GraphicsSystem.h"
#include "graphics/mlgl.h"
#include "graphics/MLEvent.h"
#include "graphics/MLEventManager.h"
#include "graphics/MLWindow.h"
#include "graphics/SystemWindow.h"

#include "graphics/MLPicture.h"

#include "interface/MLInterfaceVersion.h"

#include "MLStartup.h"
#include "Cosmic10Font.h"

/////////////////////////////////////////////////////////////////////////////

MLStartup::MLStartup()
{
  logo1= 0;
  logo2= 0;
}

/////////////////////////////////////////////////////////////////////////////

MLStartup::~MLStartup()
{
  delete logo1;
  logo1= 0;

  delete logo2;
  logo2= 0;
}

/////////////////////////////////////////////////////////////////////////////

void MLStartup::run()
{
  buildLogos();

  if( logo1 == 0 && logo2 == 0 )
    return; //???

  end= IFALSE;

  drawLogo();

  MLEventManager* eventManager= MLEventManager::getEventManager();

  MLEventGrabber* prev_grabber= eventManager->setEventGrabber(this);

  for(;;)
    {
    eventManager->get();
    eventManager->dispatch();
    if( end )
      break;
    eventManager->wait();
    }

  // flush any other event
  eventManager->get();
  eventManager->dispatch();

  eventManager->setEventGrabber( prev_grabber );

  GraphicsSystem* graphicsSystem= GraphicsSystem::getGraphicsSystem();

  SystemWindow* window= graphicsSystem->getRootWindow();
  window->currentPixel();
  mlFront();
  window->clear(0,0,0);
  glFlush();
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLStartup::eventGrabber( MLWindow* window, MLEvent& event )
{
  GraphicsSystem* graphicsSystem= GraphicsSystem::getGraphicsSystem();

  if( window == graphicsSystem->getRootWindow() &&
      event.type == MLEvent::REDRAW )
    {
    drawLogo();
    return ITRUE;
    }

  if( event.type == MLEvent::KEY_RELEASE ||
      event.type == MLEvent::MOUSE_RELEASE )
    {
    end= ITRUE;
    return ITRUE;
    }

  return window->handleEvent(event);
}

/////////////////////////////////////////////////////////////////////////////

void MLStartup::drawLogo()
{
  GraphicsSystem* graphicsSystem= GraphicsSystem::getGraphicsSystem();

  SystemWindow* window= graphicsSystem->getRootWindow();

  window->currentPixel();
  mlBack();
  window->clear(0,0,0);

  IVector wsize= window->getSize();

  int x= (wsize.x() - logo1->getWidth()) / 2;
  int y= 2 * (wsize.y() - logo1->getHeight()) / 3;
  logo1->draw(x,y);

  if( logo2 != 0 )
    {
    y-= 2 * logo2->getHeight();
    logo2->draw( (wsize.x() - logo2->getWidth()) / 2, y );
    }

  window->swapBuffers();
}

/////////////////////////////////////////////////////////////////////////////

MLPicture* MLStartup::getLogo( int logo_width, int logo_height,
                               unsigned char logo_data[] )
{
  MLPicture* logo= new MLPicture;
  logo->createRGB( logo_width, logo_height );

  MLZDeflated deflated(logo_data);

  if( ! deflated.inflate( (void*) logo->getData() ) )
    {
    delete logo;
    return 0;
    }

  return logo;
}

/////////////////////////////////////////////////////////////////////////////

void MLStartup::buildLogos()
{
#ifndef __WINDOWS__
  if( logo2 == 0 )
    {
    logo2= getLogo(logo2_width,logo2_height,logo2_data);
    logo2->gammaCorrect();
    }
#endif

  if( logo1 != 0 )
    return;

  logo1= getLogo(logo1_width,logo1_height,logo1_data);
  if( logo1 == 0 )
    return;

  OString version= "Version "+OString(MLInterfaceVersion::version())
                  +" (#"
                  + OString::itoa(MLInterfaceVersion::build_number())
                  + ")";
  OString copyright= "Copyright \x7f 1996-1998 Stephane C.F. Rehel";

  Cosmic10Font font;
  int x, y;

  x= logo1->getWidth() - 15 - font.getStringWidth(version.get());
  y= logo1->getHeight() - 107;
  font.drawString( x, y, logo1, version.get() );

  x= logo1->getWidth() - 15 - font.getStringWidth(copyright.get());
  y= y - font.getHeight() - 3;
  font.drawString( x, y, logo1, copyright.get() );

  logo1->gammaCorrect();
}

/////////////////////////////////////////////////////////////////////////////
