// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLWindow.h

  Abstract Window (either SoftWindow or SystemWindow)

  Stephane Rehel
  June 28, 1996
*/

#ifndef __MLWindow_h
#define __MLWindow_h

#ifndef __IBOOL_h
#include "tools/IBOOL.h"
#endif

#ifndef __SIList_h
#include "tools/SIList.h"
#endif

#ifndef __IVector_h
#include "tools/IVector.h"
#endif

class MLEvent;
class MLEventManager;
class MLEventHandler;

class SoftWindow;
class SystemWindow;

/////////////////////////////////////////////////////////////////////////////

class MLWindow
{
  friend MLEventManager;
  friend SoftWindow;

protected:
  SIList<SoftWindow> softChildren; // only SoftWindow children

private:
  MLEventHandler* _eventHandler;

protected:
  IPoint position;
  IVector size;
  IBOOL isMapped;

public:
  MLWindow();
  virtual ~MLWindow();

  SoftWindow* newSoftChild( int x, int y, int width, int height );

  void setEventHandler( MLEventHandler* __eventHandler );

  virtual void redraw()
    {}

  // return ITRUE if handled
  virtual IBOOL handleEvent( const MLEvent& );

  virtual IBOOL isSystemWindow() const =0;
  SystemWindow* getSystemWindow() const;

  IBOOL isSoftWindow() const
    {
    return ! isSystemWindow();
    }

  IBOOL mapped() const
    {
    return isMapped;
    }
  virtual void map( IBOOL yes = ITRUE ) = 0;

  virtual void current() =0; // current for opengl-drawing
  virtual void swapBuffers()
    {}

  virtual void setPosition( const IPoint& ) = 0;
  virtual void setSize( const IVector& ) = 0;

  const IVector& getSize() const
    {
    return size;
    }
  const IPoint& getPosition() const
    {
    return position;
    }
  IPoint getAbsolutePosition() const;
  IPoint reverseY( const IPoint& p ) const
    {
    return IPoint( p.x(), size.y()-1-p.y() );
    }

  int x1() const { return position.x(); }
  int y1() const { return position.y(); }
  int x2() const { return position.x()+size.x()-1; }
  int y2() const { return position.y()+size.y()-1; }

  IBOOL into( const IPoint& p ) const;

  void pixelMode();
  void clear( float r, float g, float b );

  void currentPixel()
    {
    current();
    pixelMode();
    }

  void postRedraw( IBOOL ignore_unmapped = IFALSE );
  IBOOL removeRedraw();

protected:
  virtual IBOOL create( int x, int y, int width, int height );
};

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __MLWindow_h

