/*
  MLRawReader.C

  Copyright (C) 1997-1998  Stephane Rehel

  December 14 1997
*/

#include <string.h>

#include "MLRawReader.h"

/////////////////////////////////////////////////////////////////////////////

MLRawReader::MLRawReader( FILE* _file, bool _verbose /* = FALSE */ )
{
  file= _file;
  verbose= _verbose;

  objects= 0;
  nObjects= 0;
  nPools= 0;
}

/////////////////////////////////////////////////////////////////////////////

MLRawReader::~MLRawReader()
{
  clear();

  file= 0;
}

/////////////////////////////////////////////////////////////////////////////

void MLRawReader::clear()
{
  delete [] objects;
  objects= 0;

  nObjects= 0;
  nPools= 0;
}

/////////////////////////////////////////////////////////////////////////////

char* MLRawReader::get_line()
{
  if( file == 0 )
    return 0;

  static char line[512];

  char* l= 0;
  char* s= 0;

  for(;;)
    {
    l= fgets( line, 511, file );
    if( l == 0 )
      break;

    s= strchr(l,'\n');
    if( s != 0 )
      *s= '\0';

    s= strchr(l,'#');
    if( s != 0 )
      *s= '\0';

    s= l;
    while( *l == ' ' || *l == '\t' )
      ++l;

    if( *l != '\0' )
      break;
    }

  return l;
}

/////////////////////////////////////////////////////////////////////////////

bool MLRawReader::get_integer( int* i )
{
  if( i == 0 )
    return FALSE;

  char* line= get_line();

  if( line == 0 )
    return 0;

  if( sscanf( line, "%d", i ) != 1 )
    return FALSE;

  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////

// read a raw format
// return FALSE if error
bool MLRawReader::read()
{
  clear();

  if( file == 0 )
    return FALSE;

  if( ! get_integer(&nObjects) )
    return FALSE;
  if( ! get_integer(&nPools) )
    return FALSE;

  if( nObjects == 0 )
    return FALSE;

  objects= new Object [ nObjects ];

  for( int i= 0; i < nObjects; ++i )
    if( ! readObject(i) )
      return FALSE;

  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////

bool MLRawReader::readObject( int object_index )
{
  if( objects == 0 )
    return FALSE;

  if( object_index < 0 || object_index >= nObjects )
    return FALSE;

  Object* o= objects + object_index;

  char* line= get_line();
  if( line == 0 )
    return FALSE;

  // object name is in *line
  if( verbose )
    fprintf( stderr, "reading object %s\n", line );

  o->name= new char [ strlen(line) + 1 ];
  strcpy( o->name, line );

  if( ! get_integer( &(o->nVertices) ) )
    return FALSE;
  if( verbose )
    fprintf( stderr, "  nVertices=%d\n", o->nVertices );

  if( ! get_integer( &(o->nPolygons) ) )
    return FALSE;
  if( verbose )
    fprintf( stderr, "  nPolygons=%d\n", o->nPolygons );

  int i;

  if( o->nVertices == 0 )
    return TRUE;
  o->vertices= new Vertex [ o->nVertices ];
  for( i= 0; i < o->nVertices; ++i )
    {
    Vertex* v= o->vertices + i;

    line= get_line();
    if( line == 0 )
      return FALSE;
    if( sscanf( line, "%f %f %f", &v->x, &v->y, &v->z ) != 3 )
      return FALSE;

    line= get_line();
    if( line == 0 )
      return FALSE;
    if( sscanf( line, "%f %f %f", &v->nx, &v->ny, &v->nz ) != 3 )
      return FALSE;

    if( nPools > 0 )
      {
      v->lighting= new Spectrum [ nPools ];
      for( int j= 0; j < nPools; ++j )
        {
        Spectrum* s= v->lighting+j;

        line= get_line();
        if( line == 0 )
          return FALSE;

        if( sscanf( line, "%f %f %f", &s->r, &s->g, &s->b ) != 3 )
          return FALSE;
        }
      }
    }

  if( o->nPolygons == 0 )
    return TRUE;
  o->polygons= new Polygon [ o->nPolygons ];
  for( i= 0; i < o->nPolygons; ++i )
    {
    Polygon* p= o->polygons + i;

    if( ! get_integer( &(p->nVertices) ) )
      return FALSE;

    if( p->nVertices <= 0 )
      continue;

    p->vertices= new int [ p->nVertices ];
    for( int j= 0; j < p->nVertices; ++j )
      if( ! get_integer( p->vertices + j ) )
        return FALSE;
    }

  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < nObjects
MLRawReader::Object* MLRawReader::getObject( int i ) const
{
  if( objects == 0 )
    return 0;
  if( i < 0 || i >= nObjects )
    return 0;

  return objects + i;
}

/////////////////////////////////////////////////////////////////////////////

