//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//
//  The TrickleHandler:
//              read/write optimized data buffers for trickling
//
//  20040811::mad@madness.at
//
//


#include "TrickleHandler.hpp"


// ----------------- FileBuffer -----------------------
FileBuffer::~FileBuffer ()
{
}

FileBuffer::FileBuffer (int _fd):fd (_fd)
{
  int
    offset = 0;

  // calc file len and seek to end
  offset = lseek (fd, 0, SEEK_SET);
  len = lseek (fd, 0, SEEK_END);

  if ((offset < 0) || (len < 0))
    {
#ifdef DGDEBUG
      std::cout << "Error seeking dgav temporary." << std::endl;
#endif
      syslog (LOG_ERR, "%s", "Error seeking dgav temporary file.");
      throw
      runtime_error (string ("Error seeking dgav temporary file: ") + strerror (errno));
    }
}

int
FileBuffer::doOpen (char *tempFileName)
{

  try
  {
    fd = open (tempFileName, O_RDONLY);
  } catch (exception & e)
  {
#ifdef DGDEBUG
    std::cout << "Error opening dgav temporary file." << std::endl;
    std::cout << e.what () << std::endl;
#endif
    syslog (LOG_ERR, "%s", "Error opening dgav temporary file.");
    syslog (LOG_ERR, "%s", e.what ());
    throw runtime_error (string ("Error opening dgav temporary file: ") + strerror (errno));
  }

  return fd;
}

int
FileBuffer::doAppend (const char *_data, int _len)
{
  int written = 0;

  // write data from buffer
  if ((written = write (fd, _data, _len)) < 0)
    {
#ifdef DGDEBUG
      std::cout << "Error writing to dgav temporary file." << std::endl;
#endif
      syslog (LOG_ERR, "%s", "Error writing to dgav temporary file.");
      throw runtime_error (string ("Error writing to dgav temporary file: ") + strerror (errno));
    }

  len += written;

  return written;
}

int
FileBuffer::doRead (char *_data, int _len, int _offset)
{
  int rd = 0;
  int offset = 0;

  // seek to offset
  offset = lseek (fd, _offset, SEEK_SET);
  if (offset < 0)
    {
#ifdef DGDEBUG
      std::cout << "Error seeking dgav temporary." << std::endl;
#endif
      syslog (LOG_ERR, "%s", "Error seeking dgav temporary file.");
      throw runtime_error (string ("Error seeking dgav temporary file: ") + strerror (errno));
    }

  // read data to buffer
  if ((rd = read (fd, _data, _len)) < 0)
    {
#ifdef DGDEBUG
      std::cout << "Error reading dgav temporary file." << std::endl;
#endif
      syslog (LOG_ERR, "%s", "Error reading dgav temporary file.");
      throw runtime_error (string ("Error reading dgav temporary file: ") + strerror (errno));
    }

  return rd;
}

int
FileBuffer::doRead (char *_data, int _len)
{
  int rd = 0;

  // read data to buffer
  if ((rd = read (fd, _data, _len)) > 0)
    {
#ifdef DGDEBUG
      std::cout << "Error reading dgav temporary file." << std::endl;
#endif
      syslog (LOG_ERR, "%s", "Error reading dgav temporary file.");
      throw runtime_error (string ("Error reading dgav temporary file: ") + strerror (errno));
    }

  return rd;
}

// ----------------- MemoryBuffer -----------------------
MemoryBuffer::MemoryBuffer (int _buffersize, int _tricklesize):buffersize (_buffersize), tricklesize (_tricklesize), len (0)
{
  // create new buffer
  buffer = new char[_buffersize];
}

MemoryBuffer::~MemoryBuffer ()
{
  // delete our buffer
  delete[]buffer;
}


char *
MemoryBuffer::prepareAppend (void)
{
  // see if tricklesize bytes fit into buffer
  if ((len + tricklesize) < buffersize)
    {
      return &buffer[len];
    }
  else
    {
      return NULL;              // nope we are full
    }
}

char *
MemoryBuffer::commitAppend (int _len)
{
  len += _len;

  return &buffer[len];
}

char *
MemoryBuffer::doAppend (const char *_data, int _len)
{
  // see if tricklesize bytes fit into buffer
  if ((len + tricklesize) < buffersize)
    {
      try
      {
        memcpy (&buffer[len], _data, _len);
      }
      catch (exception & e)
      {
#ifdef DGDEBUG
        std::cout << "Error appending dgav memory buffer." << std::endl;
        std::cout << e.what () << std::endl;
#endif
        syslog (LOG_ERR, "%s", "Error appending dgav memory buffer.");
        syslog (LOG_ERR, "%s", e.what ());
        throw runtime_error (string ("Error appending dgav memory buffer: ") + strerror (errno));
      }

      len += _len;
      return &buffer[len];
    }
  else
    {
      return NULL;              // nope we are full
    }
}


// ----------------- TrickleBuffer -----------------------
TrickleBuffer::TrickleBuffer (int _buffersize, int _tricklesize, int _fd):fbuf (_fd), mbuf (_buffersize, _tricklesize)
{
}

TrickleBuffer::~TrickleBuffer ()
{
}

int
TrickleBuffer::doOpen (char *tempFileName)
{
  int fd = 0;

  fd = fbuf.doOpen (tempFileName);
  return fd;
}

char *
TrickleBuffer::prepareAppend (void)
{
  char *p;

  // check if there is enough space for a chunk
  p = mbuf.prepareAppend ();
  if (!p)
    {
      // nope; so flush to file and retry
      doFlushToDisk (true);
      p = mbuf.prepareAppend ();
    }

  return p;
}

char *
TrickleBuffer::commitAppend (int _len)
{
  // update length
  mbuf.setLength (mbuf.getLength () + _len);
  return mbuf.getData ();
}

int
TrickleBuffer::doFlushToDisk (bool flushIt)
{
  if (mbuf.getLength () > 0)
    {
      if (flushIt)
        {
          fbuf.doAppend (mbuf.getData (), mbuf.getLength ());
        }
      mbuf.setLength (0);
    }

  return 0;
}

int
TrickleBuffer::doTrickle (char *_data, int _len, int _offset)
{
  int rd = 0;

  // make sure trickle buffer has enough data
  if ((_offset + _len) > fbuf.getFileLength ())
    {
      // nope, not enough data to trickle here
      // maybe its still in the memory buffer?
      if (mbuf.getLength () > _len)
        {
          // hm, yes looks good
          // so flush it and lets rety
          try
          {
            doFlushToDisk (true);
          }
          catch (exception & e)
          {
            throw runtime_error (string (e.what ()));

            return 0;
          }
        }
    }

  // retry
  // make sure trickle buffer now has enough data
  if ((_offset + _len) <= fbuf.getFileLength ())
    {
      // ok, now there is enough data
      rd = fbuf.doRead (_data, _len, _offset);
    }

  return rd;
}

int
TrickleBuffer::doRead (char *_data, int _len, int _offset)
{
  int size = 0;

  size = fbuf.doRead (_data, _len, _offset);

  return size;
}


// ----------------- TrickleHandler -----------------------
TrickleHandler::TrickleHandler (int _max_buf, int _size, int _fd, HTTPHeader * _docheader, bool _dlmgr):percent_scale (20), trickle_pos (0), last_size (0), proxyfd (0), peerfd (0),
maxfd (0), tempbuffsize (_size), dlmgr (_dlmgr), scanIt (true), dg_data (false), tbuf (_max_buf, _size, _fd), docheader (_docheader)
{
}

TrickleHandler::~TrickleHandler ()
{
}

int
TrickleHandler::doOpen (char *tempFileName)
{
  int fd = 0;

  fd = tbuf.doOpen (tempFileName);
  return fd;
}

int
TrickleHandler::prepareTrickle (Socket & _proxysock, Socket & _peerconn, int _contentlength, int _tricklepos)
{
  trickle_length = o.trickle_length;

  trickle_delay = first_delay = o.first_trickle_delay;
  following_delay = o.following_trickle_delay;
  content_length = _contentlength;
  trickle_pos = _tricklepos;
  percent_chunk = content_length / percent_scale;
  percent_done = 1;
  dlmgr_header = true;

  proxyfd = _proxysock.getFD ();
  peerfd = _peerconn.getFD ();
  maxfd = peerfd > proxyfd ? peerfd : proxyfd;

  // is there already data for us?
  if (tbuf.getFileLength () > 0)
    {
      dg_data = true;
    }

  time_last = time_current = time (NULL);

  return 0;
}

int
TrickleHandler::prepareTrickle (Socket & _proxysock, Socket & _peerconn, int _tricklepos)
{
  // since we are continuing trickle
  // switch back to 1 byte trickle mode now
  trickle_length = -1;

  trickle_delay = first_delay = 1;
  following_delay = 1;
  content_length = 0;
  trickle_pos = _tricklepos;
  dg_data = false;

  proxyfd = _proxysock.getFD ();
  peerfd = _peerconn.getFD ();
  maxfd = peerfd > proxyfd ? peerfd : proxyfd;

  time_last = time_current = time (NULL);

  return 0;
}

int
TrickleHandler::doReceive (Socket & _proxysock, Socket & _peerconn)
{
  int rc = 0;
  timeval timeout;
  fd_set fdSet;                 // sockets descriptor set
  fd_set tmpfdSet;              // temp file descriptor set

  timeout.tv_sec = 5;           // modify the struct so its a 20 sec timeout
  timeout.tv_usec = 0;

  FD_ZERO (&fdSet);             // clear the set
  FD_SET (proxyfd, &fdSet);     // add proxy  descriptor to the set
  FD_SET (peerfd, &fdSet);      // add client descriptor to the set

  while (true)
    {
      tmpfdSet = fdSet;
      rc = selectEINTR (maxfd + 1, &tmpfdSet, NULL, NULL, &timeout);
      
      if (rc > 0)
        break;
      else if (rc == 0)
        continue;
      else
        {
#ifdef DGDEBUG
	      std::cout << "SELECTEINTR ERROR" << std::endl;
#endif
          return -2;
	    }	
    }

  // Client is set to send data
  // Normally, only a connection close will happen
  if (FD_ISSET (peerfd, &tmpfdSet))
    {
      char *tempbuff = new char[tempbuffsize];
      rc_in = _peerconn.readFromSocket (tempbuff, tempbuffsize, 0, 60);
      if (rc_in < 1)
        {
          delete[] tempbuff;

          return -2;
        }

      doSend (_proxysock, tempbuff, rc_in);

      delete[] tempbuff;
    }
  else if (FD_ISSET (proxyfd, &tmpfdSet))
    {
      // prepare for new data (flush if necessary)
      tbuf_ptr = tbuf.prepareAppend ();

      // grab a (limited) amount of input stream
      rc_in = _proxysock.readFromSocket (tbuf_ptr, tbuf.getTrickleSize (), 0, 60);

      if (rc_in < 1)
        {
          return -1;            // an error occured so end the while()
          // or none received so pipe is closed
        }

      // commit/update actual amount of appended data
      tbuf.commitAppend (rc_in);
    }

  return rc_in;
}

bool TrickleHandler::doTrickle (Socket & _sock)
{
  // tricklemode definition
  if (dlmgr)
    {
      doDlManager (_sock);
    }
  else if (trickle_length > 0)
    {
      doPosTrickle (_sock);
    }
  else if (trickle_length == -1)
    {
      doNeg1Trickle (_sock);
    }
  else
    {
      doNegTrickle (_sock);
    }

  return scanIt;
}

int
TrickleHandler::commitTrickle ()
{
  tbuf.doFlushToDisk (true);

  return 0;
}

int
TrickleHandler::doNeg1Trickle (Socket & _sock)
{
  //
  // tricklemode (negative-trickle)
  //
  // trickle 1 byte after first_trickle_delay
  // trickle each following byte after following_trickle_delay

  time_current = time (NULL);

  // is the trickle delay over?
  if ((time_current - time_last) >= trickle_delay)
    {
#ifdef DGDEBUG
      std::cout << "It is time to send some data to client" << std::endl;
      std::cout << "file_length: " << tbuf.getFileLength () << std::endl;
#endif

      char c;
      if ((rc_in = tbuf.doTrickle (&c, 1, trickle_pos)) > 0)
        {
          // write the data block out to the stream
          if (doSend (_sock, &c, rc_in))
            {
              time_last = time_current;
              trickle_pos += rc_in;
            }
#ifdef DGDEBUG
          else
            {
              std::cout << "Could not trickle byte " << trickle_pos << std::endl;
            }
#endif

          // continue w/ following_trickle_delay
          if (trickle_pos > 0)
            {
              trickle_delay = following_delay;
            }
        }

#ifdef DGDEBUG
      std::cout << "rc_in: " << rc_in << std::endl;
#endif
    }

  return rc_in;
}

int
TrickleHandler::doNegTrickle (Socket & _sock)
{
  //
  // tricklemode (advanced negative-trickle)
  //
  // trickle ((downloaded_now-downloaded_previously)/1mb)*(-trickle_length)
  // per trickle intervall (first_trickle_delay, following_trickle_delay, ...)

  rc_in = 0;
  time_current = time (NULL);

  // is the trickle delay over?
  if ((time_current - time_last) >= trickle_delay)
    {
#ifdef DGDEBUG
      std::cout << "It is time to send some data to client" << std::endl;
#endif

      // determine how much we need to send (FIXME: should we limit this?)
      int send_len = int (((tbuf.getFileLength () - last_size) / 1048576.0) * (-trickle_length));
#ifdef DGDEBUG
      std::cout << "last_size  : " << last_size << std::endl;
      std::cout << "file_length: " << tbuf.getFileLength () << std::endl;
      std::cout << "send_len   : " << send_len << std::endl;
#endif

      // not enough data to trickle, eh?
      if (send_len < 1)
        {
          trickle_delay = following_delay;
          return 0;
        }

      // update last_size
      last_size = tbuf.getFileLength ();

      // allocate buffer
      // note: we could improve this by trickling directly to the socket, but
      //       since we do this only every first/following_delay
      //       this is not exactly time critical ..
      char tmp[send_len];

      if ((rc_in = tbuf.doTrickle (tmp, send_len, trickle_pos)) > 0)
        {
          // write the data block out to the stream
          if (doSend (_sock, tmp, rc_in))
            {
              time_last = time_current;
              trickle_pos += rc_in;
            }
#ifdef DGDEBUG
          else
            {
              std::cout << "Could not trickle byte " << trickle_pos << std::endl;
            }
#endif

          // continue w/ following_trickle_delay
          if (trickle_pos > 0)
            {
              trickle_delay = following_delay;
            }

#ifdef DGDEBUG
          std::cout << "rc_in: " << rc_in << std::endl;
#endif
        }
    }

  return rc_in;
}

int
TrickleHandler::doPosTrickle (Socket & _sock)
{
  //
  // tricklemode (positive-trickle)
  //
  // trickle data until we reach lastsize,
  // then stop and start scanning
  // flush data  to make sure bufferlength is large enough

  // note: we do not flush data to disk here, 
  // instead we just send the received data to the client
  // (flushing to disk is done within the prepareAppend() function)

  char block[tbuf.getTrickleSize ()];
  int rc = 0;
  int sum = 0;
  int offset = 0;

  if (dg_data)
    {
      // there is previously downloaded data in the trickle buffer
      // so send it first ...
      while (trickle_pos < (tbuf.getFileLength () - trickle_length))
        {
          rc = tbuf.doRead (block, tbuf.getTrickleSize (), offset);

#ifdef DGDEBUG
          std::cout << "rc=" << rc << std::endl;
#endif

          if (rc < 1)
            {
              break;            // an error occured so end the while()
              // or none received so pipe is closed
            }

          if (rc > 0)
            {
#ifdef DGDEBUG
              std::cout << "sending rc=" << rc << std::endl;
#endif
              if (doSend (_sock, block, rc))
                {
                  sum += rc;
                  trickle_pos += rc;
#ifdef DGDEBUG
                  std::cout << "Wrote block to client socket." << std::endl;
                  std::cout << "sum=" << sum << std::endl;
#endif
                }

              offset += rc;
            }
        }

      dg_data = false;
    }


  if (scanIt)
    {
      if ((o.max_content_scan_size > 0) && (tbuf.getFileLength () > o.max_content_scan_size))
        {
          trickle_pos = doReadAll (_sock, trickle_pos);
          scanIt = false;
        }
      else
        {
          offset = tbuf.getFileLength () - trickle_length;

          if ((offset > 0) && (trickle_pos < offset))
            {
              rc = tbuf.doRead (block, tbuf.getTrickleSize (), trickle_pos);

#ifdef DGDEBUG
              std::cout << "rc=" << rc << std::endl;
#endif

              if (rc > 0)
                {
#ifdef DGDEBUG
                  std::cout << "sending rc=" << rc << std::endl;
#endif
                  if (doSend (_sock, block, rc))
                    {
                      trickle_pos += rc;
#ifdef DGDEBUG
                      std::cout << "Wrote block to client socket." << std::endl;
                      std::cout << "sum=" << sum << std::endl;
#endif
                    }
#ifdef DGDEBUG
                  else
                    {
                      std::cout << "Could not write block to client socket." << std::endl;
                    }
#endif
                }
            }
#ifdef DGDEBUG
          std::cout << "file length: " << tbuf.getFileLength () << std::endl;
#endif
        }
    }
  else
    {
      if (doSend (_sock, tbuf_ptr, rc_in))
        {
          trickle_pos += rc_in;
        }
#ifdef DGDEBUG
      else
        {
          std::cout << "Could not trickle byte " << trickle_pos << std::endl;
        }
#endif
    }

#ifdef DGDEBUG
  std::cout << "trickle_pos: " << trickle_pos << " rc_in=" << rc_in << std::endl;
#endif

  tbuf.doFlushToDisk (true);
  return rc_in;
}

int
TrickleHandler::doDlManager (Socket & _sock)
{
  //
  // tricklemode (Download manager)
  //
  // Html download manager to prevent sending data to clients
  // before scanning object data

  String Html;
  String message;

  if (dlmgr_header)
    {

      Html = "HTTP/1.1 200 OK\r\nContent-Type: text/html; charset=ISO-8859-1\r\n";
      Html += "\r\n";

      doSend (_sock, Html.toCharArray (), Html.length ());

      // this is awful, but makes it work with some browsers...
      for (int a = 1; a < 16; a++)
        {
          Html += "\r\n";
        }

      Html = "<html>\r\n<body>\r\n<div id='header'><center><font size=5 face=\"Arial\">";
      Html += "Dansguardian Antivirus plug-in download manager";
      Html += "</font><br><br>\r\n";
      Html += "<form name='statsform'>";
      Html += "<input type='text' id='stats' name='stats' value='Initializing...' style='text-align:center'></form></center>\r\n";
      Html += "<table align='center'><tr><td>\r\n";
      Html += "<div style='font-size:8pt;padding:2px;border:solid black 1px'>\r\n";
      Html += "<span id='progress1'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress2'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress3'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress4'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress5'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress6'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress7'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress8'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress9'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress10'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress11'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress12'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress13'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress14'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress15'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress16'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress17'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress18'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress19'>&nbsp; &nbsp;</span>\r\n";
      Html += "<span id='progress20'>&nbsp; &nbsp;</span>\r\n";
      Html += "</div>\r\n";
      Html += "</td></tr></table>\r\n<br>\r\n";
      Html += "<table align='center'><tr><td>\r\n";
      Html += "<input type='text' id='operation' name='operation' ";
      Html += "style='border-style:none;font-family:Arial;font-size:medium;font-weigth:bold;color: #800000;text-align:center' ";
      Html += "value='Receiving file...'></td></tr></table>\r\n<br>\r\n";
      Html += "<script language='javascript'>\r\n";
      Html += "for (var i = 1; i <= 20; i++) document.getElementById('progress'+i).style.backgroundColor = 'transparent';";
      Html += "</script>\r\n";
      Html += "<noscript>\r\n<PRE>\r\n</noscript>\r\n";

      doSend (_sock, Html.toCharArray (), Html.length ());

      dlmgr_header = false;
      time_last = time (NULL);
    }

  time_current = time (NULL);
  trickle_pos = tbuf.getFileLength ();

  if ((time_current - time_last) > 0)
    {
      message = "Download status: ";
      message += String ((signed) (trickle_pos / ((time_current - time_last) * 1024)));
      message += " kbytes/s - received ";
      message += String ((signed) (trickle_pos / 1024));
      message += " of ";
      message += String ((signed) (content_length / 1024));
      message += " kbytes";
    }
  else
    {
      message = "Initializing...";
    }

  Html = "<script language='javascript'>document.statsform.stats.value='";
  Html += message;
  Html += "';document.statsform.stats.size='" + String (message.length ()) + "';</script>\r\n";
  Html += "<script language='javascript'>for (var i = 1; i <=" + String ((int) (trickle_pos / percent_chunk)) + "; i++) document.getElementById('progress'+i).style.backgroundColor = 'blue';</script>\r\n";
  Html += "<noscript>" + message + "</noscript>\r\n";
  Html += "<!-- force flush -->\r\n";

  while (percent_done < (trickle_pos / percent_chunk))
    {
      percent_done++;

      if (percent_done >= (trickle_pos / percent_chunk))
        {
          doSend (_sock, Html.toCharArray (), Html.length ());
        }
    }

  return 0;
}

bool TrickleHandler::doSend (Socket & _sock, char *_block, int _size) throw ()
{
  // check wether we need to send the header data first or not
  if ((!dlmgr) && (trickle_pos == 0))
    {
      try
      {
      _sock.readyForOutput (60); // exceptions on error/timeout
        docheader->out (&_sock);   // send body header to client
      }
      catch (exception & e)
      {
        return false;
      }
    }

  if (_size == 0)
    {
      return true;
    }

  try
  {
    // need exception or something for a bad write
    _sock.writeToSocket (_block, _size, 0, 60);
  }
  catch (exception & e)
  {
#ifdef DGDEBUG
    std::cout << "Error sending data to socket: ";
#endif

    return false;
  }                             // write the data block out to the stream

  return true;
}


int
TrickleHandler::doReadAll (Socket & _sock, int _from_pos)
{
  char block[tbuf.getTrickleSize ()];
  int send_cnt = 0;
  int offset = trickle_pos = _from_pos;
  int rc = 0;

  while (true)
    {
      try
      {
        rc = tbuf.doRead (block, tbuf.getTrickleSize (), offset);
      }
      catch (exception & e)
      {
        throw runtime_error (string (e.what ()));
      }

      if (rc < 1)
        {
          break;
        }
      if (rc > 0)
        {
          doSend (_sock, block, rc);
          trickle_pos = offset += rc;
          send_cnt++;
        }
#ifdef DGDEBUG
      else
        {
          std::cout << "Could not write block to client socket." << std::endl;
        }
#endif
    }

  if (send_cnt == 0)
    {
      // ok, seems like we did not send any data to the client
      // make sure to at least send the header now ...
      doSend (_sock, NULL, 0);
    }

  return trickle_pos;
}

// a wrapper for select to handle EINTR neater
int
TrickleHandler::selectEINTR (int numfds, fd_set * readfds, fd_set * writefds, fd_set * exceptfds, struct timeval *timeout)
{
  int rc = 0;

  while (true)
    {                           // using the while as a restart point with continue
      rc = select (numfds, readfds, writefds, exceptfds, timeout);
      if (rc < 0)
        {
          if (errno == EINTR)
            {
              continue;         // was interupted by a signal so restart
            }
        }
      break;                    // end the while
    }

  return rc;                    // return status
}
