/* $Id: cmd-save-buffer.c,v 1.11 2010/07/02 02:52:13 tcunha Exp $ */

/*
 * Copyright (c) 2009 Tiago Cunha <me@tiagocunha.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <string.h>

#include "tmux.h"

/*
 * Saves a session paste buffer to a file.
 */

int	cmd_save_buffer_exec(struct cmd *, struct cmd_ctx *);

const struct cmd_entry cmd_save_buffer_entry = {
	"save-buffer", "saveb",
	"[-a] " CMD_BUFFER_SESSION_USAGE " path",
	CMD_ARG1, "a",
	cmd_buffer_init,
	cmd_buffer_parse,
	cmd_save_buffer_exec,
	cmd_buffer_free,
	cmd_buffer_print
};

int
cmd_save_buffer_exec(struct cmd *self, struct cmd_ctx *ctx)
{
	struct cmd_buffer_data	*data = self->data;
	struct session		*s;
	struct paste_buffer	*pb;
	mode_t			mask;
	FILE			*f, *close_f;

	if ((s = cmd_find_session(ctx, data->target)) == NULL)
		return (-1);

	if (data->buffer == -1) {
		if ((pb = paste_get_top(&s->buffers)) == NULL) {
			ctx->error(ctx, "no buffers");
			return (-1);
		}
	} else {
		if ((pb = paste_get_index(&s->buffers, data->buffer)) == NULL) {
			ctx->error(ctx, "no buffer %d", data->buffer);
			return (-1);
		}
	}

	if (strcmp(data->arg, "-") == 0) {
		if (ctx->cmdclient == NULL) {
			ctx->error(ctx, "%s: can't write to stdout", data->arg);
			return (-1);
		}
		f = ctx->cmdclient->stdout_file;
		close_f = NULL;
	} else {
		mask = umask(S_IRWXG | S_IRWXO);
		if (cmd_check_flag(data->chflags, 'a'))
			f = fopen(data->arg, "ab");
		else
			f = fopen(data->arg, "wb");
		umask(mask);
		if (f == NULL) {
			ctx->error(ctx, "%s: %s", data->arg, strerror(errno));
			return (-1);
		}
		close_f = f;
	}

	if (fwrite(pb->data, 1, pb->size, f) != pb->size) {
	    	ctx->error(ctx, "%s: fwrite error", data->arg);
	    	fclose(f);
	    	return (-1);
	}

	if (close_f != NULL)
		fclose(close_f);

	return (0);
}
