/*****************************************************************************

Copyright (c) 2007, 2013, Oracle and/or its affiliates. All Rights Reserved.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; version 2 of the License.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1335 USA

*****************************************************************************/

/**************************************************//**
@file include/ha0storage.ic
Hash storage.
Provides a data structure that stores chunks of data in
its own storage, avoiding duplicates.

Created September 24, 2007 Vasil Dimov
*******************************************************/

#include "hash0hash.h"
#include "mem0mem.h"

/** Hash storage for strings */
struct ha_storage_t {
	mem_heap_t*	heap;	/*!< memory heap from which memory is
				allocated */
	hash_table_t*	hash;	/*!< hash table used to avoid
				duplicates */
};

/** Objects of this type are stored in ha_storage_t */
struct ha_storage_node_t {
	ulint			data_len;/*!< length of the data */
	const void*		data;	/*!< pointer to data */
	ha_storage_node_t*	next;	/*!< next node in hash chain */
};

/*******************************************************************//**
Creates a hash storage. If any of the parameters is 0, then a default
value is used.
@return own: hash storage */
UNIV_INLINE
ha_storage_t*
ha_storage_create(
/*==============*/
	ulint	initial_heap_bytes,	/*!< in: initial heap's size */
	ulint	initial_hash_cells)	/*!< in: initial number of cells
					in the hash table */
{
	ha_storage_t*	storage;
	mem_heap_t*	heap;

	if (initial_heap_bytes == 0) {

		initial_heap_bytes = HA_STORAGE_DEFAULT_HEAP_BYTES;
	}

	if (initial_hash_cells == 0) {

		initial_hash_cells = HA_STORAGE_DEFAULT_HASH_CELLS;
	}

	/* we put "storage" within "storage->heap" */

	heap = mem_heap_create(sizeof(ha_storage_t)
			       + initial_heap_bytes);

	storage = (ha_storage_t*) mem_heap_alloc(heap,
						 sizeof(ha_storage_t));

	storage->heap = heap;
	storage->hash = hash_create(initial_hash_cells);

	return(storage);
}

/*******************************************************************//**
Empties a hash storage, freeing memory occupied by data chunks.
This invalidates any pointers previously returned by ha_storage_put().
The hash storage is not invalidated itself and can be used again. */
UNIV_INLINE
void
ha_storage_empty(
/*=============*/
	ha_storage_t**	storage)	/*!< in/out: hash storage */
{
	ha_storage_t	temp_storage;

	temp_storage.heap = (*storage)->heap;
	temp_storage.hash = (*storage)->hash;

	hash_table_clear(temp_storage.hash);
	mem_heap_empty(temp_storage.heap);

	*storage = (ha_storage_t*) mem_heap_alloc(temp_storage.heap,
						  sizeof(ha_storage_t));

	(*storage)->heap = temp_storage.heap;
	(*storage)->hash = temp_storage.hash;
}

/*******************************************************************//**
Frees a hash storage and everything it contains, it cannot be used after
this call.
This invalidates any pointers previously returned by ha_storage_put(). */
UNIV_INLINE
void
ha_storage_free(
/*============*/
	ha_storage_t*	storage)	/*!< in, own: hash storage */
{
	/* order is important because the pointer storage->hash is
	within the heap */
	hash_table_free(storage->hash);
	mem_heap_free(storage->heap);
}

/*******************************************************************//**
Gets the size of the memory used by a storage.
@return bytes used */
UNIV_INLINE
ulint
ha_storage_get_size(
/*================*/
	const ha_storage_t*	storage)	/*!< in: hash storage */
{
	ulint	ret;

	ret = mem_heap_get_size(storage->heap);

	/* this assumes hash->heap and hash->heaps are NULL */
	ret += sizeof(hash_table_t);
	ret += sizeof(hash_cell_t) * hash_get_n_cells(storage->hash);

	return(ret);
}
