/* 
 *	HT Editor
 *	analy.cc
 *
 *	Copyright (C) 1999, 2000, 2001 Sebastian Biallas (sb@web-productions.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "analy.h"
#include "analy_names.h"
#include "analy_register.h"
#include "codeanaly.h"
#include "dataanaly.h"
#include "global.h"
#include "htatom.h"
#include "htctrl.h"
#include "htdebug.h"
#include "httag.h"
#include "language.h"
#include "tools.h"
#include "x86dis.h"


analyser *theanaly=NULL;
char* (*analy_addr_sym_val_func)(CPU_ADDR addr)=NULL;
analyser *testanaly=NULL;

//#undef dprintf
//#define dprintf(msg...) printf(##msg)


/*
 *
 */
addrqueueitem::addrqueueitem() {}

addrqueueitem::addrqueueitem(ADDR Addr, ADDR Func)
{
	addr = Addr;
	func = Func;
}

OBJECT_ID	addrqueueitem::object_id()
{
	return ATOM_ADDRQUEUEMEMBER;
}

int	addrqueueitem::load(ht_object_stream *f)
{
	GET_INT_HEX(f, addr);
	GET_INT_HEX(f, func);
	return f->get_error();
}

void	addrqueueitem::store(ht_object_stream *f)
{
	PUT_INT_HEX(f, addr);
	PUT_INT_HEX(f, func);
}

/*
 *
 */
void comment_list::init()
{
	ht_clist::init();
}

void comment_list::append_comment(char *s)
{
	append(new ht_data_string(s));
}

void comment_list::append_comment(int special)
{
	append(new ht_data_uint(special));
}

char *comment_list::get_name(UINT i)
{
	ht_data *d = get(i);
	return d ? ((d->object_id()==ATOM_HT_DATA_UINT) ? comment_lookup(((ht_data_uint*)d)->value): ((ht_data_string*)d)->value) : NULL;
}

/*
 *
 */
void	analyser::init()
{
	testanaly = this;

	active = false;
	dirty = false;
	addr_queue = new ht_queue();
	addr_queue->init();
	addrs = NULL;
	labels = NULL;
	set_addr_tree_optimize_threshold(1000);
	set_label_tree_optimize_threshold(1000);
	cur_addr_ops = cur_label_ops = 1;
	cur_func = NULL;
	ops_parsed = 0;
	next_explored = first_explored = last_explored = INVALID_ADDR;
	explored = new	area();
	explored->init();
	init_code_analyser();
	init_data_analyser();
	disasm = NULL;
	analy_disasm = NULL;
	maxopcodelength = 1;
	init_unasm();
	mode = ANALY_TRANSLATE_SYMBOLS;
}

void loadaddrs(ht_object_stream *st, taddr *&addr, int level, int &left)
{
	if (left<=0) {
		addr = NULL;
		return;
	}
	if (st->get_error()) {
		addr = NULL;
		return;
	}

	addr = (taddr *) smalloc0(sizeof(taddr));

	bool sdf=!((level<= 1) || (left == 1));
	if (sdf) loadaddrs(st, addr->left, level / 2, left);
	if (st->get_error()) {
		free(addr);
		addr = NULL;
		return;
	}
/*	printf("%d, %d\n", level, left);
	if (left==18286) {
		int asdf=234;
	}*/
	
	addr->addr = st->get_int_hex(sizeof(ADDR), "addr");
	// xrefs
	addr->xreflist = NULL;
	txref **xref = &addr->xreflist;
	int count = st->get_int_dec(4, "count_of_xrefs");
	if (st->get_error()) {
		free(addr);
		addr = NULL;
		return;
	}
	while (count--) {
		if (st->get_error()) {
			*xref = NULL;
			addr = NULL;
			return;
		}
		*xref = (txref *)smalloc(sizeof(txref));
		(*xref)->addr = st->get_int_hex(sizeof(ADDR), "xref_addr");
		(*xref)->type = (txreftype)st->get_int_hex(sizeof(txreftype), "xref_type");
		xref = &(*xref)->next;
	}
	*xref = NULL;
	// comments
	st->get_object((object *)addr->comments, "comments");
	if (st->get_error()) {
		free(addr);
		addr = NULL;
		return;
	}

	analyser_get_addrtype(st, &addr->type);


	// must be resolved later
	addr->thisfunc = (taddr *)st->get_int_hex(4, "func");

	addr->flags = st->get_int_hex(4, "flags");
	
	left--;

	if (st->get_error()) {
		free(addr);
		addr = NULL;
		return;
	}
	if (sdf) loadaddrs(st, addr->right, level / 2 -1, left);
}

void loadlabels(analyser *analy, ht_object_stream *st, tlabel *&label, int level, int &left)
{
	if (left<=0) {
		label = NULL;
		return;
	}
	if (st->get_error()) {
		free(label);
		label = NULL;
		return;
	}

	label = (tlabel *) smalloc0(sizeof(tlabel));

	bool sdf=!((level<= 1) || (left == 1));
	if (sdf) loadlabels(analy, st, label->left, level / 2, left);

	ADDR a = st->get_int_hex(sizeof a, "addr");
	(label->addr = analy->new_addr(a))->label = label;
//     label->addr->label = label;

//	fstrget(st, label->name);
//	          st->put_string(label->name, "name");
	label->name = st->get_string("name");

	label->type = (labeltype)st->get_int_hex(1, "type");

	left--;

	if (st->get_error()) {
		free(label);
		label = NULL;
		return;
	}
	if (sdf) loadlabels(analy, st, label->right, level / 2 -1, left);
}

void resolveaddrs(analyser *a, taddr *addr)
{
	if (addr) {
		resolveaddrs(a, addr->left);
		if ((ADDR)addr->thisfunc != INVALID_ADDR) {
			addr->thisfunc = a->find_addr((ADDR)addr->thisfunc);
		} else {
			addr->thisfunc = NULL;
		}
		resolveaddrs(a, addr->right);
	}
}

/*
 *
 */
int analyser::load(ht_object_stream *st)
{
	GET_OBJECT(st, addr_queue);
	GET_INT_DEC(st, ops_parsed);
	GET_BOOL(st, active);
	if (active) {
		some_analyser_active++;
	}

	GET_INT_HEX(st, next_explored);
	GET_INT_HEX(st, first_explored);
	GET_INT_HEX(st, last_explored);

	GET_INT_HEX(st, mode);

	GET_OBJECT(st, explored);

	int addr_count;
	GET_INT_DEC(st, addr_count);
	loadaddrs(st, addrs, addr_count, addr_count);

	if (st->get_error()) return st->get_error();

	resolveaddrs(this, addrs);
	
	int label_count;
	GET_INT_DEC(st, label_count);
	loadlabels(this, st, labels, label_count, label_count);

	if (st->get_error()) return st->get_error();

	GET_OBJECT(st, analy_disasm);
	GET_OBJECT(st, disasm);     
	if (analy_disasm) {
		analy_disasm->analy = this;
		analy_disasm->disasm = disasm;
	}
	GET_OBJECT(st, code);
	GET_OBJECT(st, data);
	if (data) {
		data->analy = this;
	}
	GET_INT_DEC(st, addrthreshold);
	GET_INT_DEC(st, labelthreshold);

	GET_INT_DEC(st, maxopcodelength);

	if (st->get_error()) return st->get_error();

	ADDR curfuncaddr = st->get_int_hex(sizeof curfuncaddr, "cur_func");
	if (curfuncaddr==INVALID_ADDR) {
		cur_func = NULL;
	} else {
		cur_func = new_addr(curfuncaddr);
	}

	cur_addr_ops = 0;
	cur_label_ops = 0;
	dirty = false;
	return st->get_error();
}

/*
 *
 */
void	analyser::done()
{
	set_active(false);
	free_addrs(addrs);
	free_labels(labels);
	if (addr_queue) {
		addr_queue->destroy();
		delete addr_queue;
	}
	if (explored) {
		explored->done();
		delete explored;
	}
	if (code) {
		code->done();
		delete code;
	}
	if (data) {
		data->done();
		delete data;
	}
	if (disasm) {
		disasm->done();
		delete disasm;
	}
	if (analy_disasm) {
		analy_disasm->done();
		delete analy_disasm;
	}
}

/*
 *	addaddrlabel will never overwrite an existing label (like addlabel)
 */
bool analyser::add_addr_label(ADDR Addr, char *Prefix, labeltype type, taddr *infunc=NULL)
{
	if (!valid_addr(Addr, scvalid)) return false;


	char *prefix = label_prefix(Prefix);

	char	label[1024];

	sprintf(label, "%s"HEX8FORMAT, prefix, Addr);

	if (add_label(Addr, label, type, infunc)) {
		return true;
	} else {
		return false;
	}
}

/*
 *
 */
void	analyser::add_comment(ADDR Addr, int line, char *c)
{
	// line 0 meens append (at the moment assume append every time ;-))

//	if (!validaddr(Addr, scvalid)) return;

	taddr *a = new_addr(Addr);

	comment_list *com = a->comments;

	if (!com) {
		com = new comment_list();
		com->init();
		a->comments = com;
	}
	com->append_comment(c);

	dprintf("#(%08lx) comment `%s'\n", Addr, c);
}

/*
 * addlabel: create label if there isnt one
 *           fail if label exist on another address
 *
 */
bool analyser::add_label(ADDR Addr, char *label, labeltype type, taddr *infunc=NULL)
{
	if (!valid_addr(Addr, scvalid)) return false;

	taddr *a = new_addr(Addr);

	if (!a->label) {

		tlabel *l = new_label(label, a, type, infunc);

		if (l->addr->addr != Addr) {
			// this label already exists at a different address
			return false;
		}

		a->label = l;

		switch (type) {
			case label_unknown:
				break;
			case label_func:
				data->setcodeaddrtype(a, dstfunction);
				break;
			case label_loc:
				data->setcodeaddrtype(a, dstlocation);
				break;
			case label_data:
				data->setaddrtype(a, dtunknowndata, 0, 0);
				break;
		}

		return true;

	} else {

		// adress already has a label
		return false;
	}
}

/*
 *
 */
void	analyser::add_xref(ADDR from, ADDR to, txreftype action)
{
	if ((!valid_addr(from, scvalid)) || (!valid_addr(to, scvalid))) return;

	taddr	*a =	new_addr(from);
	txref	*x = a->xreflist;

	if (x) {
		while (x->next) {
			if (x->addr==to) {
				// update xref
				analyser__addxref_raus:
				x->type = action;
				dprintf("xref %08lx->%08lx updated\n", from, to);
				return;
			}
			x = x->next;
		}
		if (x->addr==to) goto analyser__addxref_raus;
		x->next = (txref *) smalloc(sizeof(txref));
		x = x->next;
	} else {
		x = (txref *) smalloc(sizeof(txref));
		a->xreflist = x;
	}

	x->next =	NULL;
	x->addr =	to;
	x->type = action;

	dprintf("xref %08lx->%08lx\n", from, to);
}

/*
 *
 */
void	analyser::assign_comment(ADDR Addr, int line, char *c)
{
	/* not really implemented */
	add_comment(Addr, line, c);
}

/*
 *
 */
bool analyser::assign_label(ADDR Addr, char *label, labeltype type, taddr *infunc=NULL)
{
	if (!valid_addr(Addr, scvalid)) return false;

	taddr *a = new_addr(Addr);

	tlabel *l = new_label(label, a, type, infunc);
	if (l->addr->addr != Addr) {
		// label already exists at a different address
		return false;
	}

	switch (type) {
		case label_unknown:
			break;
		case label_func:
			data->setcodeaddrtype(a, dstfunction);
			break;
		case label_loc:
			data->setcodeaddrtype(a, dstlocation);
			break;
		case label_data:
			data->setaddrtype(a, dtunknowndata, 0, 0);
			break;
	}

	if (a->label) {
		// overwrite
		if (a->label != l) {
			// label has to be renamed
			disable_label(a->label);
			a->label = l;
		}
	} else {
		a->label = l;
	}
	return true;
}

/*
 *
 */
void	analyser::begin_analysis()
{
	if (query_config(Q_DO_ANALYSIS)) {
		dprintf("################################\nAnalysis started at %08lx:\n", Addr);
		if (analy_disasm && disasm) {
			ops_parsed = 0;
			if (goto_addr(INVALID_ADDR, INVALID_ADDR)) set_active(true);
		} else {
			dprintf("Analysis can't be started. No disassembler available.");
		}
	}
}

/*
 *
 */
bool	analyser::continue_analysis()
{
	byte			buf[16];
	OPCODE		*instr;
	int			len;
	tbranchtype	branch;

	if (!active) return	false;
	do {
		if (!savety_change_addr(addr)) {
			finish();
			return false;
		}

		int bz = bufptr(addr, buf, sizeof(buf));

		instr = disasm->decode(buf, MIN(bz, maxopcodelength), map_addr(addr));
		dprintf("opcode @%08lx [func: %08lx]: %s\n", addr, (cur_func) ? cur_func->addr : INVALID_ADDR, disasm->str(instr, 0));

		ops_parsed++;
		num_ops_parsed++;

		len = disasm->getsize(instr);
		last_explored += len;

		if (disasm->valid_insn(instr)) {
			branch = analy_disasm->is_branch(instr);
			if (branch != brnobranch) do_branch(branch, instr,	len); else {
				examine_opcode(instr);
				addr	+= len;
			}
		} else {
			dprintf("invalid opcode @%08lx\n", addr);
//             log("invalid opcode at address %08lx\n", addr);
			new_addr(addr+len)->flags |= AF_FUNCTION_END;
			goto_addr(INVALID_ADDR, INVALID_ADDR);
		}
	} while ((ops_parsed % MAX_OPS_PER_CONTINUE) !=0);
	notify_progress(addr);
	return true;
}

/*
 *
 */
void analyser::continue_analysis_at(ADDR Addr)
{
	if (!valid_code_addr(Addr)) return;
	if (query_config(Q_DO_ANALYSIS)) {
		dprintf("continueing analysis at %08lx\n", Addr);
		if (active) {
			push_addr(Addr, Addr);
		} else {
			if (disasm) {
				push_addr(Addr, Addr);
				begin_analysis();
			} else {
				dprintf("couldn't start analysis: no disasm available\n");
			}
		}
	}
}

/*
 *
 */
void	analyser::data_access(ADDR Addr, taccess access)
{
	if (!valid_addr(Addr, scvalid)) {
		char	msg[100];
		sprintf(msg, "access of invalid addr %08x at addr %08x", Addr, addr);
		log(msg);
		return;
	}

	dprintf("dataaccess of %08x\n", Addr);

	// string test
	byte buffer[1024];

	if (valid_addr(Addr, scinitialized)) {
		UINT bz = bufptr(Addr, buffer, sizeof(buffer));
		if (bz > 2) {
			analy_string *str = string_test(buffer, bz);
			if (str) {
				char string1[256], string2[31];
				str->render_string(string2, 30);
				sprintf(string1, "%s_%s_", str->name(), string2);
				make_valid_name(string2, string1);
				add_addr_label(Addr, string2, label_data);
				str->done();
				delete str;
				return;
			}
		}
	}

	if (valid_code_addr(Addr)) {
		if (add_addr_label(Addr, LPRFX_OFS, label_func)) {
			add_comment(Addr, 0, "");
		}
		taddr *a = find_addr(Addr);
		assert(a);
		// test if Addr points to code
		if ((a->type.type == dtunknown) || (a->type.type == dtcode)) {
			// test if Addr points to valid code (test not yet complete)
			byte buf[16];
			int bz = bufptr(Addr, buf, sizeof(buf));
			OPCODE *instr = disasm->decode(buf, MIN(bz, maxopcodelength), map_addr(Addr));
			if (disasm->valid_insn(instr)) {
				data->setcodeaddrtype(Addr, dstcunknown);
				push_addr(Addr, Addr);
			}
		}
	} else {
		if (valid_addr(Addr, scinitialized)) {
//               data->setaddrtype(Addr, dtunknowndata, 0, 0);
			add_addr_label(Addr, LPRFX_DTA, label_data);
		} else {
//               data->setaddrtype(Addr, dtunknowndata, 0, 0);
			add_addr_label(Addr, LPRFX_DTU, label_data);
		}
	}
}

/*
 *	disables address, frees misc
 */
void	analyser::delete_addr(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (a) {
		disable_label(a->label);
		a->label = NULL;
		a->flags |= AF_DELETED;
	}
}

/*
 *	disables label of an address and unassigns address' label
 */
void analyser::delete_label(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (a) {
		disable_label(a->label);
		a->label = NULL;
	}
}

/*
 *	an disabled label will be overwritten as soon as possible,
 *   and never be returned nor saved.
 *	performed this way to preserve the labeltrees structure
 *
 */
void	analyser::disable_label(tlabel *label)
{
	if (label) {
		label->addr = NULL;
	}
}

/*
 *
 */
void	analyser::do_branch(tbranchtype branch, OPCODE *opcode, int len)
{

	ADDR	branch_addr = analy_disasm->branch_addr(opcode);

	if (branch != brreturn) {
		if (!valid_code_addr(branch_addr)) {
			char	msg[100];
			sprintf(msg, "branch to invalid addr %08x from addr %08x", branch_addr, addr);
			log(msg);
			// a little hack
			examine_opcode(opcode);
		}
	}
	switch (branch) {
		case	brjump:
			new_addr(addr+len)->flags |= AF_FUNCTION_END;

			add_xref(branch_addr, addr, xrefjump);
			if (add_addr_label(branch_addr, LPRFX_LOC, label_loc, cur_func)) {
				add_comment(branch_addr, 0, "");
			}
			goto_addr(branch_addr, INVALID_ADDR);
			break;
		case	brreturn:
			new_addr(addr+len)->flags |= AF_FUNCTION_END;

			goto_addr(INVALID_ADDR, INVALID_ADDR);
			break;
		case	brcall:
			add_xref(branch_addr, addr, xrefcall);
			if (add_addr_label(branch_addr, LPRFX_SUB, label_func)) {
				add_comment(branch_addr, 0, "");
				add_comment(branch_addr, 0, ";-----------------------");
				add_comment(branch_addr, 0, ";  S U B R O U T I N E");
				add_comment(branch_addr, 0, ";-----------------------");
				if (branch_addr==cur_func->addr) {
					add_comment(branch_addr, 0, ";  (recursive)");
				}
			}
			push_addr(addr+len, cur_func->addr);
			goto_addr(branch_addr, branch_addr);
			break;
		case	brjXX:
			add_xref(branch_addr, addr, xrefjump);
			if (add_addr_label(branch_addr, LPRFX_LOC, label_loc, cur_func)) {
				add_comment(branch_addr, 0, "");
			}
			push_addr(addr+len, cur_func->addr);
			goto_addr(branch_addr, INVALID_ADDR);
			break;
		default:{}
			// stupid but neccessary
	}
}

/*
 *
 */
void	analyser::engage_codeanalyser()
{
	if (query_config(Q_ENGAGE_CODE_ANALYSER)) {
		dprintf("starting code analyser.\n");
	}
}

void analyserenum_addrs(taddr *addrs, ADDR at, taddr *&addr)
{
	if ((at < addrs->addr) || (at == INVALID_ADDR)) {
		addr = addrs;
		if (addrs->left) analyserenum_addrs(addrs->left, at, addr);
	} else /*if (at >= addrs->addr)*/ {
		if (addrs->right) analyserenum_addrs(addrs->right, at, addr);
	}
}

/*
 *
 */
taddr *analyser::enum_addrs(ADDR Addr)
{
	taddr *result = NULL;
	if (addrs) analyserenum_addrs(addrs, Addr, result);
	while ((result) && (result->flags & AF_DELETED)) {
		ADDR a = result->addr;
		result = NULL;
		analyserenum_addrs(addrs, a, result);
	}
	return result;
}

void analyserenum_addrs_back(taddr *addrs, ADDR at, taddr *&addr)
{
	if (at <= addrs->addr) {
		if (addrs->left) analyserenum_addrs_back(addrs->left, at, addr);
	} else /*if (at > addrs->addr)*/ {
		addr = addrs;
		if (addrs->right) analyserenum_addrs_back(addrs->right, at, addr);
	}
}

/*
 *
 */
taddr *analyser::enum_addrs_back(ADDR Addr)
{
	taddr *result = NULL;
	if (addrs) analyserenum_addrs_back(addrs, Addr, result);
	while ((result) && (result->flags & AF_DELETED)) {
		ADDR a = result->addr;
		result = NULL;
		analyserenum_addrs_back(addrs, a, result);
	}
	return result;
}

void analyserenum_labels(tlabel *labels, char *at, tlabel *&label)
{
	int i = strcmp(at, labels->name);
	if (i < 0) {
		label = labels;
		if (labels->left) analyserenum_labels(labels->left, at, label);
	} else /*if (i >= 0)*/ {
		if (labels->right) analyserenum_labels(labels->right, at, label);
	}
}

/*
 *   returns the (alphanumerically) next label to "at"
 *	or the first if "at" is NULL
 *	returns NULL if there is no preceding label
 *
 */
/* e.g.:
		tlabel *label = enum_labels(NULL);
		while (label) {
			printf("%d %08x %s\n", label->type, label->addr->addr, label->name);
			label = enum_labels(label->name);
		}
 */
tlabel *analyser::enum_labels(char *at)
{
	tlabel *result = NULL;
	if (!at) at="";
	if (labels) analyserenum_labels(labels, at, result);

	// label with !addr mustnt be returned
	while ((result) && (!result->addr)) {
		char *name = result->name;
		result = NULL;
		analyserenum_labels(labels, name, result);
	}

	return result;
}

void analyserenum_labels_back(tlabel *labels, char *at, tlabel *&label)
{
	int i = strcmp(at, labels->name);
	if (i <= 0) {
		if (labels->left) analyserenum_labels_back(labels->left, at, label);
	} else /*if (i >= 0)*/ {
		label = labels;
		if (labels->right) analyserenum_labels_back(labels->right, at, label);
	}
}

/*
 *
 */
tlabel *analyser::enum_labels_back(char *at)
{
	tlabel *result = NULL;
	// FIXME:
	if (!at) at="\xff\xff\xff";
	if (labels) analyserenum_labels_back(labels, at, result);

	// labels with !addr mustnt be returned
	while ((result) && (!result->addr)) {
		char *name = result->name;
		result = NULL;
		analyserenum_labels_back(labels, name, result);
	}

	return result;
}

/*
 *
 */
taddrtypetype analyser::examine_data(ADDR Addr)
{
	if ((valid_read_addr(Addr)) && (valid_addr(Addr, scinitialized))) {
		dprintf("examinating data @%08lx:\n", Addr);

	} else return dtunknown;
	return dtunknown;
}

/*
 *
 */
taddr *analyserfindaddr(taddr *addrs, ADDR Addr)
{
	if (addrs) {
		if (Addr < addrs->addr) return analyserfindaddr(addrs->left, Addr);
		if (Addr > addrs->addr) return analyserfindaddr(addrs->right, Addr);
		return (addrs->flags & AF_DELETED) ? NULL : addrs;
	}
	return NULL;
}

/*
 *
 */
taddr *analyser::find_addr(ADDR Addr)
{
	return analyserfindaddr(addrs, Addr);
}

/*
 *	findaddrfunc finds the function the Addr belongs to (if possible/applicable).
 */
taddr *analyser::find_addr_func(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (!a) a = enum_addrs_back(Addr);
	while ((a) && (!(a->flags & AF_FUNCTION_SET))) {
		if (a->flags & AF_FUNCTION_END) return NULL;
		a = enum_addrs_back(a->addr);
	}
	return (a) ? a->thisfunc : NULL;
}

/*
 *	findaddrlabel searches back to the last label
 */
taddr *analyser::find_addr_label(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (!a) a = enum_addrs_back(Addr);
	while ((a) && (!a->label)) {
		if (a->flags & AF_FUNCTION_END) return NULL;
		a = enum_addrs_back(a->addr);
	}
	return (a) ? a : NULL;
}

/*
 *
 */
tlabel *analyserfindlabel(tlabel *labels, char *label)
{
	if (labels) {
		int i=strcmp(label, labels->name);
		if (i < 0) return analyserfindlabel(labels->left, label);
		if (i > 0) return analyserfindlabel(labels->right, label);
		return labels->addr ? labels : NULL;
	}
	return NULL;
}

/*
 *
 */
tlabel *analyser::find_label(char *label)
{
	return analyserfindlabel(labels, label);
}

/*
 *	called once every time the analyser has nothing more to do.
 */
void	analyser::finish()
{
	dprintf("the analyser finished (for now).\n");
	cur_func = NULL;
	addr = INVALID_ADDR;
	set_active(false);
}

/*
 *
 */
void analyser::free_addr(taddr *Addr)
{
	if (Addr) {
		// label will be freed separatly
		txref *x = Addr->xreflist;
		while (x) {
			txref *tmp = x->next;
			free(x);
			x = tmp;
		}
		free_comments(Addr);
		free(Addr);
	}
}

/*
 *
 */
void	analyser::free_addrs(taddr *addrs)
{
	if (addrs) {
		free_addrs(addrs->left);
		free_addrs(addrs->right);
		free_addr(addrs);
	}
}

/*
 *
 */
void analyser::free_comments(taddr *Addr)
{
	comment_list *c = Addr->comments;
	if (c) {
		c->destroy();
		delete c;
	}
	Addr->comments = NULL;
}

/*
 *
 */
void analyser::free_label(tlabel *label)
{
	if (label) {
		if (label->name) free(label->name);
		free(label);
	}
}


/*
 *
 */
void analyser::free_labels(tlabel *labels)
{
	if (labels) {
		free_labels(labels->left);
		free_labels(labels->right);
		free_label(labels);
	}
}

void analysergetaddrcount(taddr *addr, int *c)
{
	if (addr) {
		analysergetaddrcount(addr->left, c);
		if (!(addr->flags & AF_DELETED)) (*c)++;
		analysergetaddrcount(addr->right, c);
	}
}

/*
 *
 */
int	analyser::get_addr_count()
{
	int c=0;
	analysergetaddrcount(addrs, &c);
	return c;
}


void analysergetlabelcount(tlabel *l, int *c)
{
	if (l) {
		analysergetlabelcount(l->left, c);
		if (l->addr) (*c)++;
		analysergetlabelcount(l->right, c);
	}
}

/*
 *
 */
int	analyser::get_label_count()
{
	int c=0;
	analysergetlabelcount(labels, &c);
	return c;
}
/*
 *
 */
tlabel *analyser::get_addr_label(ADDR Addr)
{
	taddr	*a = find_addr(Addr);

	return (a) ? a->label : NULL;
}

char *analyser::get_addr_section_name(ADDR Addr)
{
	return NULL;
}

/*
 *
 */
bool	analyser::goto_addr(ADDR Addr, ADDR func)
{
	if (first_explored < last_explored) {
		dprintf("explored->add(%08lx - %08lx)\n", first_explored, last_explored);
		explored->add(first_explored, last_explored);
	}

	addr	= Addr;

	if (!valid_code_addr(addr) || explored->contains(addr)) {
		dprintf("Address: %08lx Valid: %d Explored: %d\n", addr, validcodeaddr(addr), explored->contains(addr));
		do {
			if (!pop_addr(&addr, &func)) {
				first_explored	= last_explored = INVALID_ADDR;
				addr	= INVALID_ADDR;
				return false;
			}

			dprintf("pop %08lx   (Valid: %d  Explored: %d)\n", addr, validcodeaddr(addr), explored->contains(addr));
		} while ((explored->contains(addr)) || (!valid_code_addr(addr)));
	}

	if (func!=INVALID_ADDR) {
		cur_func = new_addr(func);
		set_addr_func(cur_func, cur_func);
	}

	next_explored = explored->findnext(addr);
	first_explored	= last_explored = addr;
	return true;
}

/*
 *
 */
void analyser::init_code_analyser()
{
	code = new codeanalyser();
	code->init(this);
}

/*
 *
 */
void analyser::init_data_analyser()
{
	data = new dataanalyser();
	data->init(this);
}

/*
 *
 */
void	analyser::log(char *s)
{
	// stub
}

/*
 *
 */
CPU_ADDR analyser::map_addr(ADDR Addr)
{
	/*
	 * 	this function should map the independent address Addr to a
	 * 	processor dependent
	 * 	e.g.    .23423 --> 00234324    (or something like this)
	 *   it is only used for relativ calls/jumps
	 */
	CPU_ADDR a;
	a.addr32=Addr;
	return a;
}

taddr *analysernewaddr(taddr *&addrs, ADDR Addr)
{
	if (addrs) {
		if (Addr < addrs->addr) return analysernewaddr(addrs->left, Addr);
		if (Addr > addrs->addr) return analysernewaddr(addrs->right, Addr);
	} else {
		addrs = (taddr *) smalloc0(sizeof(taddr));
		addrs->addr = Addr;
	}
	addrs->flags &= ~AF_DELETED;
	return addrs;
}

/*
 *
 */
taddr *analyser::new_addr(ADDR Addr)
{
	if (!(cur_addr_ops--)) {
		optimize_addr_tree();
	}
	return analysernewaddr(addrs, Addr);
}

tlabel *analysernewlabel(tlabel *&labels, char *label, taddr *Addr, labeltype type)
{
	if (labels) {
		int i = strcmp(label, labels->name);
		if (i < 0) return analysernewlabel(labels->left, label, Addr, type);
		if (i > 0) return analysernewlabel(labels->right, label, Addr, type);
		if (!(labels->addr)) goto w;
	} else {
		labels = (tlabel *) smalloc0(sizeof(tlabel));
		labels->name = ht_strdup(label);
	    w:
		labels->addr = Addr;
		labels->type = type;
	}
	return labels;
}

/*
 *
 */
tlabel *analyser::new_label(char *label, taddr *Addr, labeltype type, taddr *infunc)
{
	if (!(cur_label_ops--)) {
		optimize_label_tree();
	}
	tlabel *result = analysernewlabel(labels, label, Addr, type);
	if ((result) && (result->addr==Addr) && (infunc)) set_addr_func(Addr, infunc);
	return result;
}

/*
 *
 */
void	analyser::notify_progress(ADDR Addr)
{
	// stub
}

/*
 * optimizes(=balances) addr tree
 */
void analyser::optimize_addr_tree()
{
	cur_addr_ops = addrthreshold;
	// implement me!
}

/*
 * see optimize_addr_tree()
 */
void analyser::optimize_label_tree()
{
	cur_label_ops = labelthreshold;
	// implement me!
}

/*
 *
 */
bool analyser::pop_addr(ADDR *Addr, ADDR *func)
{
	if (addr_queue->count()) {

		addrqueueitem *aqi = (addrqueueitem *) addr_queue->dequeue();
		*Addr = aqi->addr;
		*func = aqi->func;
		delete aqi;

		dprintf("addr %08lx (from sub %08lx) poped\n", *Addr, *func);
		return true;

	} else {
		dprintf("pop failed -> analyser obviously finished\n");
		return false;
	}
}

/*
 *
 */
void analyser::push_addr(ADDR Addr, ADDR func)
{
	if (valid_code_addr(Addr)) {
		dprintf("addr %08lx (from func %08lx) pushed\n", Addr, func);
		addrqueueitem *aqi = new addrqueueitem(Addr, func);
		addr_queue->enqueue(aqi);
	}
}

void saveaddrs(ht_object_stream *st, taddr *addr)
{
	if (addr) {
		saveaddrs(st, addr->left);

		if (!(addr->flags & AF_DELETED)) {
			st->put_int_hex(addr->addr, sizeof(ADDR), "addr");

			// xrefs
			txref	*xref = addr->xreflist;
			int		count = 0;
			while (xref) {
				count++;
				xref = xref->next;
			}
			st->put_int_dec(count, 4, "count_of_xrefs");
			xref = addr->xreflist;
			while (xref) {
				st->put_int_hex(xref->addr, sizeof(ADDR), "xref_addr");
				st->put_int_hex(xref->type, sizeof(txreftype), "xref_type");
				xref = xref->next;
			}
			st->put_object(addr->comments, "comments");

			analyser_put_addrtype(st, &addr->type);
			
			if (addr->thisfunc) {
				st->put_int_hex(addr->thisfunc->addr, 4, "func");
			} else {
				st->put_int_hex(INVALID_ADDR, 4, "func");
			}

			st->put_int_hex(addr->flags, 4, "flags");
		}

		saveaddrs(st, addr->right);
	}
}

void savelabels(ht_object_stream *st, tlabel *label)
{
	if (label) {
		savelabels(st, label->left);

		if (label->addr) {
			// label isn't deleted

			st->put_int_hex(label->addr->addr, sizeof(ADDR), "addr");

			st->put_string(label->name, "name");

			int a = (int)label->type;
			st->put_int_hex(a, 1, "type");
		}

		savelabels(st, label->right);
	}
}

/*
 *
 */
void analyser::store(ht_object_stream *st)
{
	PUT_OBJECT(st, addr_queue);
	PUT_INT_DEC(st, ops_parsed);
	PUT_BOOL(st, active);
	PUT_INT_HEX(st, next_explored);
	PUT_INT_HEX(st, first_explored);
	PUT_INT_HEX(st, last_explored);
	PUT_INT_HEX(st, mode);
	PUT_OBJECT(st, explored);

	st->put_info("addresses");
	st->put_int_dec(get_addr_count(), 4, "addr_count");
	saveaddrs(st, addrs);

	st->put_info("labels");
	st->put_int_dec(get_label_count(), 4, "label_count");
	savelabels(st, labels);

	PUT_OBJECT(st, analy_disasm);
	PUT_OBJECT(st, disasm);
	PUT_OBJECT(st, code);
	PUT_OBJECT(st, data);

	PUT_INT_DEC(st, addrthreshold);
	PUT_INT_DEC(st, labelthreshold);

	PUT_INT_DEC(st, maxopcodelength);
	if (cur_func) {
		st->put_int_hex(cur_func->addr, sizeof(ADDR), "cur_func");
	} else {
		st->put_int_hex(INVALID_ADDR, sizeof(ADDR), "cur_func");
	}
	dirty = false;
}

/*
 *
 */
int	analyser::query_config(int mode)
{
	// stub
	return 0;
}

/*
 *
 */
bool	analyser::savety_change_addr(ADDR Addr)
{
	addr	= Addr;
	return ((addr>=next_explored) || (!valid_code_addr(addr))) ? goto_addr(addr, INVALID_ADDR) : true;
}

/*
 *
 */
void analyser::set_active(bool mode)
{
	if (mode) {
		if (!active) {
			active = true;
			some_analyser_active++;
		}
	} else {
		if (active) {
			active = false;
			some_analyser_active--;
		}
	}
}


/*
 *
 */
void analyser::set_addr_func(taddr *a, taddr *func)
{
	if (a) {
		a->thisfunc = func;
		a->flags &= !AF_FUNCTION_END;
		a->flags |= AF_FUNCTION_SET;
	}
}

/*
 *
 */
void analyser::set_disasm(disassembler *Disasm)
{
	disasm = Disasm;
	if (disasm) {
		maxopcodelength = disasm->getmaxopcodelength();
	} else {
		maxopcodelength = 1;
	}
}

/*
 *	sets addr_threshold. after threshold addr_tree ops the tree will
 *   be optimized
 */
void analyser::set_addr_tree_optimize_threshold(int threshold)
{
	addrthreshold = threshold;
	if (cur_addr_ops > addrthreshold) cur_addr_ops = addrthreshold;
}

/*
 *	see set_addr_tree_optimize_threshold
 */
void analyser::set_label_tree_optimize_threshold(int threshold)
{
	labelthreshold = threshold;
	if (cur_label_ops > labelthreshold) cur_label_ops = labelthreshold;
}

/*
 *
 */
bool	analyser::valid_code_addr(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (a) {
		if ((a->type.type != dtcode) && (a->type.type != dtunknown)) return false;
	}
	return valid_addr(Addr, sccode);
}

/*
 *
 */
bool	analyser::valid_read_addr(ADDR Addr)
{
	return valid_addr(Addr, scread);
}

/*
 *
 */
bool	analyser::valid_write_addr(ADDR Addr)
{
	return valid_addr(Addr, scwrite);
}

/*
 *
 *		interface only (there's no internal use)
 *
 */

char bufbuf[1024];

char *analy_addr_sym_func(CPU_ADDR Addr, int *symstrlen)
{
	/* should only be called/used/assigned if theanaly is set */
	taddr *a = theanaly->find_addr(Addr.addr32);
	if ((a) && (a->label)){
		char *b=tag_make_ref(bufbuf, Addr.addr32, 0, a->label->name);
		*b=0;
		if (symstrlen) *symstrlen=b-bufbuf;
		return bufbuf;
	} else {
		if (theanaly->valid_addr(Addr.addr32, scvalid)) {
			char m[9];
			sprintf(m, "%xh", Addr.addr32);
			char *b=tag_make_ref(bufbuf, Addr.addr32, 0, m);
			*b=0;
			if (symstrlen) *symstrlen=b-bufbuf;
			return bufbuf;
		}
	}
	return 0;
}

char *analy_addr_sym_func2(CPU_ADDR Addr, int *symstrlen)
{
	/* should only be called/used/assigned if theanaly is set */
	taddr *a = theanaly->find_addr(Addr.addr32);
	if ((a) && (a->label)) {
		if (symstrlen) *symstrlen = strlen(a->label->name);
		return a->label->name;
	}
	return NULL;
}

/*
 *	should return a new instance of an apropriate assembler
 *
 */
assembler *analyser::create_assembler()
{
	return NULL;
}

/*
 *
 */
comment_list *analyser::get_comments(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (a) {
		return a->comments;
	}
	return NULL;
}

/*
 *
 */
char	*analyser::get_disasm_str(ADDR Addr)
{
	if (valid_code_addr(Addr)) {
		if (disasm) {
			addr_sym_func = 0;
			byte buf[16];
			int bz = bufptr(Addr, buf, sizeof(buf));
			OPCODE *o=disasm->decode(buf, MIN(bz, maxopcodelength), map_addr(Addr));
			return disasm->strf(o, X86DIS_STYLE_HEX_NOZEROPAD+X86DIS_STYLE_HEX_ASMSTYLE, DISASM_STRF_SMALL_FORMAT);
		} else {
			return "<no disassembler!>";
		}
	} else {
		return "";
	}
}

/*
 *
 */
char	*analyser::get_disasm_str_formatted(ADDR Addr)
{
	if (disasm) {
		theanaly = this;
		addr_sym_func = &analy_addr_sym_func2;

		byte buf[16];
		int bz = bufptr(Addr, buf, sizeof(buf));
		OPCODE *o=disasm->decode(buf, MIN(bz, maxopcodelength), map_addr(Addr));
		char *res = disasm->strf(o, X86DIS_STYLE_HEX_NOZEROPAD+X86DIS_STYLE_HEX_ASMSTYLE, DISASM_STRF_SMALL_FORMAT);

		theanaly = NULL;
		addr_sym_func = 0;

		return res;
	} else {
		return "<no disassembler!>";
	}
}

char tempbuf[1024];
/*
 *
 */
bool	analyser::get_formatted_line(ADDR Addr, int line, char *buf, int *length)
{
	taddr *a = find_addr(Addr);
	int level	= 0;

	*length = 0;

	if (line==1) {
		char temp[9];
		sprintf(temp, HEX8FORMAT8, Addr);
		buf=tag_make_sel(buf, temp);
	} else {
		buf=tag_make_sel(buf, "   ...  ");
	}

	if (explored->contains(Addr))	{
		strcpy(buf, " ! ");
	} else {
		strcpy(buf, "   ");
	}
	buf+=3;

/*
	if (mode & ANALY_EDIT_BYTES) {
		// indent labels (if desired)
		j = strlen(buf);
		for (int i=0; i<=(maxopcodelength*2); i++) {
			buf[i+j]=' ';
		}
		buf[j+(maxopcodelength*2)+1]=0;
	}
 */

	if (a) {
		// comments
		comment_list *c = a->comments;
		if (c) {
			int c1 = c->count();
			for (int i=0; i < c1; i++) {
				level++;
				if (level==line) {
					buf=tag_make_color(buf, VCP(VC_GREEN, VC_TRANSPARENT));
					strcpy(buf, c->get_name(i));
					return true;
				}               
			}
		}
		// label && xrefs
		bool collapsed_xrefs = false;
		if (mode & ANALY_COLLAPSE_XREFS) {
			// count xrefs
			int i=0;
			txref *x = a->xreflist;
			while (x && (i<21)) {
				x = x->next;
				i++;
			}
			if (i==21) {
				collapsed_xrefs = true;
			}
		}

		if ((a->label) || (a->xreflist)) {
			level ++;
			int labellength = (a->label) ? strlen(a->label->name) : 0;
			if (level==line) {
#define LABELINDENT 30
#define LABELINDENTSTR "30"
				if (labellength) {
					int l;
					buf=tag_make_color(buf, VCP(VC_LIGHT(VC_YELLOW), VC_TRANSPARENT));
					l=sprintf(buf, "%-"LABELINDENTSTR"s", a->label->name);
					buf[labellength]=':';
					buf+=l;
					*(buf+1)=0;
				} else {
					buf+=sprintf(buf, "%"LABELINDENTSTR"s", " ");
				}
				// xrefs
				if (labellength>=LABELINDENT) return true;
#define MAX_XREFS 3
				if (collapsed_xrefs) {
					*buf++ = ' ';
					buf = tag_make_ref(buf, Addr, 1, "<< show xrefs >>");
				} else {
					txref *x = a->xreflist;
					if (x) {
						buf=tag_make_color(buf, VCP(VC_GREEN, VC_TRANSPARENT));
						strcpy(buf, " ;xref");
						buf+=6;
					} else {
//						*(buf-l+labellength+1)=0;
					}
					int i=0;
					while (x)	{
						if (i++<MAX_XREFS) {
							char buf3[9];
							sprintf(buf3, HEX8FORMAT, x->addr);
							*(buf++)=' ';
							*(buf++)=xref_type_short(x->type);
							buf=tag_make_ref(buf, x->addr, 0, buf3);
						} else {
							*buf=0;
							return true;
						}
						x = x->next;
					}
				}
				*buf=0;
				return true;
			}
			// xrefsend
			if ((labellength>=LABELINDENT) && (a->xreflist)) level++;
		}
		// !! evil, worse, devil, stupid, fucking code: !!
		if (!collapsed_xrefs) {
			txref *x = a->xreflist;
			int i=0, j=0;
			bool	onetimeshit= false;
			while (x)	{
				if (level==line) {

					if (!onetimeshit) {
						onetimeshit = true;
						buf=tag_make_color(buf, VCP(VC_GREEN, VC_TRANSPARENT));
						strcpy(buf, "                               ;xref");
						buf+=strlen("                               ;xref");
					}
					char buf3[9];
					sprintf(buf3, HEX8FORMAT, x->addr);
					*(buf++)=' ';
					*(buf++)=xref_type_short(x->type);
					buf=tag_make_ref(buf, x->addr, 0, buf3);
					*buf=0;
					if ((j++)==MAX_XREFS) return true;
				}
				if ((((++i) % MAX_XREFS)==0) && (x->next)) level++;

				x = x->next;
			}
			if (onetimeshit) return true;
			// xrefsend
		} else {
			if (level==line) {
				strcpy(buf, "                               ");
				buf+=strlen("                               ");
				buf = tag_make_ref(buf, Addr, 1, "<< show xrefs >>");
				*buf = 0;
				return true;
			}
		}
		// end
	}

	level ++;
	if (level==line) {
		//
		if (a) {
			if ((a->type.type == dtcode) || ((a->type.type == dtunknown) && (valid_code_addr(Addr)))) {
				// code
			} else {
				// data
			}
		} else {
			if (valid_code_addr(Addr)) {
			} else {
			}
		}
		//
		if (valid_code_addr(Addr)) {
			taddr *nextaddr = enum_addrs(Addr);
			int op_len;

			if (nextaddr) {
				op_len = MIN((dword)maxopcodelength, (nextaddr->addr - Addr));
			} else {
				op_len = maxopcodelength;
			}

			byte buf[16];
			int bz = bufptr(Addr, buf, sizeof(buf));
			if (disasm && bz) {
				OPCODE *o=disasm->decode(buf, MIN(bz, op_len), map_addr(Addr));
				/* inits for addr-sym transformations */
				theanaly = this;
				addr_sym_func = &analy_addr_sym_func;
				/**/
				char *x=disasm->str(o, X86DIS_STYLE_HEX_NOZEROPAD+X86DIS_STYLE_HEX_ASMSTYLE);
				char *t=tempbuf;
				int xl=tag_strlen(x);
				*(t++)=' ';
				*(t++)=' ';
				memmove(t, x, xl);
				t+=xl;
				*t=0;
				/* deinits for addr-sym transformations */
				theanaly = NULL;
				addr_sym_func = 0;
				/**/
				*length=disasm->getsize(o);
			} else {
				if (bz>=1) {
					sprintf(tempbuf, "   db      %02xh", buf[0]);
				} else {
					strcpy(tempbuf, "  db      ??");
				}
				*length=1;
			}
		} else {
			if (valid_addr(Addr, scvalid)) {
				if ((a) && (a->type.type == dtint)) {
					*length = a->type.length;
					assert(*length);
					if (valid_addr(Addr, scinitialized)) {
						switch (a->type.intsubtype) {
							case dstiword: {
								word c;
								bufptr(Addr, (byte *)&c, 2);
								sprintf(tempbuf, "  dw      %04xh", c);
								break;
							}
							case dstidword: {
								dword c;
								bufptr(Addr, (byte *)&c, 4);
								sprintf(tempbuf, "  dd      %08xh", c);
								break;
							}
							case dstibyte:
							default: {
								byte c;
								if (bufptr(Addr, (byte *)&c, 1)==1) {
									sprintf(tempbuf, "  db      %02xh ; '%c'", c, (c<32)?32:c);
								} else {
									strcpy(tempbuf, " db      ?");
								}
							}
						}
					} else {
						switch (a->type.intsubtype) {
							case dstiword:
								strcpy(tempbuf, "  dw      ????");
								break;
							case dstidword:
								strcpy(tempbuf, "  dd      ????????");
								break;
							case dstibyte:
							default:
								strcpy(tempbuf, "  db      ??");
						}
					}
				} else {
					*length =	1;
					byte c;
					if (valid_addr(Addr, scinitialized) &&
					(bufptr(Addr, &c, 1)==1)) {
						sprintf(tempbuf, "  db      %02xh ; '%c'", c, (c<32)?32:c);
					} else {
						strcpy(tempbuf, "  db      ??");
					}
				}
			} else {
				ADDR next = next_valid(Addr);
				if (next != INVALID_ADDR) {
					*length = next - Addr;
					sprintf(tempbuf, "  db      ?? * %d", next - Addr);
				} else {
					*length =	1;
					strcpy(tempbuf, "  db      ??");
				}
			}
		}
		if (mode & ANALY_EDIT_BYTES) {
			if (valid_addr(Addr, scinitialized)) {

				FILEADDR a=file_addr(Addr);
				assert(a != INVALID_FILE_OFS);
				for (int i=0; i<(*length); i++) {
					buf = tag_make_edit_byte(buf, a+i);
				}
				for (int i=0; i<=(maxopcodelength*2)-(*length)*2; i++) {
					*(buf++)=' ';
				}
				*buf=0;
			}
		}
		int l=tag_strlen(tempbuf);
		memmove(buf, tempbuf, l);
		buf+=l;
		*buf=0;
		return true;
	}
	*buf=0;
	return false;
}

bool analyser::get_formatted_line_length(ADDR Addr, int line, int *length)
{
	taddr *a = find_addr(Addr);
	int level	= 0;

	*length = 0;

	if (a) {
		// comments
		comment_list *c = a->comments;
		if (c) {
			int c1 = c->count();
			for (int i=0; i < c1; i++) {
				level++;
				if (level==line) return true;
			}
		}
		// label && xrefs
		bool collapsed_xrefs = false;
		if (mode & ANALY_COLLAPSE_XREFS) {
			// count xrefs
			int i=0;
			txref *x = a->xreflist;
			while (x && (i<21)) {
				x = x->next;
				i++;
			}
			if (i==21) {
				collapsed_xrefs = true;
			}
		}
		if (a->label || a->xreflist) {
			level ++;
			int labellength = (a->label) ? strlen(a->label->name) : 0;
			if (level==line) {
				if (labellength>=LABELINDENT) return true;
				// xrefs
				txref *x = a->xreflist;
				int i=0;
				while (x)	{
					if (i++<MAX_XREFS) {
					} else return true;
					x = x->next;
				}
				return true;
				// xrefsend
			}
			if ((labellength>=LABELINDENT) && (a->xreflist)) level++;
		}
		// !! evil, worse, devil, stupid, fucking code: !!
		if (!collapsed_xrefs) {
			txref *x = a->xreflist;
			int i=0,j=0;
			bool	onetimeshit= false;
			while (x)	{
				if (level==line) {
					onetimeshit = true;
					if ((j++)==MAX_XREFS) return true;
				}
				if ((((++i) % MAX_XREFS)==0) && (x->next)) level ++;
				x = x->next;
			}
			if (onetimeshit) return true;
			// xrefsend
		} else {
			if (level==line) return true;
		}
		// end
	}

	level ++;
	if (level==line) {
		if (valid_code_addr(Addr))	{
			taddr *nextaddr = enum_addrs(Addr);
			int op_len;
			if (nextaddr) {
				op_len = MIN((dword)maxopcodelength, (nextaddr->addr - Addr));
			} else {
				op_len = maxopcodelength;
			}
			if (disasm) {
				byte buf[16];
				int bz = bufptr(Addr, buf, 16);
				OPCODE *o=disasm->decode(buf, MIN(bz, op_len), map_addr(Addr));
				*length=disasm->getsize(o);
			} else {
				*length=1;
			}
		} else {
			if (valid_addr(Addr, scvalid)) {
				if ((a) && (a->type.type == dtint)) {
					*length = a->type.length;
					assert(*length);
				} else {
					*length =	1;
				}
			} else {
				ADDR next = next_valid(Addr);
				if (next != INVALID_ADDR) {
					*length = next - Addr;
				} else {
					*length =	1;
				}
			}
		}
		return true;
	}

	return false;
}

/*
 *
 */
char *analyser::get_name()
{
	return "generic";
}

/*
 *
 */
txref *analyser::get_xrefs(ADDR Addr)
{
	taddr *a = find_addr(Addr);
	if (!a) return NULL;
	return a->xreflist;
}

/*
 *
 */
bool analyser::is_dirty()
{
	return dirty;
}

/*
 *
 */
void analyser::make_dirty()
{
	dirty = true;
}


/*
 *
 */
void analyser::set_display_mode(int enable, int disable)
{
	mode |= enable;
	mode &= ~disable;
}

/*
 *
 */
void analyser::toggle_display_mode(int toggle)
{
	mode ^= toggle;
}

/*
 *	converts FILEADDR fileaddr to ADDR
 */
ADDR analyser::vaddr(FILEADDR fileaddr)
{
	// abstract / stub
	return INVALID_ADDR;
}

/****************************************************************************/

/*
 *
 */
void analy_disassembler::init(analyser *A)
{
	analy = A;
	disasm = NULL;
	init_disasm();
}

/*
 *
 */
int  analy_disassembler::load(ht_object_stream *st)
{
	return st->get_error();
}

/*
 *
 */
void analy_disassembler::done()
{
}

/*
 *
 */
void analy_disassembler::init_disasm()
{
	if (analy) {
		analy->set_disasm(disasm);
	}
}

/*
 *
 */
void analy_disassembler::store(ht_object_stream *f)
{
}

