/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: gstring.h,v 1.2.24.1 2004/07/09 01:54:36 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#ifndef _GSTRING_H
#define _GSTRING_H

/*******************************************************************************
 * Defines
 */
#define GSTRING_REFMODE	    TRUE
#define GSTRING_VALMODE	    FALSE

/*******************************************************************************
 * Includes
 */
#include <string.h>
#include "hxtypes.h"
#include "hxassert.h"
#include "hxstrutl.h"

/*******************************************************************************
 * Class definition
 */
class GString
{
protected:
    enum {kDefaultCapacity = 32,
	  kMinimumCapacity = 32,
	  kMaximumCapacity = 65536};
    
    char    *m_pszBuffer;
    ULONG32  m_ulCapacity;
    ULONG32  m_ulLength;
    LONG32   m_lError;
    BOOL     m_bRefMode;
    
    ULONG32 NextPowerOfTwo(ULONG32 ulX)
    {
        if (ulX > 0)
        {
            ulX--;
        }
        else
        {
            return 1;
        }
        ULONG32 i = 0;
        while (ulX > 0)
        {
            ulX >>= 1;
            i++;
        }
        return (1 << i);
    };

public:
    enum {ErrorMemoryFull = -1,
	  ErrorNone       = 0};

    GString() : m_pszBuffer(0L), m_ulCapacity(0), m_ulLength(0), m_lError(ErrorNone), m_bRefMode(FALSE) {};
    
    GString(ULONG32 ulCapacity)
	: m_bRefMode(FALSE)
    {
        if (ulCapacity < kMinimumCapacity)
        {
            ulCapacity = kMinimumCapacity;
        }
        else if (ulCapacity > kMaximumCapacity)
        {
            ulCapacity = kMaximumCapacity;
        }
        m_ulCapacity = m_ulLength = 0;
        m_lError     = ErrorNone;
        m_pszBuffer  = 0L;
        if (ulCapacity > 0)
        {
            m_pszBuffer  = new char [ulCapacity];
            if (m_pszBuffer)
            {
                m_ulCapacity = ulCapacity;
            }
            else
            {
                m_lError = ErrorMemoryFull;
            }
        }
    };
    
    GString(const GString & rString)
    {
        m_lError     = ErrorNone;
	
	m_bRefMode = rString.m_bRefMode;

	if (m_bRefMode)
	{
	    m_pszBuffer = rString.m_pszBuffer;
	    m_ulCapacity = 0;
	    m_ulLength = rString.length();
	    m_bRefMode = TRUE;
	}
	else
	{
	    m_pszBuffer  = new char [rString.capacity()];
	    if (m_pszBuffer)
	    {
		strcpy(m_pszBuffer, rString.m_pszBuffer); /* Flawfinder: ignore */
		m_ulCapacity = rString.capacity();
		m_ulLength = rString.length();
	    }
	    else
	    {
		m_ulCapacity = 0;
		m_ulLength = 0;
		m_lError = ErrorMemoryFull;
	    }
	}
    }
    
    GString(const char *pszString)
	: m_bRefMode(FALSE)
    {
        m_ulCapacity = m_ulLength = 0;
        m_lError     = ErrorNone;
        m_pszBuffer  = 0L;
	
        ULONG32 ulCapacity = NextPowerOfTwo(strlen(pszString) + 1);
        if (ulCapacity < kMinimumCapacity)
        {
            ulCapacity = kMinimumCapacity;
        }
        else if (ulCapacity > kMaximumCapacity)
        {
            ulCapacity = kMaximumCapacity;
        }
	
        m_pszBuffer = new char [ulCapacity];
        if (m_pszBuffer)
        {
            strcpy(m_pszBuffer, pszString); /* Flawfinder: ignore */
            m_ulCapacity = ulCapacity;
            m_ulLength   = strlen(pszString);
        }
        else
        {
            m_lError = ErrorMemoryFull;
        }
    };
    
    GString(const char *pszString, LONG32 lLen, BOOL bRefMode = FALSE)
	: m_bRefMode(bRefMode)
    {
	LONG32 strLen;
	char* tracer;
	m_ulCapacity = 0;
	m_lError = ErrorNone;
	
	if (m_bRefMode)
	{
	    m_pszBuffer = (char *) pszString;
	}
	else
	{
	    ULONG32 ulCapacity = NextPowerOfTwo(lLen + 1);

	    m_pszBuffer = 0L;
	    m_ulLength = 0;

	    if (ulCapacity < kMinimumCapacity)
	    {
		ulCapacity = kMinimumCapacity;
	    }
	    else if (ulCapacity > kMaximumCapacity)
	    {
		ulCapacity = kMaximumCapacity;
	    }
	
	    m_pszBuffer = new char [ulCapacity];
	    if (m_pszBuffer)
	    {
		strncpy(m_pszBuffer, pszString, lLen); /* Flawfinder: ignore */
		m_pszBuffer[lLen] = '\0';
		m_ulCapacity = ulCapacity;
	    }
	    else
	    {
		m_lError = ErrorMemoryFull;
		return;
	    }
	}

	for (strLen = 0, tracer = m_pszBuffer; strLen < lLen; strLen++, tracer++)
	{
	    if (*tracer == '\0')
	    {
		lLen = strLen;
	    }
	}

	m_ulLength = lLen;
    };

    ~GString()
    {
        if ((!m_bRefMode) && m_pszBuffer)
        {
            delete [] m_pszBuffer;
        }
    };

    ULONG32 length()       const { return m_ulLength;   };
    ULONG32 size()         const { return m_ulLength;   };
    ULONG32 capacity()     const { return m_ulCapacity; };
    ULONG32 max_capacity() const { return kMaximumCapacity; }
    BOOL    empty()        const { return (m_ulLength == 0 ? TRUE : FALSE); };
    LONG32  error()        const { return m_lError; };
    LONG32  isref()        const { return m_bRefMode; }; 
    
    // c_str() const method is supported for bakward compatibility 
    // with the code that expects this method to be const.
    // The const method does not support reference mode (m_bRefMode).
    const char *c_str()    const
    {
	HX_ASSERT(!m_bRefMode);

	if (m_bRefMode)
	{
	    return NULL;
	}
	
	return m_pszBuffer;
    }

    const char *c_str()
    {
	if (m_bRefMode)
	{
	    m_lError = reserve(m_ulLength + 1);

	    if (m_lError != ErrorNone)
	    {
		m_pszBuffer = NULL;
		m_ulCapacity = 0;
		m_ulLength = 0;
		m_lError = ErrorMemoryFull;
	    }
	}
	
	return m_pszBuffer;
    }

    LONG32 reserve(ULONG32 ulN)
    {
        if (ulN > m_ulCapacity)
        {
            ULONG32 ulCapacity = NextPowerOfTwo(ulN);
            if (ulCapacity < kMinimumCapacity)
            {
                ulCapacity = kMinimumCapacity;
            }
            else if (ulCapacity > kMaximumCapacity)
            {
                ulCapacity = kMaximumCapacity;
            }

            char *pszTmp = 0L;
            pszTmp       = new char [ulCapacity];
            if (!pszTmp)
            {
                return ErrorMemoryFull;
            }
            if (m_ulLength > ulCapacity)
            {
                m_ulLength = ulCapacity - 1;
            }
            if (m_ulLength > 0)
            {
                strncpy(pszTmp, m_pszBuffer, m_ulLength); /* Flawfinder: ignore */
		pszTmp[m_ulLength] = '\0';
            }

	    if (!m_bRefMode)
	    {
		delete [] m_pszBuffer;
	    }
            m_pszBuffer  = pszTmp;
            m_ulCapacity = ulCapacity;
	    m_bRefMode = FALSE;
        }

        return ErrorNone;
    };

    GString & operator = (const GString &rString)
    {
	if (rString.m_bRefMode)
	{
	    if (!m_bRefMode)
	    {
		delete [] m_pszBuffer;
	    }

	    m_pszBuffer = rString.m_pszBuffer;
	    m_ulLength = rString.m_ulLength;
	    m_ulCapacity = rString.m_ulCapacity;
	    m_bRefMode = TRUE;
	}
	else
	{
	    m_lError = reserve(rString.m_ulLength + 1);
	    if (m_lError == ErrorNone)
	    {
		strcpy(m_pszBuffer, rString.m_pszBuffer); /* Flawfinder: ignore */
		m_ulLength = rString.m_ulLength;
	    }
	}

        return *this;
    };

    GString & operator = (const char *pszString)
    {
        if (pszString == NULL)
        {
            return *this;
        }
        m_lError = reserve(strlen(pszString) + 1);
        if (m_lError == ErrorNone)
        {
            strcpy(m_pszBuffer, pszString); /* Flawfinder: ignore */
            m_ulLength = strlen(pszString);
        }

        return *this;
    };

    GString & operator += (const GString &rString)
    {
        m_lError = reserve(m_ulLength + rString.m_ulLength + 1);
        if (m_lError == ErrorNone)
        {
            strncat(m_pszBuffer, rString.m_pszBuffer, rString.m_ulLength); /* Flawfinder: ignore */
            m_ulLength += rString.m_ulLength;
        }

        return *this;
    };

    GString & operator += (const char *pszString)
    {
        m_lError = reserve(m_ulLength + strlen(pszString) + 1);
        if (m_lError == ErrorNone)
        {
            strcat(m_pszBuffer, pszString); /* Flawfinder: ignore */
            m_ulLength += strlen(pszString);
        }

        return *this;
    };

    GString operator + (const GString &rString)
    {
        GString cTmp(*this);
        cTmp += rString;
        return cTmp;
    };

    GString operator + (const char *pszString)
    {
        GString cTmp(*this);
        cTmp += pszString;
        return cTmp;
    };

    BOOL operator == (const GString &rString)
    {
        return (((rString.m_ulLength != m_ulLength) ||
	         strncmp(m_pszBuffer, rString.m_pszBuffer, m_ulLength)) ? FALSE : TRUE);
    };

    BOOL operator == (const char *pszString)
    {
        return (((m_ulLength != strlen(pszString)) ||
	         strncmp(m_pszBuffer, pszString, m_ulLength)) ? FALSE : TRUE);
    };

    BOOL operator != (const GString &rString)
    {
        return (((m_ulLength != rString.m_ulLength) ||
	         strncmp(m_pszBuffer, rString.m_pszBuffer, m_ulLength)) ? TRUE : FALSE);
    };

    BOOL operator != (const char *pszString)
    {
        return (((m_ulLength != strlen(pszString)) ||
	         strncmp(m_pszBuffer, pszString, m_ulLength)) ? TRUE : FALSE);
    };

    const char & operator [] (LONG32 lPos) const
    {
        if (lPos < 0)
        {
            lPos = 0;
        }
        else if ((ULONG32) lPos > m_ulLength)
        {
            lPos = m_ulLength;
        }
        return m_pszBuffer[lPos];
    };

    LONG32 find(const char *pszString, LONG32 lPos = 0) const
    {
	char *pSubStr;

	if (m_bRefMode)
	{
	    pSubStr = StrNStr(&m_pszBuffer[lPos], pszString, m_ulLength - lPos, kMaximumCapacity);
	}
	else
	{
	    pSubStr = strstr(&m_pszBuffer[lPos], pszString);
	}

        LONG32 lRetVal;
        if (pSubStr)
        {
            lRetVal = pSubStr - m_pszBuffer;
        }
        else
        {
            lRetVal = -1;
        }

        return lRetVal;
    };

    LONG32 find(const GString &rString, LONG32 lPos = 0) const
    {
	char *pSubStr;

	if (m_bRefMode || rString.m_bRefMode)
	{
	    pSubStr = StrNStr(&m_pszBuffer[lPos], rString.m_pszBuffer, m_ulLength - lPos, rString.m_ulLength);
	}
	else
	{
	    pSubStr = strstr(&m_pszBuffer[lPos], rString.m_pszBuffer);
	}

        LONG32 lRetVal;
        if (pSubStr)
        {
            lRetVal = pSubStr - m_pszBuffer;
        }
        else
        {
            lRetVal = -1;
        }

        return lRetVal;
    };

    LONG32 find(char c, LONG32 lPos = 0) const
    {
        char *pSubStr;
	
	if (m_bRefMode)
	{
	    pSubStr = StrNChr(&m_pszBuffer[lPos], c, m_ulLength - lPos);
	}
	else
	{
	    pSubStr = strchr(&m_pszBuffer[lPos], c);
	}

        LONG32 lRetVal;
        if (pSubStr)
        {
            lRetVal = pSubStr - m_pszBuffer;
        }
        else
        {
            lRetVal = -1;
        }

        return lRetVal;
    };

    LONG32 find_last_of(char c) const
    {
	char* pSubStr;
	
	if (m_bRefMode)
	{
	    pSubStr = StrNRChr(m_pszBuffer, c, m_ulLength);
	}
	else
	{
	    pSubStr = strrchr(m_pszBuffer, c);
	}

	LONG32 lRet;
        if (pSubStr)
	{
	    lRet = pSubStr - m_pszBuffer;
	}
	else
	{
	    lRet = -1;
	}

        return lRet;
    };

    LONG32 find_first_of(const char *pszString, LONG32 lPos = 0) const
    {
	LONG32 lRet;

	if (m_bRefMode)
	{
	    lRet = StrNCSpn(&m_pszBuffer[lPos], pszString, m_ulLength - lPos, kMaximumCapacity);
	}
	else
	{
	    lRet = strcspn(&m_pszBuffer[lPos], pszString);
	}

        return lRet + lPos;
    };

    LONG32 find_first_of(const GString &rString, LONG32 lPos = 0) const
    {
	LONG32 lRet;

	if (m_bRefMode || rString.m_bRefMode)
	{
	    lRet = StrNCSpn(&m_pszBuffer[lPos], rString.m_pszBuffer, m_ulLength - lPos, rString.m_ulLength);
	}
	else
	{
	    lRet = strcspn(&m_pszBuffer[lPos], rString.m_pszBuffer);
	}

        return lRet + lPos;
    };

    LONG32 find_first_of(char c, LONG32 lPos = 0) const
    {
        return find(c, lPos);
    };

    LONG32 find_first_not_of(const char *pszString, LONG32 lPos = 0) const
    {
        LONG32 lRet;
	
	if (m_bRefMode)
	{
	    lRet = StrNSpn(&m_pszBuffer[lPos], pszString, m_ulLength - lPos, kMaximumCapacity);
	}
	else
	{
	    lRet = strspn(&m_pszBuffer[lPos], pszString);
	}

        return lRet + lPos;
    };

    LONG32 find_first_not_of(const GString &rString, LONG32 lPos = 0) const
    {
	LONG32 lRet;
	
	if (m_bRefMode || rString.m_bRefMode)
	{
	    lRet = StrNSpn(&m_pszBuffer[lPos], rString.m_pszBuffer, m_ulLength - lPos, rString.m_ulLength);
	}
	else
	{
	    lRet = strspn(&m_pszBuffer[lPos], rString.m_pszBuffer);
	}

        return lRet + lPos;
    };

    GString substr(LONG32 lPos, LONG32 rPos) const
    {
        if (lPos < 0) lPos = 0;
        if ((ULONG32) lPos >= m_ulLength) lPos = m_ulLength - 1;
        if (rPos < 0) rPos = 0;
        if ((ULONG32) rPos >= m_ulLength) rPos = m_ulLength - 1;
        if (rPos < lPos) rPos = lPos;

        GString retStr(&m_pszBuffer[lPos], rPos - lPos + 1, m_bRefMode);

        return retStr;
    };

    void CopyN(unsigned char *pBuffer, ULONG32 ulN)
    {
        m_lError = reserve(ulN + 1);
        if (m_lError == ErrorNone)
        {
            strncpy(m_pszBuffer, (const char *) pBuffer, ulN); /* Flawfinder: ignore */
            m_pszBuffer[ulN] = '\0';
            m_ulLength = strlen(m_pszBuffer);
        }
    };

    void LowerCase()
    {
        for (UINT32 i = 0; i < m_ulLength; i++)
        {
            char c = m_pszBuffer[i];

            // If character is uppercase ASCII, convert to lowercase
            if (c >= 65 && c <= 90)
            {
                c             += 32;
                m_pszBuffer[i] = c;
            }
        }
    };

    void UpperCase()
    {
        for (UINT32 i = 0; i < m_ulLength; i++)
        {
            char c = m_pszBuffer[i];

            // If character is lowercase ASCII, convert to uppercase
            if (c >= 97 && c <= 122)
            {
                c             -= 32;
                m_pszBuffer[i] = c;
            }
        }
    };
};

#endif  // _GSTRING_H

