<?php
/*
 * $RCSfile: module.inc,v $
 *
 * Gallery - a web based photo album viewer and editor
 * Copyright (C) 2000-2006 Bharat Mediratta
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/**
 * @version $Revision: 1.7.2.1 $ $Date: 2006/04/05 22:10:59 $
 * @package RSS
 * @author Jonatan Heyman <http://heyman.info>
 * @author Pierre-Luc Paour
 * @author Daniel Grund <http://www.photogrund.nl>
 */

/**
 * Implementation of the RSS module
 *
 * @package RSS
 */
class RssModule extends GalleryModule /* and GalleryEventListener */ {

    function RssModule() {
	global $gallery;
	$this->setId('rss');
	$this->setName($gallery->i18n('RSS'));
	$this->setDescription($gallery->i18n('RSS'));
	$this->setVersion('1.0.6');
	$this->setGroup('export', $gallery->i18n('Export'));
	$this->setCallbacks('getSiteAdminViews|getItemLinks|registerEventListeners');
	$this->setRequiredCoreApi(array(7, 0));
	$this->setRequiredModuleApi(array(3, 0));
    }

    /**
     * @see GalleryModule::getSiteAdminViews
     */
    function getSiteAdminViews() {
	return array(null, array(array('name' => $this->translate('RSS'),
				       'view' => 'rss.RssSiteAdmin')));
    }

    /**
     * @see GalleryModule::getItemLinks()
     */
    function getItemLinks($items, $wantsDetailedLinks, $permissions) {
	GalleryCoreApi::requireOnce('modules/rss/classes/RssMapHelper.class');
	global $gallery;
	$links = array();

	/* check that this type of link is allowed */
	list ($ret, $param) = GalleryCoreApi::fetchAllPluginParameters('module', 'rss');
	if ($ret) {
	    return array($ret->wrap(__FILE__, __LINE__), null);
	}

	foreach ($items as $item) {
	    $itemId = $item->getId();
	    if (isset($wantsDetailedLinks[$itemId])) {
		if ($param['allowSimpleFeed'] == 1) {
		    $itemTypeNames = array_merge($item->itemTypeName(), $item->itemTypeName(false));

		    /* 
		     * Specific translations: 
		     * _('RSS Feed for this Album') 
		     * _('RSS Feed for this Photo') 
		     * _('RSS Feed for this Movie') 
		     */
		    $links[$itemId][] = array(
			'text' => $this->_translate(array(
			    'text' => 'RSS Feed for this %s',
			    'arg1' => $itemTypeNames[0]), $itemTypeNames[2]),
			'params' => array(
			    'view' => 'rss.SimpleRender',
			    'itemId' => $itemId));
		}

		list ($ret, $canConfigure) = RssMapHelper::canConfigureFeed($item, $param);
		if ($ret) {
		    return array($ret->wrap(__FILE__, __LINE__), null);
		}

		if ($canConfigure) {
		    $links[$itemId][] = array(
			    'text' => $this->translate('Configure RSS Feeds'),
			    'params' => array(
				'view' => 'core.ItemAdmin',
				'subView' => 'rss.EditFeed',
				'itemId' => $itemId,
				'return' => 1));
		}
	    }
	}

	return array(null, $links);
    }

    /**
     * @see GalleryModule::getRewriteRules
     */
    function getRewriteRules() {
	$rules = array();
	$rules[] = array(
	    'comment' => $this->translate('Simple RSS Feed'),
	    'match' => array('view' => 'rss.SimpleRender'),
	    'pattern' => 'srss/%itemId%',
	    'help' => $this->translate('Short URL for Simple RSS Feeds'));

	$rules[] = array(
	    'comment' => $this->translate('RSS Feed'),
	    'match' => array('view' => 'rss.Render'),
	    'pattern' => 'rss/%name%',
	    'keywords' => array(
		'name' => array(
		    'pattern' => '([^\/\?]+)',
		    'help' => $this->translate('The name of the configured RSS Feed'))),
	    'help' => $this->translate('Short URL for Configurable RSS Feeds'));

	return $rules;
    }

    /**
     * @see GalleryModule::registerEventListeners();
     */
    function registerEventListeners() {
	GalleryCoreApi::registerEventListener('GalleryEntity::delete', new RssModule());
    }

    /**
     * Event Handler: get rid of any feeds for items that are deleted
     *
     * @see GalleryEventListener::handleEvent
     */
    function handleEvent($event) {
	$data = $event->getData();

	if ($event->getEventName() == 'GalleryEntity::delete') {
	    $entity = $event->getEntity();
	    if (GalleryUtilities::isA($entity, 'GalleryItem')) {
		GalleryCoreApi::requireOnce('modules/rss/classes/RssMapHelper.class');
		list ($ret, $results) = RssMapHelper::fetchFeedsForItem($entity->getId());
		if ($ret) {
		    return array($ret->wrap(__FILE__, __LINE__), null);
		}

		foreach ($results as $feed) {
		    $ret = RssMapHelper::deleteFeed($feed['name']);
		    if ($ret) {
			return array($ret->wrap(__FILE__, __LINE__), null);
		    }
		}
	    }
	}

	return array(null, null);
    }

    /**
     * @see GalleryModule::upgrade()
     */
    function upgrade($currentVersion) {
	switch($currentVersion) {
	    case null:
		/* Initial install */

		foreach (array('defaultVersion' => '2.0',
			       'defaultCount' => 10,
			       'maxCount' => 50,
			       'allowSimpleFeed' => 1,
			       'allowConfigurableFeed' => 0,
			       'allowAlbums' => 0,
			       'allowPhotos' => 0,
			       'allowPhotosRecurse' => 0,
			       'allowCommentsAlbum' => 0,
			       'allowCommentsPhoto' => 0,
			       'defaultCopyright' => '',
			       'defaultTtl' => 120,
			    ) as $key => $value) {
		    $ret = $this->setParameter($key, $value);
		    if ($ret) {
			return $ret->wrap(__FILE__, __LINE__);
		    }
		}
		/* keep going */

	    case '1.0.0':
		foreach (array('sfAlbumType' => 'photos',
			       'sfDate' => 'new',
			       'sfPhotosRecurseLimit' => '10',
			    ) as $key => $value) {
		    $ret = $this->setParameter($key, $value);
		    if ($ret) {
			return $ret->wrap(__FILE__, __LINE__);
		    }
		}
	}

	return null;
    }
}
?>
