// Copyright 2013 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "src/platform/mutex.h"

#include <errno.h>

namespace v8 {
namespace internal {

#if V8_OS_POSIX

static V8_INLINE void InitializeNativeHandle(pthread_mutex_t* mutex) {
  int result;
#if defined(DEBUG)
  // Use an error checking mutex in debug mode.
  pthread_mutexattr_t attr;
  result = pthread_mutexattr_init(&attr);
  ASSERT_EQ(0, result);
  result = pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK);
  ASSERT_EQ(0, result);
  result = pthread_mutex_init(mutex, &attr);
  ASSERT_EQ(0, result);
  result = pthread_mutexattr_destroy(&attr);
#else
  // Use a fast mutex (default attributes).
  result = pthread_mutex_init(mutex, NULL);
#endif  // defined(DEBUG)
  ASSERT_EQ(0, result);
  USE(result);
}


static V8_INLINE void InitializeRecursiveNativeHandle(pthread_mutex_t* mutex) {
  pthread_mutexattr_t attr;
  int result = pthread_mutexattr_init(&attr);
  ASSERT_EQ(0, result);
  result = pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
  ASSERT_EQ(0, result);
  result = pthread_mutex_init(mutex, &attr);
  ASSERT_EQ(0, result);
  result = pthread_mutexattr_destroy(&attr);
  ASSERT_EQ(0, result);
  USE(result);
}


static V8_INLINE void DestroyNativeHandle(pthread_mutex_t* mutex) {
  int result = pthread_mutex_destroy(mutex);
  ASSERT_EQ(0, result);
  USE(result);
}


static V8_INLINE void LockNativeHandle(pthread_mutex_t* mutex) {
  int result = pthread_mutex_lock(mutex);
  ASSERT_EQ(0, result);
  USE(result);
}


static V8_INLINE void UnlockNativeHandle(pthread_mutex_t* mutex) {
  int result = pthread_mutex_unlock(mutex);
  ASSERT_EQ(0, result);
  USE(result);
}


static V8_INLINE bool TryLockNativeHandle(pthread_mutex_t* mutex) {
  int result = pthread_mutex_trylock(mutex);
  if (result == EBUSY) {
    return false;
  }
  ASSERT_EQ(0, result);
  return true;
}

#elif V8_OS_WIN

static V8_INLINE void InitializeNativeHandle(PCRITICAL_SECTION cs) {
  InitializeCriticalSection(cs);
}


static V8_INLINE void InitializeRecursiveNativeHandle(PCRITICAL_SECTION cs) {
  InitializeCriticalSection(cs);
}


static V8_INLINE void DestroyNativeHandle(PCRITICAL_SECTION cs) {
  DeleteCriticalSection(cs);
}


static V8_INLINE void LockNativeHandle(PCRITICAL_SECTION cs) {
  EnterCriticalSection(cs);
}


static V8_INLINE void UnlockNativeHandle(PCRITICAL_SECTION cs) {
  LeaveCriticalSection(cs);
}


static V8_INLINE bool TryLockNativeHandle(PCRITICAL_SECTION cs) {
  return TryEnterCriticalSection(cs);
}

#endif  // V8_OS_POSIX


Mutex::Mutex() {
  InitializeNativeHandle(&native_handle_);
#ifdef DEBUG
  level_ = 0;
#endif
}


Mutex::~Mutex() {
  DestroyNativeHandle(&native_handle_);
  ASSERT_EQ(0, level_);
}


void Mutex::Lock() {
  LockNativeHandle(&native_handle_);
  AssertUnheldAndMark();
}


void Mutex::Unlock() {
  AssertHeldAndUnmark();
  UnlockNativeHandle(&native_handle_);
}


bool Mutex::TryLock() {
  if (!TryLockNativeHandle(&native_handle_)) {
    return false;
  }
  AssertUnheldAndMark();
  return true;
}


RecursiveMutex::RecursiveMutex() {
  InitializeRecursiveNativeHandle(&native_handle_);
#ifdef DEBUG
  level_ = 0;
#endif
}


RecursiveMutex::~RecursiveMutex() {
  DestroyNativeHandle(&native_handle_);
  ASSERT_EQ(0, level_);
}


void RecursiveMutex::Lock() {
  LockNativeHandle(&native_handle_);
#ifdef DEBUG
  ASSERT_LE(0, level_);
  level_++;
#endif
}


void RecursiveMutex::Unlock() {
#ifdef DEBUG
  ASSERT_LT(0, level_);
  level_--;
#endif
  UnlockNativeHandle(&native_handle_);
}


bool RecursiveMutex::TryLock() {
  if (!TryLockNativeHandle(&native_handle_)) {
    return false;
  }
#ifdef DEBUG
  ASSERT_LE(0, level_);
  level_++;
#endif
  return true;
}

} }  // namespace v8::internal
