#!/usr/bin/python

# Copyright 2014 Google Inc.
#
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Function for generating the SkUserConfig file, customized for Android."""

import os
import shutil


AUTOGEN_WARNING = (
"""
///////////////////////////////////////////////////////////////////////////////
//
// THIS FILE IS AUTOGENERATED BY GYP_TO_ANDROID.PY. DO NOT EDIT.
//
// This file contains Skia's upstream include/config/SkUserConfig.h as a
// reference, followed by the actual defines set for Android.
//
///////////////////////////////////////////////////////////////////////////////

"""
)

BUILD_GUARD = 'SkUserConfig_Android_DEFINED'


def generate_user_config(original_sk_user_config, require_sk_user_config,
                         target_dir, ordered_set):
  """Generate the SkUserConfig file specific to the Android framework.

  Android needs its #defines in its skia/include/core directory, so that other
  libraries which use Skia's headers get the right definitions. This function
  takes the existing sample version of SkUserConfig, checked into Skia, and
  appends the defines from ordered_set, which is expected to be a
  vars_dict_lib.OrderedSet containing the defines. The result is written to
  target_dir/SkUserConfig.h

  Args:
      original_sk_user_config: Path to original SkUserConfig.h
      require_sk_user_config: If True, raise an AssertionError if
          SkUserConfig.h does not exist. Either way, if it does exist, copy it
          into the new file.
      target_dir: Directory within which the modified SkUserConfig.h will be
          written. Its name will be the same basename as
          original_sk_user_config. If None, the new file will be written to the
          working directory.
      ordered_set: A vars_dict_lib.OrderedSet, containing a list of defines to
          be appended to SkUserConfig.

  Raises:
      AssertionError: If original_sk_user_config does not exist.
  """

  sk_user_config_exists = os.path.exists(original_sk_user_config)
  if require_sk_user_config:
    assert sk_user_config_exists

  dst_filename = os.path.basename(original_sk_user_config)
  if target_dir:
    dst_filename = os.path.join(target_dir, dst_filename)

  with open(dst_filename, 'w') as dst:
    dst.write(AUTOGEN_WARNING)

    # Copy the original exactly. This is merely for reference. Many of the
    # defines written to the file below, either manually or generated from the
    # gyp files, have explanations in the original SkUserConfig.h
    if sk_user_config_exists:
      with open(original_sk_user_config, 'r') as original:
        shutil.copyfileobj(original, dst)

    # Now add the defines specific to Android. Write a custom build guard to
    # ensure they don't get defined more than once.
    dst.write('\n// Android defines:\n')
    dst.write('#ifndef ' + BUILD_GUARD + '\n')
    dst.write('#define ' + BUILD_GUARD + '\n')

    # Add conditional defines manually:

    # do this build check for other tools that still read this header
    dst.write('#ifdef ANDROID\n')
    dst.write('    #include <utils/misc.h>\n')
    dst.write('#endif\n\n')

    dst.write('#if __BYTE_ORDER == __BIG_ENDIAN\n')
    dst.write('    #define SK_CPU_BENDIAN\n')
    dst.write('    #undef  SK_CPU_LENDIAN\n')
    dst.write('#else\n')
    dst.write('    #define SK_CPU_LENDIAN\n')
    dst.write('    #undef  SK_CPU_BENDIAN\n')
    dst.write('#endif\n\n')

    # Now add the defines from the gyp files.
    for item in ordered_set:
      # Although our defines may have '=' in them, when written to the header
      # there should be a space between the macro and what it replaces.
      dst.write('#define ' + item.replace('=', ' ') + '\n')

    dst.write('\n#endif // ' + BUILD_GUARD + '\n')
