/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#define SYSLOG_NAMES
#include "vpnd.h"

/*============================================================================*/
/* the big league: config parser                                              */
/*============================================================================*/

/* configuration data structure */

typedef struct
{
	char *item;	/* command		*/
	char option;	/* 1 if optional	*/
	char min;	/* minimum parameters	*/
	char max;	/* maximum parameters	*/
} CONFIG;

/* configuration data array, keep sequence! */

static CONFIG config[]=
{
{ "mode",      0,1,1 },
{ "local",     0,1,1 },
{ "remote",    0,1,1 },
{ "server",    0,1,2 },
{ "client",    0,1,2 },
{ "keepalive", 1,0,1 },
{ "autoroute", 1,0,0 },
{ "keyfile",   1,1,1 },
{ "route1",    1,3,3 },
{ "route2",    1,3,3 },
{ "route3",    1,3,3 },
{ "route4",    1,3,3 },
{ "route5",    1,3,3 },
{ "route6",    1,3,3 },
{ "route7",    1,3,3 },
{ "route8",    1,3,3 },
{ "route9",    1,3,3 },
{ "pidfile",   1,1,1 },
{ "keyttl",    1,1,1 },
{ "randomdev", 1,1,1 },
{ "mtu",       1,1,1 },
{ "speed",     1,1,1 },
{ "retry",     1,1,1 },
{ "modemchat", 1,1,1 },
{ "localline", 1,0,0 },
{ "nortscts",  1,0,0 },
{ "noanswer",  1,1,1 },
{ "nocslip",   1,0,0 },
{ "keysize",   1,1,1 },
{ "nocompress",1,0,0 },
{ "xfilter",   1,0,0 },
{ "threshold", 1,1,1 },
{ "linktest",  1,1,1 },
{ "peerroute", 1,0,1 },
{ "oobfix",    1,0,0 },
{ "suspend",   1,1,1 },
{ "rxmax",     1,1,1 },
{ "txmax",     1,1,1 },
{ "linkup",    1,1,1 },
{ "linkdown",  1,1,1 },
{ "ipopts",    1,1,1 },
{ "sendbuf",   1,1,1 },
{ "connwait",  1,1,1 },
{ "hmac",      1,1,2 },
{ "slipup",    1,1,1 },
{ "slipdown",  1,1,1 },
{ "facility",  1,1,1 },
{ NULL,        0,0,0 }
};

/* configuration data array access */

#define IDX_MODE		0
#define IDX_LOCAL		1
#define IDX_REMOTE		2
#define IDX_SERVER		3
#define IDX_CLIENT		4
#define IDX_KEEPALIVE		5
#define IDX_AUTOROUTE		6
#define IDX_KEYFILE		7
#define IDX_PIDFILE		17
#define IDX_KEYTTL		18
#define IDX_RANDOMDEV		19
#define IDX_MTU			20
#define IDX_SPEED		21
#define IDX_RETRY		22
#define IDX_MODEMCHAT		23
#define IDX_LOCALLINE		24
#define IDX_NORTSCTS		25
#define IDX_NOANSWER		26
#define IDX_NOCSLIP		27
#define IDX_KEYSIZE		28
#define IDX_NOCOMPRESS		29
#define IDX_XFILTER		30
#define IDX_THRESHOLD		31
#define IDX_LINKTEST		32
#define IDX_PEERROUTE		33
#define IDX_OOBFIX		34
#define IDX_SUSPEND		35
#define IDX_RXMAX		36
#define IDX_TXMAX		37
#define IDX_LINKUP		38
#define IDX_LINKDOWN		39
#define IDX_IPOPTS		40
#define IDX_SENDBUF		41
#define IDX_CONNWAIT		42
#define IDX_HMAC		43
#define IDX_SLIPUP		44
#define IDX_SLIPDOWN		45
#define IDX_FACILITY		46

#define CFG_MODE(a)		anchor->config[IDX_MODE].a
#define CFG_LOCAL(a)		anchor->config[IDX_LOCAL].a
#define CFG_REMOTE(a)		anchor->config[IDX_REMOTE].a
#define CFG_SERVER(a)		anchor->config[IDX_SERVER].a
#define CFG_CLIENT(a)		anchor->config[IDX_CLIENT].a
#define CFG_KEEPALIVE(a)	anchor->config[IDX_KEEPALIVE].a
#define CFG_AUTOROUTE(a)	anchor->config[IDX_AUTOROUTE].a
#define CFG_KEYFILE(a)		anchor->config[IDX_KEYFILE].a
#define CFG_FIRSTROUTE		8
#define CFG_ROUTE(a,b)		anchor->config[a].b
#define CFG_LASTROUTE		16
#define CFG_PIDFILE(a)		anchor->config[IDX_PIDFILE].a
#define CFG_KEYTTL(a)		anchor->config[IDX_KEYTTL].a
#define CFG_RANDOMDEV(a)	anchor->config[IDX_RANDOMDEV].a
#define CFG_MTU(a)		anchor->config[IDX_MTU].a
#define CFG_SPEED(a)		anchor->config[IDX_SPEED].a
#define CFG_RETRY(a)		anchor->config[IDX_RETRY].a
#define CFG_MODEMCHAT(a)	anchor->config[IDX_MODEMCHAT].a
#define CFG_LOCALLINE(a)	anchor->config[IDX_LOCALLINE].a
#define CFG_NORTSCTS(a)		anchor->config[IDX_NORTSCTS].a
#define CFG_NOANSWER(a)		anchor->config[IDX_NOANSWER].a
#define CFG_NOCSLIP(a)		anchor->config[IDX_NOCSLIP].a
#define CFG_KEYSIZE(a)		anchor->config[IDX_KEYSIZE].a
#define CFG_NOCOMPRESS(a)	anchor->config[IDX_NOCOMPRESS].a
#define CFG_XFILTER(a)		anchor->config[IDX_XFILTER].a
#define CFG_THRESHOLD(a)	anchor->config[IDX_THRESHOLD].a
#define CFG_LINKTEST(a)		anchor->config[IDX_LINKTEST].a
#define CFG_PEERROUTE(a)	anchor->config[IDX_PEERROUTE].a
#define CFG_OOBFIX(a)		anchor->config[IDX_OOBFIX].a
#define CFG_SUSPEND(a)		anchor->config[IDX_SUSPEND].a
#define CFG_RXMAX(a)		anchor->config[IDX_RXMAX].a
#define CFG_TXMAX(a)		anchor->config[IDX_TXMAX].a
#define CFG_LINKUP(a)		anchor->config[IDX_LINKUP].a
#define CFG_LINKDOWN(a)		anchor->config[IDX_LINKDOWN].a
#define CFG_IPOPTS(a)		anchor->config[IDX_IPOPTS].a
#define CFG_SENDBUF(a)		anchor->config[IDX_SENDBUF].a
#define CFG_CONNWAIT(a)		anchor->config[IDX_CONNWAIT].a
#define CFG_HMAC(a)		anchor->config[IDX_HMAC].a
#define CFG_SLIPUP(a)		anchor->config[IDX_SLIPUP].a
#define CFG_SLIPDOWN(a)		anchor->config[IDX_SLIPDOWN].a
#define CFG_FACILITY(a)		anchor->config[IDX_FACILITY].a

/* numeric configuration data keep sequence! */

typedef struct
{
	char idx;	/* index into configuration data	*/
	char lower;	/* lower limit or -1 for boolean	*/
	WORD16 upper;	/* upper limit or true value if boolean	*/
	WORD16 dft;	/* default value if option not given	*/
} NUMDATA;

static NUMDATA numdata[]=
{
	{ -IDX_SERVER,     1, 65535,           379             },
	{ -IDX_CLIENT,     1, 65535,           0               },
	{ IDX_KEEPALIVE,   1, 65535,           0               },
	{ IDX_KEYTTL,      0, 65535,           60              },
	{ IDX_MTU,         1, 65535,           1500            },
	{ IDX_RETRY,       1, 65535,           10              },
	{ IDX_NOANSWER,    1, 65535,           11              },
	{ IDX_KEYSIZE,     0, BLOWFISH_MAXKEY, BLOWFISH_MAXKEY },
	{ IDX_THRESHOLD,   1, MAXBLOCK-1,      16              },
	{ IDX_LINKTEST,    1, MAXIDLE,         0               },
	{ IDX_SUSPEND,     1, MAXIDLE,         0               },
	{ IDX_RXMAX,       1, MAXIDLE,         10              },
	{ IDX_TXMAX,       1, MAXIDLE,         10              },
	{ IDX_IPOPTS,      1, MAXIPOPT,        0               },
	{ IDX_SENDBUF,     1, 65535,           0               },
	{ IDX_CONNWAIT,    1, MAXIDLE,         0               },
	{ IDX_AUTOROUTE,  -1, 1,               0               },
	{ IDX_LOCALLINE,  -1, 1,               0               },
	{ IDX_NORTSCTS,   -1, 0,               1               },
	{ IDX_NOCSLIP,    -1, 0,               1               },
	{ IDX_NOCOMPRESS, -1, 1,               0               },
	{ IDX_XFILTER,    -1, 1,               0               },
	{ IDX_OOBFIX,     -1, 1,               0               },
	{ IDX_HMAC,        1, 2,               0               },
	{ 0,               0, 0,               0               }

};

/* serial line speed conversion structure */

typedef struct
{
	int speednum;	/* numeric speed	*/
	int speedval;	/* speed setup value	*/
} SPEED;

/* serial line speeds */

static SPEED speeddata[]=
{
	{ 19200,  B19200  },
	{ 38400,  B38400  },
	{ 57600,  B57600  },
	{ 115200, B115200 },
	{ 230400, B230400 },
#ifdef LINUX2
	{ 460800, B460800 },
#endif
	{ 0,      0       }
};

/* host netmask */

static char ip255[]="255.255.255.255";

/*
 * checkfile
 *
 * input:  pathname - the pathname of the file to be checked
 *         perms    - the minimal file parmissions or 0 for device file
 *
 * return: 0 in case of error, the file size for regular files,
 *         the file permissions for device files
 *
 * This procedure checks the file access mode of the given file.
 */

int checkfile(char *pathname,int perms)
{
	struct stat stb;	/* used for file check	*/


	/* get file information */

	if(lstat(pathname,&stb))
	{
		printf("Can't stat '%s'.\n",pathname);
		return 0;
	}

	/* if check for device file */

	if(!perms)
	{
		/* assert that target is character special file */

		if(!S_ISCHR(stb.st_mode))
		{
			printf("'%s' is not a character special device.\n",
				pathname);
			return 0;
		}

		if(!stb.st_mode)printf("'%s' has permission 0.\n",pathname);

		/* signal success or error */

		return (int)(stb.st_mode);
	}

	/* check that the file is a regular file */

	if(!S_ISREG(stb.st_mode))
	{
		printf("'%s': not a regular file.\n",pathname);
		return 0;
	}

	/* check that the file is owned by root:root */

	if(stb.st_uid||stb.st_gid)
	{
#ifdef LINUX
		printf("'%s' must be owned by root:root.\n",pathname);
#elif FreeBSD
		printf("'%s' must be owned by root:wheel.\n",pathname);
#endif
		return 0;
	}

	/* check that config file can only be written to by owner */

	if(stb.st_mode&perms)
	{
		printf("'%s' permissions must be %03o or better.\n",
			pathname,0777^perms);
		return 0;
	}

	if(!stb.st_size)printf("'%s' has size 0.\n",pathname);

	/* return file size (0=error) */

	return (int)(stb.st_size);
}

/*
 * chkip
 *
 * input:  ip - pointer to ip in string form
 *
 * return: 1 in case of valid ip, else 0
 *
 * This procedure checks the given ip for validity
 */

int chkip(char *ip)
{
	/* error, if given string is too long to hold valid ip */

	if(!dns&&strlen(ip)>15)return 0;

	/* success, if conversion to ip succeeds */

	if(getaddr(ip)!=-1)return 1;

	/* error, if not special case */

	if(strcmp(ip,ip255))return 0;

	/* success if special case */

	return 1;
}

/*
 * chksvrcln
 *
 * input:  mode   - 0 for server data, 1 for client data
 *	   p1     - pointer to server/client parameter 1
 *	   p2     - pointer to server/client parameter 2
 *	   txt    - pointer to server/client message text
 *	   anchor - pointer to vpnd global data
 *
 * return: 1 in case of success, 0 in case of error
 *
 * This procedure checks the server/client parameters for plausibility
 * and sets up some required data.
 */

int chksvrcln(char mode,char *p1,char *p2,char *txt,VPN *anchor)
{
	/* check command ip and port if not device mode */

	if(p1[0]!='/')
	{
		/* check server ip */

		if(chkip(p1))
		{
			if(mode)anchor->client_ip=p1;
			else anchor->server_ip=p1;
		}
		else
		{
			printf("Illegal %s ip.\n",txt);
			return 0;
		}
	}

	/* check command device */

	else
	{
		if(strncmp(p1,"/dev/",5))
		{
			printf("Illegal %s device (not in /dev).\n",txt);
			return 0;
		}
		if(p2)
		{
			printf("Do not use port for %s device.\n",txt);
			return 0;
		}
		if(anchor->mode==mode)
		{
			if(!(anchor->normalmode=checkfile(p1,0)))return 0;
			anchor->serialdev=p1;
		}
		anchor->serialmode=1;
	}

	/* signal success */

	return 1;
}

/*
 * parse
 *
 * input: anchor - pointer to vpnd global data
 *
 * return: 0 in case of error, 1 in case of success
 *
 * This procedure performs basic configuration file
 * parsing and data gathering.
 */

int parse(VPN *anchor)
{
	FILE *fp;		/* config file access	*/
	char *item;		/* current item		*/
	int i;			/* counter/index	*/
	int j;			/* counter/index	*/
	union
	{
		CONFIG *c;	/* config format access	*/
		NUMDATA *n;	/* numeric parameters	*/
		WORD16 *w16;	/* 16 bit parameters	*/
	} u;			/* fast access union	*/
	union
	{
		CFGDATA *c;	/* configuration data	*/
		WORD16 *w16;	/* 16 bit parameters	*/
	} v;			/* fast access union	*/
	WORD16 val[32];		/* numeric data buffer	*/
	char *bfr;		/* input buffer		*/


	/* debug message */

	ENTER("parse");

	/* set up buffer pointer */

	bfr=anchor->u1.bfr;

	/* get check config file permissions */

	if(!checkfile(anchor->conffile,022))return 0;

	/* try to open config file, handle errors */

	if((fp=fopen(anchor->conffile,"r"))==NULL)
	{
		printf("'%s': no such file\n",anchor->conffile);
		return 0;
	}

	/* for all lines of the config file */

	while(fgets(bfr,sizeof(anchor->u1.bfr),fp))
	{
		/* get first token, skip empty lines */

		if((item=strtok(bfr," \t\r\n"))==NULL)continue;

		/* next line if line is comment */

		else if(*item=='#')continue;

		/* for all commands if the current item matches the command */

		else for(u.c=config,v.c=anchor->config;u.c->item;u.c++,v.c++)
			if(!strcmp(item,u.c->item))
		{
			/* if command previously encountered signal error */

			if(v.c->found)
			{
				printf("Command '%s' repeated.\n",item);
				fclose(fp);
				return 0;
			}

			/* set comand encountered flag */

			v.c->found=1;

			/* reset parameter counter */

			j=0;

			/* for all remaining tokens of the line */

			while((item=strtok(NULL," \t\r\n"))!=NULL)
			{
				/* done if comment or empty element */

				if(*item=='#'||*item==0)break;

				/* signal error if maximum parameters 
				   exceeded */

				if(j==u.c->max)
				{
					printf("Command '%s' takes only %d " \
						"parameters.\n",u.c->item,
						u.c->max);
					fclose(fp);
					return 0;
				}

				/* store parameter and adjust parameter
				   count */

				switch(j++)
				{
					case 0:	v.c->p1=strdup(item);
						break;
					case 1:	v.c->p2=strdup(item);
						break;
					case 2:	v.c->p3=strdup(item);
						break;
				}
			}

			/* signal error if not minimum parameters given */

			if(j<u.c->min)
			{
				printf("Command '%s' takes at least %d " \
					"parameters.\n",u.c->item,u.c->min);
				fclose(fp);
				return 0;
			}

			/* exit command search loop */

			break;
		}

		/* if item didn't match any command signal error */

		if(!u.c->item)
		{
			printf("Illegal command '%s' encountered.\n",item);
			fclose(fp);
			return 0;
		}
	}

	/* close config file */

	fclose(fp);

	/* signal error if mandatory command not given */

	for(u.c=config,v.c=anchor->config;u.c->item;u.c++,v.c++)
		if(!u.c->option)if(!v.c->found)
	{
		printf("Missing command '%s'.\n",u.c->item);
		return 0;
	}

	/* check mode command parameter */

	if(!strcmp(CFG_MODE(p1),"client"))anchor->mode=1;
	else if(!strcmp(CFG_MODE(p1),"server"))anchor->mode=0;
	else
	{
		printf("Mode must be 'client' or 'server'.\n");
		return 0;
	}

	/* check local command ip */

	if(chkip(CFG_LOCAL(p1)))anchor->local_ip=CFG_LOCAL(p1);
	else
	{
		printf("Illegal local ip.\n");
		return 0;
	}

	/* check remote command ip */

	if(chkip(CFG_REMOTE(p1)))anchor->remote_ip=CFG_REMOTE(p1);
	else
	{
		printf("Illegal remote ip.\n");
		return 0;
	}

	/* check client and server parameters */

	if(!chksvrcln(0,CFG_SERVER(p1),CFG_SERVER(p2),"server",anchor))return 0;
	if(!chksvrcln(1,CFG_CLIENT(p1),CFG_CLIENT(p2),"client",anchor))return 0;

	/* assert that server and client are in same (non)device mode */

	if((CFG_SERVER(p1[0])=='/'&&CFG_CLIENT(p1[0])!='/')||
		(CFG_SERVER(p1[0])!='/'&&CFG_CLIENT(p1[0])=='/'))
	{
		printf("Use either ip or device for both client and server.\n");
		return 0;
	}

	/* for all numeric parameters */

	for(v.w16=val,u.n=numdata;u.n->idx;u.n++,v.w16++)
	{
		/* preset buffer with default value */

		*v.w16=u.n->dft;

		/* get index into configuration array */

		j=u.n->idx;

		/* in case of boolean parameter */

		if(u.n->lower==-1)
		{
			/* copy true value to buffer if option is given */

			if(anchor->config[j].found)*v.w16=u.n->upper;

			/* next numeric parameter */

			continue;
		}

		/* if second parameter has to be converted */

		if(j<0)
		{
			/* normalize index */

			j=-j;

			/* get pointer to second parameter */

			item=anchor->config[j].p2;
		}

		/* get pointer to first parameter */

		else item=anchor->config[j].p1;

		/* if option is given and parameter is given */

		if(anchor->config[j].found&&item)
		{
			/* convert parameter to number */

			i=atoi(item);

			/* if parameter is numeric and within given limits */

			if(!(i==0&&strcmp(item,"0")))
				if(i>=u.n->lower&&i<=u.n->upper)
			{
				/* copy value to buffer */

				*v.w16=(WORD16)(i);

				/* next numeric parameter */

				continue;
			}

			/* signal error if value is out of range */

			printf("Illegal %s value, valid range is %d to %d.\n",
				config[j].item,u.n->lower,u.n->upper);
			return 0;
		}
	}

	/* copy buffer values to actual global configuration data */

	u.w16=val;
	anchor->server_port=*u.w16++;
	anchor->client_port=*u.w16++;
	anchor->keepalive=*u.w16++;
	anchor->keyttl=((int)(*u.w16++))*60;
	anchor->mtu=*u.w16++;
	anchor->retry=*u.w16++;
	anchor->noresponse=*u.w16++;
	anchor->keysize=(WORD08)(*u.w16++);
	anchor->threshold=*u.w16++;
	anchor->linktest=*u.w16++;
	anchor->suspend=*u.w16++;
	anchor->rxmax=*u.w16++;
	anchor->txmax=*u.w16++;
	anchor->ipopts=*u.w16++;
	anchor->sendbuf=(int)(*u.w16++);
	anchor->connwait=*u.w16++;
	anchor->autoroute=(WORD08)(*u.w16++);
	anchor->localline=(WORD08)(*u.w16++);
	anchor->rtscts=(WORD08)(*u.w16++);
	anchor->cslip=(WORD08)(*u.w16++);
	anchor->nocompress=(WORD08)(*u.w16++);
	anchor->xfilter=(WORD08)(*u.w16++);
	anchor->oobfix=(WORD08)(*u.w16++);
	anchor->hmacopt=(int)(*u.w16++);

	/* check all routes, if given */

	for(i=CFG_FIRSTROUTE;i<=CFG_LASTROUTE;i++)if(CFG_ROUTE(i,found))
	{
		if(!chkip(CFG_ROUTE(i,p1))||!chkip(CFG_ROUTE(i,p2))||
			!chkip(CFG_ROUTE(i,p3)))
		{
			printf("Illegal destination, netmask or gateway " \
				"for '%s' command.\n",config[i].item);
			return 0;
		}
		anchor->rtab[anchor->rmax][0]=getaddr(CFG_ROUTE(i,p1));
		anchor->rtab[anchor->rmax][1]=
			strcmp(CFG_ROUTE(i,p2),ip255)?
				getaddr(CFG_ROUTE(i,p2)):-1;
		anchor->rtab[anchor->rmax][2]=getaddr(CFG_ROUTE(i,p3));
		anchor->rmax++;
	}

	/* get keyfile if given */

	if(CFG_HMAC(found))
	{
		if(CFG_HMAC(p2))
		{
			if(!strcmp(CFG_HMAC(p2),"md5"))anchor->hmacmode=1;
			else if(!strcmp(CFG_HMAC(p2),"sha1"))anchor->hmacmode=2;
			else if(!strcmp(CFG_HMAC(p2),"rmd160"))
				anchor->hmacmode=3;
			else
			{
				printf("unknown mode for hmac option.\n");
				return 0;
			}
		}
		else anchor->hmacmode=1;
	}

	/* get syslog facility if given */

	if(CFG_FACILITY(found))for(i=0;;i++)
	{
		if(!facilitynames[i].c_name)
		{
			printf("unknown syslog facility name.\n");
			return 0;
		}
		if(!strcmp(facilitynames[i].c_name,CFG_FACILITY(p1)))
		{
			facility=facilitynames[i].c_val;
			break;
		}
	}

	/* get keyfile if given */

	if(CFG_KEYFILE(found))anchor->keyfile=CFG_KEYFILE(p1);

	/* use optional pid file */

	if(CFG_PIDFILE(found))pidfile=CFG_PIDFILE(p1);

	/* use optional random device file */

	if(CFG_RANDOMDEV(found))anchor->randomdev=CFG_RANDOMDEV(p1);

	/* set pathname of linkup process if requested */

	if(CFG_LINKUP(found))anchor->linkup=CFG_LINKUP(p1);

	/* set pathname of linkdown process if requested */

	if(CFG_LINKDOWN(found))anchor->linkdown=CFG_LINKDOWN(p1);

	/* set pathname of linkup process if requested */

	if(CFG_SLIPUP(found))anchor->slipup=CFG_SLIPUP(p1);

	/* set pathname of linkdown process if requested */

	if(CFG_SLIPDOWN(found))anchor->slipdown=CFG_SLIPDOWN(p1);

	/* use optional modem chat file, check file permissions, too */

	if(CFG_MODEMCHAT(found))
	{
		anchor->modemchat=CFG_MODEMCHAT(p1);
		if(!checkfile(anchor->modemchat,022))return 0;
	}

	/* check serial line speed if given */

	if(CFG_SPEED(found))
	{
		j=atoi(CFG_SPEED(p1));
		for(i=0;speeddata[i].speednum;i++)if(j==speeddata[i].speednum)
			break;
		if(speeddata[i].speednum)anchor->speed=speeddata[i].speedval;
		else
		{
			printf("Illegal speed value.\n");
			return 0;
		}
	}

	/* process peer priority route if given if not in serial line mode */

	if(CFG_PEERROUTE(found))if(!anchor->serialmode)
	{
		/* in client mode peer is server ip,
		   in server mode peer is client ip */

		if(anchor->mode)anchor->peerip=getaddr(anchor->server_ip);
		else anchor->peerip=getaddr(anchor->client_ip);

		/* if a fixed device is given */

		if(CFG_PEERROUTE(p1))
		{
			/* set pointer to device name */

			strncpy(anchor->peerdev,CFG_PEERROUTE(p1),16);
			anchor->peerdev[15]=0;

			/* set priority route mode to device mode */

			anchor->peerroute=1;
		}

		/* if no device is given, try to get routing data to peer
		   from system routing tables, handle errors */

		else if(!peerroute(anchor))
		{
			printf("Can't find route to peer.\n");
			return 0;
		}
	}

	/* if encryption has to be used */

	if(anchor->keysize)
	{
		/* check key file permissions and size, set extended mode */

		switch(checkfile(anchor->keyfile,0377))
		{
		case BLOWFISH_MAXKEY:	anchor->extended=0;
					break;
		case XKEYSIZE:		anchor->extended=1;
					break;
		default:		printf("'%s' has wrong size.\n",
						anchor->keyfile);
		case 0:			return 0;
		}
	}

	/* check that passphrase is given only for extended mode
	   master key file and that optional hmac is used only in
	   extended mode */

	if(!anchor->extended)
	{
		if(anchor->passwd)
		{
			printf("-p option is only valid for extended master" \
				" key files.\n");
			return 0;
		}
		if(anchor->hmacopt==1)
		{
			printf("Optional hmac is only valid for extended " \
				"master key files.\n");
			return 0;
		}
	}

	/* sanity settings */

	anchor->mackey=macinit[anchor->hmacmode];
	anchor->macproc=mactab[anchor->hmacmode];
	anchor->mac=maclen[anchor->hmacmode];
	if(!anchor->serialmode)anchor->xfilter=0;
	else anchor->linktest=0;
	if(!anchor->keysize||anchor->mode)anchor->keyttl=0;
	if(!anchor->keysize)
	{
		anchor->cipher=NONE;
		anchor->encrypt=(void (*)(WORD08 *,WORD32 *))(dummy);
	}

	/* debug message */

	LEAVE("parse");

	/* signal success */

	return 1;
}
