/*
 * Ras server thread for OpenGate
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: RasThread.cxx,v $
 * Revision 1.14  2000/05/15 19:01:29  aunitt
 * Added IP address to log messages and generally tidied up logging.
 *
 * Revision 1.13  2000/05/12 14:01:30  aunitt
 * Renamed PRasLog to Log.
 *
 * Revision 1.12  2000/05/05 11:12:56  aunitt
 * Added comment noting potential problem we need to fix.
 *
 * Revision 1.11  2000/04/27 17:43:22  aunitt
 * Added support for handling registration timeouts.
 *
 * Revision 1.10  2000/04/20 18:43:31  aunitt
 * Fixed bug in not correctly forming unregister requests.
 * Added support for handling endpoints that haven't registered with us in the
 * Endpoint table.
 *
 * Revision 1.9  2000/04/12 13:25:51  aunitt
 * Changed to use new convenience functions in AddrUtils.
 * Now logs unregistration requests in RasLog.
 *
 * Revision 1.8  2000/04/10 19:11:56  aunitt
 * Moved environment data into new environment object to tidy things up.
 *
 * Revision 1.7  2000/04/05 15:15:29  aunitt
 * Added RasLog object.
 *
 * Revision 1.6  2000/03/21 21:12:30  aunitt
 * Added additional informational messages.
 *
 * Revision 1.5  2000/03/19 17:55:15  aunitt
 * Fixed bug in not reading RAS Message introduced in last change
 *
 * Revision 1.4  2000/03/12 21:18:24  aunitt
 * Fixed bug in not handling  messages from endpoints that were not registered
 *
 * Revision 1.3  2000/02/27 20:45:02  aunitt
 * Fixed problems with replying to wrong RAS address
 *
 * Revision 1.2  2000/02/15 20:47:23  aunitt
 * Added support for using system log
 *
 * Revision 1.1.1.1  2000/02/01 22:25:35  aunitt
 * Initial revision
 *
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#if (_MSC_VER >= 1200)
#include <q931.h>
#include <h245.h>               // To stop "ostream" ambiguous symbol error
#endif
#include "RasThread.h"
#include "AddrUtils.h"
#include "RasServer.h"
#include "Log.h"

static const int BufferSize = 4096;	// Size of network buffers

RasThread::RasThread( const Environ &   AkaEnviron,
                      const WORD        Port
                     ) 
	: PThread(1000, NoAutoDeleteThread, NormalPriority, "RasThread"),
	  MyEnviron(AkaEnviron), RASSocket(Port)
{
	RasServ = new RasServer( AkaEnviron, Port );
	Resume();
}

RasThread::~RasThread()
{
	delete RasServ;
}

BOOL RasThread::ReadRasReq( H225_RasMessage &		Mesg,
    						H225_TransportAddress & ReplyTo
						  )
// Task: to read a RAS message from the given socket
{
	PBYTEArray  Buffer(BufferSize);
	PPER_Stream ReadStream(Buffer);

	if ( RASSocket.Read( ReadStream.GetPointer(), ReadStream.GetSize() ) )
	{
 		PIPSocket::Address	SenderAddr;
		WORD				SenderPort;
		
		RASSocket.GetLastReceiveAddress( SenderAddr, SenderPort );
		ReplyTo = AddrUtils::ConvertToH225TransportAddr( SenderAddr, SenderPort );

		if ( Mesg.Decode( ReadStream ) )
		{
		    MyEnviron.Log->LogRasMsg( Mesg, OpengateLog::Receiving, ReplyTo );
		    return TRUE;
		}
	}
	return FALSE;
}

BOOL RasThread::WriteRasReply( const H225_TransportAddress & ReplyTo,
                                     H225_RasMessage &       Mesg
                             )
// Task: to write a RAS message to the given destination
{
	PBYTEArray         Buffer(BufferSize);
	PPER_Stream        WriteStream(Buffer);
	
	MyEnviron.Log->LogRasMsg( Mesg, OpengateLog::Sending, ReplyTo );
	
    if ( ReplyTo.GetTag() == H225_TransportAddress::e_ipAddress )
    {
        const H225_TransportAddress_ipAddress & ReplyToIP = ReplyTo;
        PIPSocket::Address ReplyToAddr;
        WORD ReplyToPort;

        AddrUtils::ConvertToIPAddress( ReplyToIP, ReplyToAddr, ReplyToPort );
    	Mesg.Encode( WriteStream );
	    WriteStream.CompleteEncoding();
    	RASSocket.SetSendAddress( ReplyToAddr, ReplyToPort  );
	    return RASSocket.Write( WriteStream.GetPointer(), WriteStream.GetSize() );
	}
	else
	{
	    PSYSTEMLOG( Error, "No IP address to reply to!" );
	    return false;
	}
}

void RasThread::Main()
{
    // Register our handler with the endpoint table
    MyEnviron.EPTable->SetTimeoutHdlr( this );

	if ( RASSocket.Listen( MyEnviron.LocalAddr ) )
	{
		H225_RasMessage         Request;
		H225_RasMessage         Reply;
		H225_TransportAddress   ReplyTo;

		while ( RASSocket.IsOpen() )
		{
			if ( ReadRasReq( Request, ReplyTo ) )
			{
			    // We need to solve the problem of Win32 multicasts coming here!
				if ( RasServ->HandleRasRequest( Request, false, Reply, ReplyTo ) )
				{
					WriteRasReply( ReplyTo, Reply );
				}
			}
			else
			{
			    PSYSTEMLOG( Info, "Failed to read RAS mesg" );
			}
		}
	}
	else
	{
		// Listen failed
		PSYSTEMLOG( Error, "Listen failed" );
	}
}

static void Unregister( const Endpoint &        EP,
                              H225_RasMessage & UnregReq,
                              OpengateLog *     Log
                      )
// Task: To send an unregistration message to the given endpoint
{
	PPER_Stream             WriteStream;
	PUDPSocket              Socket;
	H225_TransportAddress   IpAddr;
	PIPSocket::Address      Addr;
    WORD                    Port;

	UnregReq.Encode( WriteStream );
	WriteStream.CompleteEncoding();
	if( EP.GetRasIPAddress(IpAddr) )
	{
	    H225_TransportAddress_ipAddress & RasIP = IpAddr;
        AddrUtils::ConvertToIPAddress( RasIP, Addr, Port );
    	
        Socket.SetSendAddress( Addr, Port );
   	    Log->LogRasMsg( UnregReq, OpengateLog::Sending, IpAddr );

	    Socket.Write( WriteStream.GetPointer(), WriteStream.GetSize() );
	}
}

void RasThread::Close()
{
	RASSocket.Close();

	// Clear out the table and send unregister requests to endpoints
	H225_RasMessage UnregReq;
	while( !MyEnviron.EPTable->IsEmpty() )
	{
		Endpoint EP = MyEnviron.EPTable->Pop();
		if ( EP.IsInLocalZone() )
		{
		    RasServ->MakeUnregMessage( EP, UnregReq );
		    Unregister( EP, UnregReq, MyEnviron.Log );
		}
	}
}

BOOL RasThread::SendIRR( const Endpoint & EP )
// Task: to send an IRR to the given endpoint
{
    H225_RasMessage         Mesg;
    H225_CallReferenceValue CRV;
                	
    CRV.SetValue(0);        // All / any calls
    RasServ->MakeIRR( CRV, Mesg );

    H225_TransportAddress SendTo;
   	if( EP.GetRasIPAddress(SendTo) )
  	{
		return WriteRasReply( SendTo, Mesg );
    }

    return FALSE;
}

BOOL RasThread::SendURQ( const Endpoint &                   EP,
                         const H225_UnregRequestReason &    Reason
                       )
// Task: to send an URQ to the given endpoint
{
    H225_RasMessage Mesg;

    RasServ->MakeUnregMessage( EP, Reason, Mesg );

    H225_TransportAddress SendTo;
  	if( EP.GetRasIPAddress(SendTo) )
  	{
		return WriteRasReply( SendTo, Mesg );
    }

    return FALSE;
}

bool RasThread::OnRegistrationTimeout( const H225_EndpointIdentifier & Id,
                                             unsigned                  Chance,
                                             PTimeInterval &           NextPeriod
                                     )
// Task: To handle a registration timeout
{
    try
    {
        Endpoint EP = MyEnviron.EPTable->FindByEndpointId( Id );
        switch( Chance )
        {
            case 0  :
            case 1  :   // First and second time round send an IRR to the endpoint
                		if ( EP.IsInLocalZone() )
		                {
		                    SendIRR( EP );
                	        NextPeriod.SetInterval( 0, 5 );     // Retry in five seconds
                	        return true;
                	    }
                	    else
                	    {
                	        // There is a problem with this, if we are called by
                	        // a non local endpoint and the call is still active then
                	        // we will delete it!
                	
                	        // Just delete cached endpoints from other zones
                	        MyEnviron.EPTable->DeferredRemove( Id );
                            return false;
                	    }
                    	break;
            case 2  :
            default :   // Okay, assume it's dead
                        if ( EP.IsInLocalZone() )
                        {
                            // Better tell them that the timeout has occured
                            H225_UnregRequestReason Reason;
                            Reason.SetTag( H225_UnregRequestReason::e_ttlExpired );
                            SendURQ( EP, Reason );
                        }
                        MyEnviron.EPTable->DeferredRemove( Id );
                        return false;
        }
    }
    catch( EndpointTable::NotFoundError )
    {
        PSYSTEMLOG( Warning, "Id not found while handling timeout" );
        return false;
    }
}
