///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2018 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// bug_dg test
//   with mkgeo_grid -region
// TODO:  mkgeo_ugrid -region : add interface domain
#include "rheolef.h"
using namespace rheolef;
using namespace std;
// 2d test for orientation, 3d test also for face rotation
struct u_west {
  Float operator() (const point& x) const {
    return cos(0.5*pi*x[0])
          *cos(0.5*pi*x[1])
          *cos(0.5*pi*x[2]); }
  static Float integrate(size_t d) {
    Float pi = acos(Float(-1));
    switch (d) {
      case 1:  return 1/sqrt(Float(2));
      case 2:  return sqrt(Float(2))/pi;
      default: return 2*sqrt(Float(2))/sqr(pi);
    }
  }
  u_west (size_t d1) : d(d1), pi(acos(Float(-1))) {}
  size_t d; Float pi;
};
// u_east(x) > u_west(x) on the interface x0=1/2:
struct u_east {
  Float operator() (const point& x) const {
    return (1+sin(0.5*pi*x[0]))
          *(1+sin(0.5*pi*x[1]))
          *(1+sin(0.5*pi*x[2])); }
  static Float integrate(size_t d) {
    Float pi = acos(Float(-1));
    switch (d) {
      case 1:  return (1+1/sqrt(Float(2)));
      case 2:  return (1+1/sqrt(Float(2)))*(1+2/pi);
      default: return (1+1/sqrt(Float(2)))*sqr(1+2/pi);
    }
  }
  u_east (size_t d1) : d(d1), pi(acos(Float(-1))) {}
  size_t d; Float pi;
};
int main(int argc, char**argv) {
  environment rheolef(argc, argv);
  geo omega (argv[1]);
  string approx          = (argc > 2) ?      argv[2]  : "P1d";
  Float tol              = (argc > 3) ? atof(argv[3]) : 1e300;
  space Xh (omega, approx),
        Xh_west (omega["west"], approx),
        Xh_east (omega["east"], approx);
  size_t d = omega.dimension();
  field uh (Xh);
  uh["west"] = interpolate (Xh_west, u_west(d));
  uh["east"] = interpolate (Xh_east, u_east(d));
  size_t k = Xh.degree();
  quadrature_option qopt;
  qopt.set_family(quadrature_option::gauss);
  qopt.set_order(2*k+1);
  point n_interf; // BUG_INT_BDR_JUMP: fix the sign of the interface: should be done when using jump
	          // when the "ds" interface measure has the opposite sign, e.g. "interface2"
  if (d != 3) n_interf = point(1, 0, 0);
  else        n_interf = point(0, 1, 0);
  Float value_jump    = integrate (omega["interface"],  jump(uh)*dot(n_interf,normal()), qopt);
  Float value_average = integrate (omega["interface"],  average(uh), qopt);
  Float value_jump_expected    =  u_west::integrate(d) - u_east::integrate(d);
  Float value_average_expected = (u_west::integrate(d) + u_east::integrate(d))/2;
  Float err = max(abs(value_jump    - value_jump_expected),
                  abs(value_average - value_average_expected));
  derr << setprecision(16);
  derr << "value_jump             = " << value_jump << endl;
  derr << "value_jump_expected    = " << value_jump_expected << endl;
  derr << "value_average          = " << value_average << endl;
  derr << "value_average_expected = " << value_average_expected << endl;
  derr << "err                    = " << err << endl;
#ifdef TO_CLEAN
  // BUG_INT_BDR_JUMP : int on interface2 should have the opposite sign
  // when removing the dot(.,.) sign fix
  Float value_jump2   = integrate (omega["interface2"], jump(uh)*dot(n_interf,normal()), qopt);
  Float value_jump3   = integrate (omega.internal_sides(), jump(uh)*dot(n_interf,normal()), qopt);
  Float value_average2= integrate (omega["interface2"], average(uh), qopt);
  Float value_west_expected    = u_west::integrate(d);
  Float value_east_expected    = u_east::integrate(d);
  Float value_west_approx      = integrate (omega["interface"], u_west(d), qopt);
  Float value_east_approx      = integrate (omega["interface"], u_east(d), qopt);
  derr << "value_west_expected    = " << value_west_expected << endl;
  derr << "value_west_approx      = " << value_west_approx   << endl;
  derr << "value_east_expected    = " << value_east_expected << endl;
  derr << "value_east_approx      = " << value_east_approx   << endl;
  derr << "value_jump2            = " << value_jump2 << endl;
  derr << "value_jump3            = " << value_jump3 << endl;
  derr << "value_average2         = " << value_average2 << endl;
#endif // TO_CLEAN
  return (err < tol)  ? 0 : 1;
}
