///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// 
// evaluate a field on a predefined point set: hat_x[q], q=0..nq
// See also piola_transformation.h
//
#include "rheolef/field_evaluate.h"
namespace rheolef { 

// -----------------------------------------------------
// scalar-valued case:
// -----------------------------------------------------
template<class T, class M>
T
field_evaluate (
  const field_basic<T,M>&       uh,
  const basis_on_pointset<T>&   bops,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_idof,
  size_t                        q)
{
  typedef typename field_basic<T,M>::size_type size_type;
  T value = 0;
  size_type loc_idof = 0;
  bops.evaluate (hat_K, q);
  for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; ++loc_idof) {
    const T& cdof = bops.value (loc_idof);
    const T& udof = uh.dis_dof (dis_idof[loc_idof]);
    value += cdof*udof;
  }
  return value;
}
// -----------------------------------------------------
// vector-valued case:
// -----------------------------------------------------
template<class T, class M>
point_basic<T>
vector_field_evaluate (
  const field_basic<T,M>&       uh,
  const basis_on_pointset<T>&   bops,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_idof_tab,
  size_t                        q)
{
  typedef typename field_basic<T,M>::size_type size_type;
  size_type dis_ndof = uh.dis_ndof();
  size_type loc_ndof = dis_idof_tab.size();
  bops.evaluate (hat_K, q);
  size_type loc_comp_ndof = std::distance (bops.begin(), bops.end());
  size_type loc_comp_idof = 0;
  size_type n_comp = uh.get_geo().dimension();
  point_basic<T> value (0,0,0);
  for (typename basis_on_pointset<T>::const_iterator iter = bops.begin(), last = bops.end();
          iter != last;
          iter++, loc_comp_idof++) {
    const T& cdof = *iter;
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      size_type loc_idof = loc_comp_idof + i_comp*loc_comp_ndof;
      assert_macro (loc_idof < loc_ndof, "invalid local index "<<loc_idof<<" out of range [0:"<<loc_ndof<<"[");
      size_type dis_idof = dis_idof_tab[loc_idof];
      assert_macro (dis_idof < dis_ndof, "invalid distr index");
      const T& udof = uh.dis_dof (dis_idof);
      value[i_comp] += cdof*udof;
    }
  }
  return value;
}
// -----------------------------------------------------
// tensor-valued case:
// -----------------------------------------------------
template<class T, class M>
tensor_basic<T>
tensor_field_evaluate (
  const field_basic<T,M>&       uh,
  const basis_on_pointset<T>&   bops,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_idof_tab,
  size_t                        q)
{
  typedef typename field_basic<T,M>::size_type size_type;
  size_type dis_ndof = uh.dis_ndof();
  size_type loc_ndof = dis_idof_tab.size();
  bops.evaluate (hat_K, q);
  size_type loc_comp_ndof = std::distance (bops.begin(), bops.end());
  size_type loc_comp_idof = 0;
  size_type d = uh.get_geo().dimension();
  space_constant::coordinate_type sys_coord = uh.get_geo().coordinate_system();
  size_type n_comp = space_constant::n_component (space_constant::tensor, d, sys_coord);
  tensor_basic<T> value; // initiated to zero
  for (typename basis_on_pointset<T>::const_iterator iter = bops.begin(), last = bops.end();
          iter != last;
          iter++, loc_comp_idof++) {
    const T& cdof = *iter;
    for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
      size_type loc_idof = loc_comp_idof + ij_comp*loc_comp_ndof;
      assert_macro (loc_idof < loc_ndof, "invalid local index "<<loc_idof<<" out of range [0:"<<loc_ndof<<"[");
      size_type dis_idof = dis_idof_tab[loc_idof];
      assert_macro (dis_idof < dis_ndof, "invalid distr index");
      const T& udof = uh.dis_dof (dis_idof);
      T tmp = cdof*udof;
      std::pair<size_type,size_type> ij
          = space_constant::tensor_subscript (space_constant::tensor, sys_coord, ij_comp);
      size_type i = ij.first;
      size_type j = ij.second;
      value(i,j) += tmp;
      if (i != j) value(j,i) += tmp;
    }
  }
  return value;
}
// -----------------------------------------------------
// homogeneous multi-component case: get the i-th value
// -----------------------------------------------------
template<class T, class M>
T
field_component_evaluate (
  const field_basic<T,M>&       uh,
  const basis_on_pointset<T>&   bops,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_idof_tab,
  size_t                        q,
  size_t                        i_comp)
{
  typedef typename field_basic<T,M>::size_type size_type;
  bops.evaluate (hat_K, q);
  size_type dis_ndof = uh.dis_ndof();
  size_type loc_ndof = dis_idof_tab.size();
  size_type loc_comp_ndof = std::distance (bops.begin(), bops.end());
  size_type loc_comp_idof = 0;
  size_type n_comp = uh.get_geo().dimension();
  T value = 0;
  for (typename basis_on_pointset<T>::const_iterator iter = bops.begin(), last = bops.end();
          iter != last;
          iter++, loc_comp_idof++) {
     const T& cdof = *iter;
     size_type loc_idof = loc_comp_idof + i_comp*loc_comp_ndof;
     assert_macro (loc_idof < loc_ndof, "invalid local index "<<loc_idof<<" out of range [0:"<<loc_ndof<<"[");
     size_type dis_idof = dis_idof_tab[loc_idof];
     assert_macro (dis_idof < dis_ndof, "invalid distr index");
     const T& udof = uh.dis_dof (dis_idof);
     value += cdof*udof;
  }
  return value;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M)	 			\
template							\
T								\
field_evaluate<T,M>(						\
  const field_basic<T,M>&       uh,				\
  const basis_on_pointset<T>&   bops,				\
  reference_element             hat_K,				\
  const std::vector<size_t>&    dis_idof,			\
  size_t                        q);				\
template							\
point_basic<T>							\
vector_field_evaluate<T,M> (					\
  const field_basic<T,M>&       uh,				\
  const basis_on_pointset<T>&   bops,				\
  reference_element             hat_K,				\
  const std::vector<size_t>&    dis_idof_tab,			\
  size_t                        q);				\
template							\
tensor_basic<T>							\
tensor_field_evaluate<T,M> (					\
  const field_basic<T,M>&       uh,				\
  const basis_on_pointset<T>&   bops,				\
  reference_element             hat_K,				\
  const std::vector<size_t>&    dis_idof_tab,			\
  size_t                        q);				\
template							\
T								\
field_component_evaluate<T,M> (					\
  const field_basic<T,M>&       uh,				\
  const basis_on_pointset<T>&   bops,				\
  reference_element             hat_K,				\
  const std::vector<size_t>&    dis_idof_tab,			\
  size_t                        q,				\
  size_t                        i_comp);

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
