///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:field
NAME: @code{field} -- plot a field 
@pindex field
@cindex plotting
@cindex plotting data
@fiindex @file{.field} field

SYNOPSIS:
  @example
  	field @var{options} @var{filename}[.field[.gz]]
  @end example
DESCRIPTION:       
  Read and output a finite element field from file.
EXAMPLE: 
  @example
	field square.field
	field square.field -bw
	field box.field
  @end example
INPUT FILE SPECIFICATION:
    @table @code

    @item @var{filename}
	specifies the name of the file containing
      	the input field.

    @item -
	read field on standard input instead on a file.
    @item -name
        when the field comes from standard input, the file base name
        is not known and is set to "output" by default.
        This option allows one to change this default.
        Useful when dealing with output formats (graphic, format conversion)
        that creates auxiliary files, based on this name.
@cindex RHEOPATH environment variable
    @item -I@var{dir} 
    @itemx -I @var{dir} 
        Add @var{dir} to the rheolef file search path.
	This option is useful e.g. when the mesh .geo and the .field fikes are in
	different directories.
        This mechanism initializes a search path given by the environment variable @samp{RHEOPATH}.
        If the environment variable @samp{RHEOPATH} is not set, the default value is the current directory.


@clindex catchmark
    @item  -mark @var{string}
    @itemx -catch @var{string}
    @itemx -catchmark @var{string}
	Jump across the file to the specifield @var{string}.
        Label start at the beginning of a line, preceded by a @file{#} mark (see @pxref{catchmark algorithm}).
    @end table

OUTPUT FILE FORMAT OPTIONS:
  @table @code
  @item -text
  @itemx -field
	output field on standard output stream in Rheolef ascii (field or geo) text file format.
@toindex @code{gmsh}
@fiindex @file{.gmsh} mesh file
@fiindex @file{.gmsh_pos} data file
  @item -gmsh
	output field on standard output stream in gmsh text file format.
  @item -gmsh-pos
	output field on standard output stream in gmsh-pos text file format,
        suitable for mesh adaptation purpose.
@toindex @code{bamg}
@fiindex @file{.bamg} mesh file
@fiindex @file{.bamg_bb} data file
  @item -bamg-bb
	output field on standard output stream in bamg-bb text file format,
        suitable for mesh adaptation purpose.
@cindex graphic render
@cindex image file format
@cindex graphic render
@fiindex @file{.png} image
@fiindex @file{.jpg} image
@fiindex @file{.gif} image
@fiindex @file{.tif} image
@fiindex @file{.ppm} image
@fiindex @file{.bmp} image
@fiindex @file{.pdf} image
@fiindex @file{.eps} image
@fiindex @file{.ps} image
@fiindex @file{.svg} image
  @item -image-format @code{string}
        The argument is any valid image format, such as
        bitmap @code{png}, @code{jpg}, @code{gif}, @code{tif}, @code{ppm}, @code{bmp}
        or vectorial @code{pdf}, @code{eps}, @code{ps}, @code{svg}
	image file formats.
	this option can be used with the @code{paraview} and the @code{gnuplot} renders.
	The output file is e.g. @emph{basename}.@code{png} when @emph{basename}
	is the name of the mesh, or can be set with the @code{-name} option.
  @item -resolution @code{int} @code{int}
        The argument is a couple of sizes, for the image resolution.
	This option can be used together with the @code{-image-format}
	for any of the bitmap image formats.
	This option requires the @code{paraview} render.
  @end table

GETTING INFORMATION:
    @table @code
    @item -min
    @itemx -max
        print the min (resp. max) value of the scalar field and then exit.
    @item -get-geo
        print the name of the mesh associated to the field and exit.
    @end table

RENDER OPTIONS:
  @table @code

@toindex @code{gnuplot}
  @item -gnuplot
	use @code{gnuplot} tool.
	This is the default in one dimension.
@toindex @code{paraview}
  @item -paraview
        use @code{paraview} tool.
        This is the default for two- and tri-dimensional geometries.
@toindex @code{mayavi}
  @item -mayavi
        use @code{mayavi} tool.
        This is maintained for backward compatibility purpose:
	it has been supersetted by the paraview render.
  @end table

RENDERING OPTIONS:
    @table @code
    @item -color
    @itemx -gray
    @itemx -black-and-white
    @itemx -bw
	Use (color/gray scale/black and white) rendering.
	Color rendering is the default.
    @item -label
    @itemx -nolabel
        Show or hide title, color bar and various anotations.
        Default is to show labels.
    @item -elevation
    @itemx -noelevation
	For two dimensional field, represent values as elevation
	in the third dimension.
	The default is no evelation.
    @item -scale @var{float}
        applies a multiplicative factor to the field.
        This is useful e.g. in conjonction with the @code{elevation} option.
        The default value is 1.
    @item -stereo
    @itemx -nostereo
        Rendering mode suitable for red-blue anaglyph 3D stereoscopic glasses.
        This option is only available with @code{mayavi} and @code{paraview}.
    @item -fill
	isoline intervals are filled with color.
	This is the default.
    @item -nofill
	draw isolines by using lines.
    @item -volume
    @itemx -novolume
	for 3D data, render values using a colored translucid volume.
        This option requires the @code{paraview} code.
    @item -cut
    @itemx -nocut
        Cut by a specified plane.
        The cutting plane is specified by its origin point and normal vector.
        This option requires the @code{paraview} code.
    @item -origin @var{float} [@var{float} [@var{float}]]
        set the origin of the cutting plane.
        Default is (0.5, 0.5, 0.5).
    @item -normal @var{float} [@var{float} [@var{float}]]
        set the normal of the cutting plane.
        Default is (1, 0, 0).
    @item -iso [@var{float}]
        do draw 3d isosurface. When the optional float is not provided,
	a median value is used.
        This option requires the @code{paraview} code.
    @item -noiso
        do not draw isosurface.
    @item -n-iso @var{int}
        For 2D visualizations, the isovalue table contains
        regularly spaced values from fmin to fmax, the bounds
        of the field.
@cindex projection
    @item -proj @var{approx}
        Convert all selected fields to approximation @var{approx} by using a L2 projection.
    @item -proj
        Convert all selected fields to Pk-continuous approximation by using a L2 projection.
    @item -lumped-proj
        Force P1 approximation for L2 projection and use a lumped mass matrix for it.
    @item -round [@var{float}]
        Round the input up to the specified precision.
        This option, combined with @code{-field}, leads to a round filter.
        Useful for non-regression test purpose, in order
  	to compare numerical results between files with a limited precision,
        since the full double precision is machine-dependent.

@cindex vorticity
@cindex stream function
    @item -n-iso-negative @var{int}
        The isovalue table is split into negatives and positives values.
        Assume there is n_iso=15 isolines: if 4 is requested by this option,
        then, there will be 4 negatives isolines, regularly spaced
        from fmin to 0 and 11=15-4 positive isolines, regularly spaced
        from 0 to fmax.
        This option is useful when plotting e.g. vorticity or stream
        functions, where the sign of the field is representative.
    @item -subdivide int
        When using a high order geometry, the number of points per edge used to draw
        a curved element. Default value is the mesh order.

    @item -deformation
        Render vector-valued fields as deformed mesh
        using @code{mayavi}, @code{paraview} or @code{gnuplot}.
        This is the default vector field representation.

    @item -velocity
        Render vector-valued fields as arrows using @code{mayavi} or  @code{paraview}.

    @end table
COMPONENT EXTRACTION AND DOMAIN REDUCTION:
    @table @code
    @item -comp @var{int}
    @itemx -comp @var{string}
	Extract the i-th component of a vector-valued field.
        For a tensor-valued field, indexing components as "00", "01", "11"... is supported.
    @item -domain @var{name}
	Reduce the visualization to the specified domain.
    @end table

OTHERS OPTIONS:
    @table @code
    @item -verbose
	print messages related to graphic files created and
       command system calls (this is the default).

    @item -noverbose
	does not print previous messages.

    @item -clean
	clear temporary graphic files (this is the default).
    
    @item -noclean
	does not clear temporary graphic files.

    @item -execute
	execute graphic command (this is the default).
    
    @item -noexecute
	does not execute graphic command. Generates only 
	graphic files. This is useful in conjunction with the
	@code{-noclean} command.
    @end table
FIELD FILE FORMAT:
@noindent
It contains a header and a list values at degrees of freedom.
The header contains the @code{field}
keyword followed by a line containing a format version number (presently 1),
the number of degrees of freedom (i.e. the number of values listed),
the mesh file name without the @file{.geo} extension
the approximation (e.g. P1, P2, etc), and finally the list of values:
@noindent
A sample field file (compatible with the sample mesh example presented in
command manual; see @ref{geo command}) writes:
@example
        field
        1 4
        square
        P1
        0.0
        1.0
        2.0
        3.0
@end example
EXAMPLES:
  @example
    field cube.field -cut -normal 0 1 0 -origin 0.5 0.5 0.5 -vtk
  @end example
  This command send to @code{vtk} the cutted 2d plane of the 3d field.
  @example
    field cube.field -cut -normal 0 1 0 -origin 0.5 0.5 0.5 -text > cube-cut.field
  @end example
  This command generates the cutted 2d field and its associated mesh.
  @example
    field cube.field -iso 0.5
  @end example
  This command draws the isosurface.
  @example
    field cube.field -iso 0.5 -text > isosurf.geo
  @end example
  This command generates the isosurface as a 3d surface mesh in
  @file{.geo} format. This is suitable for others treatments.

AUTHOR: Pierre.Saramito@imag.fr
DATE:
    last change: 11 oct 2011. initial version: 7 july 1997
End:
*/

# include "rheolef_seq.h"
# include "rheolef/iofem.h"
using namespace rheolef;
using namespace std;
void usage() {
      cerr << "field: usage:" << endl
           << "field "
           << "{-|file[.field[.gz]]}"
           << "[-Idir|-I dir] "
           << "[-[catch]mark string] "
           << "[-min|-max|-get-geo] "
           << "[-field|-text|-gmsh|-gmsh-pos|-bamg-bb] "
           << "[-gnuplot|-paraview|-mayavi] "
           << "[-image-format string] "
           << "[-resolution int [int]] "
           << "[-name string] "
           << "[-color|-gray|-black-and-white|-bw] "
           << "[-[no]label] "
           << "[-[no]fill] "
           << "[-[no]elevation] "
           << "[-[no]stereo] "
           << "[-[no]cut] "
	   << "[-noiso|-iso float] "
           << "[-n-iso int] "
           << "[-n-iso-negative int] "
           << "[-subdivide int] "
           << "[-comp string] "
           << "[-domain string] "
           << "[-proj [string]] "
           << "[-lumped-proj] "
           << "[-round [float]] "
           << "[-velocity|-deformation] "
           << "[-elevation] "
           << "[-scale float] "
           << "[-[no]clean] [-[no]execute] [-[no]verbose] "
           << endl;
      exit (1);
}
// extern:
namespace rheolef {
  template <class T> geo_basic<T,sequential> paraview_isosurface (
    const field_basic<T,sequential>& uh);
  template <class T> field_basic<T,sequential> paraview_plane_cut (
    const field_basic<T,sequential>& uh,
    const point_basic<T>&            origin,
    const point_basic<T>&            normal);
} // namespace rheolef

typedef enum {
        no_render,
        gnuplot_render,
        mayavi_render,
        paraview_render,
        plotmtv_render,
        vtk_render,
        atom_render,
        x3d_render,
        file_render
} render_type;

typedef enum {
        deformation_style,
        velocity_style,
        undef_vector_style
} vector_style_type;

typedef enum {
        show_none,
        show_min,
        show_max,
        show_geo,
} show_type;

int main(int argc, char**argv) {
    environment_option_type eopt;
    //eopt.thread_level = environment_option_type::no_thread;
    environment rheolef (argc, argv, eopt);
    check_macro (communicator().size() == 1, "field: command may be used as mono-process only");
    // ----------------------------
    // NEW: for child processes (TODO: move in environment)
    // ----------------------------
#ifdef TO_CLEAN
    MPI_Comm parent_comm;
    MPI_Comm_get_parent (&parent_comm);
    if (parent_comm == MPI_COMM_NULL) { // no upstream mpi process: create new_np=1 more processes
      cerr << "field: I'm NOT a child..." << endl;
    } else {
      cerr << "field: I'm a child..." << endl;
    }
#endif // _RHEOLEF_HAVE_MPI
    // ----------------------------
    // default options
    // ----------------------------
    string filename = "";
    string name = "output";
    dout.os() << setbasename(name);
    string format = "";
    dout.os() << verbose; bool bverbose = true;
    std::string mark = "";
    render_type render = no_render;
    vector_style_type vector_style = deformation_style;
    field_basic<Float,sequential> uh;
    bool def_plane_cut_opt = false;
    bool def_fill_opt = false;
    bool do_iso3d = false;
    std::string i_comp_name = "";
    size_t i_comp = std::numeric_limits<size_t>::max();
    bool do_proj = false;
    string use_proj_approx = "";
    bool do_lumped_mass = false;
    bool do_cut  = false;
    bool do_round = false;
    std::string reduce_to_domain = "";
    Float round_prec = sqrt(std::numeric_limits<Float>::epsilon());
    show_type show = show_none;
    dout.os() << label;
    // ----------------------------
    // scan the command line
    // ----------------------------
    for (int i = 1; i < argc; i++) {

	// general options:
             if (strcmp (argv[i], "-clean") == 0)     dout.os() << clean;
        else if (strcmp (argv[i], "-noclean") == 0)   dout.os() << noclean;
        else if (strcmp (argv[i], "-execute") == 0)   dout.os() << execute;
        else if (strcmp (argv[i], "-noexecute") == 0) dout.os() << noexecute;
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; dout.os() << verbose; }
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; dout.os() << noverbose; }
        else if (strcmp (argv[i], "-I") == 0)         {
            if (i+1 == argc) { cerr << "geo -I: option argument missing" << endl; usage(); }
            append_dir_to_rheo_path (argv[++i]);
        }
        else if (argv [i][0] == '-' && argv [i][1] == 'I')  { append_dir_to_rheo_path (argv[i]+2); }
	// output file option:
        else if (strcmp (argv[i], "-field") == 0)           { dout.os() << rheo;     render = file_render; }
        else if (strcmp (argv[i], "-text") == 0)            { dout.os() << rheo;     render = file_render; }
        else if (strcmp (argv[i], "-gmsh") == 0)            { dout.os() << gmsh;     render = file_render; }
        else if (strcmp (argv[i], "-gmsh-pos") == 0)        { dout.os() << gmsh_pos; render = file_render; }
        else if (strcmp (argv[i], "-bamg-bb") == 0)         { dout.os() << bamg;     render = file_render; }
        else if (strcmp (argv[i], "-name") == 0)            {
            if (i+1 == argc) { std::cerr << "field -name: option argument missing" << std::endl; usage(); }
            name = argv[++i];
	}
        // render spec:
        else if (strcmp (argv[i], "-gnuplot") == 0)         { dout.os() << gnuplot;  render = gnuplot_render; }
        else if (strcmp (argv[i], "-mayavi")  == 0)         { dout.os() << mayavi;   render = mayavi_render; }
        else if (strcmp (argv[i], "-paraview")  == 0)       { dout.os() << paraview; render = paraview_render; }

	// filter 
        else if (strcmp (argv[i], "-proj") == 0)            {
	    do_proj = true;
            if (i+1 < argc && argv[i+1][0] != '-') {
              use_proj_approx = argv[++i];
            }
	}
        else if (strcmp (argv[i], "-lumped-proj") == 0)     {
	    do_proj = true;
	    do_lumped_mass = true;
	    use_proj_approx = "P1";
        }
        // inquire option
        else if (strcmp (argv[i], "-min") == 0)       { show = show_min; }
        else if (strcmp (argv[i], "-max") == 0)       { show = show_max; }
        else if (strcmp (argv[i], "-get-geo") == 0)   { show = show_geo; }

	// render options:
        else if (strcmp (argv[i], "-velocity") == 0)        { dout.os() << velocity;    vector_style = velocity_style; }
        else if (strcmp (argv[i], "-deformation") == 0)     { dout.os() << deformation; vector_style = deformation_style; }
        else if (strcmp (argv[i], "-fill") == 0)            { dout.os() << fill;   def_fill_opt = true; }
        else if (strcmp (argv[i], "-nofill") == 0)          { dout.os() << nofill; def_fill_opt = false; }
        else if (strcmp (argv[i], "-elevation") == 0)       { dout.os() << elevation; }
        else if (strcmp (argv[i], "-noelevation") == 0)     { dout.os() << noelevation; }
        else if (strcmp (argv[i], "-color") == 0)           { dout.os() << color; }
        else if (strcmp (argv[i], "-gray") == 0)            { dout.os() << gray; }
        else if (strcmp (argv[i], "-black-and-white") == 0) { dout.os() << black_and_white; }
        else if (strcmp (argv[i], "-bw") == 0)              { dout.os() << black_and_white; }
        else if (strcmp (argv[i], "-label") == 0)           { dout.os() << label; }
        else if (strcmp (argv[i], "-nolabel") == 0)         { dout.os() << nolabel; }
        else if (strcmp (argv[i], "-stereo") == 0)          { dout.os() << stereo; 
							      if (render != mayavi_render && 
                                                                  render != paraview_render) {
                                                                dout.os() << paraview; 
							        render  = paraview_render;
                                                              }
                                                            }
        else if (strcmp (argv[i], "-nostereo") == 0)        { dout.os() << nostereo; }
        else if (strcmp (argv[i], "-volume") == 0)          { dout.os() << paraview << volume;   
							      render  = paraview_render; }
        else if (strcmp (argv[i], "-novolume") == 0)        { dout.os() << novolume; }
        else if (strcmp (argv[i], "-cut") == 0)             { do_cut = true; }
        else if (strcmp (argv[i], "-nocut") == 0)           { do_cut = false; }
        else if (strcmp (argv[i], "-show-cut") == 0)        { dout.os() << cut;   def_plane_cut_opt = true; }
        else if (strcmp (argv[i], "-no-show-cut") == 0)     { dout.os() << nocut; def_plane_cut_opt = true; }
        else if (strcmp (argv[i], "-noiso") == 0)           { dout.os() << noiso; do_iso3d = false; }
        else if (strcmp (argv[i], "-iso") == 0)   {

	    do_iso3d = true;
            dout.os() << iso;
            if (i+1 < argc && is_float(argv[i+1])) {
              Float iso_value = to_float (argv[++i]);
              dout.os() << setisovalue(iso_value);
            }
        } else if (strcmp (argv[i], "-n-iso") == 0)   {

            if (i+1 == argc || !isdigit(argv[i+1][0])) usage();
            size_t idx = atoi (argv[++i]);
            dout.os() << setn_isovalue(idx);

        } else if (strcmp (argv[i], "-n-iso-negative") == 0)   {

            if (i+1 == argc || !isdigit(argv[i+1][0])) usage();
            size_t idx = atoi (argv[++i]);
            dout.os() << setn_isovalue_negative(idx);

        } else if (strcmp (argv[i], "-scale") == 0)   {

            if (i+1 == argc || !is_float(argv[i+1])) usage();
            Float scale = to_float (argv[++i]);
            cout << setvectorscale (scale);

	} else if (strcmp (argv[i], "-image-format") == 0) {
            if (i == argc-1) {
                cerr << "field -image-format: option argument missing" << endl;
                usage();
            }
            string format = argv[++i];
            if (format == "jpeg")       format = "jpg";
            if (format == "postscript") format = "ps";
            dout.os() << setimage_format(format);
        }
        else if (strcmp (argv[i], "-resolution") == 0) {
            if (i == argc-1 || !isdigit(argv[i+1][0])) { std::cerr << "geo -resolution: option argument missing" << std::endl; usage(); }
            size_t nx = atoi(argv[++i]);
            size_t ny = (i < argc-1 && isdigit(argv[i+1][0])) ? atoi(argv[++i]) : nx;
	    dout.os() << setresolution(point_basic<size_t>(nx,ny));
        }
        else if (strcmp (argv[i], "-mark") == 0 || strcmp (argv[i], "-catch") == 0 || strcmp (argv[i], "-catchmark") == 0) {
            if (i == argc-1) {
                cerr << "field -mark: option argument missing" << endl;
                usage();
            }
	    mark = argv[++i];
        }
        else if (strcmp (argv[i], "-subdivide") == 0) {
            if (i == argc-1) { cerr << "field -subdivide: option argument missing" << endl; usage(); }
            size_t nsub = atoi(argv[++i]);
            dout.os() << setsubdivide (nsub);
        }
        else if (strcmp (argv[i], "-comp") == 0)   {

            if (i+1 == argc || !isdigit(argv[i+1][0])) usage();
            i_comp_name = argv[++i];
        }
        else if (strcmp (argv[i], "-domain") == 0)   {

            if (i+1 == argc) usage();
            reduce_to_domain = argv[++i];
        }
        else if (strcmp (argv[i], "-round") == 0)   {

	    do_round = true;
            if (i+1 < argc && is_float(argv[i+1])) {
              round_prec = to_float (argv[++i]);
            }
        }
        else if ((strcmp (argv[i], "-origin") == 0) || (strcmp (argv[i], "-normal") == 0))   {

	    point x;
	    unsigned int io = i;
	    if (i+1 == argc || !is_float(argv[i+1])) {
	    	warning_macro ("invalid argument to `" << argv[i] << "'");
	    	usage();
	    }
	    x[0] = to_float (argv[++i]);
	    if (i+1 < argc && is_float(argv[i+1])) {
	        x[1] = to_float (argv[++i]);
	        if (i+1 < argc && is_float(argv[i+1])) {
	            x[2] = to_float (argv[++i]);
	        }	
	    }	
            if (strcmp (argv[io], "-origin") == 0)   {
		cout << setorigin(x);
	    } else {
		cout << setnormal(x);
	    }
        }
	// input options:
        else if (strcmp (argv[i], "-") == 0) {

	    // field on stdin
            filename = "-";
	}
        else if (argv[i][0] != '-') {
            // input field on file	
	    filename = argv[i];
        }
        else {
            cerr << "field: unknown option `" << argv[i] << "'" << endl;
            usage();
        }
    }
    // ----------------------------
    // field treatment
    // ----------------------------
    if (filename == "") {
        cerr << "field: no input file specified" << endl;
        usage();
    } else if (filename == "-") {
        // field on stdin
        std::string thename;
        if (name != "output") thename = name;
        std::cin >> setbasename(thename);
        if (mark != "") din >> catchmark(mark);
        dout.os() << setbasename(name) << reader_on_stdin;
        din >> uh;
    } else {
        idiststream ids;
        ids.open (filename, "field");
        check_macro(ids.good(), "\"" << filename << "[.field[.gz]]\" not found.");
        if (mark != "") ids >> catchmark(mark);
        ids >> uh;
        std::string root_name = delete_suffix (delete_suffix(filename, "gz"), "field");
        name = get_basename (root_name);
        dout.os() << setbasename(name);
    }
    if (uh.valued_tag() == space_constant::vector) { 
      if (vector_style == velocity_style && (render != mayavi_render && render != paraview_render)) {
        // gnuplot does not support yet velocity style: requieres mayavi or paraview
        dout.os() << paraview; render = paraview_render;
      }
    }
    if (render == no_render) {
        // try to choose the best render from dimension
        if (uh.get_geo().map_dimension() >= 2 ||
            uh.get_geo().dimension() == 3) {
          dout.os() << paraview;
        } else {
          dout.os() << gnuplot;
        }
    }
    if (uh.get_geo().map_dimension() == 3) {
      // default 3D is iso+cut and nofill; default 2D is nocut and fill...
      if (!def_plane_cut_opt)  dout.os() << cut;
      if (!def_fill_opt)       dout.os() << nofill;
    }
    if (i_comp_name != "") {
      // compute uh[i_comp]:
      switch (uh.valued_tag()) {
        case space_constant::vector: {
          i_comp = atoi (i_comp_name.c_str());
          break;
        }
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor: {
	  // string as "00" "21" "22" etc
	  check_macro (i_comp_name.size() == 2, "invalid component `"<<i_comp_name<<"'");
          size_t i = i_comp_name[0] - '0';
          size_t j = i_comp_name[1] - '0';
          i_comp = space_constant::tensor_index (uh.valued_tag(), uh.get_geo().coordinate_system(), i, j);
          break;
        }
        default: error_macro ("component: unexpected "<<uh.valued()<<"-valued field");
      }
      field_basic<Float,sequential> vh = uh[i_comp];
      uh = vh;
    }
    if (reduce_to_domain != "") {
      geo_basic<Float,sequential> dom = uh.get_geo()[reduce_to_domain];
      field_basic<Float,sequential> vh;
      if (uh.get_space().get_numbering().is_discontinuous() && dom.map_dimension()+1 == uh.get_geo().map_dimension()) {
        // brdy trace of a discontinuous field
        space_basic<Float,sequential> Wh (dom, uh.get_approx(), uh.valued());
	uh.get_geo().neighbour_guard();
        vh = interpolate (Wh, uh);
      } else {
        vh = uh[reduce_to_domain];
      }
      uh = vh;
      if (render == file_render) {
	// put field on dout: save also the cutted geo:
	odiststream geo_out (uh.get_geo().name(), "geo");
        geo_out << uh.get_geo();
      }
    }
    if (! do_proj && render == mayavi_render && uh.valued() != "scalar"
        && uh.get_space().get_numbering().has_compact_support_inside_element()) {
      // non-scalar vector & tensor mayavi visualization may be continuous
      warning_macro ("mayavi: perform L2-projection to continuous space");
      do_proj = true;
    }
    if (do_proj && uh.get_space().get_numbering().has_compact_support_inside_element()) {
      const space_basic<Float,sequential>& Uh = uh.get_space();
      size_t k = Uh.degree();
      if (k == 0) k++;
      std::string approx = (use_proj_approx == "") ? "P" + itos(k) : use_proj_approx;
      space_basic<Float,sequential> Vh (uh.get_geo(), approx, uh.valued());
      test_basic<Float,sequential,details::vf_tag_10> u (Uh), v (Vh);
      test_basic<Float,sequential,details::vf_tag_01> vt (Vh);
      integrate_option fopt;
      fopt.lump = do_lumped_mass;
      form_basic<Float,sequential> m, p;
      switch (Uh.valued_tag()) {
        case space_constant::scalar:
	  m = integrate(v*vt, fopt);
	  p = integrate(u*vt);
	  break;
        case space_constant::vector:
	  m = integrate(dot(v,vt), fopt);
	  p = integrate(dot(u,vt));
	  break;
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor:
	  m = integrate(ddot(v,vt), fopt);
	  p = integrate(ddot(u,vt));
	  break;
	default:
	  error_macro ("proj: unexpected valued field: " << Uh.valued());
      }
      solver_basic<Float,sequential> sm = ldlt(m.uu());
      field_basic<Float,sequential> vh (Vh);
      vh.set_u() = sm.solve((p*uh).u());
      uh = vh;
    }
    if (do_cut) {
       point origin = iofem::getorigin(cout);
       point normal = iofem::getnormal(cout);
       uh = paraview_plane_cut (uh, origin, normal);
       if (render == file_render) {
	 // put field on dout: save also the cutted geo:
	 odiststream geo_out (uh.get_geo().name(), "geo");
	 geo_out << uh.get_geo();
       }
    }
    if (do_round) {
      uh = round (uh, round_prec);
    }
    if (show == show_min) {
      cout << std::setprecision(std::numeric_limits<Float>::digits10)
           << uh.min() << endl;
      return 0;
    }
    if (show == show_max) {
      cout << std::setprecision(std::numeric_limits<Float>::digits10)
           << uh.max() << endl;
      return 0;
    }
    if (show == show_geo) {
      cout << uh.get_geo().name() << endl;
      return 0;
    }
    if (do_iso3d) {
      geo_basic<Float,sequential> iso_surface = paraview_isosurface (uh);
      dout << iso_surface;
      return 0;
    }
    // generates marks (mayavi support it now):
    {
      string root_filename = iorheo::getbasename(dout.os());
      string full_mark = (root_filename == "output") ? "" : root_filename;
      if (mark != "") {
        // full_mark = (full_mark == "") ? mark : full_mark + "." + mark;
        full_mark = mark;
      }
      if (i_comp_name != "") { 
         full_mark = (full_mark == "") ? "value" : full_mark;
         full_mark = full_mark + "[" + i_comp_name + "]";
      }
      if (reduce_to_domain != "") {
         full_mark = (full_mark == "") ? "value" : full_mark;
         full_mark = full_mark + "[" + reduce_to_domain + "]";
      }
      dout.os() << setmark (full_mark);
    }
    // final output:
    dout << uh;
    // ----------------------------
    // NEW: for child processes (TODO: move in environment)
    // ----------------------------
#ifdef TO_CLEAN
    if (parent_comm != MPI_COMM_NULL) {
      cerr << "field: I'm a child and I send signal to parents" << endl;
      MPI_Barrier (parent_comm);
    }
#endif // _RHEOLEF_HAVE_MPI
}
