///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "Pk_numbering.h"
#include "rheolef/geo.h"
#include "rheolef/rheostream.h"
namespace rheolef {
using namespace std;

template <class T, class M>
numbering_Pk<T,M>::numbering_Pk (std::string name)
 : base(),
   _loc_ndof_by_variant()
{
  base::_basis = basis_basic<T> (name);
  check_macro (base::_basis.is_continuous(),
        "invalid Pk continuous basis \""<<base::_basis.name()<<"\"");
  reference_element::init_local_nnode_by_variant (base::_basis.degree(), _loc_ndof_by_variant);
}
template <class T, class M>
void
numbering_Pk<T,M>::set_degree (size_type degree) const
{
  // TODO: virer set_degree et passer en nocopy
  string            family_name = base::_basis.family_name();
  basis_option sopt        = base::_basis.option();
  std::string name = basis_rep<T>::standard_naming (family_name, degree, sopt);
  base::_basis = basis_basic<T> (name);
  reference_element::init_local_nnode_by_variant (degree, _loc_ndof_by_variant);
}
template <class T, class M>
typename numbering_Pk<T,M>::size_type
numbering_Pk<T,M>::degree () const
{
  return base::_basis.degree();
}
template <class T, class M>
std::string
numbering_Pk<T,M>::name() const
{
  return base::_basis.name();
}
template <class T, class M>
bool
numbering_Pk<T,M>::is_continuous () const
{
  return true;
}
template <class T, class M>
bool
numbering_Pk<T,M>::has_compact_support_inside_element () const
{
  return false;
}
template <class T, class M>
typename numbering_Pk<T,M>::size_type
numbering_Pk<T,M>::ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
  size_type ndof = 0;
  for (size_type variant = 0; variant < reference_element::max_variant; variant++) { 
    ndof += gs.ownership_by_variant [variant].size() * _loc_ndof_by_variant [variant];
  }
  return ndof;
}
template <class T, class M>
typename numbering_Pk<T,M>::size_type
numbering_Pk<T,M>::dis_ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
  size_type dis_ndof = 0;
  for (size_type variant = 0; variant < reference_element::max_variant; variant++) { 
    dis_ndof += gs.ownership_by_variant [variant].dis_size() * _loc_ndof_by_variant [variant];
  }
  return dis_ndof;
}
template <class T, class M>
void
numbering_Pk<T,M>::dis_idof (
        const geo_size&       gs, 
	const geo_element&    K, 
	std::vector<size_type>& dis_idof1) const
{
  dis_idof1.resize (reference_element::n_node (K.variant(), degree()));
  // PARANO mode:
  std::fill (dis_idof1.begin(), dis_idof1.end(), std::numeric_limits<size_type>::max());

  for (size_type subgeo_variant = 0, variant = K.variant(); subgeo_variant <= variant; subgeo_variant++) {
    size_type subgeo_dim   = reference_element::dimension (subgeo_variant);
    size_type loc_sub_ndof = _loc_ndof_by_variant [subgeo_variant];
    for (size_type first_loc_idof = reference_element::first_inod_by_variant (variant, degree(), subgeo_variant),
                    last_loc_idof = reference_element:: last_inod_by_variant (variant, degree(), subgeo_variant), 
                         loc_idof = first_loc_idof;
		         loc_idof <  last_loc_idof; loc_idof++) {
      // 1) local loc_igev on subgeo
      size_type loc_igev   = (loc_idof - first_loc_idof) / loc_sub_ndof;
      size_type loc_igev_j = (loc_idof - first_loc_idof) % loc_sub_ndof;
      // 2) then compute dis_ige, global distributed by dimension; computation depends upon the subgeo dimension:
      size_type dis_ige;
      switch (subgeo_dim) {
        case 0: {
          // convert node numbering to vertex numbering, for geo order > 1
          size_type loc_inod = loc_idof; // only one dof per vertex
          size_type dis_inod = K [loc_inod];
          size_type iproc           = gs.node_ownership.find_owner (dis_inod);
          size_type first_dis_inod  = gs.node_ownership.first_index(iproc);
          assert_macro (dis_inod >= first_dis_inod, "invalid index");
          size_type inod = dis_inod - first_dis_inod;
          size_type first_dis_iv    = gs.ownership_by_variant [reference_element::p].first_index(iproc);
          size_type iv = inod;
          size_type dis_iv = first_dis_iv + iv;
          dis_ige = dis_iv;
	  break;
        }
        case 1: {
          loc_igev_j = geo_element::fix_edge_indirect (K, loc_igev, loc_igev_j, degree());
          size_type loc_ige = loc_igev;
          dis_ige = K.edge(loc_ige);
	  break;
        }
        case 2: {
          size_type loc_ige = loc_igev;
	  if (subgeo_variant == reference_element::t) {
             loc_igev_j = geo_element::fix_triangle_indirect (K, loc_igev, loc_igev_j, degree());
          } else {
             size_type loc_ntri = (K.variant() == reference_element::P) ? 2 : 0;
             loc_ige += loc_ntri;
             loc_igev_j = geo_element::fix_quadrangle_indirect (K, loc_ige, loc_igev_j, degree());
          }
          dis_ige = K.face(loc_ige);
	  break;
        }
        case 3:
        default: {
          dis_ige = K.dis_ie();
        }
      }
      // 3) then compute dis_igev, distributed by variant
      size_type iproc         = gs.ownership_by_dimension [subgeo_dim].find_owner (dis_ige);
      size_type first_dis_ige = gs.ownership_by_dimension [subgeo_dim].first_index(iproc);
      assert_macro (dis_ige >= first_dis_ige, "invalid index");
      size_type ige = dis_ige - first_dis_ige;
      size_type igev = ige;
      for (size_type prev_subgeo_variant = reference_element::first_variant_by_dimension(subgeo_dim);
                     prev_subgeo_variant < subgeo_variant; 
                     prev_subgeo_variant++) {
        size_type nprev = gs.ownership_by_variant [prev_subgeo_variant].size(iproc);
        assert_macro (ige >= nprev, "invalid index");
        igev -= nprev;
      } 
      size_type first_dis_igev = gs.ownership_by_variant [subgeo_variant].first_index(iproc);
      size_type dis_igev = first_dis_igev + igev;
      // 3) finally compute dis_idof
      size_type dis_idof = 0;
      for (size_type prev_subgeo_variant = 0;
                     prev_subgeo_variant < subgeo_variant; 
                     prev_subgeo_variant++) {
        dis_idof += gs.ownership_by_variant [prev_subgeo_variant]. last_index(iproc) * _loc_ndof_by_variant [prev_subgeo_variant];
      }
      dis_idof +=  dis_igev*_loc_ndof_by_variant [subgeo_variant] + loc_igev_j;
      for (size_type next_subgeo_variant = subgeo_variant+1;
                     next_subgeo_variant < reference_element::max_variant; 
                     next_subgeo_variant++) {
        dis_idof += gs.ownership_by_variant [next_subgeo_variant].first_index(iproc) * _loc_ndof_by_variant [next_subgeo_variant];
      }
      assert_macro (loc_idof < dis_idof1.size(), "invalid index");
      dis_idof1 [loc_idof] = dis_idof;
    }
  }
}
// -------------------------------------------------------------
// permut to/from ios dof numbering, for distributed environment
// -------------------------------------------------------------
template <class T, class M>
static 
void
Pk_set_ios_permutations (
    const geo_basic<T,M>& omega,
    size_t			       degree,
    disarray<distributor::size_type,M>&   idof2ios_dis_idof,
    disarray<distributor::size_type,M>&   ios_idof2dis_idof)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
static 
void
Pk_set_ios_permutations (
    const geo_basic<T,distributed>&              omega,
    size_t					 degree,
    disarray<distributor::size_type,distributed>&   idof2ios_dis_idof,
    disarray<distributor::size_type,distributed>&   ios_idof2dis_idof)
{
  typedef size_t size_type;
  std::array<size_type,reference_element::max_variant> loc_ndof_by_variant ;
  reference_element::init_local_nnode_by_variant (degree, loc_ndof_by_variant);
  omega.set_ios_permutation (loc_ndof_by_variant, idof2ios_dis_idof);
  size_type dis_ndof = idof2ios_dis_idof.ownership().dis_size();
  distributor ios_dof_ownership (dis_ndof, idof2ios_dis_idof.comm(), distributor::decide);
  ios_idof2dis_idof.resize (ios_dof_ownership, std::numeric_limits<size_type>::max());
  idof2ios_dis_idof.reverse_permutation (ios_idof2dis_idof);
}
#endif // _RHEOLEF_HAVE_MPI
template <class T, class M>
void
numbering_Pk<T,M>::set_ios_permutations (
    const geo_basic<T,M>& omega,
    disarray<size_type,M>&   idof2ios_dis_idof,
    disarray<size_type,M>&   ios_idof2dis_idof) const
{
    Pk_set_ios_permutations (omega, degree(), idof2ios_dis_idof, ios_idof2dis_idof);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

template class numbering_Pk<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class numbering_Pk<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
