///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/basis_on_pointset.h"
namespace rheolef {

// -----------------------------------------------------------------------
// utility: local class:
// uniformize hat_node access for quadrature or Lagrange basis
// => allows code merge for basis_on_pointset
// -----------------------------------------------------------------------
template<class T>
class pointset_on_geo : public std::vector<point_basic<T> > {
  typedef std::vector<point_basic<T> > base;
public:
  pointset_on_geo ();
  pointset_on_geo (const quadrature<T>&     quad, reference_element hat_K);
  void initialize (const quadrature<T>&     quad, reference_element hat_K);
};
template<class T>
pointset_on_geo<T>::pointset_on_geo ()
 : base ()
{
}
template<class T>
void
pointset_on_geo<T>::initialize (const quadrature<T>& quad, reference_element hat_K)
{
  base::resize (quad.size(hat_K));
  typename base::iterator dest = base::begin();
  for (typename quadrature<T>::const_iterator
      iter = quad.begin(hat_K),
      last = quad.end  (hat_K); iter != last; ++iter, ++dest) {
    *dest = (*iter).x;
  }
}
template<class T>
pointset_on_geo<T>::pointset_on_geo (const quadrature<T>& quad, reference_element hat_K)
 : base (quad.size(hat_K))
{
  initialize(quad, hat_K);
}
// ----------------------------------------------------------
// cstors
// ----------------------------------------------------------
template<class T>
basis_on_pointset_rep<T>::basis_on_pointset_rep ()
 : _b(),
   _mode(max_mode),
   _restricted_on_side(false),
   _specific_value(false),
   _quad(),
   _nb(),
   _basis_size(),
   _val(),
   _grad_val(),
   _spec_val(),
   _spec_grad_val(),
   _sid_val(),
   _sid_grad_val(),
   _initialized(),
   _grad_initialized(),
   _sid_initialized(),
   _curr_K_variant(std::numeric_limits<size_type>::max()),
   _curr_q        (std::numeric_limits<size_type>::max()),
   _curr_sid()
{
   _initialized.fill(false);
   _grad_initialized.fill(false);
   _sid_initialized.fill(false);
}
template<class T>
basis_on_pointset_rep<T>::basis_on_pointset_rep (const basis_on_pointset_rep<T>& x)
 : _b(x._b),
   _mode(x._mode),
   _restricted_on_side(x._restricted_on_side),
   _specific_value(x._specific_value),
   _quad(x._quad),
   _nb(x._nb),
   _basis_size(x._basis_size),
   _val(x._val),
   _grad_val(x._grad_val),
   _spec_val(x._spec_val),
   _spec_grad_val(x._spec_grad_val),
   _sid_val(x._sid_val),
   _sid_grad_val(x._sid_grad_val),
   _initialized     (x._initialized),
   _grad_initialized(x._grad_initialized),
   _sid_initialized (x._sid_initialized),
   _curr_K_variant  (x._curr_K_variant),
   _curr_q          (x._curr_q),
   _curr_sid        (x._curr_sid)
{
  trace_macro("PHYSICAL COPY***************");
}
template<class T>
basis_on_pointset_rep<T>&
basis_on_pointset_rep<T>::operator= (const basis_on_pointset_rep<T>& x)
{
   _b = x._b;
   _mode = x._mode;
   _restricted_on_side = x._restricted_on_side;
   _specific_value = x._specific_value;
   _quad = x._quad;
   _nb = x._nb;
   _basis_size = x._basis_size;
   _val = x._val;
   _grad_val = x._grad_val;
   _spec_val = x._spec_val;
   _spec_grad_val = x._spec_grad_val;
   _sid_val = x._sid_val;
   _sid_grad_val = x._sid_grad_val;
   _initialized      = x._initialized;
   _grad_initialized = x._grad_initialized;
   _sid_initialized  = x._sid_initialized;
   _curr_K_variant   = x._curr_K_variant;
   _curr_q           = x._curr_q;
   _curr_sid         = x._curr_sid;
  return *this;
}
#ifdef TO_CLEAN
template<class T>
basis_on_pointset_rep<T>::basis_on_pointset_rep (const quadrature<T>& quad, const basis_basic<T>& b)
 : _b(b),
   _mode(quad_mode),
   _restricted_on_side(false),
   _specific_value(false),
   _quad(quad),
   _nb(),
   _basis_size(),
   _val(),
   _grad_val(),
   _spec_val(),
   _spec_grad_val(),
   _sid_val(),
   _sid_grad_val(),
   _initialized     (reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _sid_initialized (reference_element::max_variant, false),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q         (std::numeric_limits<size_type>::max()),
   _curr_sid       ()
{
}
template<class T>
basis_on_pointset_rep<T>::basis_on_pointset_rep (const basis_basic<T>& nb, const basis_basic<T>& b)
 : _b(b),
   _mode(nodal_mode),
   _restricted_on_side(false),
   _specific_value(false),
   _quad(),
   _nb(nb),
   _basis_size(),
   _val(),
   _grad_val(),
   _spec_val(),
   _spec_grad_val(),
   _sid_val(),
   _sid_grad_val(),
   _initialized     (reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _sid_initialized (reference_element::max_variant, false),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q         (std::numeric_limits<size_type>::max()),
   _curr_sid       ()
{
}
#endif // TO_CLEAN
template<class T>
void
basis_on_pointset_rep<T>::set (const quadrature<T>& quad, const basis_basic<T>& b)
{
  // data was not defined on this quadrature formulae : reset
  _b  = b;
  _mode = quad_mode;
  _quad = quad;
  _restricted_on_side = false;
  _specific_value = false;
  _initialized.fill(false);
  _grad_initialized.fill(false);
  _sid_initialized.fill(false);
}
template<class T>
void
basis_on_pointset_rep<T>::set (const basis_basic<T>& nb, const basis_basic<T>& b)
{
  // data was not defined on this quadrature formulae : reset
  _b = b;
  _mode = nodal_mode;
  _nb = nb;
  _restricted_on_side = false;
  _specific_value = false;
  _initialized.fill(false);
  _grad_initialized.fill(false);
  _sid_initialized.fill(false);
}
// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
template<class T>
void 
basis_on_pointset_rep<T>::_initialize (reference_element hat_K) const
{
  check_macro (!_specific_value, "unsupported eval in specific mode");
  reference_element::variant_type K_variant = hat_K.variant();
  _basis_size[K_variant] = _b.size (hat_K);
  if (_mode == quad_mode) {
    pointset_on_geo<T> hat_node;
    hat_node.initialize (_quad, hat_K);
    _b.eval (hat_K, hat_node, _val[K_variant]);
  } else {
    const std::vector<point_basic<T> >& hat_node = _nb.hat_node (hat_K);
    _b.eval (hat_K, hat_node, _val[K_variant]);
  }
  _restricted_on_side = false;
  _initialized [K_variant] = true;
}
template<class T>
void 
basis_on_pointset_rep<T>::_grad_initialize (reference_element hat_K) const
{
  check_macro (!_specific_value, "unsupported eval in specific mode");
  reference_element::variant_type K_variant = hat_K.variant();
  _basis_size[K_variant] = _b.size (hat_K);
  if (_mode == quad_mode) {
    pointset_on_geo<T> hat_node;
    hat_node.initialize (_quad, hat_K);
    _b.grad_eval (hat_K, hat_node, _grad_val[K_variant]);
  } else {
    const std::vector<point_basic<T> >& hat_node = _nb.hat_node (hat_K);
    _b.grad_eval (hat_K, hat_node, _grad_val[K_variant]);
  }
  _restricted_on_side = false;
  _grad_initialized [K_variant] = true;
}
// ----------------------------------------------------------
// side restriction (for DG)
// ----------------------------------------------------------
// We have a side tilde_K on an element tilde_L
// and a Piola transform from hat_K to tilde_L
// We have a quadrature formula or a point-set on hat_K:
//	(hat_xq) q=0..nq
// and we are able to transform it on tilde_K subset partial tilde_L:
//	(tilde_xq) q=0..nq
// We want to evaluate on this pointset a basis "b" of tilde_L.
// 
// basis_on_pointset<T>::init_all_sides (reference_element tilde_L)
//   => loop on all possible tilde_K side of tilde_L 
//     build the (tilde_xq)q set and evaluate on it the basis "b"
//     (possibly, have to reverse or reorder (ie rotate in 3d) the tilde_xq set)
//     store all results contiguously, side by side, in the same array
//
// basis_on_pointset<T>::restrict_on_side (reference_element tilde_L, const side_information_type& sid)
//   => if not initialized, call init_all_sides
//      then, from the current side index, compute begin and end pointers
//      that will be used by accessors
// 
// AVANTAGES:
//  - transparent : no changes in downstream codes (field_evaluate.cc etc)
//  - fast : the basis is evaluate only once for each reference element
// TECHNIQUE:
//  possibly, have to reverse or reorder (ie rotate in 3d) the tilde_xq set
//  - first, do it in 2d, with only an optional reverse ordering
//    when the reference side has the opposite orientation.
//    This is not known at "init_all_sides()" call, as it is given later
//    by the "side_information_type" argument of "restrict_on_side()" call.
//    So, we cannot use begin() and end() iterators.
//    Thus, the random accessor is preferable: "value()"
//    and any call to iterators should check that we are not in the "side" mode.
//  - second, extend it to 3d: the reordering is more complex when there is shift
//    * nodes on the vertices are rotated
//    * nodes on the boundary edges of a face are shifted: iedge=(iedge0+shift)%nedge
//    * nodes inside the face are completely reordered
//    => a table of reordering is required, it should be build by restrict_on_side()
//    Note that there is a finite number of possible shift+reverse combinations
//    and we can enumerate it and compute one time for all the all reordering
//    at init_all_sides(). Then, at restrict_on_side(), we have just to compute
//    the index associated to the reordering.
// IMPLEMENTATION:
//  - may have a different data structure for eval on volume or side 
//          _val(),      _grad_val(),
//     _side_val(), _side_grad_val(),
//    or with a vol/side flag:
//      mutable std::array<2, array<arma::Col<T>,
//        reference_element::max_variant> >        _val;
//    =>
//        _val [vol_side_flag] [hat_K] [idof]
//    motivation: when the test::_bops is used for volume, then for side,
//    and then volume again, we do not have to reinitialize.
//    perhaps, it should solve the BUG_TEST_CONST ?

template<class T>
void
basis_on_pointset_rep<T>::_sid_initialize (reference_element tilde_L, const side_information_type& sid) const
{
  // transform side hat_node on hat_K into tilde_node on tilde_K, the side of tilde_L:
  // => orient and shift varies for each side 
  // the transformed nodes from hat_K to tilde_K depends upon (loc_isid,orient,shift)
  size_type ori_idx = (sid.orient == 1) ? 0 : 1;
  reference_element::variant_type L_variant = tilde_L.variant();
  _basis_size[L_variant] = _b.size (tilde_L);                      // evaluate basis for tilde_L
  if (_mode == quad_mode) {
    pointset_on_geo<T> hat_node;
    hat_node.initialize (_quad, sid.hat); // ...on the side tilde_K, at its quad nodes
    for (size_type q = 0, nq = hat_node.size(); q < nq; ++q) {
      hat_node[q] = reference_element_face_transformation (tilde_L, sid, hat_node[q]);
    }
    _b.eval      (tilde_L, hat_node,      _sid_val[L_variant][sid.loc_isid][ori_idx][sid.shift]);
    _b.grad_eval (tilde_L, hat_node, _sid_grad_val[L_variant][sid.loc_isid][ori_idx][sid.shift]);
  } else {
    const std::vector<point_basic<T> >& hat_node = _nb.hat_node (sid.hat); // ... on its Lagrange nodes.
    std::vector<point_basic<T> > new_hat_node (hat_node.size());
    for (size_type q = 0, nq = hat_node.size(); q < nq; ++q) {
      new_hat_node[q] = reference_element_face_transformation (tilde_L, sid, hat_node[q]);
    }
    _b.eval      (tilde_L, new_hat_node,      _sid_val[L_variant][sid.loc_isid][ori_idx][sid.shift]);
    _b.grad_eval (tilde_L, new_hat_node, _sid_grad_val[L_variant][sid.loc_isid][ori_idx][sid.shift]);
  }
}
template<class T>
void
basis_on_pointset_rep<T>::_sid_initialize (reference_element tilde_L) const
{
  size_type d = tilde_L.dimension();
  size_type nsid = tilde_L.n_subgeo (d-1);
  side_information_type sid;
  sid.dim = d-1;
  for (sid.loc_isid = 0; sid.loc_isid < nsid; ++sid.loc_isid) {
    sid.n_vertex = tilde_L.subgeo_size (sid.dim, sid.loc_isid);
    sid.hat.set_variant (sid.n_vertex, sid.dim);
    int last_orient = (sid.dim == 0) ? 1 : -1; // no negative orient for 0d side
    for (sid.orient = 1; sid.orient >= last_orient; sid.orient -= 2) {
      size_type n_shift = (d == 3) ? nsid : 1;
      for (sid.shift = 0; sid.shift < n_shift; ++sid.shift) {
        _sid_initialize (tilde_L, sid);
      }
    }
  }
  _sid_initialized [tilde_L.variant()] = true;
}
template<class T>
void
basis_on_pointset_rep<T>::restrict_on_side (reference_element tilde_L, const side_information_type& sid)
{
  if (!_sid_initialized [tilde_L.variant()]) {
    _sid_initialize (tilde_L);
  }
  _restricted_on_side = true;
  _specific_value = false;
  _curr_K_variant = tilde_L.variant();
  _curr_sid       = sid;
  _curr_q         = 0;
}
// ----------------------------------------------------------
// accessors
// ----------------------------------------------------------
template<class T>
typename basis_on_pointset_rep<T>::size_type
basis_on_pointset_rep<T>::pointset_size (reference_element hat_K) const
{
  if (!_restricted_on_side) {
    if (_mode == quad_mode) return _quad.size (hat_K);
    else                    return _nb.size   (hat_K);
  } else { // on side of hat_K:
    // note(important): hat_K.variant() != _curr_K_variant in general => use _curr_sid.hat
    if (_mode == quad_mode) return _quad.size (_curr_sid.hat);
    else                    return _nb.size   (_curr_sid.hat);
  }
}
// -------------------
// all q-node eval
// -------------------
// basis eval
template<class T>
void
basis_on_pointset_rep<T>::evaluate (reference_element hat_K, size_type q) const
{
  _specific_value = false;
  _curr_K_variant = hat_K.variant();
  _curr_q         = q;
  if (!_restricted_on_side) {
    if (!_initialized [hat_K.variant()]) {
         _initialize  (hat_K);
    }
  } else {
    if (!_sid_initialized [hat_K.variant()]) {
         _sid_initialize  (hat_K);
    }
  }
}
// grad basis eval
template<class T>
void
basis_on_pointset_rep<T>::evaluate_grad (reference_element hat_K, size_type q) const
{
  _specific_value = false;
  _curr_K_variant = hat_K.variant();
  _curr_q         = q;
  if (!_restricted_on_side) {
    if (!_grad_initialized [hat_K.variant()]) {
         _grad_initialize  (hat_K);
    }
  } else {
    if (!_sid_initialized [hat_K.variant()]) {
         _sid_initialize  (hat_K);
    }
  }
}
// -------------------
// specific node eval
// -------------------
template<class T>
void
basis_on_pointset_rep<T>::evaluate (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  check_macro (!_restricted_on_side, "evaluate(hat_x): not yet on side");
  _restricted_on_side = false;
  _specific_value = true;
  _curr_K_variant = hat_K.variant();
  _curr_q         = 0;
  _basis_size[_curr_K_variant] = _b.size (hat_K);
  if (_basis_size[_curr_K_variant] != _spec_val[_curr_K_variant].size ()) {
    _spec_val[_curr_K_variant].resize (_basis_size[_curr_K_variant]); 
  }
  _b.eval (hat_K, hat_xq, _spec_val[_curr_K_variant]);
}
template<class T>
void
basis_on_pointset_rep<T>::evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  check_macro (!_restricted_on_side, "evaluate_grad(hat_x): not yet on side");
  _restricted_on_side = false;
  _specific_value = true;
  _curr_K_variant = hat_K.variant();
  _curr_q         = 0;
  _basis_size[_curr_K_variant] = _b.size (hat_K);
  if (_basis_size[_curr_K_variant] != _spec_grad_val[_curr_K_variant].size ()) {
    _spec_grad_val[_curr_K_variant].resize (_basis_size[_curr_K_variant]); 
  }
  _b.grad_eval (hat_K, hat_xq, _spec_grad_val[_curr_K_variant]);
}
// ----------------------------------------------------------
// here: could be inlined when code is stabilized
// ----------------------------------------------------------
// -------------------
// basis accessor
// -------------------
template<class T>
typename basis_on_pointset_rep<T>::const_iterator
basis_on_pointset_rep<T>::begin() const
{
  if (!_restricted_on_side) {
    if (!_specific_value) {
      return _val[_curr_K_variant].begin()
             + _basis_size[_curr_K_variant]*_curr_q;
    } else {
      return _spec_val[_curr_K_variant].begin()
             + _basis_size[_curr_K_variant]*_curr_q;
    }
  } else {
    size_type ori_idx = (_curr_sid.orient == 1) ? 0 : 1;
    return _sid_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift].begin()
         + _basis_size[_curr_K_variant]*_curr_q;
  }
}
template<class T>
typename basis_on_pointset_rep<T>::const_iterator
basis_on_pointset_rep<T>::end() const
{
  return begin() + _basis_size[_curr_K_variant];
}
// -------------------
// grad basis accessor
// -------------------
template<class T>
typename basis_on_pointset_rep<T>::const_iterator_grad
basis_on_pointset_rep<T>::begin_grad() const
{
  if (!_restricted_on_side) {
   if (!_specific_value) {
    return _grad_val[_curr_K_variant].begin()
         + _basis_size[_curr_K_variant]*_curr_q;
   } else {
    return _spec_grad_val[_curr_K_variant].begin()
         + _basis_size[_curr_K_variant]*_curr_q;
   }
  } else {
    size_type ori_idx = (_curr_sid.orient == 1) ? 0 : 1;
    return _sid_grad_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift].begin()
         + _basis_size[_curr_K_variant]*_curr_q;
  }
}
template<class T>
typename basis_on_pointset_rep<T>::const_iterator_grad
basis_on_pointset_rep<T>::end_grad() const
{
  return begin_grad() + _basis_size[_curr_K_variant];
}
// -------------------
// random accessors
// -------------------
template<class T>
const T&
basis_on_pointset_rep<T>::value (size_type loc_idof) const
{
  size_type iloc = _basis_size[_curr_K_variant]*_curr_q + loc_idof;
  if (!_restricted_on_side) {
   if (!_specific_value) {
#ifdef  _RHEOLEF_PARANO
    size_type loc_ndof = _basis_size[_curr_K_variant];
    size_type nq       = pointset_size (_curr_K_variant);
    size_type nloc     = _val[_curr_K_variant].size();
    if (iloc >= nloc) {
      warning_macro ("q = "<<_curr_q<<", nq="<<nq);
      warning_macro ("loc_idof = "<<loc_idof<<", loc_ndof="<<loc_ndof);
      warning_macro ("iloc = "<<iloc<<", nloc="<<nloc);
    }
    check_macro (_curr_q  < nq,       "index q="       << _curr_q  << " out of range [0:" << nq<<"[");
    check_macro (loc_idof < loc_ndof, "index loc_idof="<< loc_idof << " out of range [0:" << loc_ndof<<"[");
    check_macro (iloc     < nloc,     "index iloc="    << iloc     << " out of range [0:" << nloc<<"[");
#endif // _RHEOLEF_PARANO
    return _val[_curr_K_variant][iloc];
   } else {
    return _spec_val[_curr_K_variant][iloc];
   }
  } else {
    size_type ori_idx = (_curr_sid.orient == 1) ? 0 : 1;
#ifdef  _RHEOLEF_PARANO
    size_type loc_ndof = _basis_size[_curr_K_variant];
    size_type nq       = pointset_size (_curr_K_variant);
    size_type nloc     = _sid_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift].size();
    if (iloc >= nloc) {
      warning_macro ("value: on side: loc_idof="<<loc_idof<<"...");
      warning_macro ("q = "<<_curr_q<<", nq="<<nq);
      warning_macro ("loc_idof = "<<loc_idof<<", loc_ndof="<<loc_ndof);
      warning_macro ("iloc = "<<iloc<<", nloc="<<nloc);
    }
    check_macro (_curr_q  < nq,       "index q="       << _curr_q  << " out of range [0:" << nq<<"[");
    check_macro (loc_idof < loc_ndof, "index loc_idof="<< loc_idof << " out of range [0:" << loc_ndof<<"[");
    check_macro (iloc     < nloc,     "index iloc="    << iloc     << " out of range [0:" << nloc<<"[");
#endif // _RHEOLEF_PARANO
    return _sid_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift][iloc];
  }
}
template<class T>
const point_basic<T>&
basis_on_pointset_rep<T>::grad_value (size_type loc_idof) const
{
  size_type iloc = _basis_size[_curr_K_variant]*_curr_q + loc_idof;
  if (!_restricted_on_side) {
   if (!_specific_value) {
#ifdef  _RHEOLEF_PARANO
    size_type loc_ndof = _basis_size[_curr_K_variant];
    size_type nq       = pointset_size (_curr_K_variant);
    size_type nloc     = _grad_val[_curr_K_variant].size();
    if (iloc >= nloc) {
      warning_macro ("q = "<<_curr_q<<", nq="<<nq);
      warning_macro ("loc_idof = "<<loc_idof<<", loc_ndof="<<loc_ndof);
      warning_macro ("iloc = "<<iloc<<", nloc="<<nloc);
    }
    check_macro (_curr_q  < nq,       "index q="       << _curr_q  << " out of range [0:" << nq<<"[");
    check_macro (loc_idof < loc_ndof, "index loc_idof="<< loc_idof << " out of range [0:" << loc_ndof<<"[");
    check_macro (iloc     < nloc,     "index iloc="    << iloc     << " out of range [0:" << nloc<<"[");
#endif // _RHEOLEF_PARANO
    return _grad_val[_curr_K_variant][iloc];
   } else {
    return _spec_grad_val[_curr_K_variant][iloc];
   }
  } else {
    size_type ori_idx = (_curr_sid.orient == 1) ? 0 : 1;
#ifdef  _RHEOLEF_PARANO 
    size_type loc_ndof = _basis_size[_curr_K_variant];
    size_type nq       = pointset_size (_curr_K_variant);
    size_type nloc     = _sid_grad_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift].size();
    if (iloc >= nloc) {
      warning_macro ("value: on side: loc_idof="<<loc_idof<<"...");
      warning_macro ("q = "<<_curr_q<<", nq="<<nq);
      warning_macro ("loc_idof = "<<loc_idof<<", loc_ndof="<<loc_ndof);
      warning_macro ("iloc = "<<iloc<<", nloc="<<nloc);
    }
    check_macro (_curr_q  < nq,       "index q="       << _curr_q  << " out of range [0:" << nq<<"[");
    check_macro (loc_idof < loc_ndof, "index loc_idof="<< loc_idof << " out of range [0:" << loc_ndof<<"[");
    check_macro (iloc     < nloc,     "index iloc="    << iloc     << " out of range [0:" << nloc<<"[");
#endif // _RHEOLEF_PARANO
    return _sid_grad_val[_curr_K_variant][_curr_sid.loc_isid][ori_idx][_curr_sid.shift][iloc];
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)       	\
template class basis_on_pointset_rep<T>;

_RHEOLEF_instanciation(Float)

} // namespace rheolef
