/*
 * Copyright © 2015 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/**
 * \file params.c
 *
 * Test for set and get functions of framebuffer parameters defined by the
 * GL_ARB_framebuffer_no_attachments specification.
 *
 * There is interaction with following extensions:
 *    - EXT_texture_array
 *    - EXT_direct_state_access
 *
 * GL_ARB_framebuffer_no_attachments specification states:
 *
 *    "The error INVALID_ENUM is generated by FramebufferParameteri and
 *     GetFramebufferParameteriv if <target> is not DRAW_FRAMEBUFFER,
 *     READ_FRAMEBUFFER, or FRAMEBUFFER.
 *
 *     The error INVALID_OPERATION is generated by FramebufferParameteri and
 *     GetFramebufferParameteriv if the default (zero) framebuffer is bound to
 *     target.
 *
 *     The error INVALID_VALUE is generated by FramebufferParameteri and
 *     NamedFramebufferParameteriEXT if <param> is less than zero or greater the
 *     implementation-dependent limits MAX_FRAMEBUFFER_WIDTH,
 *     MAX_FRAMEBUFFER_HEIGHT, MAX_FRAMEBUFFER_LAYERS, MAX_FRAMEBUFFER_SAMPLES if
 *     <pname> is FRAMEBUFFER_DEFAULT_WIDTH, FRAMEBUFFER_DEFAULT_HEIGHT,
 *     FRAMEBUFFER_DEFAULT_LAYERS, or FRAMEBUFFER_DEFAULT_SAMPLES, respectively.
 *
 *     The error INVALID_VALUE is generated by NamedFramebufferParameteriEXT and
 *     GetNamedFramebufferParameterivEXT if <framebuffer> is not a name returned
 *     by GenFramebuffers."
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_core_version = 31;

PIGLIT_GL_TEST_CONFIG_END

static struct test_t {
	GLenum param;
	GLint default_value;
	GLint value;
	GLint max_value;
	const char *extension;
} tests[] = {
	{ GL_FRAMEBUFFER_DEFAULT_WIDTH,
		0, 0, GL_MAX_FRAMEBUFFER_WIDTH, NULL },
	{ GL_FRAMEBUFFER_DEFAULT_HEIGHT,
		0, 0, GL_MAX_FRAMEBUFFER_HEIGHT, NULL },
	{ GL_FRAMEBUFFER_DEFAULT_LAYERS,
		0, 0, GL_MAX_FRAMEBUFFER_LAYERS, "EXT_texture_array" },
	{ GL_FRAMEBUFFER_DEFAULT_SAMPLES,
		0, 0, GL_MAX_FRAMEBUFFER_SAMPLES, NULL },
	{ GL_FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS,
		0, 0, 0, NULL }
};

enum piglit_result
piglit_display(void)
{
	return PIGLIT_FAIL;
}

static bool
set_value(GLenum par, GLint val, GLenum target, GLenum error, GLuint fbo)
{
	if (fbo)
		glNamedFramebufferParameteriEXT(fbo, par, val);
	else
		glFramebufferParameteri(target, par, val);
	if (!piglit_check_gl_error(error))
		return false;
	return true;
}

static bool
get_value(GLenum par, GLint *val, GLenum target, GLenum error, GLuint fbo)
{
	if (fbo)
		glGetNamedFramebufferParameterivEXT(fbo, par, val);
	else
		glGetFramebufferParameteriv(target, par, val);
	if (!piglit_check_gl_error(error))
		return false;
	return true;
}

static bool
test_values(struct test_t *test, GLenum target, GLuint fbo)
{
	GLint max = -1;
	GLint value;

	/* If has a max value, do range test. */
	if (test->max_value != 0)
		glGetIntegerv(test->max_value, &max);

	if (!piglit_check_gl_error(GL_NO_ERROR))
		return false;

	/* If parameter has no max, it is a boolean type. */
	if (max == -1) {
		if (!set_value(test->param, GL_TRUE, target, GL_NO_ERROR, fbo))
			return false;

		if (!get_value(test->param, &value, target, GL_NO_ERROR, fbo))
			return false;

		if (value != GL_TRUE)
			return false;
		return true;
	}

	/* Parameter has a max, test that values max + 1 and -1 give error */
	if (!set_value(test->param, max + 1, target, GL_INVALID_VALUE, fbo))
		return false;
	if (!set_value(test->param, -1, target, GL_INVALID_VALUE, fbo))
		return false;

	/* Valid value (max - 1) for set and get. */
	if (!set_value(test->param, max - 1, target, GL_NO_ERROR, fbo))
		return false;
	if (!get_value(test->param, &value, target, GL_NO_ERROR, fbo))
		return false;

	if (value != max - 1)
		return false;

	return true;
}

static bool
invalid_enums_check()
{
	bool pass = true;
	GLint value;

	/* Test invalid enums for functions. */
	if (!set_value(GL_NO_ERROR, 0, GL_FRAMEBUFFER, GL_INVALID_ENUM, 0))
		pass = false;
	if (!get_value(GL_NO_ERROR, &value, GL_FRAMEBUFFER, GL_INVALID_ENUM, 0))
		pass = false;

	piglit_report_subtest_result(pass ? PIGLIT_PASS : PIGLIT_FAIL,
		"invalid enums");

	return pass;
}

static bool
dsa_subtest(GLuint fbo)
{
	unsigned i;
	GLint value;
	bool pass = true;

	if (!piglit_is_extension_supported("GL_EXT_direct_state_access")) {
		piglit_report_subtest_result(PIGLIT_SKIP, "dsa");
		return pass;
	}

	for (i = 0; i < sizeof(tests)/sizeof(struct test_t); i++) {
		/* If extension required, see that it is supported. */
		if (tests[i].extension &&
			!piglit_is_extension_supported(tests[i].extension))
			continue;

		pass = pass && test_values(&tests[i], GL_DRAW_FRAMEBUFFER, fbo);
		pass = pass && test_values(&tests[i], GL_READ_FRAMEBUFFER, fbo);
		pass = pass && test_values(&tests[i], GL_FRAMEBUFFER, fbo);
	}

	/* Test using fbo name that was not returned by glGenFramebuffers. */
	if (!set_value(GL_FRAMEBUFFER_DEFAULT_WIDTH, 42, GL_FRAMEBUFFER,
		GL_INVALID_VALUE, 666))
		pass = false;
	if (!get_value(GL_FRAMEBUFFER_DEFAULT_WIDTH, &value, GL_FRAMEBUFFER,
		GL_INVALID_VALUE, 666))
		pass = false;

	piglit_report_subtest_result(pass ? PIGLIT_PASS : PIGLIT_FAIL, "dsa");
	return pass;
}

static bool
default_fbo_errors()
{
	bool pass = true;
	unsigned i;
	GLint value;
	glBindFramebuffer(GL_FRAMEBUFFER, 0);

	/* Error cases with default fbo. */
	for (i = 0; i < ARRAY_SIZE(tests); i++) {
		/* If extension required, see that it is supported. */
		if (tests[i].extension &&
			!piglit_is_extension_supported(tests[i].extension))
			continue;
		if (!set_value(tests[i].param, tests[i].default_value, GL_FRAMEBUFFER,
			GL_INVALID_OPERATION, 0))
			pass = false;
		if (!get_value(tests[i].param, &value, GL_FRAMEBUFFER,
			GL_INVALID_OPERATION, 0))
			pass = false;
	}
	piglit_report_subtest_result(pass ? PIGLIT_PASS : PIGLIT_FAIL,
		"default fbo bound");
	return pass;
}

/* Idea copied from arb_direct_state_access/dsa-utils.h */
#define SUBTEST(subtest, global, ...) \
do { \
        piglit_report_subtest_result(subtest ? PIGLIT_PASS : PIGLIT_FAIL, \
                                     __VA_ARGS__); \
        global = global && subtest; \
	subtest = true; \
} while (0)

void
piglit_init(int argc, char **argv)
{
	GLuint fbo, i;
	bool pass = true;
	bool sub = true;

        if (piglit_get_gl_version() < 30 ||
                !piglit_is_extension_supported("GL_ARB_framebuffer_object"))
                piglit_report_result(PIGLIT_SKIP);

	piglit_require_extension("GL_ARB_framebuffer_no_attachments");

	pass = default_fbo_errors();

	/* Create fbo with no attachments. */
	glGenFramebuffers(1, &fbo);
	glBindFramebuffer(GL_FRAMEBUFFER, fbo);

	/* Check that fbo is marked initially not complete. */
	if (glCheckFramebufferStatus(GL_FRAMEBUFFER) ==
		GL_FRAMEBUFFER_COMPLETE)
		sub = false;

	SUBTEST(sub, pass, "initially incomplete");

	pass = pass && invalid_enums_check();

	/* Verify initial values. */
	for (i = 0; i < ARRAY_SIZE(tests); i++) {
		/* If extension required, see that it is supported. */
		if (tests[i].extension &&
			!piglit_is_extension_supported(tests[i].extension))
			continue;

		glGetFramebufferParameteriv(GL_FRAMEBUFFER, tests[i].param,
			&tests[i].value);
		if (tests[i].value != tests[i].default_value)
			sub = false;
	}

	SUBTEST(sub, pass, "default values");

	/* Check that there are no errors. */
	if (!piglit_check_gl_error(GL_NO_ERROR))
		piglit_report_result(PIGLIT_FAIL);

	/* Set some values and verify them (for bound fbo). */
	for (i = 0; i < ARRAY_SIZE(tests); i++) {
		/* If extension required, see that it is supported. */
		if (tests[i].extension &&
			!piglit_is_extension_supported(tests[i].extension))
			continue;

		sub = sub && test_values(&tests[i], GL_DRAW_FRAMEBUFFER, 0);
		sub = sub && test_values(&tests[i], GL_READ_FRAMEBUFFER, 0);
		sub = sub && test_values(&tests[i], GL_FRAMEBUFFER, 0);
	}

	SUBTEST(sub, pass, "value setting");

	/* Check that fbo is marked complete now. */
	if (glCheckFramebufferStatus(GL_FRAMEBUFFER) !=
		GL_FRAMEBUFFER_COMPLETE)
		sub = false;

	SUBTEST(sub, pass, "fbo complete");

	pass = pass && dsa_subtest(fbo);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}
