
#include "SimTKmolmodel.h"
#include "molmodel/internal/Pdb.h"
#include "SimTKcommon/Testing.h"

#include <sstream>

using namespace SimTK;
using namespace std;

void testProteinDihedrals() { 
        CompoundSystem system;
        SimbodyMatterSubsystem matter(system);
        DuMMForceFieldSubsystem dumm(system);
        dumm.loadAmber99Parameters();
        dumm.setGbsaGlobalScaleFactor(0);
        dumm.setCoulombGlobalScaleFactor(0.0);

        // vpfString is useful for ensuring cis-proline is set correctly
        const char* vpfString = ""
            "ATOM     66  N   VAL A  14       1.688 -35.161  -4.509  1.00 10.89           N  \n"
            "ATOM     67  CA  VAL A  14       2.776 -34.186  -4.429  1.00  9.30           C  \n"
            "ATOM     68  C   VAL A  14       2.595 -33.225  -3.228  1.00  9.34           C  \n"
            "ATOM     69  O   VAL A  14       1.506 -32.751  -2.953  1.00  9.03           O  \n"
            "ATOM     70  CB  VAL A  14       2.855 -33.367  -5.740  1.00 11.74           C  \n"
            "ATOM     71  CG1 VAL A  14       3.946 -32.344  -5.654  1.00 12.55           C  \n"
            "ATOM     72  CG2 VAL A  14       3.097 -34.263  -6.929  1.00 14.44           C  \n"
            "ATOM     73  N   PRO A  15       3.665 -32.948  -2.479  1.00 12.35           N  \n"
            "ATOM     74  CA  PRO A  15       5.036 -33.449  -2.569  1.00 11.72           C  \n"
            "ATOM     75  C   PRO A  15       5.379 -34.583  -1.577  1.00 13.93           C  \n"
            "ATOM     76  O   PRO A  15       6.533 -34.744  -1.195  1.00 18.82           O  \n"
            "ATOM     77  CB  PRO A  15       5.845 -32.202  -2.289  1.00  5.84           C  \n"
            "ATOM     78  CG  PRO A  15       5.035 -31.517  -1.262  1.00  9.84           C  \n"
            "ATOM     79  CD  PRO A  15       3.580 -31.870  -1.485  1.00 10.51           C  \n"
            "ATOM     80  N   PHE A  16       4.402 -35.387  -1.186  1.00  8.41           N  \n"
            "ATOM     81  CA  PHE A  16       4.666 -36.483  -0.284  1.00  8.06           C  \n"
            "ATOM     82  C   PHE A  16       5.243 -37.743  -0.959  1.00  9.43           C  \n"
            "ATOM     83  O   PHE A  16       6.202 -38.330  -0.453  1.00  8.42           O  \n"
            "ATOM     84  CB  PHE A  16       3.396 -36.818   0.461  1.00 13.56           C  \n"
            "ATOM     85  CG  PHE A  16       2.973 -35.765   1.423  1.00 12.62           C  \n"
            "ATOM     86  CD1 PHE A  16       2.157 -34.730   1.015  1.00 19.89           C  \n"
            "ATOM     87  CD2 PHE A  16       3.354 -35.829   2.747  1.00 13.73           C  \n"
            "ATOM     88  CE1 PHE A  16       1.727 -33.767   1.923  1.00 16.04           C  \n"
            "ATOM     89  CE2 PHE A  16       2.927 -34.885   3.645  1.00 13.91           C  \n"
            "ATOM     90  CZ  PHE A  16       2.114 -33.853   3.232  1.00 14.41           C  \n"
            ;

        const char* gbpFrag1String = ""
            "ATOM      1  N   LEU A   5       9.888 -56.161   4.597  1.00 11.43           N  \n"
            "ATOM      2  CA  LEU A   5       8.638 -55.368   4.770  1.00 15.74           C  \n"
            "ATOM      3  C   LEU A   5       8.247 -54.783   3.426  1.00 15.40           C  \n"
            "ATOM      4  O   LEU A   5       9.120 -54.406   2.670  1.00 16.47           O  \n"
            "ATOM      5  CB  LEU A   5       8.870 -54.246   5.786  1.00 10.24           C  \n"
            "ATOM      6  CG  LEU A   5       9.191 -54.738   7.194  1.00 11.95           C  \n"
            "ATOM      7  CD1 LEU A   5       9.455 -53.563   8.053  1.00  8.81           C  \n"
            "ATOM      8  CD2 LEU A   5       8.028 -55.526   7.782  1.00 15.35           C  \n"
            "ATOM      9  N   VAL A   6       6.963 -54.862   3.079  1.00 16.79           N  \n"
            "ATOM     10  CA  VAL A   6       6.398 -54.229   1.880  1.00 13.57           C  \n"
            "ATOM     11  C   VAL A   6       5.624 -52.996   2.304  1.00 13.60           C  \n"
            "ATOM     12  O   VAL A   6       4.613 -53.132   2.972  1.00  8.66           O  \n"
            "ATOM     13  CB  VAL A   6       5.386 -55.148   1.199  1.00 15.73           C  \n"
            "ATOM     14  CG1 VAL A   6       4.937 -54.539  -0.110  1.00 12.94           C  \n"
            "ATOM     15  CG2 VAL A   6       5.982 -56.537   1.004  1.00 17.64           C  \n"
            "ATOM     16  N   VAL A   7       6.094 -51.809   1.929  1.00  9.96           N  \n"
            "ATOM     17  CA  VAL A   7       5.446 -50.564   2.320  1.00  9.01           C  \n"
            "ATOM     18  C   VAL A   7       4.806 -49.930   1.098  1.00  6.87           C  \n"
            "ATOM     19  O   VAL A   7       5.413 -49.867   0.054  1.00  8.60           O  \n"
            "ATOM     20  CB  VAL A   7       6.465 -49.587   2.995  1.00 12.36           C  \n"
            "ATOM     21  CG1 VAL A   7       7.822 -49.808   2.438  1.00 13.83           C  \n"
            "ATOM     22  CG2 VAL A   7       6.065 -48.116   2.797  1.00  9.85           C  \n"
            "ATOM     23  N   ALA A   8       3.529 -49.593   1.197  1.00 13.39           N  \n"
            "ATOM     24  CA  ALA A   8       2.773 -49.057   0.069  1.00 13.79           C  \n"
            "ATOM     25  C   ALA A   8       2.602 -47.552   0.172  1.00  9.25           C  \n"
            "ATOM     26  O   ALA A   8       2.500 -47.028   1.279  1.00 15.37           O  \n"
            "ATOM     27  CB  ALA A   8       1.448 -49.689   0.031  1.00 12.95           C  \n"
            "ATOM     28  N   THR A   9       2.586 -46.876  -0.980  1.00 11.17           N  \n"
            "ATOM     29  CA  THR A   9       2.309 -45.447  -1.099  1.00  9.76           C  \n"
            "ATOM     30  C   THR A   9       1.358 -45.161  -2.252  1.00 11.27           C  \n"
            "ATOM     31  O   THR A   9       1.157 -46.000  -3.122  1.00 11.74           O  \n"
            "ATOM     32  CB  THR A   9       3.557 -44.649  -1.423  1.00 16.25           C  \n"
            "ATOM     33  OG1 THR A   9       4.402 -45.395  -2.311  1.00 16.45           O  \n"
            "ATOM     34  CG2 THR A   9       4.289 -44.276  -0.187  1.00 21.89           C  \n"
            "ATOM     35  N   ASP A  10       0.912 -43.916  -2.338  1.00 10.94           N  \n"
            "ATOM     36  CA  ASP A  10       0.087 -43.441  -3.442  1.00 13.74           C  \n"
            "ATOM     37  C   ASP A  10       0.911 -43.463  -4.735  1.00 15.06           C  \n"
            "ATOM     38  O   ASP A  10       2.125 -43.629  -4.690  1.00 19.02           O  \n"
            "ATOM     39  CB  ASP A  10      -0.369 -41.999  -3.147  1.00 15.41           C  \n"
            "ATOM     40  CG  ASP A  10      -1.770 -41.707  -3.633  1.00 19.42           C  \n"
            "ATOM     41  OD1 ASP A  10      -2.203 -42.274  -4.658  1.00 21.59           O  \n"
            "ATOM     42  OD2 ASP A  10      -2.450 -40.884  -2.985  1.00 25.31           O  \n"
            "ATOM     43  N   THR A  11       0.258 -43.262  -5.878  1.00 15.47           N  \n"
            "ATOM     44  CA  THR A  11       0.955 -43.164  -7.162  1.00 17.91           C  \n"
            "ATOM     45  C   THR A  11       1.383 -41.737  -7.432  1.00 13.11           C  \n"
            "ATOM     46  O   THR A  11       2.346 -41.509  -8.165  1.00 17.17           O  \n"
            "ATOM     47  CB  THR A  11       0.069 -43.624  -8.337  1.00 19.89           C  \n"
            "ATOM     48  OG1 THR A  11      -1.219 -43.985  -7.840  1.00 26.13           O  \n"
            "ATOM     49  CG2 THR A  11       0.670 -44.827  -9.035  1.00 24.20           C  \n"
            "ATOM     50  N   ALA A  12       0.597 -40.783  -6.944  1.00 10.88           N  \n"
            "ATOM     51  CA  ALA A  12       1.015 -39.392  -6.880  1.00 15.07           C  \n"
            "ATOM     52  C   ALA A  12       0.247 -38.601  -5.834  1.00 16.92           C  \n"
            "ATOM     53  O   ALA A  12      -0.957 -38.376  -5.971  1.00 18.82           O  \n"
            "ATOM     54  CB  ALA A  12       0.851 -38.739  -8.220  1.00 17.27           C  \n"
            "ATOM     55  N   PHE A  13       0.976 -38.090  -4.852  1.00 12.98           N  \n"
            "ATOM     56  CA  PHE A  13       0.439 -37.164  -3.870  1.00 16.31           C  \n"
            "ATOM     57  C   PHE A  13       1.552 -36.139  -3.617  1.00 15.19           C  \n"
            "ATOM     58  O   PHE A  13       2.357 -36.311  -2.714  1.00 16.10           O  \n"
            "ATOM     59  CB  PHE A  13       0.072 -37.936  -2.585  1.00 15.61           C  \n"
            "ATOM     60  CG  PHE A  13      -0.553 -37.084  -1.485  1.00 21.35           C  \n"
            "ATOM     61  CD1 PHE A  13      -1.391 -36.012  -1.784  1.00 22.62           C  \n"
            "ATOM     62  CD2 PHE A  13      -0.343 -37.401  -0.138  1.00 22.88           C  \n"
            "ATOM     63  CE1 PHE A  13      -2.008 -35.280  -0.759  1.00 23.34           C  \n"
            "ATOM     64  CE2 PHE A  13      -0.954 -36.674   0.890  1.00 19.43           C  \n"
            "ATOM     65  CZ  PHE A  13      -1.786 -35.616   0.582  1.00 21.44           C  \n"
            "ATOM     66  N   VAL A  14       1.688 -35.161  -4.509  1.00 10.89           N  \n"
            "ATOM     67  CA  VAL A  14       2.776 -34.186  -4.429  1.00  9.30           C  \n"
            "ATOM     68  C   VAL A  14       2.595 -33.225  -3.228  1.00  9.34           C  \n"
            "ATOM     69  O   VAL A  14       1.506 -32.751  -2.953  1.00  9.03           O  \n"
            "ATOM     70  CB  VAL A  14       2.855 -33.367  -5.740  1.00 11.74           C  \n"
            "ATOM     71  CG1 VAL A  14       3.946 -32.344  -5.654  1.00 12.55           C  \n"
            "ATOM     72  CG2 VAL A  14       3.097 -34.263  -6.929  1.00 14.44           C  \n"
            "ATOM     73  N   PRO A  15       3.665 -32.948  -2.479  1.00 12.35           N  \n"
            "ATOM     74  CA  PRO A  15       5.036 -33.449  -2.569  1.00 11.72           C  \n"
            "ATOM     75  C   PRO A  15       5.379 -34.583  -1.577  1.00 13.93           C  \n"
            "ATOM     76  O   PRO A  15       6.533 -34.744  -1.195  1.00 18.82           O  \n"
            "ATOM     77  CB  PRO A  15       5.845 -32.202  -2.289  1.00  5.84           C  \n"
            "ATOM     78  CG  PRO A  15       5.035 -31.517  -1.262  1.00  9.84           C  \n"
            "ATOM     79  CD  PRO A  15       3.580 -31.870  -1.485  1.00 10.51           C  \n"
            "ATOM     80  N   PHE A  16       4.402 -35.387  -1.186  1.00  8.41           N  \n"
            "ATOM     81  CA  PHE A  16       4.666 -36.483  -0.284  1.00  8.06           C  \n"
            "ATOM     82  C   PHE A  16       5.243 -37.743  -0.959  1.00  9.43           C  \n"
            "ATOM     83  O   PHE A  16       6.202 -38.330  -0.453  1.00  8.42           O  \n"
            "ATOM     84  CB  PHE A  16       3.396 -36.818   0.461  1.00 13.56           C  \n"
            "ATOM     85  CG  PHE A  16       2.973 -35.765   1.423  1.00 12.62           C  \n"
            "ATOM     86  CD1 PHE A  16       2.157 -34.730   1.015  1.00 19.89           C  \n"
            "ATOM     87  CD2 PHE A  16       3.354 -35.829   2.747  1.00 13.73           C  \n"
            "ATOM     88  CE1 PHE A  16       1.727 -33.767   1.923  1.00 16.04           C  \n"
            "ATOM     89  CE2 PHE A  16       2.927 -34.885   3.645  1.00 13.91           C  \n"
            "ATOM     90  CZ  PHE A  16       2.114 -33.853   3.232  1.00 14.41           C  \n"
            "ATOM     91  N   GLU A  17       4.682 -38.144  -2.097  1.00 12.52           N  \n"
            "ATOM     92  CA  GLU A  17       5.309 -39.136  -2.968  1.00  6.67           C  \n"
            "ATOM     93  C   GLU A  17       4.919 -38.967  -4.435  1.00  5.89           C  \n"
            "ATOM     94  O   GLU A  17       3.741 -38.853  -4.764  1.00  8.72           O  \n"
            "ATOM     95  CB  GLU A  17       4.959 -40.549  -2.512  1.00  8.35           C  \n"
            "ATOM     96  CG  GLU A  17       3.459 -40.852  -2.452  1.00  8.92           C  \n"
            "ATOM     97  CD  GLU A  17       2.821 -40.477  -1.131  1.00 12.62           C  \n"
            "ATOM     98  OE1 GLU A  17       3.519 -40.401  -0.107  1.00  9.84           O  \n"
            "ATOM     99  OE2 GLU A  17       1.599 -40.280  -1.108  1.00 12.75           O  \n"
            "ATOM    100  N   PHE A  18       5.911 -38.937  -5.320  1.00  5.14           N  \n"
            "ATOM    101  CA  PHE A  18       5.660 -38.911  -6.757  1.00  9.50           C  \n"
            "ATOM    102  C   PHE A  18       6.898 -39.324  -7.527  1.00 10.39           C  \n"
            "ATOM    103  O   PHE A  18       8.011 -39.141  -7.067  1.00 17.79           O  \n"
            "ATOM    104  CB  PHE A  18       5.218 -37.514  -7.199  1.00 11.65           C  \n"
            "ATOM    105  CG  PHE A  18       6.210 -36.421  -6.860  1.00 16.07           C  \n"
            "ATOM    106  CD1 PHE A  18       6.411 -36.029  -5.547  1.00 10.03           C  \n"
            "ATOM    107  CD2 PHE A  18       6.983 -35.835  -7.852  1.00 13.69           C  \n"
            "ATOM    108  CE1 PHE A  18       7.358 -35.084  -5.225  1.00 10.95           C  \n"
            "ATOM    109  CE2 PHE A  18       7.924 -34.894  -7.533  1.00 11.46           C  \n"
            "ATOM    110  CZ  PHE A  18       8.111 -34.519  -6.218  1.00 13.40           C  \n"
            "ATOM    111  N   LYS A  19       6.688 -39.933  -8.679  1.00 17.27           N  \n"
            "ATOM    112  CA  LYS A  19       7.757 -40.517  -9.483  1.00 18.53           C  \n"
            "ATOM    113  C   LYS A  19       8.684 -39.410  -9.874  1.00 16.86           C  \n"
            "ATOM    114  O   LYS A  19       8.246 -38.277 -10.033  1.00 23.82           O  \n"
            "ATOM    115  CB  LYS A  19       7.172 -41.134 -10.760  1.00 19.89           C  \n"
            "ATOM    116  CG  LYS A  19       7.690 -42.510 -11.172  1.00 22.98           C  \n"
            "ATOM    117  CD  LYS A  19       6.811 -43.079 -12.310  1.00 32.56           C  \n"
            "ATOM    118  CE  LYS A  19       7.493 -43.024 -13.707  1.00 37.16           C  \n"
            "ATOM    119  NZ  LYS A  19       8.111 -44.366 -14.175  1.00 43.17           N  \n"
            "ATOM    120  N   GLN A  20       9.933 -39.765 -10.149  1.00 17.99           N  \n"
            "ATOM    121  CA  GLN A  20      10.913 -38.819 -10.646  1.00 18.84           C  \n"
            "ATOM    122  C   GLN A  20      11.956 -39.566 -11.453  1.00 22.14           C  \n"
            "ATOM    123  O   GLN A  20      12.839 -40.223 -10.906  1.00 19.00           O  \n"
            "ATOM    124  CB  GLN A  20      11.576 -38.108  -9.489  1.00 25.85           C  \n"
            "ATOM    125  CG  GLN A  20      12.681 -37.175  -9.906  1.00 34.40           C  \n"
            "ATOM    126  CD  GLN A  20      12.430 -35.762  -9.438  1.00 40.67           C  \n"
            "ATOM    127  OE1 GLN A  20      11.338 -35.204  -9.642  1.00 36.36           O  \n"
            "ATOM    128  NE2 GLN A  20      13.444 -35.159  -8.806  1.00 43.33           N  \n"
            "ATOM    129  N   GLY A  21      11.838 -39.490 -12.770  1.00 25.64           N  \n"
            "ATOM    130  CA  GLY A  21      12.625 -40.363 -13.613  1.00 24.65           C  \n"
            "ATOM    131  C   GLY A  21      12.170 -41.774 -13.349  1.00 22.51           C  \n"
            "ATOM    132  O   GLY A  21      10.966 -42.067 -13.451  1.00 26.14           O  \n"
            "ATOM    133  N   ASP A  22      13.064 -42.542 -12.728  1.00 24.83           N  \n"
            "ATOM    134  CA  ASP A  22      12.844 -43.972 -12.478  1.00 28.01           C  \n"
            "ATOM    135  C   ASP A  22      12.513 -44.316 -11.018  1.00 26.08           C  \n"
            "ATOM    136  O   ASP A  22      12.033 -45.405 -10.736  1.00 28.92           O  \n"
            "ATOM    137  CB  ASP A  22      14.080 -44.759 -12.922  1.00 29.85           C  \n"
            "ATOM    138  CG  ASP A  22      13.760 -45.856 -13.932  1.00 31.93           C  \n"
            "ATOM    139  OD1 ASP A  22      12.647 -45.842 -14.519  1.00 29.44           O  \n"
            "ATOM    140  OD2 ASP A  22      14.649 -46.714 -14.153  1.00 30.85           O  \n"
            "ATOM    141  N   LEU A  23      12.812 -43.409 -10.097  1.00 21.05           N  \n"
            "ATOM    142  CA  LEU A  23      12.541 -43.628  -8.680  1.00 21.62           C  \n"
            "ATOM    143  C   LEU A  23      11.271 -42.910  -8.193  1.00 21.36           C  \n"
            "ATOM    144  O   LEU A  23      10.470 -42.421  -8.992  1.00 18.41           O  \n"
            "ATOM    145  CB  LEU A  23      13.758 -43.207  -7.821  1.00 23.15           C  \n"
            "ATOM    146  CG  LEU A  23      14.539 -41.903  -8.077  1.00 27.51           C  \n"
            "ATOM    147  CD1 LEU A  23      13.584 -40.747  -7.901  1.00 35.59           C  \n"
            "ATOM    148  CD2 LEU A  23      15.749 -41.723  -7.120  1.00 24.86           C  \n"
            "ATOM    149  N   TYR A  24      11.059 -42.944  -6.881  1.00 19.66           N  \n"
            "ATOM    150  CA  TYR A  24      10.053 -42.139  -6.222  1.00 11.94           C  \n"
            "ATOM    151  C   TYR A  24      10.743 -41.202  -5.271  1.00 12.46           C  \n"
            "ATOM    152  O   TYR A  24      11.758 -41.550  -4.684  1.00 12.11           O  \n"
            "ATOM    153  CB  TYR A  24       9.105 -43.013  -5.440  1.00  8.67           C  \n"
            "ATOM    154  CG  TYR A  24       7.833 -43.272  -6.173  1.00 13.15           C  \n"
            "ATOM    155  CD1 TYR A  24       7.824 -44.054  -7.328  1.00 12.14           C  \n"
            "ATOM    156  CD2 TYR A  24       6.634 -42.705  -5.745  1.00 16.31           C  \n"
            "ATOM    157  CE1 TYR A  24       6.659 -44.262  -8.039  1.00 14.20           C  \n"
            "ATOM    158  CE2 TYR A  24       5.455 -42.899  -6.461  1.00 15.50           C  \n"
            "ATOM    159  CZ  TYR A  24       5.481 -43.675  -7.608  1.00 16.96           C  \n"
            "ATOM    160  OH  TYR A  24       4.339 -43.828  -8.353  1.00 21.52           O  \n"
            "ATOM    161  N   VAL A  25      10.100 -40.076  -5.007  1.00 12.67           N  \n"
            "ATOM    162  CA  VAL A  25      10.667 -39.003  -4.203  1.00 14.06           C  \n"
            "ATOM    163  C   VAL A  25       9.494 -38.373  -3.409  1.00 15.53           C  \n"
            "ATOM    164  O   VAL A  25       8.338 -38.545  -3.795  1.00 11.06           O  \n"
            "ATOM    165  CB  VAL A  25      11.318 -37.926  -5.136  1.00 16.19           C  \n"
            "ATOM    166  CG1 VAL A  25      11.840 -36.788  -4.347  1.00 21.62           C  \n"
            "ATOM    167  CG2 VAL A  25      12.444 -38.515  -5.922  1.00 16.97           C  \n"
            "ATOM    168  N   GLY A  26       9.800 -37.659  -2.316  1.00 11.81           N  \n"
            "ATOM    169  CA  GLY A  26       8.794 -36.947  -1.548  1.00 14.13           C  \n"
            "ATOM    170  C   GLY A  26       9.090 -36.916  -0.055  1.00 12.53           C  \n"
            "ATOM    171  O   GLY A  26      10.047 -37.507   0.401  1.00 13.15           O  \n"
            "ATOM    172  N   PHE A  27       8.255 -36.250   0.722  1.00  6.71           N  \n"
            "ATOM    173  CA  PHE A  27       8.449 -36.150   2.157  1.00  6.44           C  \n"
            "ATOM    174  C   PHE A  27       8.265 -37.513   2.867  1.00 13.51           C  \n"
            "ATOM    175  O   PHE A  27       8.938 -37.810   3.855  1.00 11.16           O  \n"
            "ATOM    176  CB  PHE A  27       7.466 -35.105   2.678  1.00  6.47           C  \n"
            "ATOM    177  CG  PHE A  27       7.433 -34.976   4.155  1.00  9.33           C  \n"
            "ATOM    178  CD1 PHE A  27       6.661 -35.833   4.922  1.00  9.03           C  \n"
            "ATOM    179  CD2 PHE A  27       8.193 -34.013   4.788  1.00 11.46           C  \n"
            "ATOM    180  CE1 PHE A  27       6.652 -35.747   6.305  1.00  8.35           C  \n"
            "ATOM    181  CE2 PHE A  27       8.193 -33.913   6.166  1.00  8.19           C  \n"
            "ATOM    182  CZ  PHE A  27       7.421 -34.789   6.929  1.00 10.12           C  \n"
            "ATOM    183  N   ASP A  28       7.220 -38.241   2.470  1.00 14.07           N  \n"
            "ATOM    184  CA  ASP A  28       6.920 -39.574   2.993  1.00 14.99           C  \n"
            "ATOM    185  C   ASP A  28       8.018 -40.545   2.622  1.00 14.48           C  \n"
            "ATOM    186  O   ASP A  28       8.456 -41.330   3.450  1.00 16.21           O  \n"
            "ATOM    187  CB  ASP A  28       5.619 -40.098   2.402  1.00 13.40           C  \n"
            "ATOM    188  CG  ASP A  28       4.416 -39.662   3.173  1.00 12.04           C  \n"
            "ATOM    189  OD1 ASP A  28       4.554 -39.091   4.260  1.00 12.89           O  \n"
            "ATOM    190  OD2 ASP A  28       3.309 -39.880   2.689  1.00  8.53           O  \n"
            "ATOM    191  N   VAL A  29       8.486 -40.451   1.382  1.00  4.87           N  \n"
            "ATOM    192  CA  VAL A  29       9.555 -41.299   0.886  1.00 11.18           C  \n"
            "ATOM    193  C   VAL A  29      10.903 -41.070   1.570  1.00 11.07           C  \n"
            "ATOM    194  O   VAL A  29      11.591 -42.030   1.891  1.00 17.10           O  \n"
            "ATOM    195  CB  VAL A  29       9.694 -41.172  -0.632  1.00  9.96           C  \n"
            "ATOM    196  CG1 VAL A  29      10.821 -42.039  -1.122  1.00 14.26           C  \n"
            "ATOM    197  CG2 VAL A  29       8.416 -41.617  -1.303  1.00  8.50           C  \n"
            "ATOM    198  N   ASP A  30      11.244 -39.824   1.881  1.00 15.02           N  \n"
            "ATOM    199  CA  ASP A  30      12.423 -39.537   2.702  1.00 10.85           C  \n"
            "ATOM    200  C   ASP A  30      12.219 -39.839   4.180  1.00 14.35           C  \n"
            "ATOM    201  O   ASP A  30      13.172 -40.144   4.880  1.00 16.20           O  \n"
            "ATOM    202  CB  ASP A  30      12.834 -38.080   2.571  1.00 14.69           C  \n"
            "ATOM    203  CG  ASP A  30      13.384 -37.757   1.226  1.00 16.86           C  \n"
            "ATOM    204  OD1 ASP A  30      13.872 -38.658   0.537  1.00 19.60           O  \n"
            "ATOM    205  OD2 ASP A  30      13.371 -36.579   0.858  1.00 20.02           O  \n"
            "ATOM    206  N   LEU A  31      10.996 -39.679   4.673  1.00 10.19           N  \n"
            "ATOM    207  CA  LEU A  31      10.713 -39.909   6.075  1.00  9.62           C  \n"
            "ATOM    208  C   LEU A  31      10.804 -41.366   6.448  1.00  7.52           C  \n"
            "ATOM    209  O   LEU A  31      11.319 -41.694   7.497  1.00 14.41           O  \n"
            "ATOM    210  CB  LEU A  31       9.331 -39.398   6.443  1.00 12.00           C  \n"
            "ATOM    211  CG  LEU A  31       9.047 -39.590   7.930  1.00 12.23           C  \n"
            "ATOM    212  CD1 LEU A  31       9.826 -38.576   8.725  1.00 18.71           C  \n"
            "ATOM    213  CD2 LEU A  31       7.593 -39.413   8.203  1.00 15.30           C  \n"
            "ATOM    214  N   TRP A  32      10.214 -42.233   5.646  1.00  6.24           N  \n"
            "ATOM    215  CA  TRP A  32      10.280 -43.646   5.923  1.00  6.70           C  \n"
            "ATOM    216  C   TRP A  32      11.717 -44.182   5.856  1.00 10.70           C  \n"
            "ATOM    217  O   TRP A  32      12.102 -45.034   6.653  1.00 15.23           O  \n"
            "ATOM    218  CB  TRP A  32       9.395 -44.418   4.950  1.00  8.19           C  \n"
            "ATOM    219  CG  TRP A  32       9.486 -45.874   5.160  1.00  6.64           C  \n"
            "ATOM    220  CD1 TRP A  32      10.044 -46.779   4.313  1.00  9.05           C  \n"
            "ATOM    221  CD2 TRP A  32       9.212 -46.576   6.375  1.00  7.01           C  \n"
            "ATOM    222  NE1 TRP A  32      10.168 -47.998   4.931  1.00  7.95           N  \n"
            "ATOM    223  CE2 TRP A  32       9.663 -47.906   6.200  1.00  9.49           C  \n"
            "ATOM    224  CE3 TRP A  32       8.640 -46.210   7.607  1.00 11.61           C  \n"
            "ATOM    225  CZ2 TRP A  32       9.568 -48.874   7.210  1.00  8.00           C  \n"
            "ATOM    226  CZ3 TRP A  32       8.545 -47.174   8.618  1.00  6.35           C  \n"
            "ATOM    227  CH2 TRP A  32       9.011 -48.488   8.411  1.00  8.04           C  \n"
            "";

        const char* gbpFrag2String = ""
            "ATOM     80  N   PHE A  16       4.402 -35.387  -1.186  1.00  8.41           N  \n"
            "ATOM     81  CA  PHE A  16       4.666 -36.483  -0.284  1.00  8.06           C  \n"
            "ATOM     82  C   PHE A  16       5.243 -37.743  -0.959  1.00  9.43           C  \n"
            "ATOM     83  O   PHE A  16       6.202 -38.330  -0.453  1.00  8.42           O  \n"
            "ATOM     84  CB  PHE A  16       3.396 -36.818   0.461  1.00 13.56           C  \n"
            "ATOM     85  CG  PHE A  16       2.973 -35.765   1.423  1.00 12.62           C  \n"
            "ATOM     86  CD1 PHE A  16       2.157 -34.730   1.015  1.00 19.89           C  \n"
            "ATOM     87  CD2 PHE A  16       3.354 -35.829   2.747  1.00 13.73           C  \n"
            "ATOM     88  CE1 PHE A  16       1.727 -33.767   1.923  1.00 16.04           C  \n"
            "ATOM     89  CE2 PHE A  16       2.927 -34.885   3.645  1.00 13.91           C  \n"
            "ATOM     90  CZ  PHE A  16       2.114 -33.853   3.232  1.00 14.41           C  \n"
            "ATOM     91  N   GLU A  17       4.682 -38.144  -2.097  1.00 12.52           N  \n"
            "ATOM     92  CA  GLU A  17       5.309 -39.136  -2.968  1.00  6.67           C  \n"
            "ATOM     93  C   GLU A  17       4.919 -38.967  -4.435  1.00  5.89           C  \n"
            "ATOM     94  O   GLU A  17       3.741 -38.853  -4.764  1.00  8.72           O  \n"
            "ATOM     95  CB  GLU A  17       4.959 -40.549  -2.512  1.00  8.35           C  \n"
            "ATOM     96  CG  GLU A  17       3.459 -40.852  -2.452  1.00  8.92           C  \n"
            "ATOM     97  CD  GLU A  17       2.821 -40.477  -1.131  1.00 12.62           C  \n"
            "ATOM     98  OE1 GLU A  17       3.519 -40.401  -0.107  1.00  9.84           O  \n"
            "ATOM     99  OE2 GLU A  17       1.599 -40.280  -1.108  1.00 12.75           O  \n"
            "ATOM    100  N   PHE A  18       5.911 -38.937  -5.320  1.00  5.14           N  \n"
            "ATOM    101  CA  PHE A  18       5.660 -38.911  -6.757  1.00  9.50           C  \n"
            "ATOM    102  C   PHE A  18       6.898 -39.324  -7.527  1.00 10.39           C  \n"
            "ATOM    103  O   PHE A  18       8.011 -39.141  -7.067  1.00 17.79           O  \n"
            "ATOM    104  CB  PHE A  18       5.218 -37.514  -7.199  1.00 11.65           C  \n"
            "ATOM    105  CG  PHE A  18       6.210 -36.421  -6.860  1.00 16.07           C  \n"
            "ATOM    106  CD1 PHE A  18       6.411 -36.029  -5.547  1.00 10.03           C  \n"
            "ATOM    107  CD2 PHE A  18       6.983 -35.835  -7.852  1.00 13.69           C  \n"
            "ATOM    108  CE1 PHE A  18       7.358 -35.084  -5.225  1.00 10.95           C  \n"
            "ATOM    109  CE2 PHE A  18       7.924 -34.894  -7.533  1.00 11.46           C  \n"
            "ATOM    110  CZ  PHE A  18       8.111 -34.519  -6.218  1.00 13.40           C  \n"
            "ATOM    111  N   LYS A  19       6.688 -39.933  -8.679  1.00 17.27           N  \n"
            "ATOM    112  CA  LYS A  19       7.757 -40.517  -9.483  1.00 18.53           C  \n"
            "ATOM    113  C   LYS A  19       8.684 -39.410  -9.874  1.00 16.86           C  \n"
            "ATOM    114  O   LYS A  19       8.246 -38.277 -10.033  1.00 23.82           O  \n"
            "ATOM    115  CB  LYS A  19       7.172 -41.134 -10.760  1.00 19.89           C  \n"
            "ATOM    116  CG  LYS A  19       7.690 -42.510 -11.172  1.00 22.98           C  \n"
            "ATOM    117  CD  LYS A  19       6.811 -43.079 -12.310  1.00 32.56           C  \n"
            "ATOM    118  CE  LYS A  19       7.493 -43.024 -13.707  1.00 37.16           C  \n"
            "ATOM    119  NZ  LYS A  19       8.111 -44.366 -14.175  1.00 43.17           N  \n"
            "";

    std::string pdbString(gbpFrag1String);

    std::istringstream inFileStream( pdbString );
    PdbStructure pdbStructure(inFileStream);

    // Protein myProtein("LVVATDTAFVPFEFKQGDLYVGFDVDLW");
    Protein myProtein(pdbStructure);

    Compound::AtomTargetLocations atomTargets = myProtein.createAtomTargets(pdbStructure);
    cout << "Number of atom targets found = " << atomTargets.size() << endl;
    cout << "Number of atoms = " << myProtein.getNumAtoms() << endl;

    // just for testing
    // myProtein.matchDefaultConfiguration(atomTargets, Compound::Match_Idealized);

    Real residual = myProtein.getTransformAndResidual(atomTargets).residual;
    cout << "residual = " << residual << endl;

    // Write pdb files for visual comparison
    cout << pdbString << endl;
    // Use a different chain id for matched structure
    myProtein.setPdbChainId("B");
    myProtein.writeDefaultPdb(cout, myProtein.getTopLevelTransform());

    // Increased target residual from 0.05 to 0.06 after raising tolerance in ObservedPointFitter call
    // Raises actual residual to 0.052
    SimTK_ASSERT_ALWAYS(residual < 0.08, (String("RMS error was too large: ") + String(residual)).c_str());
}

int main() 
{
    SimTK_START_TEST("TestProteinDihedrals");
        SimTK_SUBTEST(testProteinDihedrals);
    SimTK_END_TEST();
}

