// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/profiles/profile_theme_update_service.h"

#include "base/notreached.h"
#include "base/optional.h"
#include "chrome/browser/chrome_notification_types.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_attributes_entry.h"
#include "chrome/browser/profiles/profile_attributes_storage.h"
#include "chrome/browser/themes/theme_properties.h"
#include "chrome/browser/themes/theme_service.h"
#include "chrome/browser/themes/theme_service_factory.h"
#include "chrome/browser/ui/signin/profile_colors_util.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "content/public/browser/notification_details.h"
#include "content/public/browser/notification_service.h"
#include "content/public/browser/notification_source.h"
#include "ui/base/theme_provider.h"

ProfileThemeUpdateService::ProfileThemeUpdateService(
    Profile* profile,
    ProfileAttributesStorage* profile_attributes_storage,
    ThemeService* theme_service)
    : profile_(profile),
      profile_attributes_storage_(profile_attributes_storage),
      theme_service_(theme_service) {
  notification_registrar_.Add(this, chrome::NOTIFICATION_BROWSER_THEME_CHANGED,
                              content::NotificationService::AllSources());
  // Kicks off an update on startup.
  UpdateProfileThemeColors();
}

ProfileThemeUpdateService::~ProfileThemeUpdateService() = default;

void ProfileThemeUpdateService::UpdateProfileThemeColors() {
  ProfileAttributesEntry* entry = nullptr;
  bool has_entry = profile_attributes_storage_->GetProfileAttributesWithPath(
      profile_->GetPath(), &entry);
  if (!has_entry)
    return;

  if (!theme_service_->UsingAutogeneratedTheme()) {
    // Only save colors for autogenerated themes.
    entry->SetProfileThemeColors(base::nullopt);
    return;
  }

  ThemeService* service = ThemeServiceFactory::GetForProfile(profile_);
  ProfileThemeColors colors =
      GetProfileThemeColorsForThemeSupplier(service->GetThemeSupplier());
  entry->SetProfileThemeColors(colors);
}

// content::NotificationObserver:
void ProfileThemeUpdateService::Observe(
    int type,
    const content::NotificationSource& source,
    const content::NotificationDetails& details) {
  switch (type) {
    case chrome::NOTIFICATION_BROWSER_THEME_CHANGED:
      UpdateProfileThemeColors();
      break;
    default:
      NOTREACHED();
  }
}
