/*
 * Copyright (c) 2018-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"

#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_fp32_mmla_4VLx4(const float *A, int lda, const float *B, float *C, int ldc, int M, int N, int K, const float *bias, Activation act, bool accumulate) {
    const int K_stride = ((K + 1) / 2) * 2;
    const long loops_count = ((K + 4) / 8) - 1;
    K -= loops_count * 8;
    const long regs_count = (K / 4) - 1;
    K -= (regs_count + 1) * 4;
    const long leftovers = K;
    const long blocks_count = (K + 1) / 2;
    float nullbias[128];
    if (!accumulate && !bias) {
        memset(nullbias, 0, (2 * get_vector_length<float>() * sizeof(float)));
    }
    float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
    float maxval =   static_cast<float>(std::numeric_limits<float>::infinity());
    const float * const minptr = &minval;
    const float * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    int rows_to_compute;

    for (int y=0; y<M; y+=rows_to_compute) {
        const float * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(float);

        float *c_ptr0 = C + (y * ldc);

        rows_to_compute = M-y;
        if (rows_to_compute > 8) {
            if (rows_to_compute % 8) {
                rows_to_compute = 8 - 1;
            } else {
                rows_to_compute = 8;
            }
        }

        for (int x0=0; x0<N; x0+=(2 * get_vector_length<float>())) {
            const long width = std::min((unsigned long)N-x0, (2 * get_vector_length<float>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = blocks_count;
            const float *a_ptr0 = a_ptr0_base;
            const float *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(float);
            const float *biasptr = bias ? bias+x0 : nullbias;

            switch(rows_to_compute) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z1.s, #0\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z14.s, #0\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "mov z1.s, #0\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "mov z14.s, #0\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z5.s, #0\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "mov z1.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z5.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "mov z1.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z5.s, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp1 z1.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "st1w z1.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z3.s, #0\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z20.d, z16.d\n"
                        "mov z21.d, z17.d\n"
                        "mov z22.d, z18.d\n"
                        "mov z23.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z3.s, #0\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "mov z14.s, #0\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "mov z14.s, #0\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z7.s, #0\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z3.s, #0\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z7.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z3.s, #0\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z7.s, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp1 z5.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "st1w z5.s, p1, [c_ptr2, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z21.d, z17.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z22.d, z18.d\n"
                        "mov z23.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p0/z, [c_ptr3]\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp2 z5.s, z20.s, z21.s\n"
                        "uzp1 z6.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "uzp2 z7.s, z22.s, z23.s\n"
                        "st1w z5.s, p0, [c_ptr3]\n"
                        "st1w z6.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z7.s, p1, [c_ptr3, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
                case 5:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "a_ptr4 .req X3\n"
                        "c_ptr1 .req X4\n"
                        "c_ptr2 .req X5\n"
                        "c_ptr3 .req X6\n"
                        "c_ptr4 .req X7\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "add a_ptr4, a_ptr3, %[lda]\n"
                        "add c_ptr4, c_ptr3, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z5.s, #0\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z21.d, z17.d\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "mov z22.d, z18.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z23.d, z19.d\n"
                        "mov z24.d, z16.d\n"
                        "mov z25.d, z17.d\n"
                        "mov z26.d, z18.d\n"
                        "mov z27.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z5.s, #0\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p0/z, [c_ptr3]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr4]\n"
                        "mov z14.s, #0\n"
                        "zip1 z24.s, z13.s, z14.s\n"
                        "zip2 z25.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr4, #1, MUL VL]\n"
                        "mov z14.s, #0\n"
                        "zip1 z26.s, z13.s, z14.s\n"
                        "zip2 z27.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z9.s, #0\n"
                        "add a_ptr4, a_ptr4, #0x20\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z5.s, #0\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z9.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "addvl a_ptr4, a_ptr4, #2\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z5.s, #0\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p6/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "addvl a_ptr4, a_ptr4, #1\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z9.s, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp2 z5.s, z20.s, z21.s\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "uzp1 z6.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "uzp2 z7.s, z22.s, z23.s\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z5.s, p0, [c_ptr3]\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "st1w z6.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "uzp1 z8.s, z24.s, z25.s\n"
                        "st1w z7.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "uzp1 z9.s, z26.s, z27.s\n"
                        "st1w z8.s, p0, [c_ptr4]\n"
                        "st1w z9.s, p1, [c_ptr4, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq a_ptr4\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        ".unreq c_ptr4\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "cc", "memory"
                    );
                    break;
                case 6:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "a_ptr4 .req X3\n"
                        "a_ptr5 .req X4\n"
                        "c_ptr1 .req X5\n"
                        "c_ptr2 .req X6\n"
                        "c_ptr3 .req X7\n"
                        "c_ptr4 .req X8\n"
                        "c_ptr5 .req X9\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "add a_ptr4, a_ptr3, %[lda]\n"
                        "add c_ptr4, c_ptr3, %[ldc]\n"
                        "add a_ptr5, a_ptr4, %[lda]\n"
                        "add c_ptr5, c_ptr4, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z21.d, z17.d\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z22.d, z18.d\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "mov z23.d, z19.d\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "mov z24.d, z16.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z25.d, z17.d\n"
                        "mov z26.d, z18.d\n"
                        "mov z27.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p0/z, [c_ptr3]\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr4]\n"
                        "ld1w z14.s, p0/z, [c_ptr5]\n"
                        "zip1 z24.s, z13.s, z14.s\n"
                        "zip2 z25.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr4, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr5, #1, MUL VL]\n"
                        "zip1 z26.s, z13.s, z14.s\n"
                        "zip2 z27.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "add a_ptr4, a_ptr4, #0x20\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "add a_ptr5, a_ptr5, #0x20\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5, #-0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl a_ptr4, a_ptr4, #2\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl a_ptr5, a_ptr5, #2\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p6/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "addvl a_ptr4, a_ptr4, #1\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p6/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl a_ptr5, a_ptr5, #1\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp2 z5.s, z20.s, z21.s\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "uzp1 z6.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "uzp2 z7.s, z22.s, z23.s\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z5.s, p0, [c_ptr3]\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "st1w z6.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "uzp1 z8.s, z24.s, z25.s\n"
                        "st1w z7.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "uzp2 z9.s, z24.s, z25.s\n"
                        "uzp1 z10.s, z26.s, z27.s\n"
                        "uzp2 z11.s, z26.s, z27.s\n"
                        "st1w z8.s, p0, [c_ptr4]\n"
                        "st1w z9.s, p0, [c_ptr5]\n"
                        "st1w z10.s, p1, [c_ptr4, #1, MUL VL]\n"
                        "st1w z11.s, p1, [c_ptr5, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq a_ptr4\n"
                        ".unreq a_ptr5\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        ".unreq c_ptr4\n"
                        ".unreq c_ptr5\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "cc", "memory"
                    );
                    break;
                case 7:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "a_ptr4 .req X3\n"
                        "a_ptr5 .req X4\n"
                        "a_ptr6 .req X5\n"
                        "c_ptr1 .req X6\n"
                        "c_ptr2 .req X7\n"
                        "c_ptr3 .req X8\n"
                        "c_ptr4 .req X9\n"
                        "c_ptr5 .req X10\n"
                        "c_ptr6 .req X11\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "add a_ptr4, a_ptr3, %[lda]\n"
                        "add c_ptr4, c_ptr3, %[ldc]\n"
                        "add a_ptr5, a_ptr4, %[lda]\n"
                        "add c_ptr5, c_ptr4, %[ldc]\n"
                        "add a_ptr6, a_ptr5, %[lda]\n"
                        "add c_ptr6, c_ptr5, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z7.s, #0\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "mov z20.d, z16.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z21.d, z17.d\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z22.d, z18.d\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "mov z23.d, z19.d\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "mov z24.d, z16.d\n"
                        "add a_ptr6, a_ptr6, #0x10\n"
                        "mov z25.d, z17.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z26.d, z18.d\n"
                        "mov z27.d, z19.d\n"
                        "mov z28.d, z16.d\n"
                        "mov z29.d, z17.d\n"
                        "mov z30.d, z18.d\n"
                        "mov z31.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z7.s, #0\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p0/z, [c_ptr3]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "ld1w z14.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "add a_ptr6, a_ptr6, #0x10\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr4]\n"
                        "ld1w z14.s, p0/z, [c_ptr5]\n"
                        "zip1 z24.s, z13.s, z14.s\n"
                        "zip2 z25.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr4, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr5, #1, MUL VL]\n"
                        "zip1 z26.s, z13.s, z14.s\n"
                        "zip2 z27.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr6]\n"
                        "mov z14.s, #0\n"
                        "zip1 z28.s, z13.s, z14.s\n"
                        "zip2 z29.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr6, #1, MUL VL]\n"
                        "mov z14.s, #0\n"
                        "zip1 z30.s, z13.s, z14.s\n"
                        "zip2 z31.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "add a_ptr4, a_ptr4, #0x20\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "add a_ptr5, a_ptr5, #0x20\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p7/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z11.s, #0\n"
                        "add a_ptr6, a_ptr6, #0x20\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6, #-0x10]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5, #-0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z7.s, #0\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p7/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z11.s, #0\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl a_ptr4, a_ptr4, #2\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl a_ptr5, a_ptr5, #2\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "addvl a_ptr6, a_ptr6, #2\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "mov z7.s, #0\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p6/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p6/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "addvl a_ptr4, a_ptr4, #1\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "addvl a_ptr5, a_ptr5, #1\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p6/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z11.s, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "addvl a_ptr6, a_ptr6, #1\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp2 z5.s, z20.s, z21.s\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "uzp1 z6.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "uzp2 z7.s, z22.s, z23.s\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z5.s, p0, [c_ptr3]\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "fmax z28.s, p7/m, z28.s, z14.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "st1w z6.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "fmin z28.s, p7/m, z28.s, z15.s\n"
                        "st1w z7.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "uzp1 z8.s, z24.s, z25.s\n"
                        "uzp2 z9.s, z24.s, z25.s\n"
                        "uzp1 z10.s, z26.s, z27.s\n"
                        "uzp2 z11.s, z26.s, z27.s\n"
                        "st1w z8.s, p0, [c_ptr4]\n"
                        "fmax z29.s, p7/m, z29.s, z14.s\n"
                        "fmax z30.s, p7/m, z30.s, z14.s\n"
                        "fmax z31.s, p7/m, z31.s, z14.s\n"
                        "st1w z9.s, p0, [c_ptr5]\n"
                        "fmin z29.s, p7/m, z29.s, z15.s\n"
                        "fmin z30.s, p7/m, z30.s, z15.s\n"
                        "st1w z10.s, p1, [c_ptr4, #1, MUL VL]\n"
                        "fmin z31.s, p7/m, z31.s, z15.s\n"
                        "uzp1 z12.s, z28.s, z29.s\n"
                        "st1w z11.s, p1, [c_ptr5, #1, MUL VL]\n"
                        "uzp1 z13.s, z30.s, z31.s\n"
                        "st1w z12.s, p0, [c_ptr6]\n"
                        "st1w z13.s, p1, [c_ptr6, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq a_ptr4\n"
                        ".unreq a_ptr5\n"
                        ".unreq a_ptr6\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        ".unreq c_ptr4\n"
                        ".unreq c_ptr5\n"
                        ".unreq c_ptr6\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "cc", "memory"
                    );
                    break;
                default:
                case 8:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "a_ptr4 .req X3\n"
                        "a_ptr5 .req X4\n"
                        "a_ptr6 .req X5\n"
                        "a_ptr7 .req X6\n"
                        "c_ptr1 .req X7\n"
                        "c_ptr2 .req X8\n"
                        "c_ptr3 .req X9\n"
                        "c_ptr4 .req X10\n"
                        "c_ptr5 .req X11\n"
                        "c_ptr6 .req X12\n"
                        "c_ptr7 .req X13\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "add a_ptr4, a_ptr3, %[lda]\n"
                        "add c_ptr4, c_ptr3, %[ldc]\n"
                        "add a_ptr5, a_ptr4, %[lda]\n"
                        "add c_ptr5, c_ptr4, %[ldc]\n"
                        "add a_ptr6, a_ptr5, %[lda]\n"
                        "add c_ptr6, c_ptr5, %[ldc]\n"
                        "add a_ptr7, a_ptr6, %[lda]\n"
                        "add c_ptr7, c_ptr6, %[ldc]\n"
                        "whilelt p6.s, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.s\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z15.s, p0/z, [%[biasptr]]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z15.s, z15.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z15.s, z15.s\n"
                        "ld1w z15.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z18.s, z15.s, z15.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z15.s, z15.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                        "mov z20.d, z16.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z21.d, z17.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z22.d, z18.d\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z23.d, z19.d\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "mov z24.d, z16.d\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "mov z25.d, z17.d\n"
                        "add a_ptr6, a_ptr6, #0x10\n"
                        "mov z26.d, z18.d\n"
                        "add a_ptr7, a_ptr7, #0x10\n"
                        "mov z27.d, z19.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "mov z28.d, z16.d\n"
                        "mov z29.d, z17.d\n"
                        "mov z30.d, z18.d\n"
                        "mov z31.d, z19.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z13.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z14.s, p0/z, [c_ptr1]\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z16.s, z13.s, z14.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                        "zip2 z17.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        "ld1w z14.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z18.s, z13.s, z14.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                        "zip2 z19.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr2]\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        "ld1w z14.s, p0/z, [c_ptr3]\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                        "add a_ptr4, a_ptr4, #0x10\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                        "zip1 z20.s, z13.s, z14.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "zip2 z21.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "add a_ptr5, a_ptr5, #0x10\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add a_ptr6, a_ptr6, #0x10\n"
                        "zip1 z22.s, z13.s, z14.s\n"
                        "add a_ptr7, a_ptr7, #0x10\n"
                        "zip2 z23.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr4]\n"
                        "ld1w z14.s, p0/z, [c_ptr5]\n"
                        "zip1 z24.s, z13.s, z14.s\n"
                        "zip2 z25.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr4, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr5, #1, MUL VL]\n"
                        "zip1 z26.s, z13.s, z14.s\n"
                        "zip2 z27.s, z13.s, z14.s\n"
                        "ld1w z13.s, p0/z, [c_ptr6]\n"
                        "ld1w z14.s, p0/z, [c_ptr7]\n"
                        "zip1 z28.s, z13.s, z14.s\n"
                        "zip2 z29.s, z13.s, z14.s\n"
                        "ld1w z13.s, p1/z, [c_ptr6, #1, MUL VL]\n"
                        "ld1w z14.s, p1/z, [c_ptr7, #1, MUL VL]\n"
                        "zip1 z30.s, z13.s, z14.s\n"
                        "zip2 z31.s, z13.s, z14.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "add a_ptr4, a_ptr4, #0x20\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "add a_ptr5, a_ptr5, #0x20\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p7/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z11.s, p7/z, [a_ptr7]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "add a_ptr6, a_ptr6, #0x20\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "add a_ptr7, a_ptr7, #0x20\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1rqw z6.s, p7/z, [a_ptr6, #-0x10]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [a_ptr4, #-0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p7/z, [a_ptr5, #-0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr7, #-0x10]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1rqw z2.s, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z3.s, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p7/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p7/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p7/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p7/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p7/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p7/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p7/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z11.s, p7/z, [a_ptr7]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl a_ptr4, a_ptr4, #2\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        "addvl a_ptr5, a_ptr5, #2\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        "addvl a_ptr6, a_ptr6, #2\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        "addvl a_ptr7, a_ptr7, #2\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        "trn1 z8.d, z0.d, z1.d\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "trn1 z9.d, z2.d, z3.d\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "trn1 z10.d, z4.d, z5.d\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "trn1 z11.d, z6.d, z7.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "trn2 z2.d, z4.d, z5.d\n"
                        "ld1rqw z4.s, p6/z, [%[a_ptr0]]\n"
                        "trn2 z3.d, z6.d, z7.d\n"
                        "ld1rqw z5.s, p6/z, [a_ptr1]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        "ld1rqw z6.s, p6/z, [a_ptr2]\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        "ld1rqw z7.s, p6/z, [a_ptr3]\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        "ld1rqw z8.s, p6/z, [a_ptr4]\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        "ld1rqw z9.s, p6/z, [a_ptr5]\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        "addvl a_ptr4, a_ptr4, #1\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        "addvl a_ptr5, a_ptr5, #1\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        "ld1rqw z10.s, p6/z, [a_ptr6]\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "ld1rqw z11.s, p6/z, [a_ptr7]\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #4\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        "addvl a_ptr6, a_ptr6, #1\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        "addvl a_ptr7, a_ptr7, #1\n"
                        "trn1 z0.d, z4.d, z5.d\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        "trn1 z1.d, z6.d, z7.d\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        "trn1 z2.d, z8.d, z9.d\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "trn1 z3.d, z10.d, z11.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z11.d, z10.d, z11.d\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0]]\n"
                        "trn2 z10.d, z8.d, z9.d\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "trn2 z9.d, z6.d, z7.d\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn2 z8.d, z4.d, z5.d\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x64a0e590 // fmmla z16.s, z12.s, z0.s\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x64a0e5b1 // fmmla z17.s, z13.s, z0.s\n"
                        ".inst 0x64a0e5d2 // fmmla z18.s, z14.s, z0.s\n"
                        ".inst 0x64a0e5f3 // fmmla z19.s, z15.s, z0.s\n"
                        ".inst 0x64a1e594 // fmmla z20.s, z12.s, z1.s\n"
                        ".inst 0x64a1e5b5 // fmmla z21.s, z13.s, z1.s\n"
                        ".inst 0x64a1e5d6 // fmmla z22.s, z14.s, z1.s\n"
                        ".inst 0x64a1e5f7 // fmmla z23.s, z15.s, z1.s\n"
                        ".inst 0x64a2e598 // fmmla z24.s, z12.s, z2.s\n"
                        ".inst 0x64a2e5b9 // fmmla z25.s, z13.s, z2.s\n"
                        ".inst 0x64a2e5da // fmmla z26.s, z14.s, z2.s\n"
                        ".inst 0x64a2e5fb // fmmla z27.s, z15.s, z2.s\n"
                        ".inst 0x64a3e59c // fmmla z28.s, z12.s, z3.s\n"
                        ".inst 0x64a3e5bd // fmmla z29.s, z13.s, z3.s\n"
                        ".inst 0x64a3e5de // fmmla z30.s, z14.s, z3.s\n"
                        ".inst 0x64a3e5ff // fmmla z31.s, z15.s, z3.s\n"
                        "b.eq 5f\n"
                        "ld1w z12.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1w z13.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1w z14.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1w z15.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x64a8e590 // fmmla z16.s, z12.s, z8.s\n"
                        ".inst 0x64a8e5b1 // fmmla z17.s, z13.s, z8.s\n"
                        ".inst 0x64a8e5d2 // fmmla z18.s, z14.s, z8.s\n"
                        ".inst 0x64a8e5f3 // fmmla z19.s, z15.s, z8.s\n"
                        ".inst 0x64a9e594 // fmmla z20.s, z12.s, z9.s\n"
                        ".inst 0x64a9e5b5 // fmmla z21.s, z13.s, z9.s\n"
                        ".inst 0x64a9e5d6 // fmmla z22.s, z14.s, z9.s\n"
                        ".inst 0x64a9e5f7 // fmmla z23.s, z15.s, z9.s\n"
                        ".inst 0x64aae598 // fmmla z24.s, z12.s, z10.s\n"
                        ".inst 0x64aae5b9 // fmmla z25.s, z13.s, z10.s\n"
                        ".inst 0x64aae5da // fmmla z26.s, z14.s, z10.s\n"
                        ".inst 0x64aae5fb // fmmla z27.s, z15.s, z10.s\n"
                        ".inst 0x64abe59c // fmmla z28.s, z12.s, z11.s\n"
                        ".inst 0x64abe5bd // fmmla z29.s, z13.s, z11.s\n"
                        ".inst 0x64abe5de // fmmla z30.s, z14.s, z11.s\n"
                        ".inst 0x64abe5ff // fmmla z31.s, z15.s, z11.s\n"
                        "5:\n"
                        "ld1rw z14.s, p7/z, [%[minptr]]\n"
                        "ld1rw z15.s, p7/z, [%[maxptr]]\n"
                        "fmax z16.s, p7/m, z16.s, z14.s\n"
                        "fmax z17.s, p7/m, z17.s, z14.s\n"
                        "fmax z18.s, p7/m, z18.s, z14.s\n"
                        "fmax z19.s, p7/m, z19.s, z14.s\n"
                        "fmin z16.s, p7/m, z16.s, z15.s\n"
                        "fmin z17.s, p7/m, z17.s, z15.s\n"
                        "fmin z18.s, p7/m, z18.s, z15.s\n"
                        "fmin z19.s, p7/m, z19.s, z15.s\n"
                        "fmax z20.s, p7/m, z20.s, z14.s\n"
                        "uzp1 z0.s, z16.s, z17.s\n"
                        "uzp2 z1.s, z16.s, z17.s\n"
                        "uzp1 z2.s, z18.s, z19.s\n"
                        "uzp2 z3.s, z18.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z20.s, p7/m, z20.s, z15.s\n"
                        "fmax z21.s, p7/m, z21.s, z14.s\n"
                        "fmax z22.s, p7/m, z22.s, z14.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z23.s, p7/m, z23.s, z14.s\n"
                        "fmax z24.s, p7/m, z24.s, z14.s\n"
                        "fmin z21.s, p7/m, z21.s, z15.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z22.s, p7/m, z22.s, z15.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #2\n"
                        "fmin z23.s, p7/m, z23.s, z15.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z20.s, z21.s\n"
                        "uzp2 z5.s, z20.s, z21.s\n"
                        "fmin z24.s, p7/m, z24.s, z15.s\n"
                        "uzp1 z6.s, z22.s, z23.s\n"
                        "st1w z4.s, p0, [c_ptr2]\n"
                        "uzp2 z7.s, z22.s, z23.s\n"
                        "fmax z25.s, p7/m, z25.s, z14.s\n"
                        "fmax z26.s, p7/m, z26.s, z14.s\n"
                        "st1w z5.s, p0, [c_ptr3]\n"
                        "fmax z27.s, p7/m, z27.s, z14.s\n"
                        "fmax z28.s, p7/m, z28.s, z14.s\n"
                        "fmin z25.s, p7/m, z25.s, z15.s\n"
                        "st1w z6.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z15.s\n"
                        "fmin z27.s, p7/m, z27.s, z15.s\n"
                        "fmin z28.s, p7/m, z28.s, z15.s\n"
                        "st1w z7.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "uzp1 z8.s, z24.s, z25.s\n"
                        "uzp2 z9.s, z24.s, z25.s\n"
                        "uzp1 z10.s, z26.s, z27.s\n"
                        "uzp2 z11.s, z26.s, z27.s\n"
                        "st1w z8.s, p0, [c_ptr4]\n"
                        "fmax z29.s, p7/m, z29.s, z14.s\n"
                        "fmax z30.s, p7/m, z30.s, z14.s\n"
                        "fmax z31.s, p7/m, z31.s, z14.s\n"
                        "st1w z9.s, p0, [c_ptr5]\n"
                        "fmin z29.s, p7/m, z29.s, z15.s\n"
                        "fmin z30.s, p7/m, z30.s, z15.s\n"
                        "st1w z10.s, p1, [c_ptr4, #1, MUL VL]\n"
                        "fmin z31.s, p7/m, z31.s, z15.s\n"
                        "uzp1 z12.s, z28.s, z29.s\n"
                        "st1w z11.s, p1, [c_ptr5, #1, MUL VL]\n"
                        "uzp2 z13.s, z28.s, z29.s\n"
                        "uzp1 z14.s, z30.s, z31.s\n"
                        "uzp2 z15.s, z30.s, z31.s\n"
                        "st1w z12.s, p0, [c_ptr6]\n"
                        "st1w z13.s, p0, [c_ptr7]\n"
                        "st1w z14.s, p1, [c_ptr6, #1, MUL VL]\n"
                        "st1w z15.s, p1, [c_ptr7, #1, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq a_ptr4\n"
                        ".unreq a_ptr5\n"
                        ".unreq a_ptr6\n"
                        ".unreq a_ptr7\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        ".unreq c_ptr4\n"
                        ".unreq c_ptr5\n"
                        ".unreq c_ptr6\n"
                        ".unreq c_ptr7\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
