(*
 * Copyright 2007 Samuel Mimram
 *
 * This file is part of ocaml-ogg.
 *
 * ocaml-ogg is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ocaml-ogg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with ocaml-ogg; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GNU Library General Public License, you may
 * link, statically or dynamically, a "work that uses the Library" with a publicly
 * distributed version of the Library to produce an executable file containing
 * portions of the Library, and distribute that executable file under terms of
 * your choice, without any of the additional requirements listed in clause 6
 * of the GNU Library General Public License.
 * By "a publicly distributed version of the Library", we mean either the unmodified
 * Library as distributed by INRIA, or a modified version of the Library that is
 * distributed under the conditions defined in clause 3 of the GNU Library General
 * Public License. This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU Library General Public License.
 *
 *)

(**
  * Functions for manipulating ogg streams files using libogg.
  *
  * @author Samuel Mimram
  *)

(**
  * The [page] struct encapsulates the data for an Ogg page.
  *
  * Ogg pages are the fundamental unit of framing and interleave in an ogg
  * bitstream. They are made up of packet segments of 255 bytes each. There can
  * be as many as 255 packet segments per page, for a maximum page size of a
  * little under 64 kB. This is not a practical limitation as the segments can be
  * joined across page boundaries allowing packets of arbitrary size. In practice
  * pages are usually around 4 kB.
  *)
type page = string

(** Multiple Ogg pages. *)
type pages = string

exception Not_enough_data

module Stream :
sig
  (**
    * The [stream] values track the current encode/decode state of the
    * current logical bitstream.
    *)
  type t

  (**
    * Create a [stream].
    *)
  val create : ?serial:int -> unit -> t

  (**
    * This function forms packets into pages. Internally,
    * it assembles the accumulated packet bodies into an Ogg page
    * suitable for writing to a stream.
    *
    * This function will only return a page when a "reasonable" amount of packet
    * data is available. Normally this is appropriate since it limits the overhead
    * of the Ogg page headers in the bitstream. Call
    * [flush] if immediate page generation is desired. This may be
    * occasionally necessary, for example, to limit the temporal latency of a
    * variable bitrate stream.
    *)
  val pageout : t -> page

  (** Same as [pageout] but may return multiple pages if available. *)
  val pagesout : t -> pages

  val pagesout_eos : t -> pages

  (**
    * This function checks for remaining packets inside the stream and forces
    * remaining packets into a page, regardless of the size of the page.
    *
    * This should only be used when you want to flush an undersized page from the
    * middle of the stream. Otherwise, [pageout] should always be used.
    *
    * This function can be used to verify that all packets have been flushed. If
    * the return value is the empty string, all packets have been placed into a page.
    *)
  val flush : t -> pages
end

module Sync :
sig
  type t

  val create : unit -> t
end
