/**
 * \file FormSearch.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Edwin Leuven
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "FormSearch.h"
#include "ControlSearch.h"
#include "forms/form_search.h"

#include "Tooltips.h"
#include "xformsBC.h"

#include "lyx_forms.h"

using std::string;

namespace lyx {
namespace frontend {

typedef FormController<ControlSearch, FormView<FD_search> > base_class;

FormSearch::FormSearch(Dialog & parent)
	: base_class(parent, _("Find and Replace"))
{}


void FormSearch::build()
{
	dialog_.reset(build_search(this));

	// Manage the ok, apply and cancel/close buttons
	bcview().setCancel(dialog_->button_close);

	// disable for read-only documents
	bcview().addReadOnly(dialog_->input_replace);
	bcview().addReadOnly(dialog_->button_replace);
	bcview().addReadOnly(dialog_->button_replaceall);

	// set up the tooltips
	string str = _("Enter the string you want to find.");
	tooltips().init(dialog_->input_search, str);
	str = _("Enter the replacement string.");
	tooltips().init(dialog_->input_replace, str);
	str = _("Continue to next search result.");
	tooltips().init(dialog_->button_findnext, str);
	str = _("Replace search result by replacement string.");
	tooltips().init(dialog_->button_replace, str);
	str = _("Replace all by replacement string.");
	tooltips().init(dialog_->button_replaceall, str);
	str = _("Do case sensitive search.");
	tooltips().init(dialog_->check_casesensitive, str);
	str = _("Search only matching words.");
	tooltips().init(dialog_->check_matchword, str);
	str = _("Search backwards.");
	tooltips().init(dialog_->check_searchbackwards, str);
}


void FormSearch::update()
{
	fl_set_input_selected(dialog_->input_search, true);
	fl_set_focus_object(dialog_->form, dialog_->input_search);
}


namespace {

string const getUntrimmedString(FL_OBJECT * ob)
{
	BOOST_ASSERT(ob->objclass == FL_INPUT);

	char const * const tmp = fl_get_input(ob);
	return tmp ? tmp : string();
}

} // namespace anon



ButtonPolicy::SMInput FormSearch::input(FL_OBJECT * ob, long)
{
	if (ob == dialog_->button_findnext) {
		controller().find(getUntrimmedString(dialog_->input_search),
				  fl_get_button(dialog_->check_casesensitive),
				  fl_get_button(dialog_->check_matchword),
				  !fl_get_button(dialog_->check_searchbackwards));

	} else if (ob == dialog_->button_replace || ob == dialog_->button_replaceall) {
		bool const all = (ob == dialog_->button_replaceall);

		controller().replace(getUntrimmedString(dialog_->input_search),
				     getUntrimmedString(dialog_->input_replace),
				     fl_get_button(dialog_->check_casesensitive),
				     fl_get_button(dialog_->check_matchword),
				     !fl_get_button(dialog_->check_searchbackwards),
				     all);
	}

	return ButtonPolicy::SMI_VALID;
}

} // namespace frontend
} // namespace lyx
