#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h> 

#include "../include/os.h"
#include "../include/string.h"

#include "sfm.h"

#include "obj.h"
#include "sarreality.h"
#include "weather.h"
#include "sar.h"
#include "config.h"


sar_weather_data_struct *SARWeatherPresetsInit(void *core_ptr);
void SARWeatherPresetsShutdown(sar_weather_data_struct *w);

void SARWeatherEntryDelete(
        sar_weather_data_struct *w, sar_weather_data_entry_struct *entry
);

void SARWeatherSetScenePreset(
        sar_weather_data_struct *w,
        sar_scene_struct *scene,
        const char *name
);

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


/*
 *	Initializes a new sar_weather_data_struct structure and returns
 *	the pointer to it.
 *
 *	This pointer must be freed by the calling function by a call to
 *	SARWeatherPresetsShutdown().
 */
sar_weather_data_struct *SARWeatherPresetsInit(void *core_ptr)
{
	sar_weather_data_struct *w = (sar_weather_data_struct *)calloc(
	    1, sizeof(sar_weather_data_struct)
	);
	if(w == NULL)
	    return(w);

	w->core_ptr = core_ptr;

        if(option.runtime_debug)
        {
            printf(
"SARWeatherPresetsInit(): Initialized preset weather resources.\n"
            );
        }

	return(w);
}

/*
 *      Deallocates all weather resources on the given
 *	sar_weather_data_struct structure and the structure itself.
 *
 *	The given pointer must not be referanced again after calling
 *	this function.
 */
void SARWeatherPresetsShutdown(sar_weather_data_struct *w)
{
	int i;

	if(w == NULL)
	    return;

	if(option.runtime_debug)
	{
	    printf(
"SARWeatherPresetsShutdown(): Shutting down preset weather resources.\n"
	    );
	}

	/* Deallocate each preset weather entry structure. */
	for(i = 0; i < w->total_presets; i++)
	    SARWeatherEntryDelete(
		w, w->preset[i]
	    );

	free(w->preset);
	w->preset = NULL;
	w->total_presets = 0;

	/* Deallocate structure itself. */
	free(w);

	return;
}

/*
 *	Deallocates the given weather entry structure and all its
 *	resources.
 */
void SARWeatherEntryDelete(
        sar_weather_data_struct *w, sar_weather_data_entry_struct *entry  
)
{
	int i;
	sar_cloud_layer_struct *cloud_layer_ptr;
	sar_cloud_bb_struct *cloud_bb_ptr;

	if(entry == NULL)
	    return;

	/* Deallocate weather name. */
	free(entry->name);
	entry->name = NULL;

	/* Destroy all cloud layers. */
	for(i = 0; i < entry->total_cloud_layers; i++)
	{
	    cloud_layer_ptr = entry->cloud_layer[i];
	    if(cloud_layer_ptr == NULL)
		continue;

	    SARCloudLayerDestroy(NULL, cloud_layer_ptr);
	}
	free(entry->cloud_layer);
	entry->cloud_layer = NULL;
	entry->total_cloud_layers = 0;

	/* Destroy all cloud `billboard' objects. */
        for(i = 0; i < entry->total_cloud_bbs; i++)
        {
            cloud_bb_ptr = entry->cloud_bb[i];
            if(cloud_bb_ptr == NULL)
                continue;

            SARCloudBBDestroy(cloud_bb_ptr);
        }
        free(entry->cloud_bb);
        entry->cloud_bb = NULL;
        entry->total_cloud_bbs = 0;


	/* Deallocate structure itself. */
	free(entry);

	return;
}


/*
 *	Sets the weather for the given scene to the given preset
 *	value.
 */
void SARWeatherSetScenePreset(
	sar_weather_data_struct *w,
	sar_scene_struct *scene,
	const char *name
)
{
	sar_weather_data_struct *wd = w;
	sar_weather_data_entry_struct *wdp_ptr = NULL;

	int i, cloud_layer_num, cloud_bb_num;
	sar_scene_horizon_struct *horizon_ptr;
	sar_cloud_layer_struct	*cloud_layer_ptr,
				*cloud_layer_src_ptr;
	sar_cloud_bb_struct	*cloud_bb_ptr,
				*cloud_bb_src_ptr;
	sar_color_struct	*sky_nominal_color,
				*sky_brighten_color,
				*sky_darken_color,
				*star_color,
				*moon_color;


	if((wd == NULL) || (scene == NULL))
	    return;

	/* Set default sky shading colors. */
	sky_nominal_color = &scene->sky_nominal_color;
	sky_nominal_color->r = 0.54;
	sky_nominal_color->g = 0.71;
	sky_nominal_color->b = 0.96;
        sky_brighten_color = &scene->sky_brighten_color;
        sky_brighten_color->r = 1.00;
        sky_brighten_color->g = 0.95;
        sky_brighten_color->b = 0.60;
        sky_darken_color = &scene->sky_darken_color;
        sky_darken_color->r = 0.96;
        sky_darken_color->g = 0.60;
        sky_darken_color->b = 0.60;

	star_color = &scene->star_color;
        star_color->r = 1.00;
        star_color->g = 1.00;
        star_color->b = 1.00;
        moon_color = &scene->moon_color;
        moon_color->r = 0.10;  
        moon_color->g = 0.10;
        moon_color->b = 0.10;


	/* Set default atmosphere parameters. */
	scene->atmosphere_dist_coeff = 0.90;
	scene->atmosphere_density_coeff = 0.30;

	/* Reset horizon's time of day value to -1 so a layer call to
	 * SARSimUpdateScene() will update the horizon resources.
	 */
	horizon_ptr = &scene->horizon;
	horizon_ptr->last_tod = -1;

	/* Delete any existing cloud layers on scene structure. */
	for(i = 0; i < scene->total_cloud_layers; i++)
	    SARCloudLayerDestroy(scene, scene->cloud_layer[i]);
	free(scene->cloud_layer);
	scene->cloud_layer = NULL;
	scene->total_cloud_layers = 0;

	/* Delete any existing cloud `billboard' objects on scene
	 * structure.
	 */
	for(i = 0; i < scene->total_cloud_bbs; i++)
            SARCloudBBDestroy(scene->cloud_bb[i]);
        free(scene->cloud_bb);
        scene->cloud_bb = NULL;
        scene->total_cloud_bbs = 0;


	/* ******************************************************** */

	/* Match preset weather data structure with given
	 * weather code value.
	 */
	if(name == NULL)
	{
            /* No name given, so first preset weather data structure
	     * as default if possable.
	     */
	    wdp_ptr = ((wd->total_presets > 0) ? wd->preset[0] : NULL);
	}
	else
	{
	    for(i = 0; i < wd->total_presets; i++)
	    {
	        wdp_ptr = wd->preset[i];
	        if(wdp_ptr == NULL)
		    continue;

		if(!strcasecmp(wdp_ptr->name, name))
		    break;
	    }
	    if(i >= wd->total_presets)
	    {
	        /* No match, assume first preset weather data structure
	         * as default if possable.
	         */
		wdp_ptr = ((wd->total_presets > 0) ? wd->preset[0] : NULL);

		if(wdp_ptr != NULL)
		    fprintf(
			stderr,
 "SARWeatherSetScenePreset(): No such weather preset named `%s', using default.\n",
			name
		    );
		else
                    fprintf(   
                        stderr,
 "SARWeatherSetScenePreset(): No such weather preset named `%s' and no default available.\n",
                        name
                    );
	    }
	    else
	    {
		/* Got match. */
		wdp_ptr = wd->preset[i];
	    }
	}

	/* No weather data preset available, then fail. */
	if(wdp_ptr == NULL)
	    return;

/* Adds a cloud layer pointer to the scene structure. */
#define DO_ADD_CLOUD_LAYER      \
{ \
 cloud_layer_num = scene->total_cloud_layers; \
 scene->total_cloud_layers++; \
 scene->cloud_layer = (sar_cloud_layer_struct **)realloc( \
  scene->cloud_layer, \
  scene->total_cloud_layers * sizeof(sar_cloud_layer_struct *) \
 ); \
 if(scene->cloud_layer == NULL) \
 { \
  scene->total_cloud_layers = 0; \
 } \
 else \
 { \
  scene->cloud_layer[cloud_layer_num] = cloud_layer_ptr; \
 } \
}

/* Adds a cloud `billboard' object pointer to the scene structure. */
#define DO_ADD_CLOUD_BB		\
{ \
 cloud_bb_num = scene->total_cloud_bbs; \
 scene->total_cloud_bbs++; \
 scene->cloud_bb = (sar_cloud_bb_struct **)realloc( \
  scene->cloud_bb, \
  scene->total_cloud_bbs * sizeof(sar_cloud_bb_struct *) \
 ); \
 if(scene->cloud_bb == NULL) \
 { \
  scene->total_cloud_bbs = 0; \
 } \
 else \
 { \
  scene->cloud_bb[cloud_bb_num] = cloud_bb_ptr; \
 } \
}


	/* Colors. */
	memcpy(
	    sky_nominal_color,
	    &wdp_ptr->sky_nominal_color,
	    sizeof(sar_color_struct)
	);
        memcpy( 
            sky_brighten_color,
            &wdp_ptr->sky_brighten_color,
            sizeof(sar_color_struct)
        );
        memcpy( 
            sky_darken_color,
            &wdp_ptr->sky_darken_color,
            sizeof(sar_color_struct)
        );

        memcpy(
            star_color,
            &wdp_ptr->star_color,
            sizeof(sar_color_struct)
        );
        memcpy(
            moon_color,
            &wdp_ptr->moon_color,
            sizeof(sar_color_struct)
        );


	/* Atmosphere. */
	scene->atmosphere_dist_coeff = wdp_ptr->atmosphere_dist_coeff;
        scene->atmosphere_density_coeff = wdp_ptr->atmosphere_density_coeff;


	/* Cloud layers. */
	for(i = 0; i < wdp_ptr->total_cloud_layers; i++)
	{
	    cloud_layer_src_ptr = wdp_ptr->cloud_layer[i];
	    if(cloud_layer_src_ptr == NULL)
		continue;

            cloud_layer_ptr = SARCloudLayerCreate(
                scene,
                cloud_layer_src_ptr->tile_width,
		cloud_layer_src_ptr->tile_height,
		cloud_layer_src_ptr->range,
		cloud_layer_src_ptr->z,
		cloud_layer_src_ptr->tex_name
            );
	    DO_ADD_CLOUD_LAYER
	}

        /* Cloud `billboard' objects. */
        for(i = 0; i < wdp_ptr->total_cloud_bbs; i++)
        { 
            cloud_bb_src_ptr = wdp_ptr->cloud_bb[i];
            if(cloud_bb_src_ptr == NULL)
                continue;

            cloud_bb_ptr = SARCloudBBCreate(
                scene,
		cloud_bb_src_ptr->tile_width,
		cloud_bb_src_ptr->tile_height,
		cloud_bb_src_ptr->x,
		cloud_bb_src_ptr->y,
		cloud_bb_src_ptr->z,
		cloud_bb_src_ptr->width,
		cloud_bb_src_ptr->height,
		cloud_bb_src_ptr->tex_name
            );
            DO_ADD_CLOUD_BB
        }

#undef DO_ADD_CLOUD_LAYER
#undef DO_ADD_CLOUD_BB

	return;
}
