#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <math.h>

#include "sfm.h"

#include "obj.h"
#include "messages.h"
#include "smoke.h"
#include "explosion.h"
#include "simmanage.h"
#include "simcb.h"
#include "simop.h"
#include "simutils.h"
#include "sarsound.h"
#include "sar.h"
#include "config.h"


void SARSimInitModelCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data
);
void SARSimDestroyModelCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data                     
);

void SARSimAirborneCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data
);
void SARSimTouchDownCB(
	void *realm_ptr, SFMModelStruct *model,
	void *client_data, double impact_coeff
);
void SARSimOverspeedCB(
	void *realm_ptr, SFMModelStruct *model,
	void *client_data, double cur_speed,
	double speed_max_expected
);
void SARSimCollisionCB(
        void *realm_ptr, SFMModelStruct *model, SFMModelStruct *obstruction,
        void *client_data, double impact_coeff 
);
void SARSimObjectCollisionCB(
        void *client_data,              /* Core structure. */
        sar_object_struct *obj_ptr, sar_object_struct *obstruction_obj_ptr,
        double impact_coeff
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


/*
 *	FDM initialize callback, sets up initial values from
 *	corresponding object (ie position and landed state values).
 */
void SARSimInitModelCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data
)
{
	int obj_num;
	sar_object_struct *obj_ptr;
	SFMRealmStruct *realm = (SFMRealmStruct *)realm_ptr;
	sar_core_struct *core_ptr = (sar_core_struct *)client_data;
	if((realm == NULL) ||
           (model == NULL) ||
           (core_ptr == NULL)
	)
	    return;

        /* Match object from FDM. */
        obj_ptr = SARSimMatchObjectFromFDM(
            core_ptr->object, core_ptr->total_objects,
            model, &obj_num
        );
        if(obj_ptr == NULL)
            return;

	/* Set up initial values, note that position and direction
	 * values will probably be reset soon afterwards since
	 * creation of FDM's usually happen in the middle of
	 * scene file parsings and their positions will be
	 * set after creation of the FDM.
	 */
/*
	model->position.x = obj_ptr->pos.x;
        model->position.y = obj_ptr->pos.y;
        model->position.z = obj_ptr->pos.z;

        model->direction.heading = obj_ptr->dir.heading;
        model->direction.pitch = obj_ptr->dir.pitch;
        model->direction.bank = obj_ptr->dir.bank;
 */

	/* Always assume landed. */
	model->landed_state = True;

	return;
}


/*
 *	FDM destroy callback.
 */
void SARSimDestroyModelCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data
)
{
	int i;
	sar_object_struct *obj_ptr;
	sar_object_aircraft_struct *obj_aircraft_ptr;
        SFMRealmStruct *realm = (SFMRealmStruct *)realm_ptr;
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        if((realm == NULL) ||    
           (model == NULL) ||
           (core_ptr == NULL)   
        )   
            return;

	/* Note that model input is considered invalid, do not
	 * referance! So scan through objects on core structure and
	 * unset fdm.
	 */
	for(i = 0; i < core_ptr->total_objects; i++)
	{
	    obj_ptr = core_ptr->object[i];
	    if(obj_ptr == NULL)
		continue;

	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr == NULL)
                    break;

		if(obj_aircraft_ptr->fdm == model)
		    obj_aircraft_ptr->fdm = NULL;

		break;
	    }
	}


	return;
}


/*
 *	FDM airborne callback, whenever the FDM just leaves the ground.
 */
void SARSimAirborneCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data
)
{
        SFMRealmStruct *realm = (SFMRealmStruct *)realm_ptr;
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        if((realm == NULL) ||
           (model == NULL) ||
           (core_ptr == NULL)
        )
            return;



        /* Skip touch down check entirly if in slew mode. */
/*
        if(SARSimIsSlew(obj_ptr))
            return;
 */



        return;
}       


/*
 *	FDM touchdown callback, whenever the FDM just lands.
 */
void SARSimTouchDownCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data, double impact_coeff
)
{
        int obj_num, *gcc_list, gcc_list_total, crash_cause = 0;
	Boolean over_water = False, have_floats = False;
        double distance_to_camera;		/* In meters. */
        sar_scene_struct *scene;
        sar_position_struct *pos, *ear_pos;
	sar_position_struct *vel = NULL;
	sar_direction_struct *dir;
	sar_object_struct *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_contact_bounds_struct *cb;
	sar_obj_flags_t crash_flags = 0;
        double contact_radius = 0.0;
	int air_worthy_state = SAR_AIR_WORTHY_FLYABLE;

        SFMRealmStruct *realm = (SFMRealmStruct *)realm_ptr;
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        if((realm == NULL) || (model == NULL) || (core_ptr == NULL))
            return;

	scene = core_ptr->scene;
	if(scene == NULL)
	    return;


#define DO_PLAY_LAND_BELLY              \
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_LAND_BELLY, \
  scene, distance_to_camera \
 ); \
}

#define DO_PLAY_LAND_SKI                \
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_LAND_SKI, \
  scene, distance_to_camera \
 ); \
}

#define DO_PLAY_LAND_SKI_SKID           \
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_LAND_SKI_SKID, \
  scene, distance_to_camera \
 ); \
}

#define DO_PLAY_LAND_WHEEL_SKID         \
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_LAND_WHEEL_SKID, \
  scene, distance_to_camera \
 ); \
}

#define DO_PLAY_CRASH_GROUND            \
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_CRASH_GROUND, \
  scene, distance_to_camera \
 ); \
}

#define DO_PLAY_SPLASH_AIRCRAFT		\
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_SPLASH_AIRCRAFT, \
  scene, distance_to_camera \
 ); \
}  


	/* Match object from FDM. */
	obj_ptr = SARSimMatchObjectFromFDM(
	    core_ptr->object, core_ptr->total_objects,
	    model, &obj_num
	);
	if(obj_ptr == NULL)
	    return;

	/* Skip touch down check entirly if in slew mode. */
	if(SARSimIsSlew(obj_ptr))
	    return;

	/* Get list of ground contact check objects, a list of objects
	 * under the given object.
	 */
	gcc_list = SARGetGCCHitList(
	    core_ptr, scene,
	    &core_ptr->object, &core_ptr->total_objects,
	    obj_num,
	    &gcc_list_total
	);

	/* Check if over water. */
	SARGetGHCOverWater(
	    core_ptr, scene,
	    &core_ptr->object, &core_ptr->total_objects,
            obj_num,
	    NULL, &over_water
	);


        /* Get positions and velocities. */
	pos = &obj_ptr->pos;
        pos->x = model->position.x;
	pos->y = model->position.y;
	pos->z = model->position.z;

	ear_pos = &scene->ear_pos;

	dir = &obj_ptr->dir;
	dir->heading = model->direction.heading;
        dir->pitch = model->direction.pitch;
        dir->bank = model->direction.bank;

	/* Get contact bounds if any. */
	cb = obj_ptr->contact_bounds;
	if(cb != NULL)
	{
	    crash_flags = cb->crash_flags;
	    contact_radius = SARSimGetFlatContactRadius(obj_ptr);
	}


        /* Calculate distance between object and camera. */
        distance_to_camera = hypot(hypot(
            pos->x - ear_pos->x,
            pos->y - ear_pos->y
            ), pos->z - ear_pos->z
        );

        /* Get object type specific values. */
        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
            if(obj_aircraft_ptr != NULL)
	    {
		int i;
		sar_obj_landing_gear_struct *lgear_ptr;

                vel = &obj_aircraft_ptr->vel;
                vel->x = model->velocity_vector.x;
                vel->y = model->velocity_vector.y;
                vel->z = model->velocity_vector.z;
                air_worthy_state = obj_aircraft_ptr->air_worthy_state;

		for(i = 0; i < obj_aircraft_ptr->total_landing_gears; i++)
		{
		    lgear_ptr = obj_aircraft_ptr->landing_gear[i];
		    if(lgear_ptr == NULL)
			continue;

		    if(lgear_ptr->flags & SAR_LGEAR_FLAG_FLOATS)
			have_floats = 1;
		}
	    }
            break;

/* Add support for other object types here. */

	}
	/* Substructure data pointer should now be set properly or
	 * NULL.
	 */


        /* If object was already in flight but was not flyable
         * (ie it hit a building and is falling to the ground),
         * then contacting the ground is a crash.
         */
        if(air_worthy_state != SAR_AIR_WORTHY_FLYABLE)
        {
            crash_cause = 3;  /* Crashed already, just contacted ground. */
        }
	/* Impact tolorance exceeded? */ 
        else if(impact_coeff > 1.0)
        {
            crash_cause = 1;  /* Impact tolorance exceeded. */
        }
	/* Landed in water and does not have floats? */
	else if(over_water && !have_floats)
	{
	    crash_cause = 4;	/* Splash! */
	}
        /* Contacted ground at bad angle? */
        else
        {
            if(dir->pitch > (1.0 * PI))
            {
                if(dir->pitch < (1.75 * PI))
                    crash_cause = 2;
            }
            else
            {
                if(dir->pitch > (0.25 * PI))
                    crash_cause = 2;
            }
            
            if(dir->bank > (1.0 * PI))
            {
                if(dir->bank < (1.75 * PI))
                    crash_cause = 2;
            }
            else
            {
                if(dir->bank > (0.25 * PI))
                    crash_cause = 2;
            }
        }
/*
printf("Touch down %i %f\n", crash_cause, impact_coeff);
 */
        /* ******************************************************** */
        /* Did the object crash? */
        if(crash_cause)
        {
            double explosion_radius = 10.0;
            sar_position_struct explosion_pos;
            char text[1024];

            /* Set object `as crashed' by its type. */
	    /* Aircraft? */
	    if(obj_aircraft_ptr != NULL)
	    {
		/* Mark as on water? */
		if(over_water)
		    obj_aircraft_ptr->on_water = 1;

		/* Set up aircraft as crashed. */
		SARSimSetAircraftCrashed(
		    scene, &core_ptr->object, &core_ptr->total_objects,
		    obj_ptr, obj_aircraft_ptr
		);
	    }
	    else
	    {
/* Add support for other types of objects that can crash. */

            }

            /* Is the crashed object the player object? */
            if(obj_ptr == scene->player_obj_ptr)
            {
		/* Mark the player object as crashed on scene. */
		scene->player_has_crashed = 1;

                /* Check if object is not set crash other, if so that implies
                 * it has already crashed into something and we should not
                 * create another explosion or change view.
                 */
		if(crash_flags & SAR_CRASH_FLAG_CRASH_OTHER)
		{
		  /* Set crash banner message with cause of crash. */
		  SARBannerMessageAppend(scene, NULL);
		  SARBannerMessageAppend(scene, SAR_MESG_CRASH_BANNER);
                  switch(crash_cause)
                  {
		   case 4:
		    strcpy(text, SAR_MESG_CRASH_SPLASH);
                    SARBannerMessageAppend(scene, text);
                    break;

                   case 2:
                    strcpy(text, SAR_MESG_CRASH_ROTATION_TOO_STEEP);
                    SARBannerMessageAppend(scene, text);
                    break;

                   case 1:
		    /* There should be exactly one occurance of "%.0f%%"
		     * in SAR_MESG_CRASH_IMPACTED_PAST_TOLORANCE for the
		     * substitution to work.
		     */
                    sprintf(
			text,
			SAR_MESG_CRASH_IMPACTED_PAST_TOLORANCE,
                        (double)(impact_coeff * 100)
                    );
                    SARBannerMessageAppend(scene, text);
                    break;
		  }

                  /* Set footer to indicate what pilot should do now. */
                  SARBannerMessageAppend(scene, SAR_MESG_POST_CRASH_BANNER);
		}
            }
            else
            {
                /* An object other than the player object has
                 * crashed.
                 */
		char numstr[256];

		sprintf(numstr, "Object #%i", obj_num);

		(*text) = '\0';
		strcat(text, "*** ");
		strncat(text, ((obj_ptr->name == NULL) ?
		    numstr : obj_ptr->name),
		    80
		);
		strcat(text, " has crashed! ***");

                SARMessageAdd(scene, text);
            }


            /* Set position and size for creation of explosion. */
            memcpy(&explosion_pos, &obj_ptr->pos, sizeof(sar_position_struct));
            explosion_radius = MAX(contact_radius * 1.8, 10.0);

	    /* Begin checking if we should create explosion or splash. */

	    /* Check if object is not set crash other, if so that implies
	     * it has already crashed into something and we should not
	     * create another explosion or change view.
	     */
            if(crash_flags & SAR_CRASH_FLAG_CRASH_OTHER)
            {
                /* Just crashed into ground and has not previously crashed
		 * into anything.
		 */
		int i, explosion_obj_num = -1;
		sar_object_struct *explosion_obj_ptr = NULL;
		sar_external_fueltank_struct *eft_ptr;
		double fuel_left = 10.0;	/* Assume some fuel, in kg. */


		/* Check if object has any fuel left. */
		if(obj_aircraft_ptr != NULL)
		{
		    fuel_left = MAX(obj_aircraft_ptr->fuel, 0.0);
		    for(i = 0; i < obj_aircraft_ptr->total_external_fueltanks; i++)
		    {
			eft_ptr = obj_aircraft_ptr->external_fueltank[i];
			if((eft_ptr == NULL) ? 0 :
			    (eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD)
			)
			    fuel_left += (eft_ptr->dry_mass + eft_ptr->fuel);
		    }
		}


		/* Atleast 1 or more kg of fuel left and not on water
		 * to create explosion?
		 */
		if((fuel_left > 1.0) && !over_water)
		{
		    sar_position_struct smoke_offset;

                    /* Delete all effects objects related to this object
                     * but only stop smoke trails from respawning.
                     */
		    SARSimDeleteEffects(
			core_ptr, scene,
			&core_ptr->object, &core_ptr->total_objects,
			obj_num,
			SARSIM_DELETE_EFFECTS_SMOKE_STOP_RESPAWN
		    );
		    /* Create explosion object. */
                    explosion_obj_num = ExplosionCreate(
                        scene,
                        &core_ptr->object, &core_ptr->total_objects,
                        &explosion_pos,     /* Position of explosion. */
                        explosion_radius,   /* Radius of size in meters. */
                        obj_num,            /* Referance object number. */
                        SAR_STD_TEXNAME_EXPLOSION
                    );
                    if(explosion_obj_num > -1)
                    {
			sar_object_explosion_struct *obj_explosion_ptr;

                        explosion_obj_ptr = core_ptr->object[explosion_obj_num];

			obj_explosion_ptr = (sar_object_explosion_struct *)explosion_obj_ptr->data;
                        if(obj_explosion_ptr != NULL)
                        {
                            /* Set lifespan for explosion. */
                            explosion_obj_ptr->life_span = cur_millitime +
                                option.crash_explosion_life_span;

			    /* Repeat frames untill life span is reached. */
			    obj_explosion_ptr->total_frame_repeats = -1;
                        }
		    }
		    /* Create smoke trails object. */
		    smoke_offset.x = 0.0;
		    smoke_offset.y = 0.0;
		    smoke_offset.z = explosion_radius;
		    SmokeCreate(
			scene, &core_ptr->object, &core_ptr->total_objects,
			&explosion_pos, &smoke_offset,
			explosion_radius * 1.0,		/* Radius start. */
			explosion_radius * 4.0,		/* Radius max. */
			-1.0,				/* Autocalc growth. */
			1,				/* Hide at max. */
			10,				/* Total units. */
			3000,				/* Respawn interval in ms. */
			SAR_STD_TEXNAME_SMOKE_DARK,
			obj_num,
			cur_millitime + option.crash_explosion_life_span
		    );
                }
		else if(over_water)
		{
                    /* Delete all effects objects related to this object
		     * but only stop smoke trails from respawning.
		     */
                    SARSimDeleteEffects(
                        core_ptr, scene,
                        &core_ptr->object, &core_ptr->total_objects,
                        obj_num,
			SARSIM_DELETE_EFFECTS_SMOKE_STOP_RESPAWN
                    );
                    /* Create splash (explosion) object. */
		    explosion_obj_num = SplashCreate(
                        scene,
                        &core_ptr->object, &core_ptr->total_objects,
                        &explosion_pos,     /* Position of explosion. */
                        explosion_radius,   /* Radius of size in meters. */
                        obj_num,            /* Referance object number. */
                        SAR_STD_TEXNAME_SPLASH
                    );
		    if(explosion_obj_num > -1)
                    {
                        explosion_obj_ptr = core_ptr->object[explosion_obj_num];

			/* No need to modify splash (explosion) object values. */
		    }
		}

                /* Is this the player object? */
                if(scene->player_obj_ptr == obj_ptr)
                {
                    /* Set spot camera position. */
                    scene->camera_ref = SAR_CAMERA_REF_SPOT;
                    scene->camera_target = scene->player_obj_num;
                }
            }
	    else
	    {
                /* Crash flag was not set, implying it has already hit
                 * something.
                 */
                int explosion_obj_num = -1;
                sar_object_struct *explosion_obj_ptr = NULL;

                /* We still need to create a splash if landed on water. */
                if(over_water)
                {
                    /* Delete all effects objects related to this object
                     * but only stop smoke trails from respawning.
                     */
                    SARSimDeleteEffects(
                        core_ptr, scene,
                        &core_ptr->object, &core_ptr->total_objects,
                        obj_num,
			SARSIM_DELETE_EFFECTS_SMOKE_STOP_RESPAWN
                    );
		    /* Create splash (explosion) object. */
                    explosion_obj_num = SplashCreate(
                        scene,
                        &core_ptr->object, &core_ptr->total_objects,
                        &explosion_pos,     /* Position of explosion. */
                        explosion_radius,   /* Radius of size in meters. */
                        obj_num,            /* Referance object number. */
                        SAR_STD_TEXNAME_SPLASH
                    );
                    if(explosion_obj_num > -1)
                    {
                        explosion_obj_ptr = core_ptr->object[explosion_obj_num];

                        /* No need to modify splash (explosion) object values. */
                    }
                }
	    }

            /* Play explosion sound. */
	    if(over_water)
	    {
		DO_PLAY_SPLASH_AIRCRAFT
	    }
	    else
	    {
		DO_PLAY_CRASH_GROUND
	    }

            /* Call mission destroy notify instead of mission land
	     * notify, to let mission know this object has crashed.
             */
            SARMissionDestroyNotify(core_ptr, obj_ptr);
        }
        else
        {
            /* Safe landing (no crash). */
            sar_obj_landing_gear_struct *lgear_ptr = NULL;


            /* Handle safe landing by object type. */
	    /* Aircraft? */
	    if(obj_aircraft_ptr != NULL)
	    {
                if(obj_aircraft_ptr->total_landing_gears > 0)
                    lgear_ptr = obj_aircraft_ptr->landing_gear[0];

                /* Mark as on water? */
                if(over_water)
                    obj_aircraft_ptr->on_water = 1;
		else
		    obj_aircraft_ptr->on_water = 0;

                /* Landed on belly? */
                if((lgear_ptr == NULL) ? 1 :
                    (lgear_ptr->flags & SAR_LGEAR_FLAG_STATE)
                )
                {
		    /* Landed on belly, check if landed at velocity
		     * great enough to cause scrape.
		     */  
                    if(obj_aircraft_ptr->speed > SFMMPHToMPC(5))
                    {
                        DO_PLAY_LAND_BELLY
                    }
                    else
                    {
                        DO_PLAY_LAND_SKI
                    }
                }
                else
                {
                    /* Landed on gear. */

                    /* Landing gear a ski? */
                    if((lgear_ptr == NULL) ? 1 :
                        (lgear_ptr->flags & SAR_LGEAR_FLAG_SKI)
                    )
                    {
                        /* Landing gear is a ski, check if landed
			 * at velocity great enough to cause scrape.
			 */
                        if(obj_aircraft_ptr->speed > SFMMPHToMPC(5))
                        {
                            DO_PLAY_LAND_SKI_SKID
                        } 
                        else
                        {
                            DO_PLAY_LAND_SKI
                        }
                    }
                    else
                    {
                        /* Landing gear is a wheel. */

                        /* Brakes on? */  
                        if(obj_aircraft_ptr->wheel_brakes_state != 0)
                        {
                            if(obj_aircraft_ptr->speed > SFMMPHToMPC(10))
                            {
                                DO_PLAY_LAND_WHEEL_SKID
                            }
                        }
                        else
                        {
                            if(obj_aircraft_ptr->speed > SFMMPHToMPC(25))
                            {
                                DO_PLAY_LAND_WHEEL_SKID
                            }
                        }
                    }
                }       /* Landed on gear. */
            }
	    /* Other object type. */
	    else
	    {
/* Add support for safe landing of other object types here. */

	    }

	    /* Call mission land notify. */
	    SARMissionLandNotify(core_ptr, obj_ptr, gcc_list, gcc_list_total);
        }

	/* Deallocate matched list of ground contact check objects. */
	free(gcc_list);
	gcc_list = NULL;
	gcc_list_total = 0;

#undef DO_PLAY_LAND_BELLY
#undef DO_PLAY_LAND_SKI
#undef DO_PLAY_LAND_SKI_SKID
#undef DO_PLAY_LAND_WHEEL_SKID
#undef DO_PLAY_CRASH_GROUND
#undef DO_PLAY_SPLASH_AIRCRAFT
}


/*
 *	FDM overspeed callback, this function is called whenever
 *	the model is exceeding its maximum expected speed.
 *
 *	This function may be called quite often.
 */
void SARSimOverspeedCB(
        void *realm_ptr, SFMModelStruct *model,
        void *client_data, double cur_speed,
        double speed_max_expected
)
{
	int obj_num;
	sar_object_struct *obj_ptr;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_contact_bounds_struct *cb;
	sar_scene_struct *scene;
        SFMRealmStruct *realm = (SFMRealmStruct *)realm_ptr;
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        if((realm == NULL) || (model == NULL) || (core_ptr == NULL))
            return;

        scene = core_ptr->scene;
        if(scene == NULL)
            return;

        /* Match object from FDM. */ 
        obj_ptr = SARSimMatchObjectFromFDM(
            core_ptr->object, core_ptr->total_objects,
            model, &obj_num
        );
        if(obj_ptr == NULL)
            return;

        /* Skip overspeed check entirly if in slew mode. */
        if(SARSimIsSlew(obj_ptr))
            return;

	/* Handle by object type. */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		/* Past 110% of tolorance? */
		if(cur_speed > (speed_max_expected * 1.10))
		{
		    /* Is aircraft still flyable? */
		    if(obj_aircraft_ptr->air_worthy_state == SAR_AIR_WORTHY_FLYABLE)
		    {
			double contact_radius = SARSimGetFlatContactRadius(obj_ptr);


			/* Set it to be out of control. */
			obj_aircraft_ptr->air_worthy_state = SAR_AIR_WORTHY_OUT_OF_CONTROL;

			/* Reduce to 10% hit points if above. */
			if(obj_ptr->hit_points > (obj_ptr->hit_points_max * 0.10))
			    obj_ptr->hit_points = obj_ptr->hit_points_max * 0.10;

			/* Get pointer to object's contact bounds structure. */
			cb = obj_ptr->contact_bounds;
			if(cb != NULL)
			{
			    /* Remove SAR_CRASH_FLAG_CRASH_OTHER flag from
			     * object's crash flags so that this object
			     * won't crash into other objects from now on.
			     *
			     * Note that this will not create an explosion when
			     * it hits the ground.
			     */
			    cb->crash_flags &= ~SAR_CRASH_FLAG_CRASH_OTHER;
			}

			/* Delete all effects objects related to this object. */
			SARSimDeleteEffects(
			    core_ptr, scene,
			    &core_ptr->object, &core_ptr->total_objects,
			    obj_num,
			    0
			);
			/* Create smoke trails object. */
			SmokeCreate(
			    scene,
			    &core_ptr->object, &core_ptr->total_objects,
			    &obj_ptr->pos, NULL,
			    contact_radius * 0.25,	/* Radius start. */
			    contact_radius * 3.0,	/* Radius max. */
			    -1.0,			/* Autocalc growth. */
			    1,				/* Hide at max. */
			    15,				/* Total units. */
			    500,			/* Respawn interval in ms. */
			    SAR_STD_TEXNAME_SMOKE_MEDIUM,
			    obj_num,
			    0				/* Life span. */
			);

			/* Is this the player object? */
			if(scene->player_obj_ptr == obj_ptr)
			{
			    /* Mark player object as has crashed. */
			    scene->player_has_crashed = 1;

			    /* Set overspeed crash banner. */
			    SARBannerMessageAppend(scene, NULL);
 			    SARBannerMessageAppend(scene, SAR_MESG_CRASH_OVERSPEED_BANNER);
/* Do not post footer, since pilot may want to try a futile safe 
   landing.
			    SARBannerMessageAppend(scene, SAR_MESG_POST_CRASH_BANNER);
 */
                            /* Set spot camera position. */
                            scene->camera_ref = SAR_CAMERA_REF_SPOT;
                            scene->camera_target = scene->player_obj_num;
			}

		    }

/* Do not call mission destroy notify, pilot may want to try a safe 
   landing before aborting.
SARMissionDestroyNotify(core_ptr, obj_ptr);
 */
		}
	    }
	    break;
	}
}


/*
 *	FDM midair collision callback. This function won't be called
 *	by the SFM library (since all crash detection values passed
 *	to the SFM are False), instead SAR will do collision detection
 *	and call this function.
 */
void SARSimCollisionCB(
        void *realm_ptr, SFMModelStruct *model, SFMModelStruct *obstruction,
        void *client_data, double impact_coeff
)
{

	/* Ignore for now, we're using our own crash detection and
	 * handlers.
	 */

	return;
}

/*
 *      Same as SARSimCollisionCB except that it takes inputs in
 *	the form of object pointers, this is for collision detection
 *	on our side (not using the SFM collision check callback).
 */
void SARSimObjectCollisionCB(
	void *client_data,		/* Core structure. */
	sar_object_struct *obj_ptr, sar_object_struct *obstruction_obj_ptr,
        double impact_coeff
)
{
        int n, ref_object = -1;
        double r, distance_to_camera;
        double  tower_offset_x = 50.0,
                tower_offset_y = 50.0,
                tower_offset_z = 50.0;
        sar_position_struct *pos, *ear_pos, explosion_pos, smoke_offset;
        sar_scene_struct *scene;
        sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_contact_bounds_struct *cb;
        sar_core_struct *core_ptr = (sar_core_struct *)client_data; 
        if((core_ptr == NULL) ||
           (obj_ptr == NULL) ||
           (obstruction_obj_ptr == NULL)
        )
            return;

        scene = core_ptr->scene;
        if(scene == NULL)
            return;

        /* Skip collision check entirly if in slew mode. */
        if(SARSimIsSlew(obj_ptr))
            return;


#define DO_PLAY_CRASH_OBSTRUCTION	\
{ \
 SARSoundPlayPredefined( \
  core_ptr->recorder, \
  SAR_SOUND_PREDEFINED_CRASH_OBSTRUCTION, \
  scene, distance_to_camera \
 ); \
}

	/* Get pointer to position of victim object. */
        pos = &obj_ptr->pos;

	/* Get pointer to position of ear. */
        ear_pos = &scene->ear_pos;

        /* Calculate distance to camera. */
        distance_to_camera = hypot(hypot(
            pos->x - ear_pos->x,
            pos->y - ear_pos->y
            ), pos->z - ear_pos->z
        );


	/* Check if obstruction object (the `crashed into' object) is
	 * valid.
	 */
        if(obstruction_obj_ptr != NULL)
        {
            double bearing;     	/* Victim obj to obstruction obj. */
            double theta, speed, d;
            sar_position_struct *pos2, *vel;


	    /* Get pointer to obstruction object's position. */
            pos2 = &obstruction_obj_ptr->pos;

            /* Begin setting up victim object's values to mark that it has
             * crashed into the obstruction object.
             */

	    /* Change some values on victim object by its type, do not
	     * set object as fully crashed since it's probably plumitting
	     * to the ground.
	     */
            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
               obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr == NULL)
                    break;

                vel = &obj_aircraft_ptr->vel;
                speed = obj_aircraft_ptr->speed;

                /* Calculate bearing to obstruction. */
                bearing = SFMSanitizeRadians(
                    (PI / 2) - atan2(pos2->y - pos->y, pos2->x - pos->x)
                );

                /* Calculate impact of velocity loss. */
                theta = bearing - obj_ptr->dir.heading;
                vel->y = vel->y - (speed * cos(theta));
                vel->x = 0;

		/* Get size of object. */
		d = SARSimGetFlatContactRadius(obj_ptr) * 4.0;
		if(d < 10.0)
		    d = 10.0;
                tower_offset_y = -cos(bearing) * d;
                tower_offset_z = d;

                /* Set air worthy state based on if object is already
		 * landed or not.
		 */
                if(obj_aircraft_ptr->landed)
                {
		    /* Call SARSimSetAircraftCrashed() procedure since
		     * aircraft is already on ground and won't crash
		     * again.
		     */
		    SARSimSetAircraftCrashed(
			scene, &core_ptr->object, &core_ptr->total_objects,
			obj_ptr, obj_aircraft_ptr
		    );
                }
                else
                {
		    /* Aircraft is still flying, mark it as out of
		     * control.
		     */
                    obj_aircraft_ptr->air_worthy_state =
                        SAR_AIR_WORTHY_OUT_OF_CONTROL;

		    /* Turn engines off. */
		    obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_OFF;
                }
                break;

/* Add support for other objects that can crash into other objects. */

            }
        }	/* if(obstruction_obj_ptr != NULL) */


        /* Get pointer to victim object's contact bounds structure. */
	cb = obj_ptr->contact_bounds;
	if(cb != NULL)
	{
	    /* Remove SAR_CRASH_FLAG_CRASH_OTHER flag from victim
	     * object's crash flags so that the victim object won't
	     * crash into other objects from now on.
	     */
	    cb->crash_flags &= ~SAR_CRASH_FLAG_CRASH_OTHER;
	}


	/* Begin creating explosion. */

        /* Get position and values for creating explosion object. */
        memcpy(&explosion_pos, &obj_ptr->pos, sizeof(sar_position_struct));
        ref_object = SARGetObjectNumberFromPointer(
            scene, core_ptr->object, core_ptr->total_objects, obj_ptr
        );
	r = MAX(SARSimGetFlatContactRadius(obj_ptr) * 1.8, 10.0);

	/* Delete all effects objects related to this object but only
	 * stop smoke trails from respawning.
	 */
	SARSimDeleteEffects(
	    core_ptr, scene, &core_ptr->object, &core_ptr->total_objects,
	    ref_object,
	    SARSIM_DELETE_EFFECTS_SMOKE_STOP_RESPAWN
	);
	/* Create explosion object. */
        n = ExplosionCreate(
            scene, &core_ptr->object, &core_ptr->total_objects,
            &explosion_pos,     /* Position of explosion. */
            r,                  /* Radius of size in meters. */
            ref_object,         /* Referance object number. */
            SAR_STD_TEXNAME_EXPLOSION
        );
        if(n > -1)  
        {
	    sar_object_explosion_struct *obj_explosion_ptr;
	    sar_object_struct *explosion_obj_ptr = core_ptr->object[n];

	    obj_explosion_ptr = (sar_object_explosion_struct *)explosion_obj_ptr->data;
	    if(obj_explosion_ptr != NULL)
            {
		/* Set lifespan for explosion. */
		explosion_obj_ptr->life_span = cur_millitime +
		    option.crash_explosion_life_span;

		/* Repeat frames untill life span is reached. */
		obj_explosion_ptr->total_frame_repeats = -1;
            }
        }
	/* Create smoke trails object. */
	smoke_offset.x = 0.0;
	smoke_offset.y = 0.0;
	smoke_offset.z = r;
	SmokeCreate(
            scene, &core_ptr->object, &core_ptr->total_objects,
	    &explosion_pos, &smoke_offset,
            r * 1.0,		/* Radius start. */
	    r * 4.0,		/* Radius max. */
	    -1.0,		/* Autocalc growth. */
	    1,			/* Hide at max. */
	    10,			/* Total units. */
	    3000,		/* Respawn interval in ms. */
	    SAR_STD_TEXNAME_SMOKE_DARK,
	    ref_object,
	    cur_millitime + option.crash_explosion_life_span
	);

	/* Play crash obstruction sound. */
        DO_PLAY_CRASH_OBSTRUCTION


        /* Is this the player object? */
        if(scene->player_obj_ptr == obj_ptr)
        {         
            char text[128];

	    /* Mark player object as has crashed. */
	    scene->player_has_crashed = 1;

            /* Set tower position. */
            scene->camera_tower_pos.x = obj_ptr->pos.x +
                tower_offset_x;
            scene->camera_tower_pos.y = obj_ptr->pos.y +
                tower_offset_y;
            scene->camera_tower_pos.z = obj_ptr->pos.z +
                tower_offset_z;

            scene->camera_ref = SAR_CAMERA_REF_TOWER;
            scene->camera_target = scene->player_obj_num;


            /* Set scene banner to indicate collision and type. */
	    SARBannerMessageAppend(scene, NULL);
	    SARBannerMessageAppend(scene, SAR_MESG_COLLISION_BANNER);

            if(obstruction_obj_ptr != NULL)
            {
		cb = obstruction_obj_ptr->contact_bounds;
                switch((cb == NULL) ? 0 : cb->crash_type)
                {
                  case SAR_CRASH_TYPE_OBSTRUCTION:
                    strcpy(text, SAR_MESG_CRASH_OBSTRUCTION);
                    break;

                  case SAR_CRASH_TYPE_GROUND:
                    /* Never should occure for midair collisions. */
                    strcpy(text, SAR_MESG_CRASH_GROUND);
                    break;

                  case SAR_CRASH_TYPE_MOUNTAIN:
                    strcpy(text, SAR_MESG_CRASH_MOUNTAIN);
                    break;

                  case SAR_GRASH_TYPE_BUILDING:
                    strcpy(text, SAR_MESG_CRASH_BOULDING);
                    break;

                  case SAR_GRASH_TYPE_AIRCRAFT:
                    strcpy(text, SAR_MESG_CRASH_AIRCRAFT);
                    break;

		  default:
		    sprintf(text,
			"UNSUPPORTED COLLISION CODE %i",
			(int)((cb == NULL) ? 0 : cb->crash_type)
		    );
		    break;
                }
                SARBannerMessageAppend(scene, text);
            }

	    /* Set footer to indicate what pilot should do now. */
	    SARBannerMessageAppend(scene, SAR_MESG_POST_CRASH_BANNER);
        }

#undef DO_PLAY_CRASH_OBSTRUCTION
}
