#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <math.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"

#include "sfm.h"

#include "sarreality.h"
#include "obj.h"
#include "messages.h"
#include "simmanage.h"
#include "simcb.h"
#include "simutils.h"
#include "weather.h"
#include "sar.h"
#include "sarfio.h"
#include "config.h"


void SARSceneDestroy(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total
);
void SARSceneLoadLocationsToList(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu_ptr,
        sar_menu_list_struct *list_ptr, int list_num,
        const char *filename
);
int SARSceneAddPlayerObject(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
	const char *model_file,
	sar_position_struct *pos,
	sar_direction_struct *dir
);
int SARSceneLoadFromFile(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
	const char *filename,
	const char *weather_preset_name,
	void *client_data,
	int (*progress_func)(void *, long, long)
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define ISCOMMENT(c)	((c) == SAR_COMMENT_CHAR)
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


/*
 *	Quick converter from degrees to radians without sanitization:
 */
#define DEGTORAD(d)	((d) * PI / 180)


/*
 *	Deallocates all resources on the scene structure and all
 *	objects on the given objects pointer array.
 *
 *	Does not deallocate the scene structure itself, but the
 *	scene structure will be reset.
 */
void SARSceneDestroy(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total
)
{
	int i;
	char **strptr;
	sar_scene_horizon_struct *horizon_ptr;


	if(core_ptr == NULL)
	    return;

        if(option.runtime_debug)
            printf("SARSceneDestroy(): Destroying scene...\n");

#define FREE_STRING	\
{ \
 if((*strptr) != NULL) \
 { \
  free(*strptr); \
  (*strptr) = NULL; \
 } \
}

	/* Check if total objects is not NULL (which implies ptr is not
	 * NULL).
	 */
	if(total != NULL)
	{
	    /* Deallocate objects pointer array from last to first. */
	    for(i = (*total) - 1; i >= 0; i--)
	        SARObjDelete(core_ptr, ptr, total, i);

	    (*total) = 0;
	}
	/* Be sure to free objects pointer array. */
	if(ptr != NULL)
	{
	    free(*ptr);
	    (*ptr) = NULL;
	}


	/* Is scene structure valid? */
	if(scene != NULL)
	{
	    /* Deallocate all resources on scene structure. */

	    /* Ground base visual model simple (far away). */
	    SARVisualModelUnref(scene, scene->base.visual_model_simple);
	    scene->base.visual_model_simple = NULL;

	    /* Ground base visual model complext (close up). */
            SARVisualModelUnref(scene, scene->base.visual_model_close);
            scene->base.visual_model_close = NULL;


            /* Cloud layers. */
	    for(i = 0; i < scene->total_cloud_layers; i++)
		SARCloudLayerDestroy(scene, scene->cloud_layer[i]);
	    free(scene->cloud_layer);
	    scene->cloud_layer = NULL;
	    scene->total_cloud_layers = 0;

            /* Cloud `billboard' objects. */
            for(i = 0; i < scene->total_cloud_bbs; i++)
                SARCloudBBDestroy(scene->cloud_bb[i]);
            free(scene->cloud_bb);
            scene->cloud_bb = NULL;
            scene->total_cloud_bbs = 0;

	    /* Horizon. */
	    horizon_ptr = &scene->horizon;
	    for(i = 0; i < horizon_ptr->total_textures; i++)
		V3DTextureDestroy(horizon_ptr->texture[i]);

	    free(horizon_ptr->texture);
	    horizon_ptr->texture = NULL;

	    horizon_ptr->total_textures = 0;


	    /* Ground objects list. */
	    free(scene->ground_object);
	    scene->ground_object = NULL;
	    scene->total_ground_objects = 0;

            /* Human that need rescue objects list. */
            free(scene->human_need_rescue_object);
	    scene->human_need_rescue_object = NULL;
	    scene->total_human_need_rescue_objects = 0;

	    /* Visual models, all visual models should have been
	     * unref'ed by now. So here we actually deallocate them
	     * and destroy the GL lists.
	     */
	    SARVisualModelDeleteAll(scene);

	    /* Delete all textures (deallocated by V3DTextureDestroy()). */
	    for(i = 0; i < scene->total_texture_refs; i++)
		V3DTextureDestroy(scene->texture_ref[i]);
	    free(scene->texture_ref);	/* Free pointer array. */
	    scene->texture_ref = NULL;
	    scene->total_texture_refs = 0;

	    /* Default sound object paths. */
	    strptr = &scene->land_wheel_skid_sndpath;
	    FREE_STRING
            strptr = &scene->land_ski_skid_sndpath;
	    FREE_STRING
            strptr = &scene->land_ski_sndpath;
            FREE_STRING
	    strptr = &scene->land_belly_sndpath;
	    FREE_STRING
            strptr = &scene->crash_obstruction_sndpath;
	    FREE_STRING
            strptr = &scene->crash_ground_sndpath;
	    FREE_STRING
	    strptr = &scene->splash_aircraft_sndpath;
	    FREE_STRING
	    strptr = &scene->splash_human_sndpath;
	    FREE_STRING

	    /* Messages. */
	    StringFreeArray(scene->message, scene->total_messages);
	    scene->message = NULL;
	    scene->total_messages = 0;

	    /* Sticky banner message. */
	    StringFreeArray(
		scene->sticky_banner_message,
		scene->total_sticky_banner_messages
	    );
	    scene->sticky_banner_message = NULL;
	    scene->total_sticky_banner_messages = 0;

	    /* Camera referance name/description title string. */
	    strptr = &scene->camera_ref_title;
	    FREE_STRING

	    /* Realm structure. */
	    if(scene->realm != NULL)
		SFMShutdown(scene->realm);

#undef FREE_STRING

	    /* Reset the rest of the scene structure. */
	    memset(scene, 0x00, sizeof(sar_scene_struct));
	}

	if(option.runtime_debug)
	    printf("SARSceneDestroy(): Scene destroyed.\n");

	glFlush();

	return;
}

/*
 *      Loads the list of registered locations specified in the scene
 *      file and stores them in the given menu list object. The contents 
 *	of the menu's list object will be cleared first.
 */
void SARSceneLoadLocationsToList(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu_ptr,
        sar_menu_list_struct *list_ptr, int list_num,
        const char *filename
)
{
        int i, status, ptype, total_parms;
	void *p, **parm;

        if(list_ptr == NULL)
            return;

        /* Delete any existing items in the menu list object, first
	 * delete their client_data structures then the actual
	 * structures.
	 */
        for(i = 0; i < list_ptr->total_items; i++)
            SARDeleteListItemData(list_ptr->item[i]);
        SARMenuListDeleteAllItems(menu_ptr, list_num);


	/* Load all parameters from file that are of type
	 * SAR_PARM_REGISTER_LOCATION.
	 */
	status = SARParmLoadFromFile(
	    filename, SAR_FILE_FORMAT_SCENE,
	    &parm, &total_parms,
	    SAR_PARM_REGISTER_LOCATION,		/* Filter. */
	    NULL, NULL
	);
	if(status)
	    return;

	/* Itterate through loaded parms. */
	for(i = 0; i < total_parms; i++)
	{
	    p = parm[i];
	    if(p == NULL)
		continue;
	    else
		ptype = (*(int *)p);

	    if(ptype == SAR_PARM_REGISTER_LOCATION)
	    {
		int n;
                sar_menu_list_item_data_struct *li_data_ptr;
		sar_parm_register_location_struct *p_register_location =
		    (sar_parm_register_location_struct *)p;


		/* Allocate a new list item data structure. */
                li_data_ptr = (sar_menu_list_item_data_struct *)calloc(
                    1,
                    sizeof(sar_menu_list_item_data_struct)
                );
                if(li_data_ptr != NULL)
                {
		    /* Position. */
		    memcpy(
			&li_data_ptr->pos,
			&p_register_location->pos,
			sizeof(sar_position_struct)
		    );

		    /* Direction. */
		    memcpy(
                        &li_data_ptr->dir,
                        &p_register_location->dir,
                        sizeof(sar_direction_struct)
                    );

		    /* Name. */
                    li_data_ptr->name = StringCopyAlloc(p_register_location->name);


		    /* Add this location to the menu's list object. */
                    n = SARMenuListAppendItem(
                        menu_ptr, list_num,
                        li_data_ptr->name, li_data_ptr,
                        0
                    );
                    if(n < 0)
                    {
                        fprintf(stderr,
 "Error appending list item for starting location `%s'\n",
                            li_data_ptr->name
                        );

                        free(li_data_ptr->name);
                        free(li_data_ptr);
                        li_data_ptr = NULL;
                    }
                }
            }
        }

	/* Deallocate loaded parms. */
	SARParmDeleteAll(&parm, &total_parms);

        return;
}


/*
 *	Adds a player object to the scene using the given model
 *	file and initial position and direction. Returns non-zero
 *	on error.
 *
 *	Updates the player object referances on the scene structure.
 *
 *	All inputs must be valid (except for pos and dir).
 */
int SARSceneAddPlayerObject(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
        const char *model_file,
	sar_position_struct *pos, sar_direction_struct *dir
)
{
	int obj_num;
	sar_object_struct *obj_ptr;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_position_struct lpos;
	sar_direction_struct ldir;


	if((core_ptr == NULL) ||
           (scene == NULL) ||
	   (model_file == NULL)
	)
	    return(-1);

	/* Create player object on to scene. */
	obj_num = SARObjCreate(
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    SAR_OBJ_TYPE_AIRCRAFT
	);
        obj_ptr = ((obj_num < 0) ? NULL : core_ptr->object[obj_num]);

	if(obj_ptr == NULL)
	    return(-1);

	/* Load its model file. */
        SARObjLoadFromFile(
            core_ptr, obj_num,
            model_file
	);

	/* Set object name to "player", so that it can be
	 * matched later by name (ie needed later in mission file parsing)
	 */
	free(obj_ptr->name);
	obj_ptr->name = strdup("player");

	/* Update player object referances on scene structure. */
	scene->player_obj_num = obj_num;
	scene->player_obj_ptr = obj_ptr;


	/* Set local position structure. */
	if(pos != NULL)
	    memcpy(&lpos, pos, sizeof(sar_position_struct));
	else
	    memset(&lpos, 0x00, sizeof(sar_position_struct));

        /* Adjust ground_elevation_msl so that it's at the position of
         * the player object, this way the player won't suddenly drop
         * and be destroyed.
         */
        obj_ptr->ground_elevation_msl = lpos.z;

        /* Move player up from ground level so that it does not added into
         * the scene while embedded in the ground (which may cause a
         * crash if the given pos is ontop of a building).
         */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		lpos.z += MAX(obj_aircraft_ptr->belly_height, 0.0);
		lpos.z += MAX(obj_aircraft_ptr->gear_height, 0.0);
	    }
	    break;

/* Add other object types that need their position modified here. */

	}

	/* Set local direction structure. */
        if(dir != NULL)
            memcpy(&ldir, dir, sizeof(sar_direction_struct));
        else
            memset(&ldir, 0x00, sizeof(sar_direction_struct));

        /* Move player object to starting location and attitude. */
	SARSimWarpObject(scene, obj_ptr, &lpos, &ldir);

	return(0);
}

/*
 *	Loads a new scene from the specified file.
 *
 *	If the given buffers contain data then they will be deallocated
 *	first before the new data is loaded from file.
 *
 *	Initial GL states will be set to defaults here.
 */
int SARSceneLoadFromFile(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
	const char *filename,
	const char *weather_preset_name,
	void *client_data,
	int (*progress_func)(void *, long, long)
)
{
	char *strptr;
	void *p, **parm;
	int i, status, ptype, total_parms;
	int obj_num = -1, *total;
	gw_display_struct *display;
	sar_direction_struct *dir;
	sar_color_struct *color;
	sar_object_struct *obj_ptr = NULL, ***ptr;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_object_human_struct *obj_human_ptr = NULL;
	sar_object_smoke_struct *obj_smoke_ptr = NULL;
	sar_object_fire_struct *obj_fire_ptr = NULL;
	sar_scene_horizon_struct *horizon_ptr;
        struct stat stat_buf;

        sar_parm_name_struct *p_name;
        sar_parm_description_struct *p_description;
        sar_parm_player_model_file_struct *p_player_model_file;
        sar_parm_weather_struct *p_weather;
        sar_parm_register_location_struct *p_register_location;
        sar_parm_scene_gps_struct *p_scene_gps;
        sar_parm_scene_map_struct *p_scene_map;
        sar_parm_scene_elevation_struct *p_scene_elevation;
        sar_parm_scene_cant_struct *p_scene_cant;
        sar_parm_ground_flags_struct *p_scene_ground_flags;  
        sar_parm_scene_ground_tile_struct *p_scene_ground_tile;
        sar_parm_texture_base_directory_struct *p_texture_base_directory;
        sar_parm_texture_load_struct *p_texture_load;
        sar_parm_new_object_struct *p_new_object;
        sar_parm_new_helipad_struct *p_new_helipad;  
        sar_parm_new_runway_struct *p_new_runway;
        sar_parm_new_human_struct *p_new_human;
	sar_parm_new_smoke_struct *p_new_smoke;
	sar_parm_new_fire_struct *p_new_fire;
        sar_parm_new_premodeled_struct *p_new_premodeled;
        sar_parm_model_file_struct *p_model_file;
        sar_parm_range_struct *p_range;
        sar_parm_translate_struct *p_translate;
        sar_parm_translate_random_struct *p_translate_random;
        sar_parm_rotate_struct *p_rotate;
        sar_parm_no_depth_test_struct *p_no_depth_test;
	sar_parm_polygon_offset_struct *p_polygon_offset;
        sar_parm_contact_bounds_spherical_struct *p_contact_bounds_spherical;
        sar_parm_contact_bounds_cylendrical_struct *p_contact_bounds_cylendrical;
        sar_parm_contact_bounds_rectangular_struct *p_contact_bounds_rectangular;
        sar_parm_ground_elevation_struct *p_ground_elevation;
        sar_parm_object_name_struct *p_object_name;
        sar_parm_object_map_description_struct *p_object_map_description;
        sar_parm_fuel_struct *p_fuel;
        sar_parm_hitpoints_struct *p_hitpoints;
        sar_parm_engine_state_struct *p_engine_state;
        sar_parm_passengers_struct *p_passengers;
        sar_parm_human_message_enter_struct *p_human_message_enter;
	sar_parm_human_referance_struct *p_human_referance;


/* Macro to reset sar object substructure pointers to NULL. */
#define DO_RESET_SUBSTRUCTURE_PTRS      \
{ \
 obj_aircraft_ptr = NULL; \
 obj_ground_ptr = NULL; \
 obj_human_ptr = NULL; \
 obj_smoke_ptr = NULL; \
 obj_fire_ptr = NULL; \
}


        if((core_ptr == NULL) || (scene == NULL) || (filename == NULL))
            return(-1);

	display = core_ptr->display;

	ptr = &core_ptr->object;
	total = &core_ptr->total_objects;

	/* Check if file exists. */
        if(stat(filename, &stat_buf))
	{
	    fprintf(
		stderr,
 "%s: No such file.\n",
		filename
	    );
            return(-1);
	}

        /* Load all parameters from file. */
        status = SARParmLoadFromFile(
            filename, SAR_FILE_FORMAT_SCENE,
            &parm, &total_parms,
            -1,			/* No filter. */
            NULL, NULL
        );
        if(status)
	{
            fprintf(
                stderr,
 "%s: Error loading scene.\n",
                filename 
            );
            return(-1);
	}


	/* Delete current scene and objects. */
	SARSceneDestroy(core_ptr, scene, ptr, total);


        /* Reset scene. */
	scene->tod = (12 * 3600);
	scene->tod_code = SAR_SCENE_TOD_DAY;
	scene->cant_angle = (0.0 * PI);
        scene->base_flags = 0;
        scene->msl_elevation = 0.0;

	scene->dms_x_offset = 0;
	scene->dms_y_offset = 0;
	scene->planet_radius = SAR_DEF_PLANET_RADIUS;

	scene->visual_model = NULL;
	scene->total_visual_models = 0;

	color = &scene->sky_nominal_color;
	color->r = 1.0;
	color->g = 1.0;
	color->b = 1.0;
        color = &scene->sky_brighten_color;
        color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;
        color = &scene->sky_darken_color;
        color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;

        color = &scene->star_color;
        color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;
        color = &scene->moon_color;
        color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;

        scene->ground_object = NULL;
        scene->total_ground_objects = 0;

        scene->player_obj_num = -1;
        scene->player_obj_ptr = NULL;
	scene->player_has_crashed = 0;

        scene->camera_ref = SAR_CAMERA_REF_COCKPIT;
        scene->camera_fovz = SFMDegreesToRadians(40);
	dir = &scene->camera_cockpit_dir;
	dir->heading = 0 * PI;
	dir->pitch = 0 * PI;
	dir->bank = 0 * PI;
	dir = &scene->camera_spot_dir;
	dir->heading = 1.25 * PI;
	dir->pitch = 1.92 * PI;
	dir->bank = 0 * PI;
        scene->camera_spot_dist = 20.0;
        dir = &scene->camera_hoist_dir;
	dir->heading = 0.75 * PI;
        dir->pitch = 1.92 * PI;
        dir->bank = 0 * PI;
	scene->camera_hoist_dist = 20.0;
        scene->camera_target = -1;

	memset(
	    scene->camera_rotmatrix,
	    0x00,
	    SAR_CAMERA_ROTMATRIX_MAX * (3 * 3) * sizeof(double)
	);
	scene->camera_rotmatrix_count = 0;

	/* Reset initial GL states. */
	if(display != NULL)
	{
	    StateGLResetAll(&display->state_gl);
	    StateGLEnable(&display->state_gl, GL_CULL_FACE);
	    StateGLFrontFace(&display->state_gl, GL_CCW);
            StateGLShadeModel(&display->state_gl, GL_FLAT);
	}
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_PACK_ALIGNMENT, 1);


	/* Allocate messages. */
	scene->total_messages = 4;
	scene->message = (char **)calloc(scene->total_messages, sizeof(char *));
	if(scene->message == NULL)
	    scene->total_messages = 0;

	scene->message_display_untill = 0;
	scene->camera_ref_title_display_untill = 0;

	/* Allocate a realm structure. */
	scene->realm = SFMInit(0, NULL);
	if(scene->realm == NULL)
	{
	    fprintf(stderr,
		"SARSceneLoadFromFile(): Could not create FDM realm.\n"
	    );
	}
	else
	{
	    SFMRealmStruct *realm = scene->realm;

	    /* Set up realm values. */
	    realm->lapsed_time = 0;
	    realm->time_compensation = 1.0;
	    realm->time_compression = 1.0;

	    realm->gravity = SFMDefaultGravity;

	    /* Set up realm callback data and callback function ptrs. */
	    realm->init_model_cb_client_data = core_ptr;
	    realm->init_model_cb = SARSimInitModelCB;
            realm->destroy_model_cb_client_data = core_ptr;
            realm->destroy_model_cb = SARSimDestroyModelCB;
            realm->airborne_cb_client_data = core_ptr;
            realm->airborne_cb = SARSimAirborneCB;
            realm->touch_down_cb_client_data = core_ptr;
            realm->touch_down_cb = SARSimTouchDownCB;
	    realm->overspeed_cb_client_data = core_ptr;
	    realm->overspeed_cb = SARSimOverspeedCB;
            realm->collision_cb_client_data = core_ptr;
            realm->collision_cb = SARSimCollisionCB;

	}

	/* Load textures specified in core structure's textures list
	 * first before loading textures specified in the scene file.
	 */
	if(1)
	{
	    int i, n, tex_fmt;
            char *name, *path;
	    sar_texture_name_struct *tn_ptr;
            char tmp_path[PATH_MAX + NAME_MAX];

/* Already reset.
	    scene->total_texture_refs = 0;
	    scene->texture_ref = NULL;
 */

	    tex_fmt = V3D_TEX_FORMAT_RGBA;

	    /* Go through texture referance names list on core
	     * structure.
	     */
	    for(i = 0; i < core_ptr->total_texture_list; i++)
	    {
		tn_ptr = core_ptr->texture_list[i];
		if(tn_ptr == NULL)
		    continue;

		name = tn_ptr->name;
		path = tn_ptr->filename;

		if((name == NULL) ||
                   (path == NULL)
		)
		    continue;

		/* Allocate a new texture. */
		n = scene->total_texture_refs;
		scene->total_texture_refs++;
		scene->texture_ref = (v3d_texture_ref_struct **)realloc(
		    scene->texture_ref,
		    scene->total_texture_refs * sizeof(v3d_texture_ref_struct *)
		);
		if(scene->texture_ref == NULL)
		{
		    scene->total_texture_refs = 0;
		    break;
		}

		strptr = PrefixPaths(dname.local_data, path);
		if(strptr != NULL)
		{
		    if(stat(strptr, &stat_buf))
			strptr = PrefixPaths(dname.global_data, path);
		}
		if(strptr != NULL)
		{
		    strncpy(tmp_path, strptr, PATH_MAX + NAME_MAX);
		    tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
		    scene->texture_ref[n] = V3DTextureLoadFromFile2D(
			tmp_path, name,
			tex_fmt,
			NULL, NULL
		    );
		    V3DTexturePriority(
			scene->texture_ref[n],
			tn_ptr->priority
		    );
		}
	    }
	}

	/* Load referances to default sound objects. */
	if(1)
	{
            char *name, *storage;

#define SAR_DO_LOAD_SOUND	\
{ \
 if(name == NULL) \
 { \
  storage = NULL; \
 } \
 else \
 { \
  strptr = PrefixPaths(dname.local_data, name); \
  if(strptr != NULL) \
  { \
   if(stat(strptr, &stat_buf)) \
    strptr = PrefixPaths(dname.global_data, name); \
  } \
  storage = StringCopyAlloc(strptr); \
 } \
}
	    name = SAR_DEF_SOUND_LAND_WHEEL_SKID;
	    SAR_DO_LOAD_SOUND
	    scene->land_wheel_skid_sndpath = storage;

            name = SAR_DEF_SOUND_LAND_SKI_SKID;
            SAR_DO_LOAD_SOUND
            scene->land_ski_skid_sndpath = storage;

            name = SAR_DEF_SOUND_LAND_SKI;
            SAR_DO_LOAD_SOUND
            scene->land_ski_sndpath = storage;

            name = SAR_DEF_SOUND_LAND_BELLY;
            SAR_DO_LOAD_SOUND
            scene->land_belly_sndpath = storage;

            name = SAR_DEF_SOUND_CRASH_OBSTRUCTION;
            SAR_DO_LOAD_SOUND
            scene->crash_obstruction_sndpath = storage;

            name = SAR_DEF_SOUND_CRASH_GROUND;
            SAR_DO_LOAD_SOUND
            scene->crash_ground_sndpath = storage;

            name = SAR_DEF_SOUND_SPLASH_AIRCRAFT;
            SAR_DO_LOAD_SOUND
            scene->splash_aircraft_sndpath = storage;

            name = SAR_DEF_SOUND_SPLASH_HUMAN;
            SAR_DO_LOAD_SOUND
            scene->splash_human_sndpath = storage;

#undef SAR_DO_LOAD_SOUND
	}

	/* Create horizon. */
	horizon_ptr = &scene->horizon;
        horizon_ptr->last_tod = -1;
        horizon_ptr->texture = NULL;
        horizon_ptr->total_textures = 0;


	/* Generate default weather settings from the given weather
	 * preset name.
	 */
	SARWeatherSetScenePreset(
	    core_ptr->weather_data,
	    scene,
	    weather_preset_name
	);

	/* Itterate through loaded parms. */
	for(i = 0; i < total_parms; i++)
	{
	    p = parm[i];
	    if(p == NULL)
		continue;
	    else
		ptype = (*(int *)p);

	    if(progress_func != NULL)
	    {
		if(progress_func(client_data, i + 1, total_parms))
		    break;
	    }

            /* Handle by parm type. */
            switch(ptype)
            {
	      case SAR_PARM_NAME:
		p_name = (sar_parm_name_struct *)p;
		break;

	      case SAR_PARM_DESCRIPTION:
		p_description = (sar_parm_description_struct *)p;
		break;

	      case SAR_PARM_PLAYER_MODEL_FILE:
		p_player_model_file = (sar_parm_player_model_file_struct *)p;
		break;

	      case SAR_PARM_WEATHER:
		p_weather = (sar_parm_weather_struct *)p;
		break;

	      case SAR_PARM_REGISTER_LOCATION:
		p_register_location = (sar_parm_register_location_struct *)p;
		break;

	      case SAR_PARM_SCENE_GPS:
		p_scene_gps = (sar_parm_scene_gps_struct *)p;
                scene->dms_x_offset = p_scene_gps->dms_x_offset;
                scene->dms_y_offset = p_scene_gps->dms_y_offset;
                scene->planet_radius = p_scene_gps->planet_radius;
		break;

	      case SAR_PARM_SCENE_MAP:
		p_scene_map = (sar_parm_scene_map_struct *)p;
		break;

	      case SAR_PARM_SCENE_ELEVATION:
		p_scene_elevation = (sar_parm_scene_elevation_struct *)p;
                scene->msl_elevation = p_scene_elevation->elevation;
		break;

	      case SAR_PARM_SCENE_CANT:
		p_scene_cant = (sar_parm_scene_cant_struct *)p;
		scene->cant_angle = p_scene_cant->cant;
		break;

	      case SAR_PARM_SCENE_GROUND_FLAGS:
		p_scene_ground_flags = (sar_parm_ground_flags_struct *)p;
		scene->base_flags = p_scene_ground_flags->flags;
		break;

	      case SAR_PARM_SCENE_GROUND_TILE:
		p_scene_ground_tile = (sar_parm_scene_ground_tile_struct *)p;
		if(1)
		{
                    double min, max;
                    GLuint list;
                    sar_visual_model_struct **vmodel;
                    v3d_texture_ref_struct *t = NULL;

		    /* Close range tiling size and range. */
                    scene->base.tile_width = ((p_scene_ground_tile->tile_width > 0) ?
                        p_scene_ground_tile->tile_width : SAR_DEF_GROUND_BASE_TILE_WIDTH
                    );
                    scene->base.tile_height = ((p_scene_ground_tile->tile_height > 0) ?
                        p_scene_ground_tile->tile_height : SAR_DEF_GROUND_BASE_TILE_HEIGHT
                    );
                    scene->base.close_range = ((p_scene_ground_tile->close_range > 0) ?
                        p_scene_ground_tile->close_range : SAR_DEF_GROUND_BASE_TILE_CLOSE_RANGE
                    );
                    /* Far solid color. */
		    memcpy(
		        &scene->base.color, &p_scene_ground_tile->color,
		        sizeof(sar_color_struct)
		    );

                    /* Close texture referance name. */
                    t = SARGetTextureRefByName(scene, p_scene_ground_tile->texture_name);

                    /* Create simple (far away) ground base visual model
                     * as needed.
                     */
                    vmodel = &scene->base.visual_model_simple;
                    if((*vmodel) != NULL)
                    {
                        fprintf(
                            stderr,
 "Warning: Ground base (far away) visual model is already defined.\n"
                        );
                    }
                    else
                    {
                        /* Create new visual model. */
                        (*vmodel) = SARVisualModelNew(
                            scene, NULL, NULL
                        );

                        /* (Re)generate GL list on visual model structure. */
                        list = (GLuint)SARVisualModelNewList(*vmodel);
                        if(list != 0)
                        {
                            /* Mark visual model as loading. */
                            (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADING;

                            /* Begin recording new list. */
                            glNewList(list, GL_COMPILE);  
                            {
                                /* Far (big) ground base tiles. */
                                min = -(SAR_MAX_VISIBILITY_DISTANCE +
                                    (0.25 * SAR_MAX_VISIBILITY_DISTANCE));
                                max = (SAR_MAX_VISIBILITY_DISTANCE +
                                    (0.25 * SAR_MAX_VISIBILITY_DISTANCE));
                                SARObjGenerateTilePlane(
                                    min, max,               /* Min and max. */
                                    (int)(max / 4),         /* Tile width and height. */
                                    (int)(max / 4)
                                );
                            }
                            glEndList();

                            /* Mark visual model as done loading. */
                            (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADED;
                        } 
                    }

                    /* Create detailed (close up) ground base visual model as
                     * needed.
                     */
                    vmodel = &scene->base.visual_model_close;
                    if((*vmodel) != NULL)
                    {
                        fprintf(
                            stderr,
 "Warning: Ground base (close up) visual model is already defined.\n"
                        );
                    }
                    else
                    {
                        /* Create new visual model. */
                        (*vmodel) = SARVisualModelNew(
                            scene, NULL, NULL
                        );

                        /* (Re)generate GL list on visual model structure. */
                        list = (GLuint)SARVisualModelNewList(*vmodel);
                        if(list != 0)
                        {
                            /* Mark visual model as loading. */
                            (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADING;

                            /* Begin recording new list. */
                            glNewList(list, GL_COMPILE);
                            {
                                /* Select tiled ground texture if defined, if
                                 * no texture then a texture unselect will be
                                 * recorded.
                                 */
                                V3DTextureSelect(t);

                                /* Generate close range textured tiles. */
                                SARObjGenerateTilePlane(
                                    -scene->base.close_range,       /* Min. */
                                    scene->base.close_range,        /* Max. */
                                    scene->base.tile_width,         /* Tile width. */
                                    scene->base.tile_height         /* Tile height. */ 
                                );
                            }
                            glEndList();

                            /* Mark visual model as done loading. */
                            (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADED;
                        }
                    }
		}
		break;

	      case SAR_PARM_TEXTURE_BASE_DIRECTORY:
		p_texture_base_directory = (sar_parm_texture_base_directory_struct *)p;
		break;

	      case SAR_PARM_TEXTURE_LOAD:
		p_texture_load = (sar_parm_texture_load_struct *)p;
                SARObjLoadTexture(
                    core_ptr, scene, p_texture_load
                );
		break;

	      /* Skip mission parms. */

	      case SAR_PARM_NEW_OBJECT:
		p_new_object = (sar_parm_new_object_struct *)p;
                obj_num = SARObjCreate(
		    scene, ptr, total,
		    p_new_object->object_type
		);
		obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);

		/* Reset all substructure type pointers. */
		DO_RESET_SUBSTRUCTURE_PTRS

		if(obj_ptr != NULL)
		{
		    /* Get pointer to substructure. */
		    switch(obj_ptr->type)
		    {
		      case SAR_OBJ_TYPE_AIRCRAFT:
		        obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		        break;

		      case SAR_OBJ_TYPE_GROUND:
		        obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data;
		        break;
		    }
		}
		break;

	      case SAR_PARM_NEW_HELIPAD:
		p_new_helipad = (sar_parm_new_helipad_struct *)p;
		DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadHelipad(
                    core_ptr, scene, p_new_helipad
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
		break;

	      case SAR_PARM_NEW_RUNWAY:
		p_new_runway = (sar_parm_new_runway_struct *)p;
		DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjCreate(
                    scene, ptr, total,
                    SAR_OBJ_TYPE_RUNWAY
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                if(obj_ptr != NULL)
                {
                    sar_object_runway_struct *obj_runway_ptr = 
			(sar_object_runway_struct *)obj_ptr->data;

                    obj_ptr->range = p_new_runway->range;

                    if(obj_runway_ptr != NULL)
                    {
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_NORTH_LABEL;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_SOUTH_LABEL;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_EDGE_LIGHTING;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_NORTH_LIGHTING;
                        obj_runway_ptr->flags |= SAR_RUNWAY_FLAG_SOUTH_LIGHTING;

                        obj_runway_ptr->length = p_new_runway->length;
                        obj_runway_ptr->width = p_new_runway->width;

                        SARObjAddContactBoundsRectangular(
                            obj_ptr,
                            SAR_CRASH_FLAG_SUPPORT_SURFACE, 0,
			    -(obj_runway_ptr->width / 2),
			    (obj_runway_ptr->width / 2),
			    -(obj_runway_ptr->length / 2),
                            (obj_runway_ptr->length / 2),
                            0.0, 0.0
			);

/*
                        obj_runway_ptr->north_label =
                        obj_runway_ptr->south_label =
 */

                        obj_runway_ptr->light_spacing =
                            obj_runway_ptr->length / 50;

                        /* Load texture. */
                        obj_runway_ptr->tex_num = SARGetTextureRefNumberByName(
                            scene, SAR_STD_TEXNAME_RUNWAY
                        );
                    }
                }
		break;

	      case SAR_PARM_NEW_HUMAN:
		p_new_human = (sar_parm_new_human_struct *)p;
		DO_RESET_SUBSTRUCTURE_PTRS
		obj_num = SARObjLoadHuman(
		    core_ptr, scene, p_new_human
		);
		obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
		obj_human_ptr = (sar_object_human_struct *)obj_ptr->data;
		break;

	      case SAR_PARM_NEW_FIRE:
		p_new_fire = (sar_parm_new_fire_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadFire(
                    core_ptr, scene, p_new_fire
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                obj_fire_ptr = (sar_object_fire_struct *)obj_ptr->data;
                break;

              case SAR_PARM_NEW_SMOKE:
                p_new_smoke = (sar_parm_new_smoke_struct *)p;
                DO_RESET_SUBSTRUCTURE_PTRS
                obj_num = SARObjLoadSmoke(
                    core_ptr, scene, p_new_smoke
                );
                obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
                obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data;
                break;

	      case SAR_PARM_NEW_PREMODELED:
		p_new_premodeled = (sar_parm_new_premodeled_struct *)p;
		DO_RESET_SUBSTRUCTURE_PTRS
		obj_num = SARObjCreatePremodeled(
		    core_ptr, scene,
		    p_new_premodeled->model_type,
		    p_new_premodeled->argc,
		    p_new_premodeled->argv
		);
		obj_ptr = ((obj_num < 0) ? NULL : (*ptr)[obj_num]);
		break;

	      case SAR_PARM_MODEL_FILE:
		p_model_file = (sar_parm_model_file_struct *)p;
                if((p_model_file->file != NULL) && (obj_ptr != NULL))
                    SARObjLoadFromFile(core_ptr, obj_num, p_model_file->file);
		break;

	      case SAR_PARM_RANGE:
		p_range = (sar_parm_range_struct *)p;
                if(obj_ptr != NULL)
                {
                    obj_ptr->range = MAX(p_range->range, 0.0);
		}
		break;

	      case SAR_PARM_TRANSLATE:
		p_translate = (sar_parm_translate_struct *)p;
                SARObjLoadTranslate(
                    core_ptr, scene,
                    obj_ptr, p_translate
                );
		break;

	      case SAR_PARM_TRANSLATE_RANDOM:
		p_translate_random = (sar_parm_translate_random_struct *)p;
/* Ignore this, for missions only). */
		break;

	      case SAR_PARM_ROTATE:
		p_rotate = (sar_parm_rotate_struct *)p;
		if(obj_ptr != NULL)
                {
                    SARSimWarpObject(  
                        scene, obj_ptr,
                        NULL, &p_rotate->rotate
                    );
                }
		break;

	      case SAR_PARM_NO_DEPTH_TEST:
		p_no_depth_test = (sar_parm_no_depth_test_struct *)p;
                if(obj_ptr != NULL)
                {
                    obj_ptr->flags |= SAR_OBJ_FLAG_NO_DEPTH_TEST;
                }
		break;

              case SAR_PARM_POLYGON_OFFSET:
                p_polygon_offset = (sar_parm_polygon_offset_struct *)p;
                if(obj_ptr != NULL)
                {
		    obj_ptr->flags |= p_polygon_offset->flags;
                }
                break;

	      case SAR_PARM_CONTACT_BOUNDS_SPHERICAL:
		p_contact_bounds_spherical = (sar_parm_contact_bounds_spherical_struct *)p;
		if(obj_ptr != NULL)
		{
		    sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                    SARObjAddContactBoundsSpherical(
                        obj_ptr,
                        (cb == NULL) ? 0 : cb->crash_flags,
                        (cb == NULL) ? 0 : cb->crash_type,
                        p_contact_bounds_spherical->radius
                    );
		}
		break;

              case SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL:
                p_contact_bounds_cylendrical = (sar_parm_contact_bounds_cylendrical_struct *)p;
                if(obj_ptr != NULL)
                {
                    sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                    SARObjAddContactBoundsCylendrical(
                        obj_ptr,
                        (cb == NULL) ? 0 : cb->crash_flags,
                        (cb == NULL) ? 0 : cb->crash_type,
                        p_contact_bounds_cylendrical->radius,
			p_contact_bounds_cylendrical->height_min,
			p_contact_bounds_cylendrical->height_max
                    );
                }
                break;

              case SAR_PARM_CONTACT_BOUNDS_RECTANGULAR:
                p_contact_bounds_rectangular = (sar_parm_contact_bounds_rectangular_struct *)p;
                if(obj_ptr != NULL)
                {
                    sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                    SARObjAddContactBoundsRectangular(
                        obj_ptr,
                        (cb == NULL) ? 0 : cb->crash_flags,
                        (cb == NULL) ? 0 : cb->crash_type,
                        p_contact_bounds_rectangular->x_min,
			p_contact_bounds_rectangular->x_max,
                        p_contact_bounds_rectangular->y_min, 
                        p_contact_bounds_rectangular->y_max,
                        p_contact_bounds_rectangular->z_min, 
                        p_contact_bounds_rectangular->z_max
                    );
                }
                break;

	      case SAR_PARM_GROUND_ELEVATION:
		p_ground_elevation = (sar_parm_ground_elevation_struct *)p;
                if(obj_ground_ptr == NULL)
                {
                    fprintf(
			stderr,
 "Warning: Cannot set ground elevation for object not type `%i'.\n",
                        SAR_OBJ_TYPE_GROUND
                    );
                }
                else
                {
                    obj_ground_ptr->elevation = p_ground_elevation->elevation;
                }
		break;

	      case SAR_PARM_OBJECT_NAME:
		p_object_name = (sar_parm_object_name_struct *)p;
                if(obj_ptr != NULL)
                {
                    free(obj_ptr->name);
                    obj_ptr->name = StringCopyAlloc(p_object_name->name);
                }
		break;

              case SAR_PARM_OBJECT_MAP_DESCRIPTION:
                p_object_map_description = (sar_parm_object_map_description_struct *)p;
                if(obj_ptr != NULL)
                {
/* Ignore this. */
                }
                break;

	      case SAR_PARM_FUEL:
		p_fuel = (sar_parm_fuel_struct *)p;
                if(obj_aircraft_ptr != NULL)
                {
		    /* Set current fuel. */
                    obj_aircraft_ptr->fuel = p_fuel->fuel;

		    /* Set max only if not negative. */
		    if(p_fuel->fuel_max >= 0.0)
			obj_aircraft_ptr->fuel_max = p_fuel->fuel_max;

		    /* Sanitize current. */
		    if(obj_aircraft_ptr->fuel > obj_aircraft_ptr->fuel_max)
			obj_aircraft_ptr->fuel = obj_aircraft_ptr->fuel_max;
                }
		break;

	      case SAR_PARM_HITPOINTS:
		p_hitpoints = (sar_parm_hitpoints_struct *)p;
		if(obj_ptr != NULL)
		{
		    /* Set current hit points. */
		    obj_ptr->hit_points = p_hitpoints->hitpoints;

		    /* Set max only if not negative. */
		    if(p_hitpoints->hitpoints_max >= 0.0)
			obj_ptr->hit_points_max = p_hitpoints->hitpoints_max;

		    /* Sanitize current. */
		    if(obj_ptr->hit_points > obj_ptr->hit_points_max)
			obj_ptr->hit_points = obj_ptr->hit_points_max;
		}
		break;

	      case SAR_PARM_ENGINE_STATE:
		p_engine_state = (sar_parm_engine_state_struct *)p;
/* Work on this later. */
		break;

	      case SAR_PARM_PASSENGERS:
		p_passengers = (sar_parm_passengers_struct *)p;
/* Work on this later. */
                break;

	      case SAR_PARM_HUMAN_MESSAGE_ENTER:
		p_human_message_enter = (sar_parm_human_message_enter_struct *)p;
                if((obj_ptr != NULL) && (obj_human_ptr != NULL))
		{
		    free(obj_human_ptr->mesg_enter);
		    obj_human_ptr->mesg_enter = StringCopyAlloc(
			p_human_message_enter->message
		    );
		}
		break;

              case SAR_PARM_HUMAN_REFERANCE:
                p_human_referance = (sar_parm_human_referance_struct *)p;
                if((obj_ptr != NULL) && (obj_human_ptr != NULL) &&
		   (p_human_referance->referance_name != NULL)
		)
                {
		    int human_ref_obj_num = -1;
		    const char *ref_name = (const char *)p_human_referance->referance_name;

		    /* Run towards? */
		    if(p_human_referance->flags & SAR_HUMAN_FLAG_RUN_TOWARDS)
		    {
			obj_human_ptr->flags |= SAR_HUMAN_FLAG_RUN_TOWARDS;
		    }
		    /* Run away? */
		    else if(p_human_referance->flags & SAR_HUMAN_FLAG_RUN_AWAY)
		    {
			obj_human_ptr->flags |= SAR_HUMAN_FLAG_RUN_AWAY;
		    }

		    /* Handle referance object name. */
		    if(!strcasecmp(ref_name, "player"))
		    {
			/* Set special intercept code to intercept the player. */
			obj_human_ptr->intercepting_object = -2;
		    }
		    else
		    {
			/* All else match by object name. */
			SARObjMatchPointerByName(
			    scene, *ptr, *total,
			    ref_name, &human_ref_obj_num
			);
			obj_human_ptr->intercepting_object = human_ref_obj_num;
		    }
                }
                break;


	      default:
		break;
	    }	/* Handle by parm type. */
	}	/* Itterate through loaded parms. */


	/* Deallocate loaded parms. */
	SARParmDeleteAll(&parm, &total_parms);

#undef DO_RESET_SUBSTRUCTURE_PTRS

        return(0);
}
