#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <math.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>

#include "../include/string.h"

#include "matrixmath.h"
#include "sfm.h"

#include "gw.h"
#include "stategl.h"
#include "textinput.h"

#include "sarreality.h"
#include "obj.h"
#include "messages.h"
#include "gctl.h"
#include "sarsound.h"
#include "simutils.h"

#include "sar.h"
#include "config.h"

#include "sardrawdefs.h"


extern void SARDrawHumanIterate( 
        gw_display_struct *display, sar_scene_struct *scene,
        double height, double mass,
        sar_obj_flags_t flags,          /* Human flags. */  
        sar_color_struct *palette,      /* Human colors. */
        int water_ripple_tex_num,
        sar_grad_anim_t anim_pos
);
extern void SARDrawHelipad(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double distance
);
extern void SARDrawHelipadMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double icon_len
);
extern void SARDrawRunwayMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double icon_len
);
extern void SARDrawRunway(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double distance
);
extern void SARDrawHuman(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_human_struct *obj_human_ptr
);


static void SARDrawGetDirFromPos(
	sar_position_struct *pos1, sar_position_struct *pos2,
	sar_direction_struct *dir_rtn
);
static void SARDrawSetColor(sar_color_struct *c);
void SARVertexBoxNS(double x_len, double y_len, double z_len);
void SARVertexBoxBaseNS(double x_len, double y_len, double z_len);

static void SARSetCameraMapDrawGCC(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
	int obj_num
);
static void SARSetCamera(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total
);
static void SARDrawSetSpotLight(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr
);
static void SARDrawHelp(sar_core_struct *core_ptr);

static void SARDrawRotor(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_rotor_struct *rotor_ptr,
	double throttle
);
static void SARDrawLandingGear(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_landing_gear_struct *lgear_ptr
);
static void SARDrawFuelTank(
        sar_scene_struct *scene,   
        sar_object_struct *obj_ptr,
        sar_external_fueltank_struct *eft_ptr
);
static void SARDrawDoor(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_door_struct *door_ptr
);
static void SARDrawHoistDeployment(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_hoist_struct *hoist_ptr
);
static void SARDrawSmoke(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_smoke_struct *obj_smoke_ptr
);
static void SARDrawExplosion(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
        sar_object_explosion_struct *obj_explosion_ptr
);

static void SARDrawHUDAttitude(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_direction_struct *dir, sar_position_struct *vel,
        int offset_x, int offset_y,
        int width, int height,
        double fovx, double fovy
);
static void SARDrawHUDHeading(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_direction_struct *dir, sar_obj_intercept_struct *intercept,
        int offset_x, int offset_y,
        int width, int height
);
static void SARDrawHUDElevatorTrim(
        gw_display_struct *display,
        GLfloat xc, GLfloat yc,
        GLfloat length, GLfloat width,
        double elevator_trim 
);
static void SARDrawHUD(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        double aspect
);
static void SARDrawOutsideAttitude(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
	int width, int height
);
static void SARDrawMapGrids(
        gw_display_struct *display, sar_scene_struct *scene,
	sar_position_struct *pos, double fovz_um, double aspect
);
static void SARDrawMapCrossHairs(
        gw_display_struct *display, sar_scene_struct *scene
);

static void SARDrawMessages(
	gw_display_struct *display, sar_scene_struct *scene
);
static void SARDrawBanner(
        gw_display_struct *display, sar_scene_struct *scene
);
static void SARDrawSlewCoordinates(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr
);

static void SARDrawLights(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_light_struct **ptr, int total
);

static void SARDrawObjectCockpit(
	sar_scene_struct *scene,  sar_object_struct *obj_ptr
);

static void SARDrawObjectShadow(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr
);

static void SARDrawSceneFoundations(
	gw_display_struct *display, sar_scene_struct *scene,
	double aspect
);

static void SARDrawCloudLayer(
	sar_scene_struct *scene, sar_cloud_layer_struct *cloud_layer_ptr,
        Boolean flip
);

static void SARDrawCloudBBItterate(
        sar_position_struct *pos,
        double hw, double height
);
static void SARDrawCloudBB(
        gw_display_struct *display,
        sar_scene_struct *scene,
	sar_cloud_bb_struct *cloud_bb_ptr
);

static void SARDrawSceneHorizon(
	gw_display_struct *display, sar_scene_struct *scene,
	double visibility_max	/* In meters. */
);
static void SARDrawSceneCelestial(
        gw_display_struct *display, sar_scene_struct *scene,
        double visibility_max,	/* In meters. */
	double fovz_um,		/* Field of view unit meters, z axis. */
	double aspect		/* Width / height. */
);


void SARDraw(sar_core_struct *core_ptr);
void SARDrawMap(
	sar_core_struct *core_ptr,
	Boolean draw_for_gcc, Boolean draw_for_ghc, int gcc_obj_num
);
        

static int player_wheel_brakes, player_air_brakes, player_overspeed;
static int flight_model_type;
static Boolean camera_in_cockpit;
static int camera_ref;
static sar_position_struct camera_pos; 
static sar_direction_struct camera_dir;
static sar_position_struct primary_light_pos;
static GLfloat sky_color[3], atmosphere_color[3];
static sar_cloud_layer_struct   *lowest_cloud_layer_ptr,    
                                *highest_cloud_layer_ptr;


/*
 *	Calculates the heading and pitch of position pos1 to
 *	position pos2, putting the results in dir_rtn.
 */
static void SARDrawGetDirFromPos(
        sar_position_struct *pos1, sar_position_struct *pos2,
        sar_direction_struct *dir_rtn
)
{
	double r;
	sar_position_struct delta;

	if((pos1 == NULL) || (pos2 == NULL) || (dir_rtn == NULL))
	    return;

	delta.x = pos2->x - pos1->x;
	delta.y = pos2->y - pos1->y;
	delta.z = pos2->z - pos1->z;
	r = hypot(delta.x, delta.y);

	dir_rtn->heading = SFMSanitizeRadians(
	    (0.5 * PI) - atan2(delta.y, delta.x)
	);
        dir_rtn->pitch = SFMSanitizeRadians(
            -atan2(delta.z, r)
        );
	dir_rtn->bank = 0.0;

	return;
}


/*
 *	Set gl color from color structure.
 */
static void SARDrawSetColor(sar_color_struct *c)
{
	if(c != NULL)
	    glColor4d(c->r, c->g, c->b, c->a);
}

/*
 *	Sets vertexes to create a 3D box at the center.
 *
 *      Normals are spread to extreme stretch.
 */
void SARVertexBoxNS(
	double x_len, double y_len, double z_len
)
{
	double x_min, x_max;
	double y_min, y_max;
	double z_min, z_max;

	x_max = x_len / 2;
	x_min = -x_max;

        y_max = y_len / 2;
        y_min = -y_max;

        z_max = z_len / 2;
        z_min = -z_max;

	/* Front and back. */
	glNormal3d(-0.33, -0.33, -0.88);
	glVertex3d(x_min, z_min, -y_max);
        glNormal3d(-0.33, 0.33, -0.88); 
        glVertex3d(x_min, z_max, -y_max);
        glNormal3d(0.33, 0.33, -0.88); 
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(0.33, -0.33, -0.88); 
        glVertex3d(x_max, z_min, -y_max);

        glNormal3d(-0.33, 0.33, 0.88); 
        glVertex3d(x_min, z_max, -y_min);
        glNormal3d(-0.33, -0.33, 0.88);
        glVertex3d(x_min, z_min, -y_min);
        glNormal3d(0.33, -0.33, 0.88);
        glVertex3d(x_max, z_min, -y_min);
        glNormal3d(0.33, 0.33, 0.88);
        glVertex3d(x_max, z_max, -y_min);

	/* Left and right. */
        glNormal3d(-0.88, 0.33, -0.33);
        glVertex3d(x_min, z_max, -y_max);
        glNormal3d(-0.88, -0.33, -0.33);
        glVertex3d(x_min, z_min, -y_max);
        glNormal3d(-0.88, -0.33, 0.33);
        glVertex3d(x_min, z_min, -y_min);
        glNormal3d(-0.88, 0.33, 0.33);
        glVertex3d(x_min, z_max, -y_min);

        glNormal3d(0.88, -0.33, -0.33);
        glVertex3d(x_max, z_min, -y_max);
        glNormal3d(0.88, 0.33, -0.33);
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(0.88, 0.33, 0.33);
        glVertex3d(x_max, z_max, -y_min);
        glNormal3d(0.88, -0.33, 0.33);
        glVertex3d(x_max, z_min, -y_min);

	/* Top and bottom. */
        glNormal3d(-0.33, 0.88, 0.33);
        glVertex3d(x_min, z_max, -y_min);
        glNormal3d(0.33, 0.88, 0.33);
        glVertex3d(x_max, z_max, -y_min);
        glNormal3d(0.33, 0.88, -0.33);
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(-0.33, 0.88, -0.33);
        glVertex3d(x_min, z_max, -y_max);

        glNormal3d(-0.33, -0.88, -0.33);
        glVertex3d(x_min, z_min, -y_max);
        glNormal3d(0.33, -0.88, -0.33);
        glVertex3d(x_max, z_min, -y_max);
        glNormal3d(0.33, -0.88, 0.33);
        glVertex3d(x_max, z_min, -y_min);
        glNormal3d(-0.33, -0.88, 0.33);
        glVertex3d(x_min, z_min, -y_min);

	return;
}

/*
 *      Sets vertexes to create a 3D box in the positive octant
 *	with respect to the x y plane.  The x and y walls are
 *	centered.
 *
 *	Bottom of box is not closed, since this is `based' box.
 *
 *	Normals are stretched.
 */
void SARVertexBoxBaseNS(
        double x_len, double y_len, double z_len
)
{
        double x_min, x_max;
        double y_min, y_max;
	double z_min, z_max;

        x_max = x_len / 2;
        x_min = -x_max;

        y_max = y_len / 2;
        y_min = -y_max;

	z_min = 0;
	z_max = z_len;

        /* Front and back. */
        glNormal3d(-0.33, -0.33, -0.88);
        glVertex3d(x_min, z_min, -y_max);
        glNormal3d(-0.33, 0.33, -0.88);
        glVertex3d(x_min, z_max, -y_max);
        glNormal3d(0.33, 0.33, -0.88);
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(0.33, -0.33, -0.88);
        glVertex3d(x_max, z_min, -y_max);

        glNormal3d(-0.33, 0.33, 0.88);
        glVertex3d(x_min, z_max, -y_min);
        glNormal3d(-0.33, -0.33, 0.88);
        glVertex3d(x_min, z_min, -y_min);
        glNormal3d(0.33, -0.33, 0.88);
        glVertex3d(x_max, z_min, -y_min);
        glNormal3d(0.33, 0.33, 0.88);
        glVertex3d(x_max, z_max, -y_min);

        /* Left and right. */
        glNormal3d(-0.88, 0.33, -0.33);
        glVertex3d(x_min, z_max, -y_max);
        glNormal3d(-0.88, -0.33, -0.33);
        glVertex3d(x_min, z_min, -y_max);
        glNormal3d(-0.88, -0.33, 0.33);
        glVertex3d(x_min, z_min, -y_min);
        glNormal3d(-0.88, 0.33, 0.33);
        glVertex3d(x_min, z_max, -y_min);

        glNormal3d(0.88, -0.33, -0.33);
        glVertex3d(x_max, z_min, -y_max);
        glNormal3d(0.88, 0.33, -0.33);
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(0.88, 0.33, 0.33);
        glVertex3d(x_max, z_max, -y_min);
        glNormal3d(0.88, -0.33, 0.33);
        glVertex3d(x_max, z_min, -y_min);

        /* Top only. */
        glNormal3d(-0.33, 0.88, 0.33);
        glVertex3d(x_min, z_max, -y_min);
        glNormal3d(0.33, 0.88, 0.33);
        glVertex3d(x_max, z_max, -y_min);
        glNormal3d(0.33, 0.88, -0.33);
        glVertex3d(x_max, z_max, -y_max);
        glNormal3d(-0.33, 0.88, -0.33);
        glVertex3d(x_min, z_max, -y_max);

	return;
}


/*
 *	Sets up camera for map draw ground contact check (called by
 *	SARDrawMap() with draw_for_gcc set to True).
 *
 *	Works the same as SARSetCamera() except sets camera position to be
 *	at position of specified object obj_num looking straight down.
 *	If obj_num is -1 then it defaults to the scene structure's player
 *	object.
 */
static void SARSetCameraMapDrawGCC(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
	int obj_num
)
{
	int i;
        sar_object_struct *obj_ptr;
        sar_position_struct *pos;


	/* Reset camera rotation matrix. */
	scene->camera_rotmatrix_count = 0;

        /* Update camera ref code. */
        camera_ref = scene->camera_ref;

	/* Camera will always not be in cockpit when drawing for
	 * ground contact check.
	 */
	camera_in_cockpit = False;

	/* Get pointer to object (if any). */
	if((obj_num < 0) || (obj_num >= (*total)))
	    obj_ptr = scene->player_obj_ptr;
	else
	    obj_ptr = (*ptr)[obj_num];

	/* Got reference object? */
	if(obj_ptr != NULL)
	{
	    pos = &obj_ptr->pos;

            /* Set up initial normal before we translate. */
            glNormal3d(0, 0, 1);

	    glRotated(90, 1.0, 0.0, 0.0);
	    glTranslated(-pos->x, -pos->z, pos->y);

	    /* Record camera position. */
	    camera_pos.x = pos->x;
	    camera_pos.y = pos->y;
	    camera_pos.z = pos->z;

	    /* Record camera direction. */
	    camera_dir.heading = (0.0 * PI);
	    camera_dir.pitch = (0.5 * PI);
	    camera_dir.bank = (0.0 * PI);

	    /* Set camera rotation matrix. */
	    i = 0;
	    if(i < SAR_CAMERA_ROTMATRIX_MAX)
	    {
		MatrixGetHeading33(
		    -0.0 * PI,
		    scene->camera_rotmatrix[i]
		);
		scene->camera_rotmatrix_count = i + 1;
	    }
            i = 1;
            if(i < SAR_CAMERA_ROTMATRIX_MAX)
            {
                MatrixGetPitch33(
                    -0.5 * PI,
                    scene->camera_rotmatrix[i] 
                );
		scene->camera_rotmatrix_count = i + 1;
            }
	}

	return;
}


/*
 *	Translates camera to proper positions defined on the given
 *	scene structure.
 *
 *	Local globals camera_pos, camera_dir, and camera_in_cockpit will
 *	be updated.
 *
 *	Scene structure's ear_pos will be updated as needed.
 *
 *	Note: if GET_CAM_DIR is False then camera_dir will be all
 *	zero'ed.
 */
static void SARSetCamera(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total
)
{
	int i, n;
	double x, y, z;
	sar_object_struct *obj_ptr;
	sar_direction_struct *dir, *dir2;
	sar_position_struct *pos, *pos2;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	GLdouble val[16];


        /* Reset camera rotation matrix. */
        scene->camera_rotmatrix_count = 0;

	/* Update camera ref code. */
	camera_ref = scene->camera_ref;

	switch(camera_ref)
	{
          case SAR_CAMERA_REF_HOIST:
            camera_in_cockpit = True;
            obj_ptr = scene->player_obj_ptr;
            if(obj_ptr != NULL)
            {
                double sin_heading, cos_heading;
                double cos_pitch;
                double x_offset, y_offset, z_offset;
		sar_obj_hoist_struct *hoist_ptr;


		/* Get first hoist on object if any. */
		hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
		if((hoist_ptr == NULL) ? 0 : (hoist_ptr->rope_cur_vis > 0.0))
		{
                    pos = &hoist_ptr->pos;
                    dir = &obj_ptr->dir;
                    dir2 = &scene->camera_hoist_dir;
		}
		else
		{
		    pos = &obj_ptr->pos;
                    dir = &obj_ptr->dir;
                    dir2 = &scene->camera_hoist_dir;
		}

                /* Pre calculate object heading trig values. */
                sin_heading = sin(dir->heading);
                cos_heading = cos(dir->heading);
                cos_pitch = cos(dir2->pitch);   /* Camera spot pitch. */

                /* Spot camera offset. */
                x_offset = sin(dir2->heading) * cos_pitch *
                    scene->camera_hoist_dist; 
                y_offset = cos(dir2->heading) * cos_pitch *
                    scene->camera_hoist_dist;
                z_offset = -sin(dir2->pitch) * scene->camera_hoist_dist;

                x = pos->x + (cos_heading * x_offset) +
                    (sin_heading * y_offset);
                y = pos->y - (sin_heading * x_offset) +
                    (cos_heading * y_offset);
                z = pos->z + z_offset;
                if(z < 0.0)
                    z = 0.0;
                gluLookAt(
                    x, z, -y,
                    pos->x, pos->z, -pos->y,
                    0, 1, 0
                );

                /* Record camera position. */
                camera_pos.x = x;
                camera_pos.y = y;   
                camera_pos.z = z;

                /* Set camera rotation matrix. */
                i = 0;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
                        -(dir2->heading + dir->heading + PI),
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 1;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        dir2->pitch,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
	    }
            break;

	  case SAR_CAMERA_REF_MAP:
            camera_in_cockpit = False;

            obj_ptr = scene->player_obj_ptr;
            if(obj_ptr != NULL)
            {
                pos = &scene->camera_map_pos;

                /* Set up initial normal before we translate. */
		glNormal3d(0, 0, 1);

                glRotatef(90, 1.0, 0.0, 0.0);
                glTranslated(-pos->x, -pos->z, pos->y);

                /* Record camera position. */
                camera_pos.x = pos->x;
                camera_pos.y = pos->y;
                camera_pos.z = pos->z;

                /* Set camera rotation matrix. */
                i = 0;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
                        -0.0 * PI,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 1;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        -0.5 * PI,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
	    }
	    break;

	  case SAR_CAMERA_REF_TOWER:
	    camera_in_cockpit = False;

            n = scene->camera_target;
            if(SARObjIsAllocated(*ptr, *total, n))
            {
		double x, y, z;


                obj_ptr = (*ptr)[n];

		/* Get position of tower. */
		x = scene->camera_tower_pos.x;
		y = scene->camera_tower_pos.y;
		z = scene->camera_tower_pos.z;
		/* Keep above ground. */
		if(z < 1.0)
		    z = 1.0;

		/* Get position of object. */
                pos = &obj_ptr->pos;

                gluLookAt(
                    x, z, -y,
                    pos->x, pos->z, -pos->y,
                    0, 1, 0
                );

                /* Record camera position. */
                camera_pos.x = x;
                camera_pos.y = y;
                camera_pos.z = z;

                /* Set camera rotation matrix. */
                i = 0;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
			-((0.5 * PI) - atan2(pos->y - y, pos->x - x)),
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 1;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        atan2(pos->z - z,
			    hypot(pos->y - y, pos->x - x)
			),
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
	    }
	    break;

	  case SAR_CAMERA_REF_SPOT:
	    camera_in_cockpit = False;

	    n = scene->camera_target;
	    if(SARObjIsAllocated(*ptr, *total, n))
	    {
		double sin_heading, cos_heading;
		double cos_pitch;
		double x_offset, y_offset, z_offset;


	        obj_ptr = (*ptr)[n];

                pos = &obj_ptr->pos;
                dir = &obj_ptr->dir;
		dir2 = &scene->camera_spot_dir;

		/* Pre calculate object heading trig values. */
		sin_heading = sin(dir->heading);
		cos_heading = cos(dir->heading);
		cos_pitch = cos(dir2->pitch);	/* Camera spot pitch. */

		/* Spot camera offset. */
		x_offset = sin(dir2->heading) * cos_pitch *
                    scene->camera_spot_dist;
		y_offset = cos(dir2->heading) * cos_pitch *
		    scene->camera_spot_dist;
		z_offset = -sin(dir2->pitch) * scene->camera_spot_dist;

		x = pos->x + (cos_heading * x_offset) +
		    (sin_heading * y_offset);
		y = pos->y - (sin_heading * x_offset) +
		    (cos_heading * y_offset);
		z = pos->z + z_offset;
		/* Keep z above ground. */
		if(z < 0.5)
		    z = 0.5;
		gluLookAt(
		    x, z, -y,
		    pos->x, pos->z, -pos->y,
		    0, 1, 0
		);

		/* Record camera position. */
		camera_pos.x = x;
		camera_pos.y = y;
		camera_pos.z = z;

                /* Set camera rotation matrix. */
                i = 0;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
                        -(dir2->heading + dir->heading + PI),
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 1;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        dir2->pitch,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
	    }
	    else
	    {
/* Resort to some other default if spot target object does not exist? */
	    }
	    break;

	  default:	/* Default to cockpit. */
	    camera_in_cockpit = True;

	    obj_ptr = scene->player_obj_ptr;
            if(obj_ptr != NULL)
            {
		if(obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT)
		    obj_aircraft_ptr = obj_ptr->data;
		if(obj_aircraft_ptr == NULL)
		    break;

                pos = &obj_ptr->pos;
                dir = &obj_ptr->dir;

		pos2 = &obj_aircraft_ptr->cockpit_offset_pos;
                dir2 = &scene->camera_cockpit_dir;

		/* Set up initial normal before we translate. */
		glNormal3d(0, 0, 1);

                glRotated(SFMRadiansToDegrees(dir2->pitch), 1.0, 0.0, 0.0);
                glRotated(SFMRadiansToDegrees(dir2->heading), 0.0, 1.0, 0.0);
                glTranslated(-pos2->x, -pos2->z, pos2->y);

                glRotated(SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);
                glRotated(SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                glRotated(SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                glTranslated(-pos->x, -pos->z, pos->y);

                /* Record camera position. */
                camera_pos.x = pos->x;
                camera_pos.y = pos->y;
                camera_pos.z = pos->z;

                /* Set camera rotation matrix. */
                i = 0;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
                        -dir->heading,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 1;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        -dir->pitch,
                        scene->camera_rotmatrix[i]
                    );
		    scene->camera_rotmatrix_count = i + 1;
                }
                i = 2;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetBank33( 
                        -dir->bank,
                        scene->camera_rotmatrix[i]
                    );
                    scene->camera_rotmatrix_count = i + 1;
                }
                i = 3;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetHeading33(
                        -dir2->heading,
                        scene->camera_rotmatrix[i]
                    );
                    scene->camera_rotmatrix_count = i + 1;
                }
                i = 4;
                if(i < SAR_CAMERA_ROTMATRIX_MAX)
                {
                    MatrixGetPitch33(
                        -dir2->pitch,
                        scene->camera_rotmatrix[i]
                    );
                    scene->camera_rotmatrix_count = i + 1;
                }
	    }
	    break;
        }

	/* Update camera direction? */
	if(GET_CAM_DIR)
	{
            /* Get matrix containing results of our camera rotations
	     * and translations.
	     */
            glGetDoublev(GL_MODELVIEW_MATRIX, val);

	    /* Record camera direction. */
	    camera_dir.heading = SFMSanitizeRadians(
	        (0.5 * PI) - atan2(val[10], val[8])
            );
	    camera_dir.pitch = SFMSanitizeRadians(
	        atan2(val[6], val[5])
	    );
	    camera_dir.bank = SFMSanitizeRadians(asin(-val[4]));
	}
	else
	{
	    memset(&camera_dir, 0x00, sizeof(sar_direction_struct));
	}

	/* Update ear position on scene structure based on calculated
	 * camera_pos.
	 */
	memcpy(
	    &scene->ear_pos,
	    &camera_pos,
	    sizeof(sar_position_struct)
	);

	return;
}

/*
 *	Sets up the search light (GL_LIGHT1) for the player object.
 *
 *	Inputs assumed valid.
 *
 *	Translation matrix should be translated to light position.
 */
static void SARDrawSetSpotLight(
	gw_display_struct *display,
	sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
	int i;
	Boolean state = False;
	int light_num = GL_LIGHT1;
	sar_position_struct pos;
	double scene_g, g = 1.0;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_light_struct *light_ptr;


	/* Get scene gamma as just the red compient of the scene light. */
	scene_g = scene->light_color.r;

	/* Handle by object type. */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = obj_ptr->data;
	    if(obj_aircraft_ptr == NULL)
		break;

	    /* Scene gamma less than 0.75? */
	    if(scene_g < 0.75)
	    {
		/* Object has a spot light? */
		for(i = 0; i < obj_ptr->total_lights; i++)
		{
		    light_ptr = obj_ptr->light[i];
		    if(light_ptr == NULL)
			continue;

		    if(light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE)
		    {
			if(light_ptr->flags & SAR_LIGHT_FLAG_ON)
			{
			    /* Has light, set state to True and get
			     * light position relative to center of
			     * object.
			     */
			    state = True;
			    memcpy(
				&pos, &light_ptr->pos,
				sizeof(sar_position_struct)
			    );
			}
			break;
		    }
		}

		/* Calculate gamma for spot light, as scene gamma
		 * gets darker, the spot light gets brighter.
		 */
		g = CLIP(1.0 - scene_g, 0.0, 1.0);
	    }
	    break;
	}

	/* If state is True then that implies a spot light is on. */
	if(state)
	{
	    GLfloat v[4];

            StateGLEnable(&display->state_gl, light_num);

	    v[0] = (GLfloat)pos.x;
	    v[1] = (GLfloat)pos.z;
	    v[2] = (GLfloat)-pos.y;
	    v[3] = 1.0;			/* Positional light. */
	    glLightfv(light_num, GL_POSITION, &(v[0]));

            v[0] = 0.0;
	    v[1] = 0.0;
	    v[2] = 0.0;
	    v[3] = 1.0;
	    glLightfv(light_num, GL_AMBIENT, &(v[0]));

            v[0] = (GLfloat)g;
            v[1] = (GLfloat)g;
            v[2] = (GLfloat)g;
            v[3] = 1.0;
            glLightfv(light_num, GL_DIFFUSE, &(v[0]));

            v[0] = 1.0;
            v[1] = 1.0;
            v[2] = 1.0;
            v[3] = 1.0;
            glLightfv(light_num, GL_SPECULAR, &(v[0]));

	    glLightf(light_num, (GLuint)GL_CONSTANT_ATTENUATION, (GLfloat)1.0);
            glLightf(light_num, (GLuint)GL_LINEAR_ATTENUATION, (GLfloat)0.01);
	    glLightf(light_num, (GLuint)GL_QUADRATIC_ATTENUATION, (GLfloat)0.0);

	    /* Set up direction. */
            v[0] = (GLfloat)(0.0);	/* x. */
            v[1] = (GLfloat)(-0.97);	/* z. */
            v[2] = (GLfloat)-(0.27);	/* -y. */
	    v[3] = (GLfloat)0.0;
            glLightfv(light_num, GL_SPOT_DIRECTION, &(v[0]));

            glLightf(light_num, GL_SPOT_CUTOFF, 75.0);
	    glLightf(light_num, GL_SPOT_EXPONENT, 0.0);
	}
	else
	{
	    StateGLDisable(&display->state_gl, light_num);
	}

	return;
}

/*
 *	Draws help message.
 *
 *	Will also check if current display_help page number exceeds the
 *	total number of drawable pages, if it does then display_help
 *	will be set back to 0 and nothing will be drawn.
 */
static void SARDrawHelp(sar_core_struct *core_ptr)
{
	gw_display_struct *display;
	int line_num, xc, yc, x0, y0, xmax, ymax, fw, fh, width, height;
        GWFont *font;
        sar_color_struct *color;
	char text[256];
	const char *mesg[] = SAR_KEYS_HELP_MESSAGES;
	const char *cstrptr;
	int	mesg_lines, mesg_strings,
		mesg_lines_drawn = 0,
		page_num, total_pages,
		mesg_lines_per_page,
		mesg_line_max = SAR_KEYS_HELP_MESSAGE_LINE_MAX;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

	width = display->width;
	height = display->height;

        color = &option.message_color;
        font = option.message_font;
        if(font == NULL)
            return;

        GWGetFontSize(font, NULL, NULL, &fw, &fh);
	if((fw <= 0) || (fh <= 0))
	    return;

	/* Get the current help page number, which corresponds to
	 * display_help that starting from index 1 (not 0, since 0
	 * means do not display help).
	 */
	page_num = core_ptr->display_help;

	/* Calculate total lines in help message list. */
        mesg_strings = (int)(sizeof(mesg) / sizeof(char *));
	/* Must be in pairs. */
	if((mesg_strings % 2) != 0)
	    return;
	if(mesg_strings < 2)
	    return;
	/* Calculate total lines. */
	mesg_lines = mesg_strings / 2;

        /* Initialize starting x and y position in window coordinates. */
        xc = x0 = 10;
        yc = y0 = 5;

        /* Calculate maximum bounds in window coordinates. */
        xmax = mesg_line_max * fw;
        ymax = height - fh - 5;

        /* Calculate lines per page and total pages, remember to excude
         * two lines for the heading.
         */
        mesg_lines_per_page = MAX(ymax - yc - (fh * 2), 0) / fh;
        total_pages = (int)ceil(
	    (double)mesg_lines / (double)mesg_lines_per_page
	);


	/* Check if current display_help page number exceeds the
	 * total number of drawable pages, if it does then display_help
	 * will be set back to 0 and nothing will be drawn.
	 */
	if(page_num > total_pages)
	{
	    core_ptr->display_help = 0;
	    return;
	}


        /* Draw shaded background. */
        StateGLEnable(&display->state_gl, GL_BLEND);
        StateGLBlendFunc(
	    &display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
	);
        glBegin(GL_QUADS);
        {
            glColor4d(0.0, 0.0, 0.0, 0.5);
            glVertex2d(0.0, 0.0);
            glVertex2d(width, 0.0);
            glVertex2d(width, height);
            glVertex2d(0.0, height);
        }
        glEnd();
        StateGLDisable(&display->state_gl, GL_BLEND);


        /* Set text color and font. */
        glColor4d(color->r, color->g, color->b, color->a);
        GWSetFont(display, font);

	/* Draw heading. */
	sprintf(
	    text, "%s (%i/%i)",
	    SAR_MESG_HELP_PAGE_HEADING,
	    page_num, total_pages
	);
        GWDrawString(display, xc, yc, text);
	/* Increment two additional lines so we have one blank line. */
	yc += (fh * 2);
	mesg_lines_drawn += 2;

	/* Set line_num to starting message line number. */
	line_num = mesg_lines_per_page * (page_num - 1);
	if(line_num >= 0)
	{
	    /* Begin drawing help message lines, remember that
	     * each line comes in 2 strings.
	     */
	    while((yc < ymax) && (line_num < mesg_lines))
	    {
		/* Seek x position back to initial. */
		xc = x0;

		/* Draw command. */
		cstrptr = mesg[(line_num * 2) + 0];
		if(cstrptr != NULL)
		{
		    glColor4d(color->r, color->g, 0.0, color->a);  
		    GWDrawString(
			display, xc, yc,
			cstrptr
		    );
		    xc += ((int)strlen(cstrptr) * fw) + (2 * fw);
		}

		/* Draw description. */
		cstrptr = mesg[(line_num * 2) + 1];
		if(cstrptr != NULL)
		{
		    glColor4d(color->r, color->g, color->b, color->a);
		    GWDrawString(
			display, xc, yc, 
			cstrptr
		    );
		}

		yc += fh;
		line_num++;
	    }
	}
}
 

/*
 *      Draws the rotor/propellar relative to the given object.
 *
 *      Inputs assumed valid.
 */
static void SARDrawRotor(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr, sar_obj_rotor_struct *rotor_ptr,
	double throttle
)
{
        sar_position_struct *pos;
        sar_direction_struct *dir;	/* In radians. */
        double anim_coeff;


	if(rotor_ptr->visual_model == NULL)
	    return;

	/* Does not rotate? */
	if(rotor_ptr->flags & SAR_ROTOR_FLAG_NO_ROTATE)
	{
	    /* Set animation coeff to 0 so that here is no rotating. */
	    anim_coeff = 0.0;
	}
	else
	{
	    /* Calculate animation coeff from 0.0 to 1.0. */
	    anim_coeff = (double)rotor_ptr->anim_pos /
		(double)((sar_grad_anim_t)-1);
	}

        /* Note, current transformation assumed at object's center. */
        glPushMatrix();
        {
	    Boolean draw_blured;

            /* Get offset from center of object. */
            pos = &rotor_ptr->pos;
            glTranslated(pos->x, pos->z, -pos->y);

            /* Get direction. */
            dir = &rotor_ptr->dir;
	    /* Rotate rotor. */
            glRotated(
                -SFMRadiansToDegrees(dir->heading),
                0.0, 1.0, 0.0
            );
	    if(rotor_ptr->flags & SAR_ROTOR_FLAG_PITCH_STATE)
		glRotated(
                    -SFMRadiansToDegrees(dir->pitch + (0.5 * PI) +
			((rotor_ptr->flags & SAR_ROTOR_FLAG_FOLLOW_PB) ?
                  rotor_ptr->control_coeff_pitch * (5.0 * PI / 180.0) : 0.0
			)
		    ),
                    1.0, 0.0, 0.0
                );
	    else
                glRotated(
                    -SFMRadiansToDegrees(dir->pitch +
                       ((rotor_ptr->flags & SAR_ROTOR_FLAG_FOLLOW_PB) ?
                  rotor_ptr->control_coeff_pitch * (5.0 * PI / 180.0) : 0.0
                        )
		    ),
                    1.0, 0.0, 0.0
                );
            glRotated(
                -SFMRadiansToDegrees(dir->bank +
                   ((rotor_ptr->flags & SAR_ROTOR_FLAG_FOLLOW_PB) ?
                  rotor_ptr->control_coeff_bank * (5.0 * PI / 180.0) : 0.0
                    )
		),
                0.0, 0.0, 1.0
            );

	    /* Draw `blured' rotors always? */
	    if(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS)
	    {
		draw_blured = True;
	    }
	    /* Draw blured when fast? */
	    else if(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_FAST)
	    {
		/* Throttle above 30%? */
/* Use a global setting for this? */
		if(throttle > 0.30)
		    draw_blured = True;
		else
		    draw_blured = False;
	    }
	    else
	    {
		draw_blured = False;
	    }

	    /* Check whether to draw from the rotor's visual model or
	     * a blured circle.
	     */
	    if(draw_blured)
	    {
		/* Draw disc representing blured rotors. */
		GLenum shade_model_mode = display->state_gl.shade_model_mode;
		GLboolean depth_mask_flag = display->state_gl.depth_mask_flag;
		GLdouble theta, r = (GLfloat)rotor_ptr->radius;
		sar_color_struct *c = &rotor_ptr->blades_blur_color;

		/* Set flat shade model. */
		StateGLShadeModel(&display->state_gl, GL_FLAT);

		/* Disable depth writing. */
		StateGLDepthMask(&display->state_gl, GL_FALSE);

		/* Disable alpha test and enable blending. */
		StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
		StateGLEnable(&display->state_gl, GL_BLEND);
		StateGLBlendFunc(
		    &display->state_gl,
		    GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
		);

		/* Set blur color. */
		glColor4d(c->r, c->g, c->b, c->a);

                glBegin(GL_POLYGON);
		glNormal3d(0.0, 1.0, -0.0);
		for(theta = 2.0; theta > 0.0; theta -= 0.1)
		{
/*		    glTexCoord2d(
			(sin(theta * PI) + 1.0) / 2.0,
			(cos(theta * PI) + 1.0) / 2.0
		    );
 */
		    glVertex3d(
			r * sin(theta * PI),
			0.0,
			-(r * cos(theta * PI))
		    );
		}
		glEnd();

                glBegin(GL_POLYGON);
                glNormal3d(0.0, -1.0, -0.0);
                for(theta = 0.0; theta < 2.0; theta += 0.1)
                {
                    glVertex3d(
                        r * sin(theta * PI),
                        0.0,
                        -(r * cos(theta * PI))
                    );
                }
		glEnd();

		/* Restore shade model state. */
		StateGLShadeModel(&display->state_gl, shade_model_mode);

                /* Restore depth writing. */
                StateGLDepthMask(&display->state_gl, depth_mask_flag);

		/* Disable blending and enable alpha testing. */
                StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
                StateGLDisable(&display->state_gl, GL_BLEND);
	    }
	    else
	    {
		glPushMatrix();
		{
                    /* Rotate counter clockwise along Z axis to simulate
                     * rotor spinning.
                     */
                    glRotated(
                        -SFMRadiansToDegrees(-anim_coeff * (2.0 * PI)),
                        0.0, 1.0, 0.0
                    );  

		    SARVisualModelCallList(rotor_ptr->visual_model);
		}
		glPopMatrix();
	    }
        }
        glPopMatrix();
}

/*
 *	Draws the landing gear relative to the given object.
 *
 *	Inputs assumed valid.
 */
static void SARDrawLandingGear(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_landing_gear_struct *lgear_ptr
)
{
	sar_position_struct *pos;
	sar_direction_struct *dir;	/* In radians. */
	double anim_coeff;


	if(lgear_ptr->visual_model == NULL)
	    return;

	anim_coeff = (double)lgear_ptr->anim_pos /
	    (double)((sar_grad_anim_t)-1);

	/* A coeff of 0.0 implies all retracted. */
	if(anim_coeff <= 0.0)
	    return;

	/* Translation assumed at object's center position. */
	glPushMatrix();
	{
	    /* Get offset from center of object. */
            pos = &lgear_ptr->pos;
            glTranslated(pos->x, pos->z, -pos->y);

	    /* Calculate direction. */
            dir = &lgear_ptr->extended_dir;

	    glRotated(
		-SFMRadiansToDegrees(dir->heading * anim_coeff),
                0.0, 1.0, 0.0
	    );
            glRotated(
		-SFMRadiansToDegrees(dir->pitch * anim_coeff),
		1.0, 0.0, 0.0
	    );
            glRotated(
		-SFMRadiansToDegrees(dir->bank * anim_coeff),
		0.0, 0.0, 1.0
	    );
	    SARVisualModelCallList(lgear_ptr->visual_model);
        }
	glPopMatrix();

	return;
}

/*
 *	External reserved fuel tank which is assumed to be associated with
 *	the object.
 *
 *	Do not confuse this with falling fuel tanks which are drawn by
 *	SARDrawObjectFuelTank.
 *
 *	All inputs assumed valid.
 */
static void SARDrawFuelTank(
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_external_fueltank_struct *eft_ptr
)
{
	sar_position_struct *offset_pos;
	sar_visual_model_struct *vmodel_ptr;

	/* Fuel tank not on board? */
	if(!(eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD))
	    return;

	/* Get pointer to visual model if any. */
        vmodel_ptr = eft_ptr->visual_model;
        if(vmodel_ptr == NULL)
            return;

	/* Get pointer to offset position relative to object center. */
	offset_pos = &eft_ptr->offset_pos;

        glPushMatrix();
        {
            /* Translate from center of object. */
            glTranslated(
		offset_pos->x, offset_pos->z, -offset_pos->y
	    );
            SARVisualModelCallList(vmodel_ptr);
        }
        glPopMatrix();

	return;
}

/*
 *	Draws door which is assumed to be associated with the object.
 *
 *	All inputs assumed valid.
 */
static void SARDrawDoor(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_obj_door_struct *door_ptr
)
{
        sar_position_struct *pos_closed, *pos_opened, delta;
        sar_direction_struct *dir;
        double anim_coeff;


	if(door_ptr->visual_model == NULL)
	    return;

        anim_coeff = (double)door_ptr->anim_pos /
            (double)((sar_grad_anim_t)-1);

	pos_closed = &door_ptr->pos_closed;
	pos_opened = &door_ptr->pos_opened;
	dir = &door_ptr->dir_opened;

	delta.x = pos_opened->x - pos_closed->x;
	delta.y = pos_opened->y - pos_closed->y;
	delta.z = pos_opened->z - pos_closed->z;

	delta.x = pos_closed->x + (delta.x * anim_coeff);
	delta.y = pos_closed->y + (delta.y * anim_coeff);
	delta.z = pos_closed->z + (delta.z * anim_coeff);

        /* Translation assumed at object's center position. */
        glPushMatrix();
        {
            /* Translate from center of object. */
            glTranslated(delta.x, delta.z, -delta.y);

            /* Rotate to opened position. */
            glRotated(
                -SFMRadiansToDegrees(dir->heading * anim_coeff),
                0.0, 1.0, 0.0
            );
            glRotated(
                -SFMRadiansToDegrees(dir->pitch * anim_coeff),
                1.0, 0.0, 0.0
            );
            glRotated(
                -SFMRadiansToDegrees(dir->bank * anim_coeff),
                0.0, 0.0, 1.0
            );
	    SARVisualModelCallList(door_ptr->visual_model);
        }
        glPopMatrix();

	return;
}

/*
 *	Draws the hoist deployment (rescue basket or diver) and the
 *	tope that leads up to the origin of the hoist.
 *
 *	Translation matrix should already be at location of the hoist
 *	deployment (end of rope).
 *
 *	All inputs assumed valid.
 */
static void SARDrawHoistDeployment(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr, sar_obj_hoist_struct *hoist_ptr
)
{
	double l, w, h, hl, hw, hh;
	double rope_len = hoist_ptr->rope_cur_vis;


	/* Get length as the diameter of the contact bounds. */
	l = hoist_ptr->contact_radius * 2.0;

	/* Get width as half of the length. */
	w = l / 2;

	/* Height of the basket is the upper z cylendrical contact
	 * bounds.
	 */
	h = hoist_ptr->contact_z_max;

	/* Calculate half dimensions. */
	hl = l / 2;
        hw = w / 2;
        hh = h / 2;



	/* Draw by what is at the end of the rope. */
	if(hoist_ptr->deployment == SAR_HOIST_DEPLOYMENT_DIVER)
	{
            V3DTextureSelect(NULL);
	    glColor4d(0.9, 0.9, 0.9, 1.0);

	    glBegin(GL_LINES);
	    {
                /* Draw rope leading back up to the hoist. */
		glNormal3d(0.0, 1.0, 0.0);
                glVertex3d(0.0, h - 1.0, 0.0);
                glVertex3d(0.0, rope_len, 0.0);
            }
            glEnd();

	    glPushMatrix();
	    {
		/* Translate back just a bit so drawn human is behind
		 * rope and the victim if any.
		 */
		glTranslated(0.00, -1.00, 0.25);

		SARDrawHumanIterate(
		    display, scene,
		    1.9,		/* Height in meters. */
		    54.0,		/* Mass in kg. */
		    SAR_HUMAN_FLAG_ALERT | SAR_HUMAN_FLAG_AWARE |
		    SAR_HUMAN_FLAG_GRIPPED | SAR_HUMAN_FLAG_DIVER_CATCHER,
		    hoist_ptr->diver_color,
		    hoist_ptr->water_ripple_tex_num,
		    hoist_ptr->anim_pos
		);
	    }
	    glPopMatrix();
	}
	else
	{
	    int tex_num;
	    GLenum shade_model_mode;
	    v3d_texture_ref_struct *side_tex, *end_tex, *bottom_tex;

	    /* Get pointers to textures. */
	    tex_num = hoist_ptr->side_tex_num;
	    if(SARIsTextureAllocated(scene, tex_num))
		side_tex = scene->texture_ref[tex_num];
	    else
		side_tex = NULL;

	    tex_num = hoist_ptr->end_tex_num;
	    if(SARIsTextureAllocated(scene, tex_num))
		end_tex = scene->texture_ref[tex_num];
	    else
		end_tex = NULL;

	    tex_num = hoist_ptr->bottom_tex_num;
	    if(SARIsTextureAllocated(scene, tex_num))
		bottom_tex = scene->texture_ref[tex_num];
	    else
		bottom_tex = NULL;

	    /* Record previous shade model and use smooth shading. */
	    shade_model_mode = display->state_gl.shade_model_mode;
	    StateGLShadeModel(&display->state_gl, GL_SMOOTH);

	    /* Begin drawing rescue basket. */

	    /* Draw bottom base. */
	    glColor4d(1.0, 1.0, 1.0, 1.0);
	    V3DTextureSelect(bottom_tex);
	    glBegin(GL_QUADS);
	    {
		/* Upward facing. */
		glNormal3d(-0.33, 0.88, 0.33);
		glTexCoord2d(0, 1 - 0);
                glVertex3d(-hl, 0.0,  hw);

                glNormal3d(0.33, 0.88, 0.33);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(hl,  0.0,  hw);

                glNormal3d(0.33, 0.88, -0.33);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(hl,  0.0, -hw);

                glNormal3d(-0.33, 0.88, -0.33);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(-hl, 0.0, -hw);

		/* Downward facing. */
                glNormal3d(-0.33, -0.88, -0.33);
		glTexCoord2d(0, 1 - 1);
		glVertex3d(-hl, 0.0, -hw);

		glNormal3d(0.33, -0.88, -0.33);
		glTexCoord2d(1, 1 - 1);
		glVertex3d(hl, 0.0, -hw);

                glNormal3d(0.33, -0.88, 0.33);
		glTexCoord2d(1, 1 - 0);
		glVertex3d(hl, 0.0,  hw);

                glNormal3d(-0.33, -0.88, 0.33);
		glTexCoord2d(0, 1 - 0);
		glVertex3d(-hl, 0,  hw);
	    }
	    glEnd();

	    /* Sides. */
            V3DTextureSelect(side_tex);
	    glBegin(GL_QUADS);
	    {
		/* Front forwards facing side. */
		glNormal3d(-0.33, 0.33, -0.88);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(-hl, hh,  -hw);

                glNormal3d(0.33, 0.33, -0.88);
                glTexCoord2d(0, 1 - 1);
                glVertex3d( hl, hh,  -hw);

                glNormal3d(0.33, -0.33, -0.88);
                glTexCoord2d(0, 1 - 0);
                glVertex3d( hl, 0.0, -hw);

                glNormal3d(-0.33, -0.33, -0.88);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(-hl, 0.0, -hw);


		/* Backwards facing side. */
                glNormal3d(-0.33, -0.33, 0.88);
		glTexCoord2d(1, 1 - 0);
                glVertex3d(-hl, 0.0, -hw);

                glNormal3d(0.33, -0.33, 0.88);
		glTexCoord2d(0, 1 - 0);
		glVertex3d( hl, 0.0, -hw);

		glNormal3d(0.33, 0.33, 0.88);
		glTexCoord2d(0, 1 - 1);
                glVertex3d( hl, hh, -hw);

                glNormal3d(-0.33, 0.33, 0.88);
	        glTexCoord2d(1, 1 - 1);
                glVertex3d(-hl, hh, -hw);


		/* Back forwards facing side. */
                glNormal3d(-0.33, 0.33, -0.88);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(-hl, hh,  hw);

                glNormal3d(0.33, 0.33, -0.88); 
                glTexCoord2d(1, 1 - 1);
                glVertex3d( hl, hh,  hw);

                glNormal3d(0.33, -0.33, -0.88); 
                glTexCoord2d(1, 1 - 0);
                glVertex3d( hl, 0.0, hw);

                glNormal3d(-0.33, -0.33, -0.88);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(-hl, 0.0, hw);

		/* Back backwards facing side. */
                glNormal3d(-0.33, -0.33, 0.88);
		glTexCoord2d(0, 1 - 0);
                glVertex3d(-hl, 0.0, hw);

                glNormal3d(0.33, -0.33, 0.88);
	        glTexCoord2d(1, 1 - 0);
                glVertex3d( hl, 0.0, hw);

                glNormal3d(0.33, 0.33, 0.88);
		glTexCoord2d(1, 1 - 1);
		glVertex3d( hl, hh, hw);

		glNormal3d(-0.33, 0.33, 0.88);
		glTexCoord2d(0, 1 - 1);
		glVertex3d(-hl, hh, hw);
	    }
	    glEnd();

	    /* Left and right side ends. */
	    V3DTextureSelect(end_tex);
	    glBegin(GL_QUADS);
	    {
		/* Left, left facing side. */
		glNormal3d(-0.88, -0.33, -0.33);
		glTexCoord2d(0, 1 - 0);
		glVertex3d(-hl, 0.0, -hw);

                glNormal3d(-0.88, -0.33, 0.33);
		glTexCoord2d(1, 1 - 0);
		glVertex3d(-hl, 0.0, hw);

		glNormal3d(-0.88, 0.33, 0.33);
		glTexCoord2d(1, 1 - 1);
		glVertex3d(-hl, hh, hw);

		glNormal3d(-0.88, 0.33, -0.33);
		glTexCoord2d(0, 1 - 1);
		glVertex3d(-hl, hh, -hw);

		/* Left, right facing side. */
                glNormal3d(0.88, 0.33, -0.33);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(-hl, hh, -hw);

                glNormal3d(0.88, 0.33, 0.33);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(-hl, hh, hw);

                glNormal3d(0.88, -0.33, 0.33);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(-hl, 0.0, hw);

                glNormal3d(0.88, -0.33, -0.33);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(-hl, 0.0, -hw);


		/* Right, left facing side. */
                glNormal3d(-0.88, -0.33, -0.33);
		glTexCoord2d(0, 1 - 0);
                glVertex3d(hl, 0.0, -hw);

                glNormal3d(-0.88, -0.33, 0.33);
		glTexCoord2d(1, 1 - 0);
                glVertex3d(hl, 0.0, hw);

                glNormal3d(-0.88, 0.33, 0.33);
		glTexCoord2d(1, 1 - 1);
                glVertex3d(hl, hh, hw);

                glNormal3d(-0.88, 0.33, -0.33);
		glTexCoord2d(0, 1 - 1);
                glVertex3d(hl, hh, -hw);

		/* Right, right facing side. */
                glNormal3d(0.88, 0.33, -0.33);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(hl, hh, -hw);

                glNormal3d(0.88, 0.33, 0.33);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(hl, hh, hw);

                glNormal3d(0.88, -0.33, 0.33);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(hl, 0.0, hw);

                glNormal3d(0.88, -0.33, -0.33);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(hl, 0.0, -hw);
	    }
	    glEnd();

	    /* Other details, floatation strips. */
	    V3DTextureSelect(NULL);
	    glColor4d(0.9, 0.1, 0.1, 1.0);
	    glBegin(GL_QUADS);
	    {
                glNormal3d(0.0, 1.0, 0.0);
                glVertex3d(-hl,       hh, hw);
                glVertex3d(-hl * 0.9, hh, hw);
                glVertex3d(-hl * 0.9, hh, -hw);
                glVertex3d(-hl,       hh, -hw);

                glNormal3d(0.0, 1.0, 0.0);
                glVertex3d(hl,       hh, -hw);
                glVertex3d(hl * 0.9, hh, -hw);
                glVertex3d(hl * 0.9, hh,  hw);
                glVertex3d(hl,       hh,  hw);
            }
            glEnd();

	    /* Begin drawing cables. */
            glBegin(GL_LINES);
            {
                double hl = l / 2;
                double hw = w / 2;
                double hh = h / 2;

                glColor4d(0.9, 0.9, 0.9, 1.0);

                glNormal3d(0, 1, 0);
                glVertex3d(hl, hh, hw);
                glVertex3d(0, h, 0);

                glVertex3d(hl, hh, -hw);
                glVertex3d(0, h, 0);

                glVertex3d(-hl, hh, -hw);
                glVertex3d(0, h, 0);

                glVertex3d(-hl, hh, hw);
                glVertex3d(0, h, 0);

		/* Draw rope leading back up to the hoist. */
                glVertex3d(0, h, 0);
                glVertex3d(0, rope_len, 0);
	    }
	    glEnd();

            /* Restore previous shade model. */
            StateGLShadeModel(&display->state_gl, shade_model_mode);
	}

	return;
}

/*
 *	Draws smoke trail object, all translations and rotations will
 *	be handled within this function.
 */
static void SARDrawSmoke(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_smoke_struct *obj_smoke_ptr
)
{
        int tex_num, unit_num;
        double r;
	GLenum shade_model_mode;
        v3d_texture_ref_struct *t;
	sar_object_smoke_unit_struct *unit_ptr;
	sar_position_struct *pos;
	sar_direction_struct to_dir;


        /* Get texture for this smoke trail. */
        tex_num = obj_smoke_ptr->tex_num;
        if(SARIsTextureAllocated(scene, tex_num))
            t = scene->texture_ref[tex_num];
        else
            return;

        /* Select texture. */
        V3DTextureSelect(t);
        glColor4d(1.0, 1.0, 1.0, 1.0);

	/* Use flat shading. */
	shade_model_mode = display->state_gl.shade_model_mode;
	StateGLShadeModel(&display->state_gl, GL_FLAT);


	if(obj_smoke_ptr->unit != NULL)
	{
	    for(unit_num = 0; unit_num < obj_smoke_ptr->total_units; unit_num++)
	    {
		unit_ptr = &obj_smoke_ptr->unit[unit_num];

		/* Not visible? */
		if(unit_ptr->visibility <= 0.0)
		    continue;

		r = unit_ptr->radius;
		pos = &unit_ptr->pos;

		glPushMatrix();
		{
                    /* Translate. */
                    glTranslated(pos->x, pos->z, -pos->y);

                    /* Adjust heading to camera heading. */
                    SARDrawGetDirFromPos(
                        &camera_pos, pos, &to_dir
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            to_dir.heading
                        ), 0.0, 1.0, 0.0
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            to_dir.pitch
                        ), 1.0, 0.0, 0.0
                    );

		    /* Begin drawing smoke puff. */
		    glBegin(GL_QUADS);
		    {
			glNormal3d(0.0, 1.0, 0.0);

			glTexCoord2d(0.0, 1.0 - 0.0);
			glVertex3d(-r, -r, 0.0);
			glTexCoord2d(1.0, 1.0 - 0.0);
			glVertex3d(r, -r, 0.0);
			glTexCoord2d(1.0, 1.0 - 1.0);
			glVertex3d(r, r, 0.0);
			glTexCoord2d(0, 1.0 - 1);
			glVertex3d(-r, r, 0);
		    }
		    glEnd();
		}
		glPopMatrix();
	    }
	}

        /* Restore shade model. */ 
	StateGLShadeModel(&display->state_gl, shade_model_mode);
}

/*
 *	Draws explosion object.
 */
static void SARDrawExplosion(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_explosion_struct *obj_explosion_ptr
)
{
	int cur_frame, tex_num;
	StateGLBoolean lighting;
	GLenum shade_model_mode;
	v3d_texture_ref_struct *t;
	double r;


	/* Select current texture and frame. */
	cur_frame = obj_explosion_ptr->cur_frame;
	tex_num = obj_explosion_ptr->tex_num;
	if(SARIsTextureAllocated(scene, tex_num))
	    t = scene->texture_ref[tex_num];
	else
	    return;

	/* Sanitize frame number. */
	if(cur_frame >= t->total_frames)
	    cur_frame = t->total_frames - 1;
	if(cur_frame < 0)
	    cur_frame = 0;

	/* Get radius in meters. */
	r = obj_explosion_ptr->radius;

	/* Select texture. */
	V3DTextureSelectFrame(t, cur_frame);
        glColor4d(1.0, 1.0, 1.0, 1.0);

	/* Set flat shade model. */
	shade_model_mode = display->state_gl.shade_model_mode;
	StateGLShadeModel(&display->state_gl, GL_FLAT);

	/* Adjust lighting. */
	lighting = display->state_gl.lighting;
	switch(obj_explosion_ptr->color_emission)
	{
	  case SAR_EXPLOSION_COLOR_EMISSION_EMIT_LIGHT:
          case SAR_EXPLOSION_COLOR_EMISSION_IS_LIGHT:
	    StateGLDisable(&display->state_gl, GL_LIGHTING);
	    break;

	  default:	/* SAR_EXPLOSION_COLOR_EMISSION_NONE */
	    /* Leave lighting as it was. */
	    break;
	}

	/* Begin drawing explosion. */
	glBegin(GL_QUADS);
	{
	    glNormal3d(0, 1, 0);

            glTexCoord2d(0, 1 - 0);
            glVertex3d(-r, -r, 0);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(r, -r, 0);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(r, r, 0);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(-r, r, 0);
	}
	glEnd();

	/* Restore shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);
	/* Restore lighting. */
	if(lighting)
            StateGLEnable(&display->state_gl, GL_LIGHTING);
	else
            StateGLDisable(&display->state_gl, GL_LIGHTING);

	return;
}

/*
 *      Draws fire object.
 */
static void SARDrawFire(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_fire_struct *obj_fire_ptr
)
{
        int cur_frame, tex_num;
        StateGLBoolean lighting;
        GLenum shade_model_mode;
        v3d_texture_ref_struct *t;
        double r, h;


        /* Select current texture and frame. */
        cur_frame = obj_fire_ptr->cur_frame;
        tex_num = obj_fire_ptr->tex_num;
        if(SARIsTextureAllocated(scene, tex_num))
            t = scene->texture_ref[tex_num];
        else
            return;

        /* Sanitize frame number. */
        if(cur_frame >= t->total_frames)
            cur_frame = t->total_frames - 1;
        if(cur_frame < 0)
            cur_frame = 0;

        /* Get size (in meters). */
        r = obj_fire_ptr->radius;
        h = obj_fire_ptr->height;

        /* Select texture. */
        V3DTextureSelectFrame(t, cur_frame);
        glColor4d(1.0, 1.0, 1.0, 1.0);

        /* Set flat shade model. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);

        /* Adjust lighting. */
        lighting = display->state_gl.lighting;
	StateGLDisable(&display->state_gl, GL_LIGHTING);

        /* Begin drawing fire. */
        glBegin(GL_QUADS);
        {
            glNormal3d(0, 1, 0);
            glTexCoord2d(0.0, 1.0);
            glVertex3d(-r, 0, 0);
            glTexCoord2d(1.0, 1.0);
            glVertex3d(r, 0, 0);
            glTexCoord2d(1.0, 0.0);
            glVertex3d(r, h, 0);
            glTexCoord2d(0.0, 0.0);
            glVertex3d(-r, h, 0);
        }
        glEnd();

        /* Restore shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);
	/* Restore lighting. */
        if(lighting)
            StateGLEnable(&display->state_gl, GL_LIGHTING);
        else
            StateGLDisable(&display->state_gl, GL_LIGHTING);

	return;
}

/*
 *	Draws HUD attitude ticks.
 *
 *      Inputs are assumed valid.
 */
static void SARDrawHUDAttitude(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_direction_struct *dir, sar_position_struct *vel,
	int offset_x, int offset_y,
	int width, int height,
	double fovx, double fovy
)
{
        int x, y, x2, y2, degi;
	double pitch_deg;
        int half_width, half_height;
	double sin_bank, cos_bank, r;
        int tick_start_y, end_min, end_max;	/* In pixels. */
	int	tick_inc_yi,			/* In pixels. */
		tick_inc_degreesi,		/* In degrees. */
		tick_x_max,			/* In pixels. */
		tick_x_min;
	double	tick_inc_y,			/* In pixels. */
		tick_inc_degrees;		/* In degrees. */
	int tick_start_pitch;			/* In degrees. */
	char text[80];

	/* Glide slope cursor. */
	int gs_w = 15, gs_h = 6;
	u_int8_t gs_bm[] = {
 0x03, 0x80,
 0x04, 0x40,
 0xfc, 0x7e,
 0x04, 0x40,
 0x03, 0x80,
 0x01, 0x00
	};

	/* Coordinates are handled in left hand rule xy plane,
	 * the y axis is inverted at the call to the X draw
	 * function.
	 */

        /* Calculate half of size. */
        half_width = width / 2;
        half_height = height / 2;

        /* Calculate limits and increments. */
        end_min = (int)(-0.5 * half_height);
        end_max = (int)(0.5 * half_height);

        tick_inc_degrees = 10;		/* Always 10 degrees. */

	tick_x_max = (int)(0.06 * width);
	tick_x_min = (int)(0.03 * width);

	sin_bank = sin(dir->bank);
	cos_bank = cos(dir->bank);

	/* Calculate tick_inc_y. */
	r = (double)height / fovy;
	tick_inc_y = r * SFMDegreesToRadians(tick_inc_degrees);

	/* Split into two cases, positive and negative pitch. */
	if(dir->pitch > PI)
	{
	    /* Positive pitch. */
	    double deg;

            pitch_deg = 360 - (int)SFMRadiansToDegrees(dir->pitch);
            deg = (int)pitch_deg % (int)tick_inc_degrees;
            tick_start_y = (int)(-deg * tick_inc_y / tick_inc_degrees);
            tick_start_pitch = (int)(pitch_deg - deg);
	}
	else
	{
	    /* Negative pitch. */
	    double deg;

            pitch_deg = (int)SFMRadiansToDegrees(dir->pitch);
            deg = tick_inc_degrees -
		((int)pitch_deg % (int)tick_inc_degrees);
            tick_start_y = (int)(-deg * tick_inc_y / tick_inc_degrees);
            tick_start_pitch = (int)(-pitch_deg - deg);
	}
	tick_inc_degreesi = (int)tick_inc_degrees;
	tick_inc_yi = (int)tick_inc_y;


/* Define procedure to draw each tick for use in the for() loop
 * just below.
 */
#define DO_DRAW_ATTITUDE_TICK	{ \
 x2 = (int)(-sin_bank * y); \
 y2 = (int)(cos_bank * y); \
 if(degi == 0) \
 { \
   glBegin(GL_LINES); \
   { \
     glVertex2d( \
	(offset_x + x2 + (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_min * sin_bank) + half_height) \
     ); \
     glVertex2d( \
	(offset_x + x2 + ((8 + tick_x_max) * cos_bank) + half_width), \
	(offset_y + y2 + ((8 + tick_x_max) * sin_bank) + half_height) \
     ); \
  \
     glVertex2d( \
	(offset_x + x2 - (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_min * sin_bank) + half_height) \
     ); \
     glVertex2d( \
	(offset_x + x2 - ((8 + tick_x_max) * cos_bank) + half_width), \
	(offset_y + y2 - ((8 + tick_x_max) * sin_bank) + half_height) \
     ); \
   } \
   glEnd(); \
 } \
 else \
 { \
   glBegin(GL_LINES); \
   { \
     glVertex2d( \
	(offset_x + x2 + (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_min * sin_bank) + half_height) \
     ); \
     glVertex2d( \
	(offset_x + x2 + (tick_x_max * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_max * sin_bank) + half_height) \
     ); \
 \
     glVertex2d( \
	(offset_x + x2 - (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_min * sin_bank) + half_height) \
     ); \
     glVertex2d( \
	(offset_x + x2 - (tick_x_max * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_max * sin_bank) + half_height) \
     ); \
  } \
  glEnd(); \
  \
    sprintf(text, "%i", degi); \
    /* Right. */ \
    GWDrawString( \
	display, \
	(int)(offset_x + x2 + (tick_x_max * cos_bank) + half_width \
	    - 8 + (12 * cos_bank)), \
	(int)(height - (offset_y + y2 + (tick_x_max * sin_bank) + \
	    half_height + 3)), \
        text \
    ); \
    /* Left. */ \
    GWDrawString( \
        (gw_display_struct *)display, \
        (int)(offset_x + x2 - (tick_x_max * cos_bank) + half_width \
            - 8 - (18 * cos_bank)), \
        (int)(height - (offset_y + y2 - (tick_x_max * sin_bank) + \
            half_height + 3)), \
        text \
    ); \
 } \
 degi += tick_inc_degreesi; \
}


        degi = tick_start_pitch;

	/* Draw upper (pitch up) ticks. */
        for(y = tick_start_y; y < end_max; y += tick_inc_yi)
	{
	    if(degi > 90)
                break;

	    DO_DRAW_ATTITUDE_TICK
	}

        degi = tick_start_pitch - tick_inc_degreesi;

        /* Draw lower (pitch down) ticks. */
        for(y = tick_start_y - tick_inc_yi; y > end_min; y -= tick_inc_yi)
	{
	    if(degi < -90)
		break;

	    DO_DRAW_ATTITUDE_TICK
	}
#undef DO_DRAW_ATTITUDE_TICK


	/* Draw nose axis marker. */
	x = offset_x + half_width;
	y = offset_y + half_height;
	glBegin(GL_LINES);
	{
	    glVertex2d(
		(x - (0.05 * width)),
		y
	    );
	    glVertex2d(
		(x - (0.02 * width)),
		y
	    );

	    glVertex2d(
		(x + (0.05 * width)),
		y
	    );
	    glVertex2d(
		(x + (0.02 * width)),
		y
	    );
	}
	glEnd();

	/* Draw glide slope marker. */
	if(vel != NULL)
	{
	    double h, theta;

	    /* Calculate angle to ground relative to velocity. */
	    theta = SFMSanitizeRadians(
		(2 * PI) - atan2(vel->z, vel->y)
	    );
	    /* Adjust angle to ground relative to aircraft's pitch. */
	    theta = SFMSanitizeRadians(
		theta - dir->pitch
	    );
	    if(theta > (0.5 * PI))
	    {
/* Don't go up beyond nose level. */
		h = 0;
		theta = 0;
	    }
	    else
	    {
	        h = (height / fovy) * theta;
	    }

	    if(h < -end_min)
	    {
                x = (int)(offset_x + half_width);
                y = (int)(offset_y + half_height - h);

		glRasterPos2i(x - (gs_w / 2), y - (gs_h / 2));
		glBitmap(
                    gs_w, gs_h,
                    0.0, 0.0,
                    (GLfloat)gs_w, 0.0,
		    gs_bm
                );
	    }
	}

	return;
}

/*
 *	Draws HUD heading ticks and intercept arrow.
 *
 *	Inputs are assumed valid.
 */
static void SARDrawHUDHeading(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_direction_struct *dir,
	sar_obj_intercept_struct *intercept,
	int offset_x, int offset_y,	/* Look offset. */
	int width, int height		/* Of window. */
)
{
	int x, y, deg, heading_deg, len;
	int half_width, half_height;
	int tick_start_x, end_min, end_max;	/* In pixels. */
	int	tick_inc_x,			/* In pixels. */
		tick_inc_degrees;		/* In degrees. */
	int tick_start_heading;			/* In degrees. */
	char text[256];
	int ia_w = 8, ia_h = 4;
	u_int8_t ia_bm[] = {0xc6, 0x6c, 0x38, 0x10};


        /* Coordinates are handled in left hand rule xy plane,
         * with the y axis already inverted!
         */

	/* Calculate half of size. */
	half_width = width / 2;
	half_height = height / 2;

	/* Calculate limits and increments. */
	end_min = (int)(half_width - (0.3 * half_width));
	end_max = (int)(half_width + (0.3 * half_width));

	tick_inc_x = (int)(0.09 * width);
	tick_inc_degrees = 10;		/* Always 10 degrees. */

	heading_deg = (int)SFMRadiansToDegrees(
	    dir->heading - scene->cant_angle
	);
	deg = heading_deg % tick_inc_degrees;
	tick_start_x = half_width - (deg * tick_inc_x / tick_inc_degrees);
	tick_start_heading = (int)SFMSanitizeDegrees(
	    heading_deg - deg
	);


#define DO_DRAW_HEADING_TICK	{ \
 glBegin(GL_LINES); \
 { \
  glVertex2d( \
     (x + offset_x), \
     (height - (y + offset_y - 10)) \
  ); \
  glVertex2d( \
     (x + offset_x), \
     (height - (y + offset_y - 14)) \
  ); \
 } \
 glEnd(); \
  \
 sprintf(text, "%i", deg / 10); \
 len = strlen(text); \
 GWDrawString( \
     display, \
     (int)(x + offset_x - ((len - 1) * 4) - 3), \
     (int)(y + offset_y - 11), \
     text \
 ); \
 deg = (int)SFMSanitizeDegrees(deg + tick_inc_degrees); \
}

	y = (int)(half_height - (0.35 * height));
	deg = tick_start_heading;
	for(x = tick_start_x; x < end_max; x += tick_inc_x)
	{
	    DO_DRAW_HEADING_TICK
	}

	deg = (int)SFMSanitizeDegrees(tick_start_heading - tick_inc_degrees);
        for(x = tick_start_x - tick_inc_x; x > end_min; x -= tick_inc_x)
        {
            DO_DRAW_HEADING_TICK
        }

#undef DO_DRAW_HEADING_TICK

	/* Draw intercept tick. */
	if(intercept != NULL)
	{
	    int intercept_tick_dx;
	    double dtheta, icpt_dist;
            double	icpt_dx = intercept->x - camera_pos.x,
			icpt_dy = intercept->y - camera_pos.y;


            /* Calculate intercept distance in meters. */
            icpt_dist = hypot(icpt_dx, icpt_dy);

	    /* Calculate delta theta from current heading to
	     * intercept bearing.
	     */
	    dtheta = SFMDeltaRadians(
		dir->heading,
		SFMSanitizeRadians(
		    (0.5 * PI) - atan2(icpt_dy, icpt_dx)
		)
	    );
	    /* Convert to degrees, no sanitize. */
	    dtheta = dtheta * (180 / PI);

	    if(tick_inc_degrees > 0)
	        intercept_tick_dx = (int)(dtheta * ((double)tick_inc_x /
		    (double)tick_inc_degrees));
	    else
		intercept_tick_dx = 0;

            /* Apply half width. */
            intercept_tick_dx += half_width;

	    if(intercept_tick_dx < end_min)
		intercept_tick_dx = end_min;
	    else if(intercept_tick_dx > end_max)
		intercept_tick_dx = end_max;

            glRasterPos2i(
		intercept_tick_dx + offset_x + 1 - (ia_w / 2),
		height - (y + offset_y + 5 - (ia_h / 2))
	    );
            glBitmap(
                ia_w, ia_h,
                0.0, 0.0,
                (GLfloat)ia_w, 0.0,
                ia_bm
            );

	    /* Draw distance to intercept. */
	    if(1)
	    {
		double v1 = SFMMetersToMiles(icpt_dist);
		if(v1 > 10.0)
		    sprintf(text, "R: %.0f", v1);
		else
		    sprintf(text, "R: %.1f", v1);

		GWDrawString(
		    display,
		    (int)(offset_x + half_width + (0.4 * half_width)),
		    (int)(offset_y + y),
		    text
		);
	    }
	}

	/* Current heading tick. */
	glBegin(GL_LINES);
	{
	    glVertex2d(
		(half_width + offset_x),
		(height - (y + offset_y + 1))
	    );
	    glVertex2d(
		half_width + offset_x,
		(height - (y + offset_y + 5))
	    );
	}
	glEnd();
}


/*
 *	Draws elevator trim on HUD display.
 *
 *	Given coordinates are in window coordinates at the center of the 
 *	trim gauge.
 *
 *	This can also be used to draw the outside attitude stuff.
 */
static void SARDrawHUDElevatorTrim(
        gw_display_struct *display,
	GLfloat xc, GLfloat yc,
	GLfloat length, GLfloat width,
	double elevator_trim
)
{
	GLfloat x[2], y[2], tick_pos, tick_spacing = length / 6;
	GLsizei icon_width = 8, icon_height = 8;
	const GLubyte icon_bm[] = {
		0x00,
		0x10,
		0x30,
		0x70,
		0xf0,
		0x70,
		0x30,
		0x10
	};

	x[0] = xc - (width / 2);
	x[1] = xc + (width / 2);
        y[0] = yc - (length / 2);
        y[1] = yc + (length / 2);

	glBegin(GL_LINE_STRIP);
	{
	    glVertex2d(x[1], y[1]);
            glVertex2d(x[0], y[1]);
            glVertex2d(x[0], y[0]);
            glVertex2d(x[1], y[0]);
	}
	glEnd();

	if(tick_spacing > 2.0)
	{
	    glBegin(GL_LINES);
	    for(tick_pos = 0.0; tick_pos < length; tick_pos += tick_spacing)
	    {
                glVertex2d(x[0], y[0] + tick_pos);
                glVertex2d(x[0] + (width / 2), y[0] + tick_pos);
	    }
	    glEnd();
	}

	/* Current position. */
	tick_pos = (-elevator_trim * (length / 2)) + (length / 2);
        glRasterPos2f(
            (GLfloat)x[0] + (width / 2),
	    (GLfloat)(y[0] + tick_pos - (icon_height / 2))
	);
        glBitmap(
            icon_width, icon_height,
            0.0, 0.0,
            0.0, 0.0,
            icon_bm
        );


        /* Center. */
/*
        tick_pos = length / 2;
        glBegin(GL_LINES);
        {
            glVertex2d(x[0], y[0] + tick_pos);
            glVertex2d(x[1], y[0] + tick_pos);
        }
        glEnd();
 */
}


/*
 *	Draw HUD.
 *
 *	The given aspect is the width of the window in pixels
 *	divided by the height of the window in pixels.
 *
 *	All units are in projected units and radians, projected
 *	units are derived from the given width and height.
 */
static void SARDrawHUD(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	double aspect
)
{
	int i;
	sar_object_struct *obj_ptr;
        sar_direction_struct *cam_dir, *dir;
        sar_position_struct *pos, *vel = NULL;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_obj_intercept_struct *intercept = NULL;
	double fovx, fovz;	/* Field of view x and z, in radians. */
	int engine_state = SAR_ENGINE_STATE_OFF;
	double cockpit_offset_z = 0;
	double speed = 0.0;	/* In meters per cycle. */
	double throttle = 0.0;
	int gear_state = 0;
	double fuel = 0.0;
	double elevator_trim = 0.0;
	sar_color_struct *color;
	GWFont *font;
	char text[80];


	/* Get initial values. */
	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr == NULL)
	    return;
	if(obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT)
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;

	if(aspect <= 0)
	    return;

        fovz = scene->camera_fovz;
	if(fovz <= 0)
	    return;
        fovx = fovz * aspect;

	color = &option.hud_color;
	font = option.hud_font;


        /* Get direction of camera in the cockpit. */
        cam_dir = &scene->camera_cockpit_dir;

	/* Get direction and position of object. */
	dir = &obj_ptr->dir;
        pos = &obj_ptr->pos;

	/* Get other values. */
	if(obj_aircraft_ptr != NULL)
	{
	    cockpit_offset_z = obj_aircraft_ptr->cockpit_offset_pos.z;
	    vel = &obj_aircraft_ptr->vel;
	    speed = obj_aircraft_ptr->speed;
	    engine_state = obj_aircraft_ptr->engine_state;
	    throttle = SARSimThrottleOutputCoeff(
		(obj_aircraft_ptr->flight_model_type != SAR_FLIGHT_MODEL_SLEW) ?
		    obj_aircraft_ptr->flight_model_type :
		    obj_aircraft_ptr->last_flight_model_type
		,
		obj_aircraft_ptr->throttle,
		obj_aircraft_ptr->collective,
		obj_aircraft_ptr->collective_range
	    );
	    fuel = obj_aircraft_ptr->fuel;
	    elevator_trim = obj_aircraft_ptr->elevator_trim;

	    i = obj_aircraft_ptr->cur_intercept;
	    if((i >= 0) && (i < obj_aircraft_ptr->total_intercepts))
		intercept = obj_aircraft_ptr->intercept[i];

	    if(obj_aircraft_ptr->total_landing_gears > 0)
		gear_state = ((obj_aircraft_ptr->landing_gear[0] == NULL) ?
		    0 :
		    (obj_aircraft_ptr->landing_gear[0]->flags & SAR_LGEAR_FLAG_STATE)
		);
	}


	/* Check if camera direction is looking forward enough. */
	if(((cam_dir->heading > (1.5 * PI)) ||
            (cam_dir->heading < (0.5 * PI))) &&
           ((cam_dir->pitch > (1.5 * PI)) ||
            (cam_dir->pitch < (0.5 * PI)))
	)
	{
	    double heading, pitch;
	    double r;
	    int x, y;			/* Center of hud in pixels. */
	    int win_width, win_height;


	    heading = cam_dir->heading;
            if(heading > PI)
                heading = heading - (2 * PI);

	    pitch = cam_dir->pitch;
            if(pitch > PI)
                pitch = pitch - (2 * PI);

	    win_width = display->width;
	    win_height = display->height;

	    /* Set color and font. */
            SARDrawSetColor(color);  
            GWSetFont(display, font);

	    /* Calculate center in window coordinates. */
	    r = win_width / fovx;
	    x = (int)((win_width / 2) - (r * heading));

            r = win_height / fovz;
            y = (int)((win_height / 2) - (r * pitch));

	    /* Draw speed in miles per hour. */
	    sprintf(text, "%.1f", SFMMPCToMPH(speed));
	    GWDrawString(
		display,
		(int)(x - (win_width * 0.20) - 25),
		(int)(y + 3),
		text
	    );

	    /* Elevator trim. */
	    SARDrawHUDElevatorTrim(
		display,
		(GLfloat)(x - (win_width * 0.23)),
                (GLfloat)(win_height - (y - (win_height * 0.22))),
		30, 5,
		elevator_trim
	    );


	    /* MSL altitude (include scene MSL elevation). */
            sprintf(
		text, "%.0f",
                SFMMetersToFeet(pos->z + cockpit_offset_z + scene->msl_elevation)
	    );
            GWDrawString(
                display, 
                (int)(x + (win_width * 0.20)),
                (int)(y + 3),
                text
            );

	    /* Throttle. */
	    switch(engine_state)
	    {
	      case SAR_ENGINE_STATE_ON:
		sprintf(text, "Thr: %.0f%%", throttle * 100);
		break;

	      case SAR_ENGINE_STATE_INIT:
		sprintf(text, "Eng(S): %.0f%%", throttle * 100);
                break;

	      case SAR_ENGINE_STATE_OFF:
                sprintf(text, "Eng(O): %.0f%%", throttle * 100);
                break;
	    }
            GWDrawString(
                display,
                (int)(x - (win_width * 0.25)),
                (int)(y + (win_height * 0.29)),
                text
            );

	    /* Air brakes. */
	    if(player_air_brakes)
	    {
                GWDrawString(
                    display,
                    (int)(x - (win_width * 0.25)),
                    (int)(y + (win_height * 0.29) - 16),
                    "Brakes"
                );
	    }


            /* Gear state. */
	    if(!gear_state)
	    {
                GWDrawString(
                    display,
                    (int)(x - (win_width * 0.25)),
                    (int)(y + (win_height * 0.29) + 16),
                    "Gear"
                );
	    }

	    /* Fuel. */
            sprintf(text, "Fuel: %.0f lbs", SFMKGToLBS(fuel));
	    GWDrawString(
                display,
                (int)(x + (win_width * 0.05)),
                (int)(y + (win_height * 0.29)),
                text  
            );

	    /* Draw HUD heading ticks. */
	    SARDrawHUDHeading(
	        display,
	        scene,
	        dir,
		intercept,
		(int)(r * -heading), (int)(r * -pitch),
		win_width, win_height
	    );

	    /* Draw HUD attitude ticks. */
	    SARDrawHUDAttitude(
		display, scene,
                dir, vel,
                (int)(r * -heading), (int)(r * pitch),
                win_width, win_height,
		fovx, fovz
	    );
	}
}

/*
 *	Draws attitude and position label strip of the specified
 *	object.
 */
static void SARDrawOutsideAttitude(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
        int width, int height
)
{
        int i, x, y, x2, y2, degi, offset_x, offset_y;
        double pitch_deg;
        int half_width, half_height;
        double sin_bank, cos_bank;
        int tick_start_y, end_min, end_max;     /* In pixels. */
        int     tick_inc_yi,                    /* In pixels. */
                tick_inc_degreesi,              /* In degrees. */
                tick_x_max,                     /* In pixels. */
                tick_x_min;
        double  tick_inc_y,                     /* In pixels. */
                tick_inc_degrees;               /* In degrees. */
        int tick_start_pitch;                   /* In degrees. */
	sar_position_struct *pos;
	sar_direction_struct *dir;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;

	int vel_dir_w = 32, vel_dir_h = 32;
	u_int8_t vel_dir_bm[] = {
0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 
0x00, 0x01, 0x00, 0x00, 
0x10, 0x00, 0x00, 0x08, 
0x08, 0x00, 0x00, 0x10, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x01, 0x00, 0x00, 
0xc0, 0x03, 0x80, 0x03, 
0x00, 0x01, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
0x08, 0x00, 0x00, 0x10, 
0x10, 0x00, 0x00, 0x08, 
0x00, 0x01, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00
	};


        /* Coordinates are handled in left hand rule xy plane,
         * the y axis is inverted at the call to the X draw
         * function.
         */

	if(obj_ptr == NULL)
	    return;

	pos = &obj_ptr->pos;
	dir = &obj_ptr->dir;

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = obj_ptr->data;
	    break;
	}

        /* Calculate half of size. */
        half_width = width / 2;   
        half_height = height / 2;

	offset_x = 0;
	offset_y = -half_height + 20 + 5;

        /* Calculate Y limits and increments. */
        end_min = -20;
        end_max = 20;

        tick_inc_degrees = 10;          /* Always 10 degrees. */

        tick_x_max = 10;		/* Tick width. */
        tick_x_min = 4;			/* Tick spacing from center. */

        sin_bank = sin(dir->bank);
        cos_bank = cos(dir->bank);

        /* Set tick increment along Y axis. */
        tick_inc_y = 10;

        /* Split into two cases, positive and negative pitch. */ 
        if(dir->pitch > PI)
        {
            /* Positive pitch. */
            double deg;

            pitch_deg = 360 - (int)SFMRadiansToDegrees(dir->pitch);
            deg = (int)pitch_deg % (int)tick_inc_degrees;
            tick_start_y = (int)(-deg * tick_inc_y / tick_inc_degrees);
            tick_start_pitch = (int)(pitch_deg - deg);
        }
        else
        {       
            /* Negative pitch. */
            double deg;

            pitch_deg = (int)SFMRadiansToDegrees(dir->pitch);
            deg = tick_inc_degrees -
                ((int)pitch_deg % (int)tick_inc_degrees);
            tick_start_y = (int)(-deg * tick_inc_y / tick_inc_degrees);
            tick_start_pitch = (int)(-pitch_deg - deg);
        }   
        tick_inc_degreesi = (int)tick_inc_degrees;
        tick_inc_yi = (int)tick_inc_y; 


/* Define procedure to draw each tick for use in the for() loop
 * just below.
 */
#define DO_DRAW_ATTITUDE_TICK	{ \
 x2 = (int)(-sin_bank * y); \
 y2 = (int)(cos_bank * y); \
 if(degi == 0) \
 { \
    glBegin(GL_LINES); \
    { \
      glVertex2d( \
	(offset_x + x2 + (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_min * sin_bank) + half_height) \
      ); \
      glVertex2d( \
	(offset_x + x2 + ((5 + tick_x_max) * cos_bank) + half_width), \
	(offset_y + y2 + ((5 + tick_x_max) * sin_bank) + half_height) \
      ); \
 \
      glVertex2d( \
	(offset_x + x2 - (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_min * sin_bank) + half_height) \
      ); \
      glVertex2d( \
	(offset_x + x2 - ((5 + tick_x_max) * cos_bank) + half_width), \
	(offset_y + y2 - ((5 + tick_x_max) * sin_bank) + half_height) \
      ); \
    } \
    glEnd(); \
} \
else \
{ \
    glBegin(GL_LINES); \
    { \
      glVertex2d( \
	(offset_x + x2 + (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_min * sin_bank) + half_height) \
      ); \
      glVertex2d( \
	(offset_x + x2 + (tick_x_max * cos_bank) + half_width), \
	(offset_y + y2 + (tick_x_max * sin_bank) + half_height) \
      ); \
 \
      glVertex2d( \
	(offset_x + x2 - (tick_x_min * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_min * sin_bank) + half_height) \
      ); \
      glVertex2d( \
	(offset_x + x2 - (tick_x_max * cos_bank) + half_width), \
	(offset_y + y2 - (tick_x_max * sin_bank) + half_height) \
      ); \
    } \
    glEnd(); \
 } \
 degi += tick_inc_degreesi; \
}


        degi = tick_start_pitch;

        /* Draw upper (pitch up) ticks. */
        for(y = tick_start_y; y < end_max; y += tick_inc_yi)   
        {
            if(degi > 90) 
                break;

            DO_DRAW_ATTITUDE_TICK
        }

        degi = tick_start_pitch - tick_inc_degreesi;

        /* Draw lower (pitch down) ticks. */
        for(y = tick_start_y - tick_inc_yi; y > end_min; y -= tick_inc_yi)
        {
            if(degi < -90)
                break;

            DO_DRAW_ATTITUDE_TICK
        }
#undef DO_DRAW_ATTITUDE_TICK


        /* Draw aircraft axis marker. */
        x = offset_x + half_width;
        y = offset_y + half_height;
	glBegin(GL_LINES);
	{
	    glVertex2d(
		x - 6,
		y
	    );
	    glVertex2d(
		x + 6,
		y
	    );
	}
	glEnd();

	/* Draw speed and direction label. */
	if(obj_aircraft_ptr != NULL)
	{
	    sar_obj_intercept_struct *intercept = NULL;
	    double intercept_dtheta = 0.0, intercept_sin_theta,
		intercept_cos_theta, intercept_dist = 0.0,
		vel_speed;
	    char text[256];


	    /* Get pointer to selected intercept. */
            i = obj_aircraft_ptr->cur_intercept;
            if((i >= 0) && (i < obj_aircraft_ptr->total_intercepts))
	    {
                intercept = obj_aircraft_ptr->intercept[i];
		if(intercept != NULL)
		{
		    double	icpt_dx = intercept->x - pos->x,
				icpt_dy = intercept->y - pos->y;

		    /* Calculate intercept delta theta in radians. */
                    intercept_dtheta = SFMDeltaRadians(
                        dir->heading,
                        SFMSanitizeRadians(
			    (0.5 * PI) - atan2(icpt_dy, icpt_dx)
                        )
                    );
		    /* Calculate intercept distance in meters. */
		    intercept_dist = hypot(icpt_dx, icpt_dy);
		}
	    }
	    intercept_sin_theta = sin(intercept_dtheta);
            intercept_cos_theta = cos(intercept_dtheta);

            /* Calculate 2D vel. */
            vel_speed = hypot(
                obj_aircraft_ptr->vel.x,
                obj_aircraft_ptr->vel.y
            );

	    /* Speed and distance to intercept label. */
	    if(intercept_dist > 0.0)
	    {
		double	v1 = SFMMPCToMPH(obj_aircraft_ptr->speed),
			v2 = SFMMetersToMiles(intercept_dist);

		x = offset_x + half_width - (width / 4) - (7 * 8);
		y = height - (offset_y + half_height - 4);
		if(v2 > 10.0)
		    sprintf(text, "%.1f MPH  %.0f", v1, v2);
		else
		    sprintf(text, "%.1f MPH  %.1f", v1, v2);
	    }
	    else
	    {
		double v1 = SFMMPCToMPH(obj_aircraft_ptr->speed);

		x = offset_x + half_width - (width / 4) - (7 * 5);
		y = height - (offset_y + half_height - 4);
		sprintf(text, "%.1f MPH", v1);
	    }
	    GWDrawString(display, x, y, text);

	    /* Draw velocity direction (object relative). */
            x = offset_x + half_width - (width / 4);
            y = offset_y + half_height + 12;
	    if(vel_speed > 0)
	    {
		glBegin(GL_LINES);
		{
		    glVertex2d(
			(x + ((obj_aircraft_ptr->vel.x / vel_speed) * 12.0)),
			(y + ((obj_aircraft_ptr->vel.y / vel_speed) * 12.0))
		    );
		    glVertex2d(x, y);
		}
		glEnd();
	    }

	    /* Intercept direction. */
	    glBegin(GL_LINES);
	    {
		glVertex2d(
		    (x + (intercept_sin_theta * 16.0)),
		    (y + (intercept_cos_theta * 16.0))
		);
		glVertex2d(
		    (x + (intercept_sin_theta * 10.0)),
		    (y + (intercept_cos_theta * 10.0))
	        );
	    }
	    glEnd();

            glRasterPos2i(
		x - (vel_dir_w / 2) + 1,
		y - (vel_dir_h / 2)
	    );
            glBitmap(
                vel_dir_w, vel_dir_h, 
                0.0, 0.0,
                (GLfloat)vel_dir_w, 0.0,
                vel_dir_bm
            );
	}

	/* Draw altitude label. */
	if(obj_aircraft_ptr != NULL)
	{
	    double v1 = SFMMetersToFeet(
		(-obj_aircraft_ptr->center_to_ground_height) +
		obj_aircraft_ptr->cockpit_offset_pos.z
	    );
	    double throttle;
	    char text[256];


	    SARDrawHUDElevatorTrim(
		display,
		(GLfloat)(offset_x + half_width + (width / 4) - 35),
		(GLfloat)(offset_y + half_height + 5),
		30, 5,
		obj_aircraft_ptr->elevator_trim
	    );


            x = offset_x + half_width + (width / 4) - (7 * 3);
            y = height - (offset_y + half_height + 32);

            throttle = SARSimThrottleOutputCoeff(
                (obj_aircraft_ptr->flight_model_type != SAR_FLIGHT_MODEL_SLEW) ?
                    obj_aircraft_ptr->flight_model_type :
                    obj_aircraft_ptr->last_flight_model_type
                ,
                obj_aircraft_ptr->throttle,
                obj_aircraft_ptr->collective,
                obj_aircraft_ptr->collective_range
            );
	    switch(obj_aircraft_ptr->engine_state)
	    {
	      case SAR_ENGINE_STATE_OFF:
                sprintf(text, "Eng(O): %.0f%%", throttle * 100);
                break;

	      case SAR_ENGINE_STATE_INIT:
		sprintf(text, "Eng(S): %.0f%%", throttle * 100);
		break;

	      default:
		sprintf(text, "Thr: %.0f%%", throttle * 100);
		break;
	    }
	    GWDrawString(display, x, y, text);


            y += 18;
            sprintf(text, "%.0f AGL", v1);
	    GWDrawString(display, x, y, text);

	    v1 = SFMMPCToMPH(obj_aircraft_ptr->vel.z);
	    y += 18;
	    if(v1 > 10.0)
	        sprintf(text, "+%.0f ROC", v1);
            else if(v1 > 0.0)
		sprintf(text, "+%.1f ROC", v1);
	    else if(v1 > -10.0)
		sprintf(text, "%.1f ROC", v1);
            else
                sprintf(text, "%.0f ROC", v1);
	    GWDrawString(display, x, y, text);
	}

	/* Draw heading label (just above attitude ticks). */
	if(obj_aircraft_ptr != NULL)
	{
	    double v1 = SFMRadiansToDegrees(dir->heading);
	    char text[80];

	    x = offset_x + half_width - 10;
	    y = height - (offset_y + half_height + 36);
	    sprintf(text, "%.0f", v1);
            GWDrawString(display, x, y, text);
	}

	/* Draw coordinates if viewing from map but not in slew mode. */
	if((scene->camera_ref == SAR_CAMERA_REF_MAP) &&
           (flight_model_type != SAR_FLIGHT_MODEL_SLEW)
	)
	{
	    double dms_x, dms_y;
	    char text[256];

	    SFMMToDMS(
		camera_pos.x, camera_pos.y,
		scene->planet_radius,
		scene->dms_x_offset, scene->dms_y_offset,
		&dms_x, &dms_y
	    );

	    sprintf(
		text,
		"%s %s  Alt: %.0f",
		SFMLongitudeToString(dms_x),
                SFMLatitudeToString(dms_y),
		SFMMetersToFeet(camera_pos.z)
	    );
            GWDrawString(display, 5, 5, text);
	}
}

/*
 *	Draws map grids, called by SARDrawMap().
 *
 *	Translation should still be in world coordinates (not 
 *	orthogonal'ed).
 *
 *	Given pos should be the camera position. Field of view fov
 *	is in radians about the Y axis.
 *
 *	Inputs assumed valid.
 */
static void SARDrawMapGrids(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_position_struct *pos,
	double fovz_um,		/* Field of view in unit meters, z axis. */
	double aspect		/* Width / height. */
)
{
	double x, y, dx, dy;
	double x_min, x_max, y_max, y_min;
	double	theta, planet_radius,
		lat_dmstom_coeff, lon_dmstom_coeff,
		dms_center_x, dms_center_y;


	/* Get planet radius, must be positive. */
	planet_radius = scene->planet_radius;
	if(planet_radius < 1.0)
	    return;

	/* Calculate bounds in meters. */
	y_max = ((pos->z * fovz_um) / 2);
	y_min = (-y_max);
	y_max = (y_max + pos->y);
        y_min = (y_min + pos->y);

	x_max = ((pos->z * fovz_um * aspect) / 2);
	x_min = (-x_max);
        x_max = (x_max + pos->x);
        x_min = (x_min + pos->x);

        /* Calculate theta in radians, representing the Y axis theta. */
        theta = CLIP(
            DEGTORAD(scene->dms_y_offset) + (pos->y / planet_radius)
        , -0.5 * PI, 0.5 * PI);

	/* Calculate conversion coefficients, degrees to meters. */
	lat_dmstom_coeff = (PI / 180.0) * planet_radius;
	lon_dmstom_coeff = cos(theta) * (PI / 180.0) * planet_radius;

	/* Calculate deltas, in meters. Delta is 1 degree. */
	dx = (1.0 * lon_dmstom_coeff);
	dy = (1.0 * lat_dmstom_coeff);

	/* Calculate dms center in dms (degrees). */
	SFMMToDMS(
	    pos->x, pos->y, scene->planet_radius,
	    scene->dms_x_offset, scene->dms_y_offset,
	    &dms_center_x, &dms_center_y
	);

	/* Note: All coordinate units in dms, convert to meters just
	 * before drawing but do all calculations in dms (degrees).
	 */
#define COLOR_LEVEL0    glColor4d(1.0, 0.0, 0.0, 1.0);
#define COLOR_LEVEL1    glColor4d(1.0, 0.0, 0.0, 0.5);
#define COLOR_LEVEL2    glColor4d(1.0, 0.0, 0.0, 0.75);

	/* Begin drawing line ticks.
	 *
	 * Note that we can use floor() instead of modulus when 
	 * calculating the starting position sinec we are rounding to
	 * the nearest 1 degree as an int.
	 */

	StateGLEnable(&display->state_gl, GL_BLEND);

	/* Latitude ticks. */
	if(dy > 0.0)
	{
	    int clevel;

#define DO_DRAW_LINE	\
{ \
 switch(clevel) \
 { \
  case 1: \
   COLOR_LEVEL1 \
   clevel = 2; \
   break; \
  case 2: \
   COLOR_LEVEL2 \
   clevel = 3; \
   break; \
  case 3: \
   COLOR_LEVEL1 \
   clevel = 0; \
   break; \
  default: \
   COLOR_LEVEL0 \
   clevel = 1; \
   break; \
 } \
 glBegin(GL_LINES); \
 { \
  glVertex3d(x_min, 0.0, -y); \
  glVertex3d(x_max, 0.0, -y); \
 } \
 glEnd(); \
}

	    for(
		y = (
		    ((int)floor(dms_center_y) * lat_dmstom_coeff) -
		    (scene->dms_y_offset * lat_dmstom_coeff)
		),
		clevel = 0;
		y < y_max;
		y += (dy / 4)
	    )
	    {
		DO_DRAW_LINE
	    }
            for(
                y = (
                    ((int)floor(dms_center_y) * lat_dmstom_coeff) -
                    (scene->dms_y_offset * lat_dmstom_coeff)
                ) - (dy / 4),
		clevel = 1;
                y > y_min;
                y -= (dy / 4)
            )
	    {
		DO_DRAW_LINE
	    }
#undef DO_DRAW_LINE
	}

	/* Longitude ticks. */
        if(dx > 0.0)
        {
            int clevel;

#define DO_DRAW_LINE	\
{ \
 switch(clevel) \
 { \
  case 1: \
   COLOR_LEVEL1 \
   clevel = 2; \
   break; \
  case 2: \
   COLOR_LEVEL2 \
   clevel = 3; \
   break; \
  case 3: \
   COLOR_LEVEL1 \
   clevel = 0; \
   break; \
  default: \
   COLOR_LEVEL0 \
   clevel = 1; \
   break; \
 } \
 glBegin(GL_LINES); \
 { \
  glVertex3d(x, 0.0, -y_min); \
  glVertex3d(x, 0.0, -y_max); \
 } \
 glEnd(); \
}
            for(
                x = (
                    ((int)floor(dms_center_x) * lon_dmstom_coeff) -
                    (scene->dms_x_offset * lon_dmstom_coeff)
                ),
                clevel = 0;
                x < x_max;
                x += (dx / 4)
            )
            {
                DO_DRAW_LINE
            }
            for(
                x = (
                    ((int)floor(dms_center_x) * lon_dmstom_coeff) -
                    (scene->dms_x_offset * lon_dmstom_coeff)
                ) - (dx / 4),
                clevel = 1;
                x > x_min;
                x -= (dx / 4)
            )
            {
                DO_DRAW_LINE
            }
#undef DO_DRAW_LINE
        }

        StateGLDisable(&display->state_gl, GL_BLEND);

#undef COLOR_LEVEL0
#undef COLOR_LEVEL1
#undef COLOR_LEVEL2
}

/*
 *      Draws cross hairs, called by SARDrawMap().
 *
 *      Translation should still be ortho'ed.
 *
 *      Inputs assumed valid.   
 */
static void SARDrawMapCrossHairs(
	gw_display_struct *display, sar_scene_struct *scene
)
{
	GLfloat x1, x2, y1, y2;
	GLfloat size_coeff = (GLfloat)0.03;


        glColor4d(1.0, 0.0, 0.0, 1.0);

	x1 = (GLfloat)(display->width / 2);
	x2 = ceil(x1 + ((double)display->width * size_coeff));
	x1 = x1 - ((double)display->width * size_coeff);

	y1 = (double)display->height / 2;

	glBegin(GL_LINES);
	{
	    glVertex2i((GLint)x1, (GLint)y1);
	    glVertex2i((GLint)x2, (GLint)y1);
	}
	glEnd();


        x1 = (double)display->width / 2;

        y1 = (double)display->height / 2;
        y2 = (double)ceil(y1 + ((double)display->height * size_coeff));
        y1 = y1 - ((double)display->height * size_coeff);

        glBegin(GL_LINES);
        {
            glVertex2i((GLint)x1, (GLint)y1);
            glVertex2i((GLint)x1, (GLint)y2);
        }
        glEnd();
}


/*
 *	Draws the standard messages defined (if any) on the given scene
 *	structure.
 *
 *	Standard messages are only drawn if message_display_untill is
 *	positive.
 */
static void SARDrawMessages(
	gw_display_struct *display, sar_scene_struct *scene
)
{
	int i, len, offset_x, offset_y, width, height,
	    fw, fh, half_width, half_height, bg_width, bg_height;
	const char *mesg_ptr;
	int messages_available;
	GWFont *font;

        /* Coordinates are handled in left hand rule xy plane,
         * the y axis is inverted at the call to the draw
         * function.
         */

	/* Not currently displaying messages? */
	if(scene->message_display_untill <= 0)
	    return;

	/* Time to stop displaying messages? */
	if(scene->message_display_untill < cur_millitime)
	{
	    /* Reset message display untill time to 0 and delete
	     * all messages on the scene. This will not change
	     * the size of the message pointer array.
	     */
	    SARMessageClearAll(scene);
	    return;
	}

	/* Calculate dimensions. */
        width = display->width;
        height = display->height;

        half_width = width / 2;
        half_height = height / 2;

	font = option.message_font;
	GWGetFontSize(font, NULL, NULL, &fw, &fh);
	offset_x = 0;
        offset_y = (int)(height * 0.20);

        /* Count number of non-NULL messages. */
	messages_available = 0;
        for(i = scene->total_messages - 1; i >= 0; i--)
        {
            if(scene->message[i] == NULL)
                break;
	    else
		messages_available++;
        }

	/* Calculate background size. */
	bg_width = width;
	bg_height = (fh * messages_available);

	/* Draw darkened background. */
	StateGLEnable(&display->state_gl, GL_BLEND);
	StateGLBlendFunc(
	    &display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
	);
	glBegin(GL_QUADS);
	{
	    glColor4d(0.0, 0.0, 0.0, 0.5);
	    glVertex2d(0, offset_y);
	    glVertex2d(bg_width, offset_y);
            glVertex2d(bg_width, (offset_y + bg_height));
            glVertex2d(0, (offset_y + bg_height));
	}
	glEnd();
	StateGLDisable(&display->state_gl, GL_BLEND);

	/* Begin drawing messages, first message is boldest color. */
	glColor4d(1.0, 1.0, 1.0, 1.0);
        GWSetFont(display, font);

	offset_y += fh;
	for(i = scene->total_messages - 1; i >= 0; i--)
	{ 
	    mesg_ptr = (const char *)scene->message[i];
	    if(mesg_ptr == NULL)
	        break;

	    len = strlen(mesg_ptr);
	    offset_x = half_width - ((len * fw) / 2);

            GWDrawString(
		display,
		offset_x,
		height - offset_y,
		mesg_ptr
	    );

	    /* Lighten color and change other attributes for drawing
	     * all subsequent lines.
	     */
            glColor4d(0.8, 0.8, 0.8, 1.0);

	    /* Move current line position `down one line'. */
	    offset_y += fh;
	}
}

/*
 *	Draws sticky banner messages.
 *
 *      Sticky banner messages only drawn if the sticky_banner_message
 *      message list contains data (not NULL).
 */
static void SARDrawBanner(
        gw_display_struct *display, sar_scene_struct *scene
)
{
        int i, len, offset_x, offset_y, width, height,
            fw, fh, half_width, half_height;
        const char *mesg_ptr;
        GWFont *font;

        /* Coordinates are handled in left hand rule xy plane,
         * the y axis is inverted at the call to the draw
         * function.
         */

        /* No banner messages? */
        if((scene->sticky_banner_message == NULL) ||
           (scene->total_sticky_banner_messages < 1)
	)
            return;

        /* Calculate dimensions. */
        width = display->width;
        height = display->height;

        half_width = width / 2;
        half_height = height / 2;

        font = option.banner_font;
        GWGetFontSize(font, NULL, NULL, &fw, &fh);
        offset_x = 0;
        offset_y = (int)(height * 0.80);

        /* Begin drawing messages, first message is boldest color. */
        glColor4d(1.0, 1.0, 1.0, 1.0);
        GWSetFont(display, font);
	/* Draw first line. */
        offset_y += fh;
	i = 0;
	while(i < MIN(scene->total_sticky_banner_messages, 1))
	{
	    mesg_ptr = (const char *)scene->sticky_banner_message[i];
            if(mesg_ptr == NULL)
                break;
            
            len = strlen(mesg_ptr);
            offset_x = half_width - ((len * fw) / 2); 
         
            GWDrawString(
                display,
                offset_x,
                height - offset_y,
                mesg_ptr
            );

	    i++;
	    offset_y -= fh;
	}

	/* Give some spacing between first line and smaller subsequent
	 * lines.
	 */
	offset_y -= 5;

	/* Begin drawing subsequent lines. */
        font = option.message_font;
	GWGetFontSize(font, NULL, NULL, &fw, &fh);
	GWSetFont(display, font);
	while(i < scene->total_sticky_banner_messages)
	{
            mesg_ptr = (const char *)scene->sticky_banner_message[i];
            if(mesg_ptr == NULL)
                break;

            len = strlen(mesg_ptr);
            offset_x = half_width - ((len * fw) / 2);

            GWDrawString(
                display,
                offset_x,
                height - offset_y,
                mesg_ptr  
            );

	    i++;
	    offset_y -= fh;
	}
}

/*
 *	Draws slew coordinates of the given object.
 *
 *	The call to this function is responsible for checking if the given
 *	object is in slew mode.
 *
 *	The font and color should already be set prior this this call.
 */
static void SARDrawSlewCoordinates(
        gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr
)
{
	double dms_x, dms_y;
	sar_position_struct *pos;
	sar_direction_struct *dir;
	char text[256];


	if(obj_ptr == NULL)
	    return;

	pos = &obj_ptr->pos;
	dir = &obj_ptr->dir;

	if(0)
	{
	    SFMMToDMS(
		pos->x, pos->y,
		scene->planet_radius,
		scene->dms_x_offset, scene->dms_y_offset,
		&dms_x, &dms_y
	    );
	    sprintf(
		text,
		"%s %s  Alt: %.0f",
		SFMLongitudeToString(dms_x),
		SFMLatitudeToString(dms_y),
		SFMMetersToFeet(pos->z)
	    );
            GWDrawString(display, 5, 5, text);
	}
	else
	{
	    sprintf(
		text,
 "%i:%2i Player: (hpb) %.0f %.0f %.0f  (xyz) %.2f %.2f %.2f(%.2f feet)\n",
                (int)(scene->tod / 3600),
                (int)((int)((int)scene->tod / 60) % 60),
                SFMRadiansToDegrees(dir->heading),
                SFMRadiansToDegrees(dir->pitch),
                SFMRadiansToDegrees(dir->bank),
                pos->x, pos->y, pos->z,
                SFMMetersToFeet(pos->z)
            );
            GWDrawString(display, 5, 5, text);
	}
}

/*
 *	Draws the light array.
 */
static void SARDrawLights(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_light_struct **ptr, int total
)
{
	int i;
	sar_light_struct *light_ptr;
	sar_color_struct *c;
	sar_position_struct *pos;
	StateGLBoolean lighting;


	/* No lights to draw? */
	if(total <= 0)
	    return;

	/* Check if lighting is currently enabled, if it is then
	 * disable it.
	 */
	lighting = display->state_gl.lighting;
	if(lighting)
	{
	    StateGLDisable(&display->state_gl, GL_LIGHTING);
	    StateGLDisable(&display->state_gl, GL_LIGHT0);
	}

	/* Enable blending and point smoothing. */
	StateGLEnable(&display->state_gl, GL_BLEND);
	StateGLBlendFunc(
	    &display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
	);
	StateGLEnable(&display->state_gl, GL_POINT_SMOOTH);

	/* Itterate through each light. */
	for(i = 0; i < total; i++)
	{
	    light_ptr = ptr[i];
	    if(light_ptr == NULL)
		continue;

	    /* Light not on? */
	    if(!(light_ptr->flags & SAR_LIGHT_FLAG_ON))
		continue;

	    /* Skip attenuation spot lights. */
	    if(light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE)
		continue;

	    /* Strobe light? */
	    if(light_ptr->flags & SAR_LIGHT_FLAG_STROBE)
	    {
		if(light_ptr->next_on > 0)
		{
		    /* Waiting to be turned on, implies it's off. */

		    if(light_ptr->next_on > cur_millitime)
		    {
			continue;
		    }
		    else
		    {
			light_ptr->next_on = 0;
			light_ptr->next_off = cur_millitime +
			    light_ptr->int_on;
		    }
		}
		else
		{
		    /* Waiting to be turned off, implies its on. */

		    if(light_ptr->next_off <= cur_millitime)
                    {
			light_ptr->next_on = cur_millitime +
			    light_ptr->int_off;
                        light_ptr->next_off = 0;

                        continue;
                    }
		}
	    }

	    /* Get position and color or light. */
	    pos = &light_ptr->pos;
	    c = &light_ptr->color;

	    /* Set point size. */
            StateGLPointSize(
		&display->state_gl,
		(GLfloat)MAX(light_ptr->radius * 2.0, 1.0)
	    );

	    /* Draw light. */
	    glBegin(GL_POINTS);
	    {
                glColor4d(c->r, c->g, c->b, c->a);
		glVertex3d(pos->x, pos->z, -pos->y);
	    }
	    glEnd();
	}

	/* Disable blending. */
	StateGLDisable(&display->state_gl, GL_BLEND);

	/* Enable lighting again if it was on prior to call. */
	if(lighting)
	{
	    StateGLEnable(&display->state_gl, GL_LIGHTING);
	    StateGLEnable(&display->state_gl, GL_LIGHT0);
	}

	return;
}


/*
 *	Draws cockpit (if available) on the object.  Translation and
 *	rotation is assumed to be at the object's center.
 */
static void SARDrawObjectCockpit(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;


        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)
                obj_ptr->data;
            if(obj_aircraft_ptr == NULL)
                break;

	    SARVisualModelCallList(
		obj_aircraft_ptr->visual_model_cockpit
	    );
            break;

          default:
            break;
        }

	return;
}

/*
 *	Draws shadow (if available) on the object. Translation
 *	is already assumed to be at the object's center and heading
 *	(except pitch and bank) of object is already set.
 *
 *	Also draws rotor and prop wash if they are relivent for the
 *	given object's values.
 *
 *	Inputs assumed valid.
 */
static void SARDrawObjectShadow(
	gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr
)
{
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_object_fueltank_struct *obj_fueltank_ptr = NULL;
	int on_water = 0, is_crashed = 0;
	double	throttle = 1.0,
		center_to_ground_height = 0.0;
	sar_visual_model_struct *vmodel = NULL;

	switch(obj_ptr->type)
	{
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
            if(obj_aircraft_ptr != NULL)
	    {
		center_to_ground_height =
		    obj_aircraft_ptr->center_to_ground_height;
		throttle = obj_aircraft_ptr->throttle;
		on_water = obj_aircraft_ptr->on_water;
		if(obj_aircraft_ptr->air_worthy_state == SAR_AIR_WORTHY_NOT_FLYABLE)
		    is_crashed = 1;
		vmodel = obj_aircraft_ptr->visual_model_shadow;
	    }
	    break;

	  case SAR_OBJ_TYPE_FUELTANK:
	    obj_fueltank_ptr = (sar_object_fueltank_struct *)obj_ptr->data; 
            if(obj_fueltank_ptr != NULL)
	    {

	    }
	    break;

/* Add support for other object types with shadows here. */

	}


	/* Draw prop wash? */
	if((obj_aircraft_ptr != NULL) &&
	   (center_to_ground_height >= -SAR_SHADOW_VISIBILITY_HEIGHT)
	)
	{
	    GLenum shade_model_mode;
	    int i, tex_num, frame_num;
	    sar_obj_rotor_struct *rotor_ptr;
	    v3d_texture_ref_struct *t;
	    double r;
            double z = center_to_ground_height;

	    if(option.textured_objects && option.prop_wash)
	    {
		GLboolean depth_mask_flag;

                glPushMatrix();
                {
                    glTranslated(0.0, z, 0.0);

                    StateGLEnable(&display->state_gl, GL_BLEND);
                    StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
                    depth_mask_flag = display->state_gl.depth_mask_flag;
                    StateGLDepthMask(&display->state_gl, GL_FALSE);  
                    StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
		    StateGLBlendFunc(
			&display->state_gl, 
			GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
		    );
                    StateGLPolygonOffset(
                        &display->state_gl,
			(GLfloat)option.gl_polygon_offset_factor, -1.0
                    );

		    for(i = 0; i < obj_aircraft_ptr->total_rotors; i++)
		    {
		        rotor_ptr = obj_aircraft_ptr->rotor[i];
		        if(rotor_ptr == NULL)
			    continue;

		        tex_num = rotor_ptr->rotor_wash_tex_num;
		        if(SARIsTextureAllocated(scene, tex_num))
			    t = scene->texture_ref[tex_num];
		        else
			    continue;

		        /* Get frame number. */
		        frame_num = t->total_frames *
			    rotor_ptr->rotor_wash_anim_pos /
			    ((sar_grad_anim_t)-1);
		        if(frame_num >= t->total_frames)
			    frame_num = t->total_frames - 1;
		        if(frame_num < 0)
			    frame_num = 0;

		        /* Select propwash texture. */
		        V3DTextureSelectFrame(t, frame_num);

			/* Set color, alpha based on distance to ground
			 * divided by shadow visibility height and
			 * multiplied by both global rotor wash visibility
			 * coeff and the throttle value.
			 */
		        glColor4d(
			    1.0, 1.0, 1.0,
			    ((1.0 + (double)(
				center_to_ground_height / SAR_SHADOW_VISIBILITY_HEIGHT)
			    ) * option.rotor_wash_vis_coeff * throttle)
			);

			/* Use flat shading. */
			shade_model_mode = display->state_gl.shade_model_mode;
			StateGLShadeModel(&display->state_gl, GL_FLAT);

			/* Calculate radius to be 4 times the rotor's size. */
		        r = rotor_ptr->radius * 4.0;

			/* Draw prop wash. */
                        glBegin(GL_QUADS);
                        {
                            glNormal3d(0, 1, 0);
                            glTexCoord2d(0, 1 - 0);
                            glVertex3d(-r, 0, r);
                            glTexCoord2d(1, 1 - 0);
                            glVertex3d(r, 0, r);
                            glTexCoord2d(1, 1 - 1);
                            glVertex3d(r, 0, -r);
                            glTexCoord2d(0, 1 - 1);
                            glVertex3d(-r, 0, -r);
                        }
                        glEnd();

			/* Restore shade model. */
                        StateGLShadeModel(&display->state_gl, shade_model_mode);
		    }

                    StateGLDisable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
                    StateGLDisable(&display->state_gl, GL_BLEND);
                    StateGLDepthMask(&display->state_gl, depth_mask_flag);
		    StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
                }
                glPopMatrix();
	    }
	}

        /* Draw shadow? */
        if((vmodel != NULL) &&
           (center_to_ground_height >= -SAR_SHADOW_VISIBILITY_HEIGHT) &&
	   (on_water ? !is_crashed : 1)
        )
        {
	    GLboolean depth_mask_flag;
	    GLenum shade_model_mode;
            double z = center_to_ground_height;

            StateGLEnable(&display->state_gl, GL_BLEND);
            StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
            depth_mask_flag = display->state_gl.depth_mask_flag;
            StateGLDepthMask(&display->state_gl, GL_FALSE);
            StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
            StateGLBlendFunc(
                &display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
            );
            StateGLPolygonOffset(
                &display->state_gl, 
		(GLfloat)option.gl_polygon_offset_factor, -1.0
            );

	    /* Use flat shading. */
	    shade_model_mode = display->state_gl.shade_model_mode;
	    StateGLShadeModel(&display->state_gl, GL_FLAT);

	    /* Draw shadow. */
            glPushMatrix();
            {
                glTranslated(0.0, z, 0.0);
                SARVisualModelCallList(vmodel);
            }
            glPopMatrix();

            StateGLDisable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
            StateGLDisable(&display->state_gl, GL_BLEND);
	    StateGLDepthMask(&display->state_gl, depth_mask_flag);
            StateGLEnable(&display->state_gl, GL_ALPHA_TEST);

            /* Restore shade model. */
            StateGLShadeModel(&display->state_gl, shade_model_mode);
        }

	return;
}


/*
 *	Draws scene foundations (ground base).
 *
 *	Inputs assumed valid.
 */
static void SARDrawSceneFoundations(
	gw_display_struct *display, sar_scene_struct *scene,
	double aspect
)
{
	int in_map_view;
	int mod_offset_x, mod_offset_y;
	sar_position_struct *pos;
	double fovx, fovz;	/* Field of view x and z, in radians. */
        sar_cloud_layer_struct *cloud_layer_ptr = NULL;
	sar_visual_model_struct *vmodel = NULL;


        pos = &camera_pos;

	/* Get field of views in radians. */
	fovz = scene->camera_fovz;
        if(fovz <= 0)
            return;
        fovx = fovz * aspect;

	/* In map view? */
	in_map_view = ((scene->camera_ref == SAR_CAMERA_REF_MAP) ?
            1 : 0
	);

        /* Get pointer to first (lowest) cloud layer. */
	cloud_layer_ptr = lowest_cloud_layer_ptr;

	/* Cloud layer valid and not in map view? */
        if((cloud_layer_ptr != NULL) && !in_map_view)
        {
            /* If we are at or above the first cloud layer,
             * then do not draw foundations since the would not
	     * be visable.
             */
            if(pos->z >= cloud_layer_ptr->z)
                return;
        }

	/* Calculate module offset. */
        mod_offset_x = (int)pos->x %
            (int)scene->base.tile_width;
        mod_offset_y = (int)pos->y %
            (int)scene->base.tile_height;


	glPushMatrix();
	{
	    /* Translate to new tiled position. */
 	    glTranslated(
                (double)((int)pos->x - mod_offset_x),
                0.0,
                (double)-((int)pos->y - mod_offset_y)
            );

	    /* Draw simple (solid color) base visual model. */
	    vmodel = scene->base.visual_model_simple;

	    /* Is visual model valid and not in map view? */
	    if((vmodel != NULL) && !in_map_view)
	    {
		SAR_DRAW_TEXTURE_1D_OFF
		SAR_DRAW_TEXTURE_2D_OFF
                glColor4d(
                    scene->base.color.r * scene->light_color.r,
                    scene->base.color.g * scene->light_color.g,
                    scene->base.color.b * scene->light_color.b,
                    1.0
                );
		SARVisualModelCallList(vmodel);
	    }

	    /* Draw close (textured) tiles. */
	    vmodel = scene->base.visual_model_close;
	    if((vmodel != NULL) && option.textured_ground)
	    {
		SAR_DRAW_TEXTURE_2D_ON
                glColor4d(
                    scene->light_color.r,
                    scene->light_color.g,
                    scene->light_color.b,
                    1.0
                );
		SARVisualModelCallList(vmodel);
	    }
	}
	glPopMatrix();


	return;
}

/*
 *	Draws the given cloud layer.
 *
 *	The winding will be flipped if flip is set to True.
 *
 *	Inputs assumed valid.
 */
static void SARDrawCloudLayer(
	sar_scene_struct *scene, sar_cloud_layer_struct *cloud_layer_ptr,
	Boolean flip
)
{
        int mod_offset_x, mod_offset_y;
        sar_position_struct *pos;


	if(cloud_layer_ptr->visual_model == NULL)
	    return;

        pos = &camera_pos;

	mod_offset_x = (int)pos->x %
	    (int)cloud_layer_ptr->tile_width;
	mod_offset_y = (int)pos->y %
	    (int)cloud_layer_ptr->tile_height;

        glPushMatrix();
        {
            glTranslated(
                (double)((int)pos->x - mod_offset_x),
                (double)cloud_layer_ptr->z,
                (double)-((int)pos->y - mod_offset_y)
            );
	    if(flip)
		glRotated(
		    180.0, 0.0, 0.0, 1.0
		);

	    glColor4d(
		scene->light_color.r,
                scene->light_color.g,
                scene->light_color.b,
		1.0
	    );
	    SARVisualModelCallList(cloud_layer_ptr->visual_model);
        }
        glPopMatrix();

	return;
}

/*
 *	Called by SARDrawCloudBB() to draw repeating `tile' of cloud
 *	`billboard' objects.
 */
static void SARDrawCloudBBItterate(
	sar_position_struct *pos,
        double hw, double height
)
{
        glPushMatrix();
        {
            sar_direction_struct dir;

            /* Adjust heading to camera heading. */
            SARDrawGetDirFromPos(
                &camera_pos, pos, &dir
            );

            /* Translate and rotate heading. */
            glTranslated(
                pos->x, pos->z, -pos->y
            );
            glRotated(
                -SFMRadiansToDegrees(dir.heading),
                0.0, 1.0, 0.0
            );
/* Don't pitch
            glRotated(
                -SFMRadiansToDegrees(dir.pitch),
                1.0, 0.0, 0.0
            );
 */
            glBegin(GL_QUADS);
            {
                glTexCoord2d(0, 1 - 0);
                glVertex3d(-hw, 0.0, 0.0);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(hw, 0.0, 0.0);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(hw, height, 0.0);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(-hw, height, 0.0);
            }
            glEnd();
        }
        glPopMatrix();

	return;
}


/*
 *      Draws cloud `billboard' object.
 *
 *      Inputs assumed valid.
 */
static void SARDrawCloudBB(
        gw_display_struct *display,
        sar_scene_struct *scene,
        sar_cloud_bb_struct *cloud_bb_ptr
)
{
	StateGLBoolean alpha_test;
	sar_position_struct pos;
	int tile_width, tile_height;	/* In meters. */
	double hw, hh;			/* Half bounds of cloud in meters. */
	int tex_num;
	v3d_texture_ref_struct *t;
	sar_color_struct *c;


        /* Select texture. */
        tex_num = cloud_bb_ptr->tex_num;
        if(SARIsTextureAllocated(scene, tex_num))
            t = scene->texture_ref[tex_num];
        else
            return;

	/* Get pointer to scene light color. */
        c = &scene->light_color;

	/* Get bounds. */
	tile_width = cloud_bb_ptr->tile_width;
	tile_height = cloud_bb_ptr->tile_height;
	hw = cloud_bb_ptr->width / 2;
	hh = cloud_bb_ptr->height / 2;

	if((tile_width < 1) || (tile_height < 1) ||
           (hw < 1.0) || (hh < 1.0)
	)
	    return;

	/* Calculate center of object relative to tiled center
	 * (tiled center is a modulus of the camera pos).
	 */
	if(camera_pos.x < 0.0)
	    pos.x = camera_pos.x - (double)((int)camera_pos.x % tile_width)
		- (double)tile_width + cloud_bb_ptr->x;
	else
	    pos.x = camera_pos.x - (double)((int)camera_pos.x % tile_width)
		+ cloud_bb_ptr->x;

        if(camera_pos.y < 0.0)
            pos.y = camera_pos.y - (double)((int)camera_pos.y % tile_height)
                - (double)tile_height + cloud_bb_ptr->y;
        else
            pos.y = camera_pos.y - (double)((int)camera_pos.y % tile_height)
                + cloud_bb_ptr->y;

	/* Z position is just the z position of cloud. */
	pos.z = cloud_bb_ptr->z;


	/* Enable alpha channel. */
	alpha_test = display->state_gl.alpha_test;
	StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);

	/* Select color based on scene lighting. */
	glColor4d(c->r, c->g, c->b, 1.0);
	/* Select texture. */
	V3DTextureSelect(t);


	/* The pos is now set to the center base of the cloud `billboard'
	 * object.
	 */

	/* Draw center oriented tile. */
	SARDrawCloudBBItterate(
	    &pos, hw, cloud_bb_ptr->height
	);

	/* Upper. */
	pos.y += (double)tile_height;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        ); 

        /* Upper right. */
        pos.x += (double)tile_width;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );
        /* Right. */
        pos.y -= (double)tile_height;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );
        /* Lower right. */
        pos.y -= (double)tile_height;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );

	/* Lower. */
        pos.x -= (double)tile_width;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );
        /* Lower left. */
        pos.x -= (double)tile_width; 
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );
        /* Left. */
        pos.y += (double)tile_height;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );

        /* Upper left. */
        pos.y += (double)tile_height;
        SARDrawCloudBBItterate(
            &pos, hw, cloud_bb_ptr->height
        );

	/* Restore alpha test. */
	if(alpha_test)
	    StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	else
	    StateGLDisable(&display->state_gl, GL_ALPHA_TEST);

	return;
}

/*
 *	Draws horizon quads, translation needs to be at scene
 *	origin.
 *
 *	Inputs assumed valid and depth test and lighting turned off.
 */
static void SARDrawSceneHorizon(
        gw_display_struct *display, sar_scene_struct *scene,
	double visibility_max		/* In meters. */
)
{
	GLboolean depth_mask_flag;
	GLenum shade_model_mode;
	int	i, n,
		afternoon = 0,
		total_horizon_textures = 7,
		total_horizon_panels = 12;
	double h_min, h_max, r;
	sar_position_struct *pos;
	sar_color_struct *c;
	sar_scene_horizon_struct *horizon_ptr = &scene->horizon;
	sar_cloud_layer_struct *cloud_layer_ptr = NULL;


        /* Get camera position. */
        pos = &camera_pos;
	c = &scene->light_color;

	/* Match cloud layer just above camera. */
	for(i = 0; i < scene->total_cloud_layers; i++)
	{
	    cloud_layer_ptr = scene->cloud_layer[i];
	    if(cloud_layer_ptr == NULL)
		continue;

	    if(cloud_layer_ptr->z > pos->z)
		break;
	    else
		cloud_layer_ptr = NULL;
	}


	/* Check if horizon has all 7 textures allocated. */
	if(horizon_ptr->total_textures < total_horizon_textures)
	{
	    /* Not enough textures allocated. */
	    return;
	}

	/* Calculate horizon radius r, this is the distance from camera
	 * to each horizon panel. Note that we make this distance a bit
	 * shorter than the maximum visibility to ensure that the horizon
	 * panels are drawn closer than the camera's far clip.
	 */
	r = visibility_max * 0.85;

	/* Calculate height minimum, always 0.0. */
	h_min = 0.0;

	/* Calculate height maximum, dependant of cloud layer existance.
	 * If a cloud layer exists then it will be 1.05 times the cloud
	 * layer's altitude (we use 1.05 to ensure that it goes up into
	 * the clouds just a bit). Otherwise it will be calculated by
	 * s = r * theta where theta is (0.25 * PI).
	 */
	if(cloud_layer_ptr != NULL)
	    h_max = cloud_layer_ptr->z * 1.05;	/* Add a bit of margin. */
	else
	    h_max = MAX(r * (0.25 * PI), pos->z);

	/* Make note if it is afternoon or later. */
	if(scene->tod > (12 * 3600))
	    afternoon = 1;

	SAR_DRAW_TEXTURE_1D_ON
	SAR_DRAW_TEXTURE_2D_OFF

	/* Use flat shading. */
	shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);

	/* No need to write depth when drawing horizon panels, this
	 * will allow some objects to be drawn farther away from
	 * the horizon panels.
	 */
	depth_mask_flag = display->state_gl.depth_mask_flag;
	StateGLDepthMask(&display->state_gl, GL_FALSE);

        glPushMatrix();
        {
	    /* Angle in radians of the current angle from camera to the
	     * the horizon panel's `previous' point.
	     */
	    double theta;

	    /* Angle in radians of normal. */
	    double n_theta;

	    /* Deltas of each panel point along xy plane where index 0
	     * is the `previous point' going clockwise from 9 o'clock.
	     */
	    double dx[2], dy[2];


	    /* Translate to camera position but at z = 0.0. */
            glTranslated(pos->x, 0.0, -pos->y);

	    /* Set color to be the scene's light color so that it
	     * controls the brightness of the horizon.
	     */
            glColor4d(c->r, c->g, c->b, 1.0);

	    /* Initialize theta. */
	    theta = SFMSanitizeRadians(
		(1.5 * PI) -
		(((2.0 * PI) / total_horizon_panels) / 2)
	    );

	    /* Initialize `previous point' for the panel delta offset. */
	    dx[0] = r * sin(theta);
	    dy[0] = r * cos(theta);

	    /* Begin drawing each horizon panel. */
	    for(i = 0; i < total_horizon_panels; i++)
	    {
		/* Increment theta to `next' angle first. */
		theta = SFMSanitizeRadians(
		    theta +
		    ((2.0 * PI) / total_horizon_panels)
		);

		/* Calculate `next' panel point. */
		dx[1] = r * sin(theta);
		dy[1] = r * cos(theta);


                /* Calculate horizon texture index. */
		if(i >= total_horizon_textures)
		    n = total_horizon_textures -
			i + total_horizon_textures - 2;
		else
		    n = i;
		/* Clip. */
		if((n >= 0) && (n < total_horizon_textures))
		    V3DTextureSelect(horizon_ptr->texture[n]);

		/* Calculate normal theta. */
		n_theta = SFMSanitizeRadians(
                    theta -
                    (((2.0 * PI) / total_horizon_panels) / 2) +
		    (1.0 * PI)
                );

		/* Begin drawing this horizon panel as a GL_QUADS. */
		glBegin(GL_QUADS);
/* No lighting so no normal needed.
		glNormal3d(
		    sin(n_theta),
		    0.0,
		    -cos(n_theta)
		);		    
 */
                glTexCoord2d(1, 0);
                glVertex3d(
		    dx[0], h_min, -dy[0]
		);
                glTexCoord2d(1, 0);
                glVertex3d(
		    dx[1], h_min, -dy[1]
		);
                glTexCoord2d(0, 0);
                glVertex3d(
		    dx[1], h_max, -dy[1]
		);
                glTexCoord2d(0, 0);
                glVertex3d(
		    dx[0], h_max, -dy[0]
		);
		glEnd();
		/* Done drawing this horizon panel. */

		/* Set next panel point to previous panel point. */
		dx[0] = dx[1];
		dy[0] = dy[1];
	    }	/* Begin drawing each horizon panel. */
        }
        glPopMatrix();

	/* Restore depth writing to what it was. */
        StateGLDepthMask(&display->state_gl, depth_mask_flag);

	/* Restore shade model. */
	StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}

/*
 *	Draws celestial objects, like the sun and moon.
 */
static void SARDrawSceneCelestial(
        gw_display_struct *display, sar_scene_struct *scene,
        double visibility_max,		/* In meters. */
	double fovz_um,			/* Field of view unit meters, z axis. */
	double aspect			/* Width / height. */
)
{
	sar_position_struct *pos;
        GLboolean depth_mask_flag;
        GLenum shade_model_mode;


	/* Turn off texturing, depth testing should already be off. */
        SAR_DRAW_TEXTURE_1D_OFF
        SAR_DRAW_TEXTURE_2D_OFF

        /* Use flat shading. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);

        /* No need to write depth when drawing celestial objects, this
         * will allow some objects to be drawn farther away from
         * the celestial objects.
         */
        depth_mask_flag = display->state_gl.depth_mask_flag;
        StateGLDepthMask(&display->state_gl, GL_FALSE);

	/* Get scene's primary light position. */
	pos = &primary_light_pos;
/* Disable this for now, since its rather cpu costly.
	if(pos != NULL)
 */
	if(0)
	{
	    int i, mm;
	    double a[3 * 1], r[3 * 1];

	    /* Calculate global light's relativity to eye coordinates. */
	    a[0] = pos->x;
	    a[1] = pos->y;
	    a[2] = pos->z;

	    for(i = 0, mm = MIN(scene->camera_rotmatrix_count, SAR_CAMERA_ROTMATRIX_MAX);
		i < mm;
		i++
	    )
	    {
		MatrixMulti3Rotate33(
		    a,				/* 3 * 1. */
		    scene->camera_rotmatrix[i],	/* 3 * 3. */
		    r
		);
		memcpy(a, r, (3 * 1) * sizeof(double));
	    }

	    /* Infront of camera? */
	    if(a[1] > 0.0)
	    {
		double half_edgez = fovz_um * a[1] / 2.0;
		double half_edgex;
		double xc, zc;

		half_edgex = half_edgez * aspect;

		if(half_edgez > 0.0)
		    zc = a[2] / half_edgez;
		else
		    zc = 2.0;		/* Off screen. */

		if(half_edgex > 0.0)
		    xc = a[0] / half_edgex;
		else
		    xc = 2.0;		/* Off screen. */

		/* Calculated coefficients xc and zc now represent
		 * light position relative to eye center. If values
		 * are greater than 1.0 or less than -1.0 then it implies
		 * it is out of the view perspective.
		 */
		if((zc <= 1.0) && (zc >= -1.0) &&
                   (xc <= 1.0) && (xc >= -1.0)
		)
		{
/*
printf("\r Result %.0f %.0f %.0f | %.2f %.2f",
 a[0], a[1], a[2], xc, zc
); fflush(stdout);
 */
		}


	    }
	}





        /* Restore depth writing to what it was. */
        StateGLDepthMask(&display->state_gl, depth_mask_flag);
                    
        /* Restore shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}



/*
 *	Redraws scene.
 */
void SARDraw(sar_core_struct *core_ptr)
{
        int i, n, *total;
	double fovz_um, view_aspect;
	Boolean need_clear_color = True;
	gw_display_struct *display;
	sar_scene_struct *scene;
        sar_object_struct ***ptr, *obj_ptr;
	gctl_struct *gc;
	sar_direction_struct *dir;
	sar_position_struct *pos;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_object_helipad_struct *obj_helipad_ptr;
	sar_object_runway_struct *obj_runway_ptr;
	sar_object_human_struct *obj_human_ptr;
	sar_obj_rotor_struct *rotor_ptr;
	sar_obj_landing_gear_struct *lgear_ptr;
	sar_external_fueltank_struct *eft_ptr;
	sar_obj_hoist_struct *hoist_ptr;
	sar_cloud_layer_struct *cloud_layer_ptr;
	sar_cloud_bb_struct *cloud_bb_ptr;
	double far_model_range, distance, distance3d, visibility_max;
	GLfloat light_val[4], mat_val[4];


	/* Get initial pointers to resources. */
	if(core_ptr == NULL)
	     return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

	scene = core_ptr->scene;
	ptr = &core_ptr->object;
	total = &core_ptr->total_objects;
	gc = core_ptr->gctl;

	if(scene == NULL)
	    return;

        /* Draw map in a different function. */
        if(scene->camera_ref == SAR_CAMERA_REF_MAP)
        {
	    SARDrawMap(core_ptr, False, False, -1);
	    return;
        }

	/* Calculate maximum clipping visibility. */
	visibility_max = SFMMilesToMeters(3 + (option.visibility_max * 3));
	/* Make sure its atleast 100 meters. */
	if(visibility_max < 100.0)
	    visibility_max = 100.0;


/* Sets persepective for outside (far) clipping. */
#define DRAW_SET_PERSPECTIVE_OUTSIDE	\
{ \
 glMatrixMode(GL_PROJECTION); \
 glLoadIdentity(); \
 gluPerspective( \
  (GLfloat)SFMRadiansToDegrees(scene->camera_fovz), /* Field of view. */ \
  (GLfloat)view_aspect, \
  (GLfloat)0.19,	/* Was 0.9. */ \
  (GLfloat)visibility_max \
 ); \
 glMatrixMode(GL_MODELVIEW); \
}

/* Sets persepective for inside cockpit (close) clipping. */
#define DRAW_SET_PERSPECTIVE_INSIDE	\
{ \
 glMatrixMode(GL_PROJECTION); \
 glLoadIdentity(); \
 gluPerspective( \
  SFMRadiansToDegrees(scene->camera_fovz), /* Field of view. */ \
  (GLfloat)view_aspect, \
  (GLfloat)0.1,			/* Very close near clip. */ \
  (GLfloat)visibility_max	/* Reasonably far far clip. */ \
 ); \
 glMatrixMode(GL_MODELVIEW); \
}

        /* Reset viewport and matrixes. */
	SARReshapeCB(display, -1, -1, -1, -1);
	if(display->height > 0)
	    view_aspect = (double)display->width / (double)display->height;
	else
	    view_aspect = (800 / 600);
	view_aspect += display->aspect_offset;

        /* Calculate field of view in unit meters about z axis. */
        fovz_um = 1.0 * tan(scene->camera_fovz / 2.0) * 2.0;

	DRAW_SET_PERSPECTIVE_OUTSIDE
        glLoadIdentity();


	/* Set camera location, this will set camera position variables
	 * for this source file.
	 */
	SARSetCamera(scene, ptr, total);

	/* Reset other variables. */
	player_wheel_brakes = 0;
	player_air_brakes = 0;
	player_overspeed = 0;
	flight_model_type = -1;

	/* Get pointer to lowest (first) cloud layer. */
	i = scene->total_cloud_layers;
	if(i > 0)
	{
	    lowest_cloud_layer_ptr = scene->cloud_layer[0];
	    highest_cloud_layer_ptr = scene->cloud_layer[i - 1];
	}
	else
	{
	    lowest_cloud_layer_ptr = NULL;
	    highest_cloud_layer_ptr = NULL;
	}

        /* Atmosphere enabled? */
        if(option.atmosphere)
        {
	    /* Atmosphere is enabled, set up atmosphere and sky color. */
            GLfloat fog_color[4];
            double g;

	    /* Get average gamme value g of scene light color. */
            g = (scene->light_color.r + scene->light_color.g +
                scene->light_color.b) / 3;

	    /* Highest cloud layer exists? */
            if(highest_cloud_layer_ptr != NULL)
            {
		/* Camera below highest cloud layer? */
                if(camera_pos.z < highest_cloud_layer_ptr->z)
                {
                    /* Below the highest cloud layer. */

		    /* Check if textured clouds is enabled, if so then do
		     * not clear the color buffer since drawing the cloud
		     * layers, horizon, and ground base will ensure that
		     * the frame buffer is effectivly cleared.
		     */
		    if(option.textured_clouds)
			need_clear_color = False;


                    sky_color[0] = (GLfloat)(
			scene->sky_nominal_color.r * pow(g, 0.5)
		    );
                    sky_color[1] = (GLfloat)(
			scene->sky_nominal_color.g * g
		    );
                    sky_color[2] = (GLfloat)(
			scene->sky_nominal_color.b * g
		    );
                }
                else
                {
                    /* Above the highest cloud layer. */
                    sky_color[0] = (GLfloat)(
			scene->sky_nominal_color.r * 0.15 * g
		    );
                    sky_color[1] = (GLfloat)(
			scene->sky_nominal_color.g * 0.15 * g
		    );
                    sky_color[2] = (GLfloat)(
			scene->sky_nominal_color.b * 0.15 * g
		    );
                }
            }
	    else
	    {
                sky_color[0] = (GLfloat)(
                    scene->sky_nominal_color.r * pow(g, 0.5)
                );
                sky_color[1] = (GLfloat)(
                    scene->sky_nominal_color.g * g
                );
                sky_color[2] = (GLfloat)(
                    scene->sky_nominal_color.b * g
                );
            }

	    /* Set atmosphere color. */
            atmosphere_color[0] = (GLfloat)MIN(sky_color[0] * 1.1, 1.0);
            atmosphere_color[1] = (GLfloat)MIN(sky_color[1] * 1.1, 1.0);
            atmosphere_color[2] = (GLfloat)MIN(sky_color[2] * 1.1, 1.0);

	    /* Set fog color. */
            fog_color[0] = atmosphere_color[0];
            fog_color[1] = atmosphere_color[1];
            fog_color[2] = atmosphere_color[2];
            fog_color[3] = 1.0;

	    /* Enable fog and set fog values. */
	    StateGLEnable(&display->state_gl, GL_FOG);
            glFogi(GL_FOG_MODE, GL_EXP);
            glFogfv(GL_FOG_COLOR, fog_color);
            glFogf(GL_FOG_DENSITY,
		(GLfloat)(
		    (1.0 / visibility_max) *
		    (8.0 * scene->atmosphere_density_coeff)
		)
	    );
	    glHint(GL_FOG_HINT, GL_DONT_CARE);
            glFogf(GL_FOG_START,
                (GLfloat)(
		    visibility_max *
		    (1.0 - scene->atmosphere_dist_coeff)
                )
            );
            glFogf(GL_FOG_END, (GLfloat)visibility_max);

	    glClearColor(
                (GLclampf)atmosphere_color[0],
                (GLclampf)atmosphere_color[1],
                (GLclampf)atmosphere_color[2],
                (GLclampf)0.0
            );
        }
	else
	{
	    /* Atmosphere is disabled, just set sky color. */

	    /* Disable fog. */
            StateGLDisable(&display->state_gl, GL_FOG);

            /* Highest cloud layer available? */
            if(highest_cloud_layer_ptr != NULL)
            {
		/* Camera below the highest cloud layer? */
                if(camera_pos.z < highest_cloud_layer_ptr->z)
                {
                    /* Below the highest cloud layer, check if textured
		     * clouds are enabled, if so then do not clear the
                     * color buffer.
                     */
                    if(option.textured_clouds)
                        need_clear_color = False;

                    sky_color[0] = (GLfloat)(
                        scene->sky_nominal_color.r * scene->light_color.r
                    );
                    sky_color[1] = (GLfloat)(
                        scene->sky_nominal_color.g * scene->light_color.g
                    );
                    sky_color[2] = (GLfloat)(
                        scene->sky_nominal_color.b * scene->light_color.b
                    );
                }
                else
                {
                    /* Above the highest cloud layer. */
                    sky_color[0] = (GLfloat)(
                        scene->sky_nominal_color.r * 0.15 * scene->light_color.r
                    );
                    sky_color[1] = (GLfloat)(
                        scene->sky_nominal_color.g * 0.15 * scene->light_color.g
                    );
                    sky_color[2] = (GLfloat)(
                        scene->sky_nominal_color.b * 0.15 * scene->light_color.b
                    );
                }
            }
            else
            {
                sky_color[0] = (GLfloat)(
                    scene->sky_nominal_color.r * scene->light_color.r
                );
                sky_color[1] = (GLfloat)(
                    scene->sky_nominal_color.g * scene->light_color.g
                );
                sky_color[2] = (GLfloat)(
                    scene->sky_nominal_color.b * scene->light_color.b
                );
            }

            glClearColor(
                (GLclampf)sky_color[0],
                (GLclampf)sky_color[1],
                (GLclampf)sky_color[2],
                (GLclampf)0.0
            );
 	}

        /* Clear back buffer. */
	glClearDepth((GLclampd)1.0);
	/* Clear color set above in fog check. */
        glClear(
            ((need_clear_color) ? GL_COLOR_BUFFER_BIT : 0) |
            GL_DEPTH_BUFFER_BIT
        );


	/* Disable lighting, texture, and depth test for drawing
	 * foundations and clouds.
	 */
	StateGLDisable(&display->state_gl, GL_LIGHTING);
	StateGLDisable(&display->state_gl, GL_LIGHT0);
	/* GL_LIGHT1 will be turned on or off later. */

	SAR_DRAW_DEPTH_TEST_OFF

	SAR_DRAW_TEXTURE_1D_OFF

	SAR_DRAW_TEXTURE_2D_OFF


	/* Turn off depth mask before drawing ground base. */
	StateGLDepthMask(&display->state_gl, GL_FALSE);

	/* Draw ground base. */
        SARDrawSceneFoundations(display, scene, view_aspect);


	/* Turn back on depth mask for drawing clouds. */
	StateGLDepthMask(&display->state_gl, GL_TRUE);

        /* Draw cloud layers and textured horizon? */
        if(option.textured_clouds)
        {
	    /* Yes, draw cloud layers and horizon. */

	    /* Horizon. */
            if(highest_cloud_layer_ptr != NULL)
            {
                if(camera_pos.z < highest_cloud_layer_ptr->z)
                    SARDrawSceneHorizon(display, scene, visibility_max);
            }
            else 
            {   
                SARDrawSceneHorizon(display, scene, visibility_max);
            }

	    SAR_DRAW_TEXTURE_1D_OFF
	    SAR_DRAW_TEXTURE_2D_ON

	    /* Note, cloud layers should be allocated lowest to
	     * highest.
	     */

	    /* Go through cloud layers above camera. */
            for(i = 0; i < scene->total_cloud_layers; i++)
            {
                cloud_layer_ptr = scene->cloud_layer[i];
                if(cloud_layer_ptr == NULL)
                    continue;

                /* Above camera? */
                if(cloud_layer_ptr->z > camera_pos.z)
                {
		    /* Draw this cloud later and that's it. */
                    SARDrawCloudLayer(scene, cloud_layer_ptr, True);
                    break;
                }
            }
	    /* Go through cloud layers below camera. */
            for(i = scene->total_cloud_layers - 1; i >= 0; i--)
            {
		cloud_layer_ptr = scene->cloud_layer[i];
                if(cloud_layer_ptr == NULL)
                    continue;

		/* Below camera? */
		if(cloud_layer_ptr->z < camera_pos.z)
		{
		    /* Draw this cloud later and that's it. */
                    SARDrawCloudLayer(scene, cloud_layer_ptr, False);
		    break;
		}
            }
        }
	else
	{
	    /* Do not draw lower cloud layers or horizon, however we
	     * still must draw the highest cloud layer if it
	     * is defined and the camera is above it.
	     */
	    if(highest_cloud_layer_ptr != NULL)
            {
                if(camera_pos.z > highest_cloud_layer_ptr->z)
		{
		    SAR_DRAW_TEXTURE_2D_ON
                    SARDrawCloudLayer(scene, highest_cloud_layer_ptr, False);
		}
            }
	}

	/* Draw `billboard' cloud objects. */
	SAR_DRAW_DEPTH_TEST_ON
	if(option.textured_clouds)
	{
            for(i = 0; i < scene->total_cloud_bbs; i++)
            {
                cloud_bb_ptr = scene->cloud_bb[i];
                if(cloud_bb_ptr == NULL)
                    continue;

		SARDrawCloudBB(
		    display, scene, cloud_bb_ptr
		);
	    }
	}

        /* Calculate and set up primary light position and store
         * position in primary_light_pos. Remember scene structure's
         * primary light position is only an offset and needs to be
         * applied relative to the calculated camera_pos.
         */  
        primary_light_pos.x = camera_pos.x + scene->light_pos.x;
        primary_light_pos.y = camera_pos.x + scene->light_pos.y;
        primary_light_pos.z = camera_pos.x + scene->light_pos.z;

	/* No depth testing for drawing celestial objects. */
	SAR_DRAW_DEPTH_TEST_OFF

	/* Draw celestial objects. */
	SARDrawSceneCelestial(
	    display, scene,
	    visibility_max, fovz_um, view_aspect
	);


	/* Enable lighting and turn on global GL_LIGHT0. */
        StateGLEnable(&display->state_gl, GL_LIGHTING);
	StateGLEnable(&display->state_gl, GL_LIGHT0);
	/* Leave GL_LIGHT1 (search light) alone for now, it will be set
	 * later.
	 */

	/* Reset material values. */
	StateGLDisable(&display->state_gl, GL_COLOR_MATERIAL);
	mat_val[0] = 0.2;
	mat_val[1] = 0.2;
	mat_val[2] = 0.2;
	mat_val[3] = 1.0;
	glMaterialfv(GL_FRONT, GL_AMBIENT, &(mat_val[0]));
        mat_val[0] = 0.8;
        mat_val[1] = 0.8;  
        mat_val[2] = 0.8;
        mat_val[3] = 1.0;
        glMaterialfv(GL_FRONT, GL_DIFFUSE, &(mat_val[0]));
        mat_val[0] = 0.0;
        mat_val[1] = 0.0;  
        mat_val[2] = 0.0;
        mat_val[3] = 1.0;
        glMaterialfv(GL_FRONT, GL_SPECULAR, &(mat_val[0]));
        mat_val[0] = 0.0;
        glMaterialfv(GL_FRONT, GL_SHININESS, &(mat_val[0]));
        mat_val[0] = 0.0;
        mat_val[1] = 0.0;
        mat_val[2] = 0.0;
        mat_val[3] = 1.0;
        glMaterialfv(GL_FRONT, GL_EMISSION, &(mat_val[0]));

	/* Enable color material for front face only with lighting
	 * of ambience and diffusion.
	 */
	StateGLColorMaterial(
            &display->state_gl, GL_FRONT, GL_AMBIENT_AND_DIFFUSE
        );
	StateGLEnable(&display->state_gl, GL_COLOR_MATERIAL);


	/* Set light 0 as the scene's global primary light. */
	light_val[0] = (GLfloat)primary_light_pos.x;
	light_val[1] = (GLfloat)primary_light_pos.z;
	light_val[2] = (GLfloat)-primary_light_pos.y;
	light_val[3] = 0.0;		/* Directional light. */
        glLightfv(GL_LIGHT0, GL_POSITION, &(light_val[0]));

	light_val[0] = (GLfloat)(0.40 * scene->light_color.r);
        light_val[1] = (GLfloat)(0.40 * scene->light_color.g);
        light_val[2] = (GLfloat)(0.40 * scene->light_color.b);
        light_val[3] = 1.0;
	glLightfv(GL_LIGHT0, GL_AMBIENT, &(light_val[0]));

	/* Set diffuse as the scene's lighting, increases it just a
	 * bit to make it last brighter around noon.
	 */
        light_val[0] = (GLfloat)CLIP(scene->light_color.r + 0.1, 0.0, 1.0);
        light_val[1] = (GLfloat)CLIP(scene->light_color.g + 0.1, 0.0, 1.0);
        light_val[2] = (GLfloat)CLIP(scene->light_color.b + 0.1, 0.0, 1.0);
        light_val[3] = 1.0;
        glLightfv(GL_LIGHT0, GL_DIFFUSE, &(light_val[0]));

	light_val[0] = 1.0;
	light_val[1] = 1.0;
        light_val[2] = 1.0;
        light_val[3] = 1.0;
	glLightfv(GL_LIGHT0, GL_SPECULAR, &(light_val[0]));

        light_val[0] = 1.0;
        glLightfv(GL_LIGHT0, GL_CONSTANT_ATTENUATION, &(light_val[0]));

        light_val[0] = 0.0;
        glLightfv(GL_LIGHT0, GL_LINEAR_ATTENUATION, &(light_val[0]));

        light_val[0] = 0.0;
        glLightfv(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, &(light_val[0]));


	/* Turn on/off and set up texturing. */
        if(option.textured_objects)
	{
	    StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	    StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);

	    SAR_DRAW_TEXTURE_2D_ON
	}
        else
	{
	    SAR_DRAW_TEXTURE_2D_OFF
	}
	/* Don't use 1D texture. */
	SAR_DRAW_TEXTURE_1D_OFF


/* Macro to enable polygon offsetting if obj_ptr->flags specifies polygon
 * offsetting. Requires that variables display, option, and obj_ptr be valid.
 */
#define ENABLE_POLYGON_OFFSET_AS_NEEDED	\
{ \
 if(obj_ptr->flags & SAR_OBJ_FLAG_POLYGON_OFFSET) \
 { \
  if(!(obj_ptr->flags & SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH)) \
   StateGLDepthMask(&display->state_gl, GL_FALSE); \
  StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL); \
  StateGLPolygonOffset( \
   &display->state_gl, \
   (GLfloat)option.gl_polygon_offset_factor, -1.0 \
  ); \
 } \
 else if(obj_ptr->flags & SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE) \
 { \
  if(!(obj_ptr->flags & SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH)) \
   StateGLDepthMask(&display->state_gl, GL_FALSE); \
  StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL); \
  StateGLPolygonOffset( \
   &display->state_gl, \
   -(GLfloat)option.gl_polygon_offset_factor, 1.0 \
  ); \
 } \
}

/* Disables polygon offsetting if obj_ptr->flags specifies polygon
 * offsetting. Requires that variables display, option, and obj_ptr be valid.
 */
#define DISABLE_POLYGON_OFFSET_AS_NEEDED	\
{ \
 if(obj_ptr->flags & (SAR_OBJ_FLAG_POLYGON_OFFSET | \
                      SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE) \
 ) \
 { \
  StateGLDepthMask(&display->state_gl, GL_TRUE); \
  StateGLDisable(&display->state_gl, GL_POLYGON_OFFSET_FILL); \
 } \
}

        /* Begin drawing each object. */
        for(i = 0; i < core_ptr->total_objects; i++)
        {
            obj_ptr = core_ptr->object[i];
            if(obj_ptr == NULL)
                continue;

	    /* Get pointer to position of object. */
            pos = &obj_ptr->pos;

            /* Camera above and object below lowest cloud layer? */
            if(lowest_cloud_layer_ptr != NULL)
            {
                if((camera_pos.z > lowest_cloud_layer_ptr->z) &&
                   (pos->z < lowest_cloud_layer_ptr->z)
                )
                    continue;
            }

            /* Check if object is to have depth testing or not. */
            if(obj_ptr->flags & SAR_OBJ_FLAG_NO_DEPTH_TEST)
            {
		SAR_DRAW_DEPTH_TEST_OFF
		StateGLDepthMask(&display->state_gl, GL_FALSE);
            }
            else
            {
		SAR_DRAW_DEPTH_TEST_ON
		StateGLDepthMask(&display->state_gl, GL_TRUE);
            }

	    /* Set object's shade model. */
	    if(obj_ptr->flags & SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH)
	    {
		StateGLShadeModel(&display->state_gl, GL_SMOOTH);
	    }
	    else
	    {
		StateGLShadeModel(&display->state_gl, GL_FLAT);
	    }

	    /* Draw by object type. */
	    switch(*((int *)obj_ptr))
	    {
              /* ************************************************** */
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr == NULL)
		    break;

		/* Get position and check if in range with camera. */
		distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
		if(distance > obj_ptr->range)
		{
		    /* Out of range with camera. */

		    /* Mute sounds since out of camera range. */
		    SARSoundEngineVolumeMute(
			core_ptr->recorder,
			obj_aircraft_ptr->engine_inside_sndplay,
			obj_aircraft_ptr->engine_outside_sndplay
		    );
		    break;
		}

		/* Get far model range, if no far visual model available
		 * then set far_model_range as the regular range.
		 */
		far_model_range = ((obj_ptr->visual_model_far == NULL) ?
		    obj_ptr->range : obj_ptr->range_far
		);

		/* Get direction. */
		dir = &obj_ptr->dir;

		/* Calculate 3D distance. */
		distance3d = hypot(distance, pos->z - camera_pos.z);

		/* Get pointer to hoist. */
		hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);

/* Draws hoist, hoist_ptr must be valid. It will be checked if rope is
 * out which implies basket is out. Will modify pos and dir pointer 
 * values.
 */
#define DO_DRAW_HOIST_AS_NEEDED		\
{ \
 if((hoist_ptr == NULL) ? 0 : (hoist_ptr->rope_cur_vis > 0.0)) \
 { \
  sar_position_struct *basket_pos = &hoist_ptr->pos; \
  sar_direction_struct *basket_dir = &hoist_ptr->dir; \
  glPushMatrix(); \
  { \
   /* Translate and rotate heading. */ \
   glTranslated(basket_pos->x, basket_pos->z, -basket_pos->y); \
   glRotated( \
    -SFMRadiansToDegrees(basket_dir->heading), 0.0, 1.0, 0.0 \
   ); \
 \
   SARDrawHoistDeployment(display, scene, obj_ptr, hoist_ptr); \
  } \
  glPopMatrix(); \
 } \
}

		/* Draw far model if current distance is in the far
		 * model's range.
		 */
		if(distance > far_model_range)
		{
		    /* Draw only far model. */
                    glPushMatrix();
                    {
                        /* Translate and rotate heading. */
                        glTranslated(
                            pos->x, pos->z, -pos->y
                        );
                        glRotated(
                            -SFMRadiansToDegrees(dir->heading),
                            0.0, 1.0, 0.0
                        );
                        glRotated(
                            -SFMRadiansToDegrees(dir->pitch),
                            1.0, 0.0, 0.0
                        );
                        glRotated(
                            -SFMRadiansToDegrees(dir->bank),
                            0.0, 0.0, 1.0
                        );

			ENABLE_POLYGON_OFFSET_AS_NEEDED

			/* Draw far model during day time only? */
			if((obj_ptr->flags & SAR_OBJ_FLAG_FAR_MODEL_DAY_ONLY) ?
			   (scene->tod_code == SAR_SCENE_TOD_DAY) : 1
			)
			{
			    /* Draw far visual model. */
                            SARVisualModelCallList(obj_ptr->visual_model_far);
			}

			DISABLE_POLYGON_OFFSET_AS_NEEDED

			/* Draw lights. */
                        SARDrawLights(
                            display, scene,
                            obj_ptr->light, obj_ptr->total_lights
                        );

                        POST_CALLLIST_RESET_STATES
                    }
                    glPopMatrix();
		}
		else
		{
		    /* Draw closeup models. */
		    DO_DRAW_HOIST_AS_NEEDED

		    glPushMatrix();
		    {
			/* Translate and rotate heading. */
			glTranslated(pos->x, pos->z, -pos->y);
			glRotated(
			    -SFMRadiansToDegrees(dir->heading),
			    0.0, 1.0, 0.0
			);

			/* Draw shadow (before applying pitch and bank
			 * matrix rotation). Polygon offsetting will be
			 * applied as needed in this call.
			 */
			SARDrawObjectShadow(display, scene, obj_ptr);

			/* Rotate pitch and bank. */
			glRotated(
			    -SFMRadiansToDegrees(dir->pitch),
			    1.0, 0.0, 0.0
			);
			glRotated(
			    -SFMRadiansToDegrees(dir->bank),
			    0.0, 0.0, 1.0
			);

			ENABLE_POLYGON_OFFSET_AS_NEEDED

			/* Check if this is the player object. */
			if((obj_ptr == scene->player_obj_ptr) &&
			   (gc != NULL)
			)
			{
			    /* Update player wheel brakes state. */
			    player_wheel_brakes = gc->wheel_brakes_state;
			    if((player_wheel_brakes == 0) &&
			       (gc->wheel_brakes_coeff > 0.0)
			    )
				player_wheel_brakes = 1;

			    /* Update player air brakes state. */
			    player_air_brakes = gc->air_brakes_state;
                            if((player_air_brakes == 0) &&
                               (gc->air_brakes_coeff > 0.0)
                            )
                                player_air_brakes = 1;

			    /* Update player flight model. */
			    flight_model_type = obj_aircraft_ptr->flight_model_type;

			    /* Update player over speed state. */
			    if(obj_aircraft_ptr->speed > obj_aircraft_ptr->speed_max_expected)
			    {
				player_overspeed = 1;
			    }


			    /* Adjust engine volume. */
			    SARSoundAdjustEngineVolume(
				core_ptr->recorder,
				obj_aircraft_ptr->engine_inside_sndplay,
				obj_aircraft_ptr->engine_outside_sndplay,
				(scene->camera_ref == SAR_CAMERA_REF_COCKPIT),
                                obj_aircraft_ptr->engine_state,
                                obj_aircraft_ptr->throttle,
                                distance3d
                            );

			    /* Adjust spot light. */
			    SARDrawSetSpotLight(display, scene, obj_ptr);

			    /* Camera referance inside cockpit? */
                            if(scene->camera_ref == SAR_CAMERA_REF_COCKPIT)
			    {
				DRAW_SET_PERSPECTIVE_INSIDE
				SARDrawObjectCockpit(scene, obj_ptr);
				DRAW_SET_PERSPECTIVE_OUTSIDE

				DISABLE_POLYGON_OFFSET_AS_NEEDED

				/* Pop one matrix and break (do not
				 * draw the rest of the aircraft).
				 */
				glPopMatrix();
				break;
			    }
			}
			else
			{
			    /* Not player object. */

			    /* Adjust engine volume. */
			    SARSoundAdjustEngineVolume(
                                core_ptr->recorder,
                                obj_aircraft_ptr->engine_inside_sndplay,
                                obj_aircraft_ptr->engine_outside_sndplay,
                                0,
                                obj_aircraft_ptr->engine_state,
                                obj_aircraft_ptr->throttle,
                                distance3d
                            );
			}

			/* Draw standard day time model? */
		        if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAY_MODEL) ?
		           (scene->tod_code == SAR_SCENE_TOD_DAY) : 1
		        )
		        {
			    SARVisualModelCallList(obj_ptr->visual_model);
		        }
                        /* Draw standard dawn time model? */
                        if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAWN_MODEL) ?
                            (scene->tod_code == SAR_SCENE_TOD_DAWN) : 1
                        )
                        {
                            /* Disable GL_LIGHTING when drawing dawn time model. */
                            StateGLDisable(&display->state_gl, GL_LIGHTING);
                            SARVisualModelCallList(obj_ptr->visual_model_dawn);
                            StateGLEnable(&display->state_gl, GL_LIGHTING);
                        }
                        /* Draw standard dusk time model? */
                        if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DUSK_MODEL) ?
                            (scene->tod_code == SAR_SCENE_TOD_DUSK) : 1
                        )
			{
			    /* Disable GL_LIGHTING when drawing dusk time model. */
                            StateGLDisable(&display->state_gl, GL_LIGHTING);
                            SARVisualModelCallList(obj_ptr->visual_model_dusk);
			    StateGLEnable(&display->state_gl, GL_LIGHTING);
			}
                        /* Draw standard night time model? */
			if(obj_ptr->visual_model_night != NULL)
			{
			    Boolean draw_night_model = False;
			    if(obj_ptr->flags & SAR_OBJ_FLAG_HIDE_NIGHT_MODEL)
			    {
			        switch(scene->tod_code)
			        {
			          case SAR_SCENE_TOD_NIGHT:
				    draw_night_model = True;
				    break;
			          case SAR_SCENE_TOD_DAWN:
				    draw_night_model = (obj_ptr->flags &
				        SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN);
				    break;
			          case SAR_SCENE_TOD_DUSK:
                                    draw_night_model = (obj_ptr->flags & 
                                        SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK);
                                    break;
			        }
			    }
			    else
			    {
			        draw_night_model = True;
			    }
			    if(draw_night_model)
			    {
			        /* Disable GL_LIGHTING when drawing night time model. */
			        StateGLDisable(&display->state_gl, GL_LIGHTING);
			        SARVisualModelCallList(obj_ptr->visual_model_night);
			        StateGLEnable(&display->state_gl, GL_LIGHTING);
			    }
		        }

		        /* Draw landing gears. */
		        for(n = 0; n < obj_aircraft_ptr->total_landing_gears; n++)
		        {
			    lgear_ptr = obj_aircraft_ptr->landing_gear[n];
			    if(lgear_ptr == NULL)
			        continue;

			    SARDrawLandingGear(
			        scene, obj_ptr, lgear_ptr
			    );
		        }

		        /* Draw external reserved fuel tanks. */
                        for(n = 0; n < obj_aircraft_ptr->total_external_fueltanks; n++)
                        {
                            eft_ptr = obj_aircraft_ptr->external_fueltank[n];
                            if(eft_ptr == NULL)
                                continue;

                            SARDrawFuelTank(
                                scene, obj_ptr, eft_ptr
                            );
                        }

		        /* Draw rescue door. */
		        SARDrawDoor(
			    scene, obj_ptr,
			    &obj_aircraft_ptr->rescue_door
		        );

                        /* Draw lights. */
                        SARDrawLights(  
                            display, scene,
                            obj_ptr->light, obj_ptr->total_lights  
                        );

                        /* Draw rotors/propellars. */
                        for(n = 0; n < obj_aircraft_ptr->total_rotors; n++)
                        {
                            rotor_ptr = obj_aircraft_ptr->rotor[n];
                            if(rotor_ptr == NULL)
                                continue;

                            SARDrawRotor(
                                display, scene, obj_ptr, rotor_ptr,
                                obj_aircraft_ptr->throttle
                            );
                        }
		        POST_CALLLIST_RESET_STATES

			DISABLE_POLYGON_OFFSET_AS_NEEDED
		    }
                    glPopMatrix();
		}
		break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_GROUND:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

		    ENABLE_POLYGON_OFFSET_AS_NEEDED

                    /* Draw standard day time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAY_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DAY) : 1
                    )
                    {
                        SARVisualModelCallList(obj_ptr->visual_model);
                    }
                    /* Draw standard dawn time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAWN_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DAWN) : 1
                    )
                    {
                        /* Disable GL_LIGHTING when drawing dawn time model. */
                        StateGLDisable(&display->state_gl, GL_LIGHTING);
                        SARVisualModelCallList(obj_ptr->visual_model_dawn);
                        StateGLEnable(&display->state_gl, GL_LIGHTING);
                    }
                    /* Draw standard dusk time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DUSK_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DUSK) : 1
                    )
                    {
                        /* Disable GL_LIGHTING when drawing dusk time model. */
                        StateGLDisable(&display->state_gl, GL_LIGHTING);
                        SARVisualModelCallList(obj_ptr->visual_model_dusk);
                        StateGLEnable(&display->state_gl, GL_LIGHTING);
                    }
                    /* Draw standard night time model? */
                    if(obj_ptr->visual_model_night != NULL)
                    {
                        Boolean draw_night_model = False;
                        if(obj_ptr->flags & SAR_OBJ_FLAG_HIDE_NIGHT_MODEL)
                        {
                            switch(scene->tod_code)
                            {
                              case SAR_SCENE_TOD_NIGHT:
                                draw_night_model = True;
                                break;
                              case SAR_SCENE_TOD_DAWN:
                                draw_night_model = (obj_ptr->flags &
                                    SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN);
                                break;
                              case SAR_SCENE_TOD_DUSK:
                                draw_night_model = (obj_ptr->flags &
                                    SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK);
                                break;
                            }
                        }
                        else
                        {    
                            draw_night_model = True;
                        }
                        if(draw_night_model)
                        {
                            /* Disable GL_LIGHTING when drawing night time model. */
                            StateGLDisable(&display->state_gl, GL_LIGHTING);
                            SARVisualModelCallList(obj_ptr->visual_model_night);   
                            StateGLEnable(&display->state_gl, GL_LIGHTING);
                        }
                    }

		    POST_CALLLIST_RESET_STATES

		    DISABLE_POLYGON_OFFSET_AS_NEEDED

                    /* Draw lights. */
                    SARDrawLights(
			display, scene,
			obj_ptr->light, obj_ptr->total_lights
		    );
                }
                glPopMatrix();
                break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_RUNWAY:
                obj_runway_ptr = (sar_object_runway_struct *)obj_ptr->data;
                if(obj_runway_ptr == NULL)
                    break;

                /* Skip if camera is under helipad. */
                if(camera_pos.z < pos->z)
                    break;

                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Calculate 3D distance. */
                distance3d = hypot(distance, pos->z - camera_pos.z);
        
                /* Get direction. */
                dir = &obj_ptr->dir;
        
                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

                    SARDrawRunway(
			display, scene,
			obj_ptr, obj_runway_ptr, distance3d
		    );
                }
                glPopMatrix();
		POST_CALLLIST_RESET_STATES
                break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_HELIPAD:
                obj_helipad_ptr = (sar_object_helipad_struct *)obj_ptr->data;
                if(obj_helipad_ptr == NULL)
                    break;

		/* Skip if camera is under helipad. */
/* Don't check this, what about recessed helipads?
		if(camera_pos.z < pos->z)
		    break;
 */
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Calculate 3D distance. */
                distance3d = hypot(distance, pos->z - camera_pos.z);

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

                    SARDrawHelipad(
			display, scene,
			obj_ptr, obj_helipad_ptr, distance3d
		    );
                }
                glPopMatrix();
		POST_CALLLIST_RESET_STATES
                break;

	      /* ************************************************** */
	      case SAR_OBJ_TYPE_HUMAN:
		obj_human_ptr = (sar_object_human_struct *)obj_ptr->data;
		if(obj_human_ptr == NULL)
		    break;

                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate heading. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

		    SARDrawHuman(display, scene, obj_ptr, obj_human_ptr);
		}
		glPopMatrix();
		POST_CALLLIST_RESET_STATES
		break;

              /* ************************************************** */
              case SAR_OBJ_TYPE_SMOKE:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

		/* Do not push matrix to translate or rotate, the call
		 * to SARDrawSmoke() will do that for us.
		 */
		if(obj_ptr->data != NULL)
		    SARDrawSmoke(
			display, scene, obj_ptr,
			(sar_object_smoke_struct *)obj_ptr->data
		    );
		POST_CALLLIST_RESET_STATES
                break;

	      /* ************************************************** */
	      case SAR_OBJ_TYPE_FIRE:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    sar_direction_struct fire_dir;

                    /* Translate. */
                    glTranslated(pos->x, pos->z, -pos->y);

                    /* Adjust heading to camera heading. */
                    SARDrawGetDirFromPos(
                        &camera_pos, pos, &fire_dir
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            fire_dir.heading
                        ), 0.0, 1.0, 0.0
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            fire_dir.pitch
                        ), 1.0, 0.0, 0.0
                    );

                    if(obj_ptr->data != NULL)
                        SARDrawFire(
                            display, scene,
                            obj_ptr,
                            (sar_object_fire_struct *)obj_ptr->data
                        );
                }
                glPopMatrix();
                POST_CALLLIST_RESET_STATES
		break;

              /* ************************************************** */
              case SAR_OBJ_TYPE_EXPLOSION:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    sar_direction_struct explosion_dir;

                    /* Translate. */
                    glTranslated(pos->x, pos->z, -pos->y);

                    /* Adjust heading to camera heading. */
                    SARDrawGetDirFromPos(
                        &camera_pos, pos, &explosion_dir
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            explosion_dir.heading
                        ), 0.0, 1.0, 0.0
                    );
                    glRotated(
                        -SFMRadiansToDegrees(
                            explosion_dir.pitch
                        ), 1.0, 0.0, 0.0   
                    );

		    if(obj_ptr->data != NULL)
			SARDrawExplosion(
			    display, scene,
			    obj_ptr,
			    (sar_object_explosion_struct *)obj_ptr->data
			);
		}
		glPopMatrix();
		POST_CALLLIST_RESET_STATES
		break;

              /* ************************************************** */
              case SAR_OBJ_TYPE_PREMODELED:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

		    SARDrawPremodeled(
			display, scene, obj_ptr, obj_ptr->data,
			&camera_pos, &camera_dir, False
		    );
                }
                glPopMatrix();
		POST_CALLLIST_RESET_STATES
                break;

	      /* **************************************************** */
	      case SAR_OBJ_TYPE_WATERCRAFT:
	      case SAR_OBJ_TYPE_AUTOMOBILE:
	      case SAR_OBJ_TYPE_FUELTANK:
	      case SAR_OBJ_TYPE_STATIC:
                /* Get position and check if in range with camera. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

		    ENABLE_POLYGON_OFFSET_AS_NEEDED

		    /* Draw standard day time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAY_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DAY) : 1
                    )
                    {
                        SARVisualModelCallList(obj_ptr->visual_model);
                    }
                    /* Draw standard dawn time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DAWN_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DAWN) : 1
                    )
                    {
                        /* Disable GL_LIGHTING when drawing dawn time model. */
                        StateGLDisable(&display->state_gl, GL_LIGHTING);
                        SARVisualModelCallList(obj_ptr->visual_model_dawn);
                        StateGLEnable(&display->state_gl, GL_LIGHTING);
                    }
                    /* Draw standard dusk time model? */
                    if((obj_ptr->flags & SAR_OBJ_FLAG_HIDE_DUSK_MODEL) ?
                        (scene->tod_code == SAR_SCENE_TOD_DUSK) : 1
                    )
                    {
                        /* Disable GL_LIGHTING when drawing dusk time model. */
                        StateGLDisable(&display->state_gl, GL_LIGHTING);
                        SARVisualModelCallList(obj_ptr->visual_model_dusk);
                        StateGLEnable(&display->state_gl, GL_LIGHTING);
                    }
                    /* Draw standard night time model? */
                    if(obj_ptr->visual_model_night != NULL)
                    {
                        Boolean draw_night_model = False;
                        if(obj_ptr->flags & SAR_OBJ_FLAG_HIDE_NIGHT_MODEL)
                        {
                            switch(scene->tod_code)
                            {
                              case SAR_SCENE_TOD_NIGHT:
                                draw_night_model = True;
                                break;
                              case SAR_SCENE_TOD_DAWN:
                                draw_night_model = (obj_ptr->flags &
                                    SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN);
                                break;
                              case SAR_SCENE_TOD_DUSK:
                                draw_night_model = (obj_ptr->flags &
                                    SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK);
                                break;
                            }
                        }
                        else
                        {    
                            draw_night_model = True;
                        }
                        if(draw_night_model)
                        {
                            /* Disable GL_LIGHTING when drawing night time model. */
                            StateGLDisable(&display->state_gl, GL_LIGHTING);
                            SARVisualModelCallList(obj_ptr->visual_model_night);   
                            StateGLEnable(&display->state_gl, GL_LIGHTING);
                        }
                    }

		    POST_CALLLIST_RESET_STATES

		    DISABLE_POLYGON_OFFSET_AS_NEEDED

		    /* Draw lights. */
		    SARDrawLights(
			display, scene,
			obj_ptr->light, obj_ptr->total_lights
		    );
                }
                glPopMatrix();
		break;
            }
        }


	StateGLDisable(&display->state_gl, GL_FOG);

	/* Go to 2D. */
	GWOrtho2D(display);

	StateGLDisable(&display->state_gl, GL_COLOR_MATERIAL);
	StateGLDisable(&display->state_gl, GL_LIGHTING);
	StateGLDisable(&display->state_gl, GL_LIGHT0);
	SAR_DRAW_TEXTURE_1D_OFF
	SAR_DRAW_TEXTURE_2D_OFF
	StateGLDisable(&display->state_gl, GL_ALPHA_TEST);

	/* Draw in cockpit HUD. */
	if(scene->camera_ref == SAR_CAMERA_REF_COCKPIT)
	{
	    SARDrawHUD(
		display,
		scene,
		ptr, total,
		view_aspect
	    );
	}
	else
	{
            SARDrawSetColor(&option.message_color);
	    GWSetFont(display, option.message_font);
	    SARDrawOutsideAttitude(
                display,
                scene,
                scene->player_obj_ptr,
                display->width, display->height
	    );
	}
	/* Draw messages. */
	SARDrawMessages(display, scene);
	/* Draw sticky banner messages. */
	SARDrawBanner(display, scene);
	/* Draw slew coordinates if in slew mode. */
	if(flight_model_type == SAR_FLIGHT_MODEL_SLEW)
	{
            SARDrawSetColor(&option.message_color);
            GWSetFont(display, option.message_font);
	    SARDrawSlewCoordinates(display, scene, scene->player_obj_ptr);
	}

        /* Draw text input prompt. */
        SARTextInputDraw(core_ptr);

        /* Draw player wheel brakes text as needed. */
        switch(player_wheel_brakes)
        {
          case 2:
	    GWSetFont(display, option.message_font);
	    glColor4d(1.0, 0.0, 0.0, 1.0);
	    GWDrawString(
                display,
                10,
		display->height - 20,
                SAR_MESG_PARKING_BRAKES
            );
	    break;

	  case 1:
            GWSetFont(display, option.message_font);
            glColor4d(
                MAX((gc == NULL) ? 1.0 : gc->wheel_brakes_coeff, 0.0),
                0.0, 0.0, 1.0
            );
            GWDrawString(
                display,
                10,
                display->height - 20,
                SAR_MESG_WHEEL_BRAKES
            );
            break;
        }

	/* Draw over speed text as needed. */
	if(player_overspeed)
	{
	    int fw, fh;

	    GWGetFontSize(
		option.message_font,
		NULL, NULL,
		&fw, &fh
	    );
            GWSetFont(display, option.message_font);
            glColor4d(1.0, 0.0, 0.0, 1.0);

            GWDrawString(
                display,
                display->width - 20 - (fw * strlen(SAR_MESG_OVERSPEED)),
                display->height - 20,
                SAR_MESG_OVERSPEED
            );
	}

	/* Draw help. */
	if(core_ptr->display_help > 0)
	    SARDrawHelp(core_ptr);

        /* Put GL buffer to X Window. */
        GWSwapBuffer(display);

	/* Report any errors. */
	if(option.runtime_debug)
	{
	    GLenum error_code = glGetError();
	    if(error_code != GL_NO_ERROR)
		SARReportGLError(core_ptr, error_code);
	}

#undef DRAW_SET_PERSPECTIVE_INSIDE
#undef DRAW_SET_PERSPECTIVE_OUTSIDE
}


/*
 *      Redraws scene as a map.
 *
 *	If draw_for_gcc is True then drawing will be done in a different
 *	style suitable for `ground contact check'. Also the core's list
 *	of draw map object names will be updated.  It will be reallocated 
 *	to contain the new list of matched objects by index numbers.
 *
 *	If draw_for_ghc is True then drawing will be done in a different
 *	style suitable for `ground hit check'. Also the core's 
 *	drawmap_ghc_result will be set to the alpha pixel read.
 *
 *	gcc_obj_num will be passed to SARSetCameraMapDrawGCC(), it
 *	defines which object is the player object (or -1 to fetch player
 *	object number from scene structure).
 */
void SARDrawMap(
	sar_core_struct *core_ptr,
	Boolean draw_for_gcc, Boolean draw_for_ghc, int gcc_obj_num
)
{
        int i, n, width, height, *total;
	double fovz_um, view_aspect;

        gw_display_struct *display;
        sar_scene_struct *scene;
        sar_object_struct ***ptr, *obj_ptr;
	gctl_struct *gc;
	sar_color_struct *color_ptr;
        sar_direction_struct *dir;
        sar_position_struct *pos;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_object_helipad_struct *obj_helipad_ptr;
	sar_object_runway_struct *obj_runway_ptr;

        double distance, distance3d, icon_len;

	GLuint select_name_base;
#define SELECT_BUF_SIZE		512
	GLuint select_buf[SELECT_BUF_SIZE];


        /* Get initial pointers to resources. */
        display = core_ptr->display;
        scene = core_ptr->scene;
        ptr = &core_ptr->object;
        total = &core_ptr->total_objects;
	gc = core_ptr->gctl;

	/* Select name base always starts at highest object index + 1
	 * which is the total number of objects (including NULL's).
	 */
	select_name_base = (GLuint)MAX(*total, 0);


	if((display == NULL) || (scene == NULL))
	    return;

	/* Get size of viewport. */
	width = display->width;
	height = display->height;


	/* Reset viewport and translations depending on draw style. */
	if(draw_for_gcc || draw_for_ghc)
	{
	    /* Set up for ground contact check (gcc) or ground hit check 
	     * (ghc) drawing.
	     */

	    /* Adjust the viewport to be a efficient small size (does
	     * not work in Windows).
	     */
	    width = 100;
	    height = 70;

	    /* Draw for ground contact check? */
	    if(draw_for_gcc)
	    {
		/* Clear map draw object names list (if any) on core
		 * structure.
		 */
		SARDrawMapObjNameListDelete(
		    &core_ptr->drawmap_objname,
		    &core_ptr->total_drawmap_objnames
		);

		/* Initialize GL select buffer and names stack, switching
		 * to GL_SELECT render mode.
		 */
		glSelectBuffer(SELECT_BUF_SIZE, select_buf);
		glRenderMode(GL_SELECT);
	    }
	    /* Setup specifics for ground hit check? */
	    if(draw_for_ghc)
	    {
		/* Reset drawmap_ghc_result. */
		core_ptr->drawmap_ghc_result = 0.0;

		/* Set stencil buffer clear value to 0x0. */
		glClearStencil(0x0);
		/* Enable stencil buffer drawing and testing. */
		StateGLEnable(&display->state_gl, GL_STENCIL_TEST);
		StateGLStencilFunc(
		    &display->state_gl,
		    GL_ALWAYS, 0x1, 0x1
		);
		StateGLStencilOp(
		    &display->state_gl,
		    GL_REPLACE, GL_REPLACE, GL_REPLACE
		);
	    }

	    /* Initialize select names, its safe to initialize this even
	     * if not in GL_SELECT render mode (OpenGL will ignore this).
	     */
	    glInitNames();
#ifdef GL_MAX_NAME_STACK_DEPTH
	    if(GL_MAX_NAME_STACK_DEPTH > 0)
#else
	    if(1)
#endif
	    {
		glPushName(select_name_base + 0);
	    }



            /* Reset viewport. */
/*
            SARReshapeCB(display, -1, -1, -1, -1);
 */
	    /* Change the viewport, since using our own width and height
	     * would differ from the actual viewport's size.
	     */
	    glViewport(0, 0, width, height);

            /* Calculate view aspect. */
            if(height > 0)
                view_aspect = (double)width / (double)height;
            else
                view_aspect = 800.0 / 600.0;

	    view_aspect += display->aspect_offset;
/* Do we need view aspect offset when drawing for contact check or
 * hit check?
 */
	    /* Calculate field of view in unit meters about z axis. */
	    fovz_um = 1.0 * tan(scene->camera_fovz / 2.0) * 2.0;

            /* Set up projection matrix. */
            glMatrixMode(GL_PROJECTION);
            glLoadIdentity();
            /* Set perspective with short far clip but very close near
	     * clip since we are only interested in objects drawn very
	     * close to the camera.
             */
            gluPerspective(
                SFMRadiansToDegrees(scene->camera_fovz),        /* Field of viev. */
                (GLfloat)view_aspect,		/* View aspect. */
                0.1,				/* Near clip. */
                10000.0				/* Far clip. */
/*              SFMFeetToMeters(SAR_MAX_MAP_VIEW_HEIGHT) */
            );

            /* Switch back to model view matrix. */
            glMatrixMode(GL_MODELVIEW);
            glLoadIdentity();

            /* Set camera location, this will set the camera position
             * at the position of gcc_obj_num looking straight down.
             */
	    SARSetCameraMapDrawGCC(scene, ptr, total, gcc_obj_num);
	}
	else
	{
	    /* Set up for regular map drawing. */

	    /* Reset viewport. */
	    SARReshapeCB(display, -1, -1, -1, -1);

	    /* Calculate view aspect. */
	    if(height > 0)
		view_aspect = (double)width / (double)height;
	    else
		view_aspect = 800.0 / 600.0;
            view_aspect += display->aspect_offset;

            /* Calculate field of view in unit meters about z axis. */
            fovz_um = 1.0 * tan(scene->camera_fovz / 2.0) * 2.0;

	    /* Set up projection matrix. */
	    glMatrixMode(GL_PROJECTION);
	    glLoadIdentity();
	    /* Set perspective with zfar to be just a bit farther than
	     * the camera altitude from altitude of 0.
	     */
	    gluPerspective(
		SFMRadiansToDegrees(scene->camera_fovz),	/* Field of view. */
		view_aspect,			/* View aspect. */
		1.0,				/* Near clip. */
		scene->camera_map_pos.z + 100.0	/* Far clip. */
/*		SFMFeetToMeters(SAR_MAX_MAP_VIEW_HEIGHT) */
	    );

	    /* Switch back to model view matrix. */
	    glMatrixMode(GL_MODELVIEW);
	    glLoadIdentity();

	    /* Set camera location, this will set camera position
	     * variables global to this module.
	     */
	    SARSetCamera(scene, ptr, total);
	}

        /* Reset other variables. */
        player_wheel_brakes = 0;
	player_air_brakes = 0;
	player_overspeed = 0;
	flight_model_type = -1;


	/* Calculate icon length based on camera z height, each icon
	 * should be 0.05 the size of the field of view in meters.
	 */
	icon_len = (scene->camera_fovz * camera_pos.z) * 0.05;

	/* Get pointer to ground base color, going to use this to clear
	 * the frame buffer if appropriate.
	 */
	color_ptr = &scene->base.color;

        /* Clear GL depth buffer and color, match color to be
	 * the color of the ground base.
	 */
        glClearDepth((GLclampd)1.0);
        glClearColor(
	    (GLfloat)color_ptr->r, (GLfloat)color_ptr->g, (GLfloat)color_ptr->b,
	    (GLfloat)((draw_for_ghc) ? 0.0 : color_ptr->a)
	);
        glClear(
            GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT |
	    ((draw_for_ghc) ? GL_STENCIL_BUFFER_BIT : 0)
        );


	/* Disable fog. */
        StateGLDisable(&display->state_gl, GL_FOG);

        /* Turn off depth testing and turn on depth writing for
         * drawing the foundations (the ground base visual models).
         */
        StateGLDisable(&display->state_gl, GL_LIGHTING);
	StateGLDisable(&display->state_gl, GL_LIGHT0);
	StateGLDisable(&display->state_gl, GL_LIGHT1);
	StateGLDisable(&display->state_gl, GL_COLOR_MATERIAL);

        StateGLDisable(&display->state_gl, GL_BLEND);

	SAR_DRAW_DEPTH_TEST_OFF
        StateGLDepthMask(&display->state_gl, GL_FALSE);

	StateGLShadeModel(&display->state_gl, GL_FLAT);

	SAR_DRAW_TEXTURE_1D_OFF

	SAR_DRAW_TEXTURE_2D_OFF

	/* Enable scissor test if drawing for ground contact check. */
/*
	if(draw_for_gcc || draw_for_ghc)
	{
	    StateGLEnable(&display->state_gl, GL_SCISSOR_TEST);
	    StateGLScissor(
		&display->state_gl,
		(GLint)((display->width / 2) - 5),
		(GLint)((display->height / 2) - 5),
		(GLsizei)10, (GLsizei)10
	    );
	}
 */

	/* Begin drawing. */

	/* Draw ground base only when NOT doing ground hit check. */
	if(!draw_for_ghc)
	{
	    /* Draw ground base. */
	    SARDrawSceneFoundations(display, scene, view_aspect);
	}

	/* Textured objects enabled? */
        if(option.textured_objects)
        {
	    /* Enable alpha testing and turn on GL_TEXTURE_2D. */
            StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	    StateGLAlphaFunc(&display->state_gl, GL_GREATER, 0.5);

	    SAR_DRAW_TEXTURE_2D_ON
        }
        else
        {
	    /* No textured objects, so no alpha testing and no
	     * GL_TEXTURE_2D.
	     */
	    SAR_DRAW_TEXTURE_2D_OFF
        }
        /* Don't use 1D texture. */
	SAR_DRAW_TEXTURE_1D_OFF

        /* Begin drawing objects. */
        for(i = 0; i < (*total); i++)
        {
            obj_ptr = (*ptr)[i];
            if(obj_ptr == NULL)
                continue;

	    /* Enable depth testing? */
            if(obj_ptr->flags & SAR_OBJ_FLAG_NO_DEPTH_TEST)
            {
		SAR_DRAW_DEPTH_TEST_OFF
		StateGLDepthMask(&display->state_gl, GL_FALSE);
            }
            else if(!(obj_ptr->flags & SAR_OBJ_FLAG_NO_DEPTH_TEST))
            {
		SAR_DRAW_DEPTH_TEST_ON
		StateGLDepthMask(&display->state_gl, GL_TRUE);
            }

            /* Set object's shade model. */
            if(obj_ptr->flags & SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH)
            {
                StateGLShadeModel(&display->state_gl, GL_SMOOTH);
            }
            else
            {
                StateGLShadeModel(&display->state_gl, GL_FLAT);
            }

	    /* Draw by object type. */
	    switch(*(int *)obj_ptr)
	    {
              /* ***************************************************** */
	      case SAR_OBJ_TYPE_STATIC:
              case SAR_OBJ_TYPE_GROUND:
                /* Get position and check if in range with camera. */
                pos = &obj_ptr->pos;
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Set GL name as this object's inedx number. */
                glLoadName((GLuint)i);

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, 0, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

                    /* Draw standard model if defined. */
                    SARVisualModelCallList(obj_ptr->visual_model);
                }
                glPopMatrix();
		POST_CALLLIST_RESET_STATES
		break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr == NULL)
		    break;

                /* Set GL name as this object's index number. */
                glLoadName((GLuint)i);

                /* Do not draw this type for ground contact check. */
                if(draw_for_gcc || draw_for_ghc)
                    break;

		/* Check if this is the player object. */
		if((obj_ptr == scene->player_obj_ptr) &&
		   (gc != NULL)
		)
		{
			    /* Update player wheel brakes state. */
			    player_wheel_brakes = gc->wheel_brakes_state;
			    if((player_wheel_brakes == 0) &&
			       (gc->wheel_brakes_coeff > 0.0)
                            )
                                player_wheel_brakes = 1;

                            /* Update player air brakes state. */
                            player_air_brakes = gc->air_brakes_state;
                            if((player_air_brakes == 0) &&
                               (gc->air_brakes_coeff > 0.0)
                            )
                                player_air_brakes = 1;

                            /* Update player flight model. */
                            flight_model_type = obj_aircraft_ptr->flight_model_type;

                            /* Update player over speed state. */
                            if(obj_aircraft_ptr->speed > obj_aircraft_ptr->speed_max_expected)
                            {
                                player_overspeed = 1;
                            }
		}

                /* Always mute sounds in map view. */
                SARSoundEngineVolumeMute(
                    core_ptr->recorder,
                    obj_aircraft_ptr->engine_inside_sndplay, 
                    obj_aircraft_ptr->engine_outside_sndplay 
                );

                /* Get position and direction, we will be drawing the
		 * intercept lines first unconditionally to range
		 * since we want intercept lines always visable.
		 */
                pos = &obj_ptr->pos;
		dir = &obj_ptr->dir;

		/* Unselect texture. */
		V3DTextureSelect(NULL);

		/* Is this the player object? */
		if((obj_ptr == scene->player_obj_ptr) &&
                   (obj_aircraft_ptr->total_intercepts > 0)
		)
		{
		    sar_obj_intercept_struct *intercept_ptr;

		    /* Draw intercept lines and waypoints. */
		    glBegin(GL_LINES);
		    {
                        glNormal3d(0.0, 1.0, 0.0);

			/* Draw subsequent (not current) intercept lines
			 * first.
			 */
			glColor4d(0.8, 0.0, 0.0, 1.0);
			for(n = MAX(obj_aircraft_ptr->cur_intercept, 0);
                            n < (obj_aircraft_ptr->total_intercepts - 1);
			    n++
			)
			{
			    intercept_ptr = obj_aircraft_ptr->intercept[n];
			    /* Give up if any intercepts are NULL. */
			    if(intercept_ptr == NULL)
				break;

                            glVertex3d(
				intercept_ptr->x, 0.0, -intercept_ptr->y
			    );

			    if((n + 1) < obj_aircraft_ptr->total_intercepts)
				intercept_ptr = obj_aircraft_ptr->intercept[n + 1];
			    else
				intercept_ptr = NULL;
			    if(intercept_ptr != NULL)
                                glVertex3d(
                                    intercept_ptr->x, 0.0, -intercept_ptr->y
                                );
			}

			/* Draw intercept from current position to current
			 * intercept waypoint.
			 */
                        glColor4d(1.0, 0.25, 0.25, 1.0);
			n = obj_aircraft_ptr->cur_intercept;
			if((n >= 0) && (n < obj_aircraft_ptr->total_intercepts))
			{
			    intercept_ptr = obj_aircraft_ptr->intercept[n];
			    if(intercept_ptr != NULL)
			    {
				glVertex3d(pos->x, 0.0, -pos->y);
                                glVertex3d(
                                    intercept_ptr->x, 0.0, -intercept_ptr->y
                                );
			    }
			}
		    }
		    glEnd();
		}

		/* Now perform range check. */
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

		/* Begin drawing aircraft `icon'. */
                glPushMatrix();
                {  
                    /* Translate and rotate. */
                    glTranslatef(pos->x, 0.0, -pos->y);
                    glRotatef(-SFMRadiansToDegrees(dir->heading),
			0.0, 1.0, 0.0
		    );

                    glBegin(GL_LINES);
                    {
                        glNormal3d(0, 1, 0);

                        if(obj_ptr == scene->player_obj_ptr)
                            glColor4d(1.0, 1.0, 0.0, 1.0);
                        else
                            glColor4d(1.0, 0.0, 0.0, 1.0);

                        glVertex3d(0, 0, 0);
                        glVertex3d(0.0, 0.0, -(1.0 * icon_len));

                        glVertex3d(-0.5 * icon_len, 0.0, 0.0);
                        glVertex3d(0.5 * icon_len, 0.0, 0.0);
                    }
                    glEnd();
                }    
                glPopMatrix();
		POST_CALLLIST_RESET_STATES
		break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_HELIPAD:
                obj_helipad_ptr = (sar_object_helipad_struct *)obj_ptr->data;
                if(obj_helipad_ptr == NULL)
                    break;

                /* Set GL name as this object's inedx number. */
                glLoadName((GLuint)i);

                /* Get position and check if in range with camera. */
                pos = &obj_ptr->pos;
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Calculate 3D distance. */
                distance3d = hypot(distance, pos->z - camera_pos.z);

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                  if(draw_for_gcc || draw_for_ghc)
                  {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);

                    /* Draw runway. */
                    SARDrawHelipad(
                        display, scene,
                        obj_ptr, obj_helipad_ptr, distance3d
                    );
                  } 
                  else 
                  {
                    /* Translate and rotate. */
                    glTranslated(pos->x, 0.0, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);
                    /* Draw helipad for map view. */
		    SARDrawHelipadMap(
			display, scene, obj_ptr, obj_helipad_ptr,
			icon_len
		    );
                  }
                }
                glPopMatrix();
                break;

              /* ***************************************************** */
              case SAR_OBJ_TYPE_RUNWAY:
                obj_runway_ptr = (sar_object_runway_struct *)obj_ptr->data;
                if(obj_runway_ptr == NULL)
                    break;

                /* Set GL name as this object's inedx number. */
                glLoadName((GLuint)i);

                /* Get position and check if in range with camera. */     
                pos = &obj_ptr->pos;
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Calculate 3D distance. */
                distance3d = hypot(distance, pos->z - camera_pos.z);

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
		  if(draw_for_gcc || draw_for_ghc)
		  {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);
		    /* Draw runway. */
                    SARDrawRunway(
                        display, scene,
                        obj_ptr, obj_runway_ptr, distance3d
                    );
		  }
		  else
		  {
                    /* Translate and rotate. */
                    glTranslated(pos->x, 0.0, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);
		    /* Draw runway for map view. */
                    SARDrawRunwayMap(
                        display, scene, obj_ptr, obj_runway_ptr,
                        icon_len
                    );
		  }
                }
                glPopMatrix();
                break;

              /* ************************************************** */
              case SAR_OBJ_TYPE_PREMODELED:
                /* Get position and check if in range with camera. */
		pos = &obj_ptr->pos;
                distance = hypot(pos->x - camera_pos.x, pos->y - camera_pos.y);
                if(distance > obj_ptr->range)
                    break;

                /* Set GL name as this object's inedx number. */
                glLoadName((GLuint)i);

                /* Get direction. */
                dir = &obj_ptr->dir;

                glPushMatrix();
                {
                    /* Translate and rotate. */
                    glTranslated(pos->x, pos->z, -pos->y);
                    glRotated(-SFMRadiansToDegrees(dir->heading), 0.0, 1.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->pitch), 1.0, 0.0, 0.0);
                    glRotated(-SFMRadiansToDegrees(dir->bank), 0.0, 0.0, 1.0);
		    /* Draw premodeled object. */
                    SARDrawPremodeled(
                        display, scene, obj_ptr,
			(sar_object_premodeled_struct *)obj_ptr->data,
                        &camera_pos, &camera_dir,
			(Boolean)((draw_for_gcc || draw_for_ghc) ? True : False)
                    );
                }
                glPopMatrix();
                POST_CALLLIST_RESET_STATES
                break;
	    }
        }


	/* Was drawing for ground contact check? */
	if(draw_for_gcc)
	{
	    /* Flush output and switch back to render mode. Tabulate
	     * hits if any.
	     */
	    GLint hits;

	    glFlush();
	    hits = glRenderMode(GL_RENDER);
	    SARDrawMapProcessHits(core_ptr, hits, select_buf);

	    /* Need to restore viewport size. */
            glViewport(0, 0, display->width, display->height);
	}
	/* Was drawing for ground hit check? */
	else if(draw_for_ghc)
	{
	    /* Flush output and switch back to render mode. Read back
             * frame buffer at center of window coordinates.
             */
	    GLubyte *bptr, *bend;
#define READ_BACK_BUF_WIDTH	2
#define READ_BACK_BUF_HEIGHT	2
#define READ_BACK_BUF_LEN	(READ_BACK_BUF_WIDTH * READ_BACK_BUF_HEIGHT)
	    GLubyte read_back_buf[READ_BACK_BUF_LEN];

	    glFlush();

	    /* Read back drawn frame buffer's alpha channel. */
	    if(display->has_double_buffer)
		glReadBuffer(GL_BACK);
	    else
		glReadBuffer(GL_FRONT);
	    glReadPixels(
		(GLint)(width / 2), (GLint)(height / 2),
		(GLsizei)READ_BACK_BUF_WIDTH, (GLsizei)READ_BACK_BUF_HEIGHT,
		GL_STENCIL_INDEX, GL_UNSIGNED_BYTE,
		read_back_buf
	    );

	    /* Set core_ptr->drawmap_ghc_result (note that it was already
	     * reset at the start of this function.
	     */
	    for(bptr = read_back_buf, 
		bend = read_back_buf + (READ_BACK_BUF_LEN * 1);
		bptr < bend;
		bptr += 1
	    )
	    {
		if((*bptr) != 0x0)
		    core_ptr->drawmap_ghc_result = 1.0;
	    }
#undef READ_BACK_BUF_WIDTH
#undef READ_BACK_BUF_HEIGHT
#undef READ_BACK_BUF_LEN

/*
if(1)
{
 char tmp_text[256];
 sprintf(tmp_text, "Got ghc 0x%.2x %.4f",
  read_back_buf[0], core_ptr->drawmap_ghc_result
 );
 SARMessageAdd(scene, tmp_text);
}
 */

	    /* Disable stencil buffer, don't need it anymore. */
	    StateGLDisable(&display->state_gl, GL_STENCIL_TEST);

            /* Need to restore viewport size. */
            glViewport(0, 0, display->width, display->height);
	}
	/* Was doing regular map drawing? */
	else
	{
	    /* Draw extra super imposed details if not drawing for ground
	     * contact check.
	     */
            StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
            StateGLDisable(&display->state_gl, GL_COLOR_MATERIAL);
	    StateGLDisable(&display->state_gl, GL_DEPTH_TEST);
	    StateGLDepthFunc(&display->state_gl, GL_ALWAYS);
            SAR_DRAW_TEXTURE_1D_OFF
            SAR_DRAW_TEXTURE_2D_OFF

	    /* Draw map grids (before ortho). */
	    SARDrawMapGrids(
		display, scene,
		&camera_pos, fovz_um, view_aspect
	    );

            /* Go to 2D after drawing grids. */
            GWOrtho2D(display);

	    /* Draw map cross hairs. */
            SARDrawMapCrossHairs(display, scene);

            /* Set color and font for drawing the following. */
            SARDrawSetColor(&option.message_color);
            GWSetFont(display, option.message_font);

            /* Draw attitude and stats of player object. */
            SARDrawOutsideAttitude(
                display,
                scene,
                scene->player_obj_ptr,
                display->width, display->height
            );

            /* Draw messages. */
            SARDrawMessages(display, scene);
            /* Draw sticky banner messages. */
            SARDrawBanner(display, scene);
            /* Draw slew coordinates if in slew mode. */
            if(flight_model_type == SAR_FLIGHT_MODEL_SLEW)
            {
                SARDrawSetColor(&option.message_color);
                GWSetFont(display, option.message_font);
                SARDrawSlewCoordinates(display, scene, scene->player_obj_ptr);
	    }

            /* Draw text input prompt. */
            SARTextInputDraw(core_ptr);

            /* Draw help. */
            if(core_ptr->display_help > 0)
                SARDrawHelp(core_ptr);

	    /* Put GL buffer to window. */
            GWSwapBuffer(display);
	}


	/* Disable scissor test (if drawing for ground contact check). */
/*
	StateGLDisable(&display->state_gl, GL_SCISSOR_TEST);
 */

        /* Report any errors. */
        if(option.runtime_debug)
        {
            GLenum error_code = glGetError();
            if(error_code != GL_NO_ERROR)
                SARReportGLError(core_ptr, error_code);
        }

#undef SELECT_BUF_SIZE
}           
