#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../include/string.h"

#include "obj.h"
#include "messages.h"
#include "sar.h"
#include "config.h"


void SARMessageAdd(sar_scene_struct *scene, const char *message);
void SARMessageClearAll(sar_scene_struct *scene);
void SARBannerMessageAppend(sar_scene_struct *scene, const char *message);
void SARCameraRefTitleSet(sar_scene_struct *scene, const char *message);


/*
 *	Adds the specified message to the list in the scene.
 *	The new message will be inserted at the last index position.
 *
 *	The message pointer array on the scene will not change in size.
 */
void SARMessageAdd(sar_scene_struct *scene, const char *message)
{
	int i, n;


	if(scene == NULL)
	    return;

	/* No message line pointers allocated? */
	if(scene->total_messages <= 0)
	    return;

	/* Free oldest message. */
	free(scene->message[0]);

	/* Get new position to insert new message. */
	n = scene->total_messages - 1;

	/* Shift message pointers. */
        for(i = 0; i < n; i++)
            scene->message[i] = scene->message[i + 1];

	/* Copy new message to pointer array. */
	scene->message[n] = ((message == NULL) ? NULL : strdup(message));

	/* Update the time the message should be shown untill. */
	scene->message_display_untill = cur_millitime +
	    SAR_MESSAGE_SHOW_INT;

	return;
}

/*
 *	Deletes all messages on the message pointer array on the scene
 *	structure and resets all pointers to NULL.
 *
 *	The size of the message pointer array will not be changed.
 *
 *	The message_display_untill member on the scene will be reset
 *	to 0.
 */
void SARMessageClearAll(sar_scene_struct *scene)
{
	int i;

	if(scene == NULL)
	    return;

	/* Reset the time to keep displaying messages to 0 so messages
	 * shouldn't be displayed anymore if they were being displayed.
	 */
	scene->message_display_untill = 0;

	/* Delete each message and set pointer to NULL. */
	for(i = 0; i < scene->total_messages; i++)
	{
	    free(scene->message[i]);
	    scene->message[i] = NULL;
	}

	return;
}

/*
 *	Appends given message line to scene's sticky banner message, can
 *	be called multiple times to set several lines.
 *
 *	Passing NULL clears the entire list.
 */
void SARBannerMessageAppend(sar_scene_struct *scene, const char *message)
{
	if(scene == NULL)
	    return;

	if(message == NULL)
	{
            StringFreeArray(
                scene->sticky_banner_message, 
                scene->total_sticky_banner_messages 
            ); 
            scene->sticky_banner_message = NULL;
            scene->total_sticky_banner_messages = 0;
	}
	else
	{
	    int i;

	    if(scene->total_sticky_banner_messages < 0)
		scene->total_sticky_banner_messages = 0;

	    i = scene->total_sticky_banner_messages;
	    scene->total_sticky_banner_messages = i + 1;

	    scene->sticky_banner_message = (char **)realloc(
		scene->sticky_banner_message,
		scene->total_sticky_banner_messages * sizeof(char *)
	    );
	    if(scene->sticky_banner_message == NULL)
	    {
		scene->total_sticky_banner_messages = 0;
		return;
	    }

	    scene->sticky_banner_message[i] = strdup(message);
	}

	return;
}

/*
 *	Replaces the current camera referance title message on the scene
 *	structure.
 *
 *	Passing NULL clears it.
 */
void SARCameraRefTitleSet(sar_scene_struct *scene, const char *message)
{
	if(scene == NULL)
	    return;

	if(message == NULL)
	{
	    free(scene->camera_ref_title);
	    scene->camera_ref_title = NULL;
	    scene->camera_ref_title_display_untill = 0;
	}
	else
	{
	    free(scene->camera_ref_title);
	    scene->camera_ref_title = strdup(message);
	    scene->camera_ref_title_display_untill = cur_millitime +
		SAR_MESSAGE_SHOW_INT;
	}


	return;
}
