/*
                              SAR Menu System

	Structures for menu and menu object/widgets.
 */

#ifndef MENU_H
#define MENU_H

#include <sys/types.h>

#include "v3dtex.h"

#include "gw.h"
#include "image.h"


/*
 *	Flags type:
 */
#define sar_menu_flags_t	unsigned long

/*
 *	Menu object types:
 */
#define SAR_MENU_OBJECT_TYPE_LABEL		0
#define SAR_MENU_OBJECT_TYPE_BUTTON		1
#define SAR_MENU_OBJECT_TYPE_PROGRESS		2
#define SAR_MENU_OBJECT_TYPE_MESSAGE_BOX	3
#define SAR_MENU_OBJECT_TYPE_LIST		4
#define SAR_MENU_OBJECT_TYPE_SWITCH		5
#define SAR_MENU_OBJECT_TYPE_SPIN		6
#define SAR_MENU_OBJECT_TYPE_MAP		7


/*
 *	Menu color structure:
 */
typedef struct {

	double a, r, g, b;

} sar_menu_color_struct;

/*
 *	Menu label structure:
 */
typedef struct {

	int type;       /* Must be SAR_MENU_OBJECT_TYPE_LABEL. */

	double x, y;
	int offset_x, offset_y;	/* Offset from center in pixels. */
	int width, height;	/* Size in pixels. */
	const sar_image_struct *image;	/* Shared bg image. */

        GWFont *font;
	sar_menu_color_struct color;
	char *label;
#define SAR_MENU_LABEL_ALIGN_CENTER	0
#define SAR_MENU_LABEL_ALIGN_LEFT	1
#define SAR_MENU_LABEL_ALIGN_RIGHT	2
	int align;	/* One of SAR_MENU_LABEL_ALIGN_*. */

} sar_menu_label_struct;

/*
 *	Menu button structure:
 */
typedef struct {

	int type;	/* Must be SAR_MENU_OBJECT_TYPE_BUTTON. */

	double x, y;    /* Position coefficient. */
	int offset_x, offset_y;	/* Offset from center in pixels. */
	int width, height;	/* Size in pixels. */
	Boolean disabled;

#define SAR_MENU_BUTTON_STATE_UNARMED		0
#define SAR_MENU_BUTTON_STATE_ARMED		1
#define SAR_MENU_BUTTON_STATE_HIGHLIGHTED	2
	int state;		/* One of SAR_MENU_BUTTON_STATE_*. */
	const sar_image_struct	*unarmed_image,		/* Shared images. */
				*armed_image,
				*highlighted_image;

        GWFont *font;
	sar_menu_color_struct color;
	char *label;	/* Button label. */

	void *client_data;
        int id_code;

        /*              object, client_data, id_code. */
	void (*func_cb)(void *, void *, int);

} sar_menu_button_struct;

/*
 *	Menu progress bar structure:
 */
typedef struct {

	int type;	/* Must be SAR_MENU_OBJECT_TYPE_PROGRESS. */

        double x, y;    /* Position coefficient. */
        int width, height; 
	Boolean disabled;

        const sar_image_struct	*bg_image,	/* Shared images. */
				*fg_image;

	GWFont *font;
	sar_menu_color_struct color;
        char *label;

	double progress;	/* 0.0 to 1.0. */

} sar_menu_progress_struct;

/*
 *	Menu message box structure:
 *
 *	For holding more than one line of text, scroll capable.
 */
typedef struct {

	int type;	/* Must be SAR_MENU_OBJECT_TYPE_MESSAGE_BOX. */

        double x, y;	/* Position coefficient. */
        double width, height;	/* Size coefficient. */
	Boolean disabled;

        GWFont *font;
        sar_menu_color_struct color;

        /* Background image (9 total) shared image (including pointer
         * array, do not deallocate).
         */
#define MENU_MESSAGE_BOX_BG_TOTAL_IMAGES	9
        const sar_image_struct **bg_image;

	int scrolled_line;
	char *message;

} sar_menu_message_box_struct;


/*
 *	Menu list structure:
 */
typedef struct {

	sar_menu_flags_t flags;

	char *name;
	void *client_data;

} sar_menu_list_item_struct;	

typedef struct {

	int type;	/* Must be SAR_MENU_OBJECT_TYPE_LIST. */

        double x, y;    /* Position coefficient. */
        double width, height;	/* Size coefficient. */
	Boolean disabled;

	/* Background image (9 total) shared image (including pointer
	 * array, do not deallocate).
	 */
#define MENU_LIST_BG_TOTAL_IMAGES   9
	const sar_image_struct **bg_image;

        GWFont *font;
        sar_menu_color_struct color;
        char *label;

	sar_menu_list_item_struct **item;
	int total_items, items_visable;

        int	scrolled_item,		/* First visable item number. */
		selected_item;

	/*                object, client_data, item_num, item_ptr */
	void (*select_cb)(void *, void *, int, void *);

} sar_menu_list_struct;


/*
 *	Menu toggle switch structure:
 */
typedef struct {

	int type;       /* Must be SAR_MENU_OBJECT_TYPE_SWITCH. */

        double x, y;	/* Position coefficient. */
	int width, height;	/* Size coefficient. */
	Boolean disabled;

	const sar_image_struct	*bg_image,		/* Shared images. */
				*switch_off_image,
				*switch_on_image;

        GWFont *font;
        sar_menu_color_struct color;
        char *label;

	Boolean state;

	void *client_data;
	int id_code;
	void (*switch_cb)(void *, void *, int, Boolean);

} sar_menu_switch_struct;


/*
 *	Spin box structure:
 */
typedef struct {

        int type;       /* Must be SAR_MENU_OBJECT_TYPE_SPIN. */

	double x, y;	/* Position coefficient. */
	double width, height;	/* Size coefficient. */
	Boolean disabled;

        const sar_image_struct	*label_image,	/* Shared image. */
				*value_image;

        GWFont *font, *value_font;
        sar_menu_color_struct label_color, value_color;
        char *label;

#define SAR_MENU_SPIN_TYPE_STRING	0
#define SAR_MENU_SPIN_TYPE_NUMERIC	1
	int value_type;	/* One of SAR_MENU_SPIN_TYPE_* */

	/* Allow end to beginning cycle warping. */
	Boolean allow_warp;

	/* Increment step (only for SAR_MENU_SPIN_TYPE_NUMERIC). */
	double step;

	/* List of values, if the value_type is set to
	 * SAR_MENU_SPIN_TYPE_NUMERIC then there will only be one value.
	 * However if value_type is set to SAR_MENU_SPIN_TYPE_STRING then
	 * there may be one or more values in this list.
	 */
	char **value;
	int total_values, cur_value;

        void *client_data;
	int id_code;
        void (*change_cb)(void *, void *, int, char *);

} sar_menu_spin_struct;


/*
 *	Map marking structure:
 */
typedef struct {

#define SAR_MENU_MAP_MARKING_TYPE_ICON			0
#define SAR_MENU_MAP_MARKING_TYPE_INTERCEPT_LINE	1
	int type;	/* One of SAR_MENU_MAP_MARKING_TYPE_* */

	sar_menu_color_struct fg_color;

	double x, y;		/* In meters. */
	double x_end, y_end;	/* In meters (for intercept line). */

	const sar_image_struct *icon;	/* Shared icon image. */
	char *desc;		/* Description (preferably short one-liner). */

} sar_menu_map_marking_struct;
/*
 *	Map structure:
 */
typedef struct {

	int type;	/* Must be SAR_MENU_OBJECT_TYPE_MAP. */

        double x, y;    /* Position coefficient. */
        double width, height;   /* Size coefficient. */
	Boolean disabled;

	GWFont *font;
	sar_menu_color_struct color;

        /* Background image (8 total) shared image (including pointer
         * array, do not deallocate).
         */
        const sar_image_struct **bg_image;

	int scroll_x, scroll_y;		/* Upper left oriented, in pixels. */

	double m_to_pixels_coeff;	/* Meters to pixels coeff. */

	v3d_texture_ref_struct *bg_tex;	/* Background texture, not shared! */
	double bg_tex_width, bg_tex_height;	/* In meters. */

#define SAR_MENU_MAP_SHOW_MARKING_ALL		0	/* Show all markings. */
#define SAR_MENU_MAP_SHOW_MARKING_SELECTED	1	/* Show only selected marking. */
	int show_markings_policy;	/* One of SAR_MENU_MAP_SHOW_MARKING_*. */

	sar_menu_map_marking_struct **marking;
	int total_markings;

	/* Marking that pointer is over, show desc of this marking.
	 * Can be -1 for none.
	 */
	int selected_marking;

} sar_menu_map_struct;


/*
 *	Menu structure.
 */
typedef struct {

#define SAR_MENU_TYPE_STANDARD		0
	int type;	/* One of SAR_MENU_TYPE_*. */

	char *name;	/* Name of this menu (for matching purposes). */

	char *bg_image_path;
	sar_image_struct *bg_image;	/* Background image (not shared). */

	void **object;
	int total_objects, selected_object;

} sar_menu_struct;



/* In menu.c */
extern void SARMenuGLStateReset(gw_display_struct *display);
extern int SARMenuIsObjectAllocated(sar_menu_struct *m, int n);
extern sar_menu_struct *SARMenuCreate(
	int type, const char *name, const char *bg_image
);
extern void SARMenuLoadBackgroundImage(sar_menu_struct *m);
extern void SARMenuUnloadBackgroundImage(sar_menu_struct *m);
extern void SARMenuDestroy(sar_menu_struct *m);

extern int SARMenuCreateLabel(
        sar_menu_struct *m,
        double x, double y,
	int width, int height,
	const char *label,
	sar_menu_color_struct *fg_color, GWFont *font,
        const sar_image_struct *image
);
extern int SARMenuCreateButton(
        sar_menu_struct *m,
        double x, double y,
        int width, int height,
        const char *label,
        sar_menu_color_struct *fg_color,
        GWFont *font,
        const sar_image_struct *unarmed_image,
	const sar_image_struct *armed_image,
        const sar_image_struct *highlighted_image,
        void *client_data, int id_code,
        void (*func_cb)(void *, void *, int)
);
extern int SARMenuCreateProgress(
        sar_menu_struct *m,
        double x, double y,
	int width, int height,
        const char *label,
        sar_menu_color_struct *fg_color,
        GWFont *font,
	const sar_image_struct *bg_image,
	const sar_image_struct *fg_image,
	double progress
);
extern int SARMenuCreateMessageBox(   
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *fg_color, GWFont *font,
	const sar_image_struct **bg_image,	/* Pointer to 9 images. */
	const char *message
);
extern int SARMenuCreateList(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *fg_color, GWFont *font,
	const char *label,
	const sar_image_struct **bg_image,	/* Pointer to 9 images. */
        void (*select_cb)(void *, void *, int, void *) 
);
extern int SARMenuListAppendItem(
	sar_menu_struct *m,
	int n,			/* List object number on m. */
	const char *name,
	void *client_data,
	sar_menu_flags_t flags
);
extern sar_menu_list_item_struct *SARMenuListGetItemByNumber(
        sar_menu_list_struct *list_ptr, int n
);
extern void SARMenuListDeleteAllItems(
        sar_menu_struct *m,
        int n			/* List object number on m. */
);
extern int SARMenuCreateSwitch(
        sar_menu_struct *m,
        double x, double y,
        int width, int height,   
        sar_menu_color_struct *fg_color, GWFont *font,
	const char *label,
	const sar_image_struct *bg_image,
	const sar_image_struct *switch_off_image,
	const sar_image_struct *switch_on_image,
	Boolean state,
        void *client_data, int id_code,
        void (*switch_cb)(void *, void *, int, Boolean)
);
extern int SARMenuCreateSpin(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        sar_menu_color_struct *label_color,
	sar_menu_color_struct *value_color,
        GWFont *font, GWFont *value_font,
        const char *label,
        const sar_image_struct *label_image,
        const sar_image_struct *value_image,
        void *client_data, int id_code,
        void (*change_cb)(void *, void *, int, char *)
);
extern void SARMenuSpinSetValueType(
	sar_menu_struct *m,
	int n,			/* Spin object number on m. */
	int type
);
extern int SARMenuSpinAddValue(
	sar_menu_struct *m,
        int n,			/* Spin object number on m. */
        const char *value
);
extern char *SARMenuSpinGetCurrentValue(
	sar_menu_struct *m,
        int n,			/* Spin object number on m. */
	int *sel_num
);
extern void SARMenuSpinDeleteAllValues(
        sar_menu_struct *m,
        int n			/* Spin object number on m. */
);

extern void SARMenuDestroyObject(sar_menu_struct *m, int n);

extern void SARMenuSetProgressSimple(
        gw_display_struct *display,
        sar_menu_struct *m,
        double progress,
        int redraw
);
extern void SARMenuSetMessageBoxSimple(
        gw_display_struct *display,
        sar_menu_struct *m,
        const char *message,
        int redraw    
);
extern void SARMenuDrawObject(
        gw_display_struct *display,
        sar_menu_struct *m,
        int n
);
extern void SARMenuDrawAll(
	gw_display_struct *display,
	sar_menu_struct *m
);
extern int SARMenuManagePointer(
        gw_display_struct *display, sar_menu_struct *m,
        int x, int y, int state, int btn_num
);
extern int SARMenuManageKey(
        gw_display_struct *display,
        sar_menu_struct *m,   
        int key, int state
);


/* In menumap.c */
extern int SARMenuCreateMap(
        sar_menu_struct *m,
        double x, double y,
        double width, double height,
        GWFont *font,
        sar_menu_color_struct *color,
	const sar_image_struct **bg_image,	/* Total 9 images. */
	double scroll_x, double scroll_y,	/* In meters. */
	double m_to_pixels_coeff,	/* Meters to pixels coeff. */
	int show_markings_policy
);
extern void SARMenuMapDeleteAllMarkings(sar_menu_map_struct *map_ptr);
extern void SARMenuMapDraw(
        gw_display_struct *display,
        sar_menu_struct *m,
        sar_menu_map_struct *map_ptr,
	int map_num
);
extern int SARMenuMapManagePointer(
        gw_display_struct *display, sar_menu_struct *m,
        sar_menu_map_struct *map_ptr, int menu_obj_num,
        int x, int y, int state, int btn_num
);


#endif	/* MENU_H */
