/*
                               Game Controller
 */
#ifndef GCTL_H
#define GCTL_H

#include <sys/types.h>
#include <sys/stat.h>
#include "gw.h"


/*
 *	Game controller types:
 */
/* Use keyboard (and pointer if available). */
#define GCTL_CONTROLLER_KEYBOARD	0
/* Use joystick(s). */
#define GCTL_CONTROLLER_JOYSTICK	1


/*
 *	Flags data type:
 */
#define gctl_flags_t	unsigned long

/*
 *	Options for controlled type GCTL_CONTROLLER_JOYSTICK:
 *
 *	Note that GCTL_OPT_JS*_INIT is for specifying the number of
 *	joysticks to initialize, which in turn updates members data and
 *	total_joysticks on the gctl_struct in function GCtlInit().
 */
#define GCTL_OPT_JS0_INIT	(1 << 0)
#define GCTL_OPT_JS1_INIT	(1 << 1)

#define GCTL_OPT_JS0_PITCH	(1 << 16)
#define GCTL_OPT_JS0_BANK	(1 << 17)
#define GCTL_OPT_JS0_HEADING	(1 << 18)
#define GCTL_OPT_JS0_THROTTLE	(1 << 19)
#define GCTL_OPT_JS0_HAT	(1 << 20)
#define GCTL_OPT_JS0_AS_THROTTLE_AND_RUDDER	(1 << 21)

#define GCTL_OPT_JS1_PITCH	(1 << 24)
#define GCTL_OPT_JS1_BANK	(1 << 25)
#define GCTL_OPT_JS1_HEADING	(1 << 26)
#define GCTL_OPT_JS1_THROTTLE	(1 << 27)
#define GCTL_OPT_JS1_HAT	(1 << 28)
#define GCTL_OPT_JS1_AS_THROTTLE_AND_RUDDER	(1 << 29)


/*
 *	Game controller positions structure:
 */
typedef struct {

	int type;		/* One of GCTL_CONTROLLER_*. */

	gctl_flags_t options;	/* Any of GCTL_OPT_*. */

	/* Last time GCtlUpdate was called passing this structure
	 * (in milliseconds).
	 */
        time_t last_updated;

	/* Control positions. */
	double	heading,	/* -1.0 to 1.0. */
		pitch,		/* -1.0 to 1.0. */
		bank,		/* -1.0 to 1.0. */
		throttle,	/*  0.0 to 1.0. */
		hat_x,		/* -1.0 to 1.0. */
		hat_y;		/* -1.0 to 1.0. */

        /* Zoom in and out:
         *      *_state members specifies a boolean on/off value.
         *      *_kb_last members record the last keyboard event for this
         *      behavour.
         *      *_coeff members determines the magnitude [0.0 to 1.0] of     
         *      the state regardless if it is on or off. This is for when    
         *      the game controller has gone from off to on to off in one 
         *      call to GCtlUpdate().
         */
        int zoom_in_state, zoom_out_state;
        time_t zoom_in_kb_last, zoom_out_kb_last;
        double zoom_in_coeff, zoom_out_coeff;   /* 0.0 to 1.0. */

        /* Hoist up and down:
         *      *_state members specifies a boolean on/off value.
         *      *_kb_last members record the last keyboard event for this
         *      behavour.
         *      *_coeff members determines the magnitude [0.0 to 1.0] of     
         *      the state regardless if it is on or off. This is for when    
         *      the game controller has gone from off to on to off in one 
         *      call to GCtlUpdate().
         */
        int hoist_up_state, hoist_down_state;
        time_t hoist_up_kb_last, hoist_down_kb_last;
        double hoist_up_coeff, hoist_down_coeff;   /* 0.0 to 1.0. */

        /* Alt, ctrl, and shift (not nessasarly keyboard) states. */
        int alt_state, ctrl_state, shift_state;

        /* Wheel and air brakes:
	 *	*_state members specifies a boolean on/off value.
	 *	*_kb_last members record the last keyboard event for this
	 *	behavour.
	 *	*_coeff members determines the magnitude [0.0 to 1.0] of
	 *	the state regardless if it is on or off. This is for when
	 *	the game controller has gone from off to on to off in one
	 *	call to GCtlUpdate().
	 */
        int air_brakes_state, wheel_brakes_state;
        time_t air_brakes_kb_last, wheel_brakes_kb_last;
        double air_brakes_coeff, wheel_brakes_coeff;


	/* Keyboard specific stuff. */

	/* Keyboard states. */
        int     heading_kb_state,
                pitch_kb_state,
                bank_kb_state,
                throttle_kb_state, 
                hat_x_kb_state,
                hat_y_kb_state; 

	/* Keyboard last press time stamps for control positions. */
	time_t	heading_kb_last,
		pitch_kb_last,
		bank_kb_last,
		throttle_kb_last,
		hat_x_kb_last,
		hat_y_kb_last;


	/* Pointer specific stuff. */

	/* Current pressed pointer button(s), any flag set to true
	 * means that button is pressed.
	 */
#define GCTL_POINTER_BUTTON1		(1 << 0)
#define GCTL_POINTER_BUTTON2		(1 << 1)
#define GCTL_POINTER_BUTTON3		(1 << 2)
        gctl_flags_t pointer_buttons;

	/* Pointer box position and size in window coordinates, upper-left
	 * corner oriented.
	 */
	int	pointer_box_x, pointer_box_y,
		pointer_box_width, pointer_box_height;

	/* Last pointer event coordinates, in window coordinates relative
	 * to pointer_box_* and upper-left corner oriented.
	 */
	int pointer_x, pointer_y;


	/* Joystick specific stuff. */

	/* Joystick data referances, the member joystick points to an
	 * array of joystick data structures determined by the platform's
	 * joystick implementation type.
	 *
	 * For example, when using libjsw the void *joystick would be
	 * pointing to js_data_struct structures. The int total_joysticks
	 * determines how many structures in the void *joystick array.
         */
        void *joystick;
        int total_joysticks;

	/* Joystick axis mappings, indicates which joystick axis
	 * number corresponds with which action.
	 */
	int	js0_axis_heading,
		js0_axis_bank,
		js0_axis_pitch,
		js0_axis_throttle,
		js0_axis_hat_x,
		js0_axis_hat_y,

		js1_axis_heading,
		js1_axis_bank,
		js1_axis_pitch,
		js1_axis_throttle,
		js1_axis_hat_x,
		js1_axis_hat_y;

        /* Joystick button mappings, indicates which joystick button
	 * number corresponds with which action.
	 */
        int	js0_btn_rotate,		/* Treat bank as heading axis. */
		js0_btn_air_brakes,
		js0_btn_wheel_brakes,
		js0_btn_zoom_in,
		js0_btn_zoom_out,
		js0_btn_hoist_up,
		js0_btn_hoist_down;

	int	js1_btn_rotate,		/* Treat bank as heading axis. */
		js1_btn_air_brakes,
		js1_btn_wheel_brakes,
		js1_btn_zoom_in,
		js1_btn_zoom_out,
		js1_btn_hoist_up,
		js1_btn_hoist_down;

} gctl_struct;

extern int GCtlInit(gctl_struct *gc, int type, gctl_flags_t options);
extern void GCtlUpdate(gctl_struct *gc);
extern void GCtlHandlePointer(
        gw_display_struct *display, gctl_struct *gc,
        int type,                       /* One of GWEventType*. */
        int button,                     /* Button number. */
        int x, int y,
        long t                          /* Time stamp. */
);
extern void GCtlHandleKey(
        gw_display_struct *display,
        gctl_struct *gc,  
        int k, int kstate,
        int alt_state, int ctrl_state, int shift_state,
	long t
);
extern void GCtlShutdown(gctl_struct *gc);

#endif	/* GCTL_H */
